/******************************************************************************\
 gnofin/record.h   $Revision: 1.8 $
 Copyright (C) 1999 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#ifndef FIN_RECORD_H
#define FIN_RECORD_H

#include <glib.h>

typedef enum _FinRecordField	 FinRecordField;
typedef enum _FinRecordType	 FinRecordType;
typedef union _FinRecordTypeData FinRecordTypeData;
typedef struct _FinRecordInfo	 FinRecordInfo;
typedef struct _FinTransferLink  FinTransferLink;
typedef struct _FinRecord	 FinRecord;

#include "account.h"
#include "money.h"

#define FIN_RECORD_FIELD_MAXLEN_DATE		32	/* allow many formats */
#define FIN_RECORD_FIELD_MAXLEN_TYPE		4	/* TYP */
#define FIN_RECORD_FIELD_MAXLEN_TYPE_DATA	50	/* check_no|acct_name */
#define FIN_RECORD_FIELD_MAXLEN_TYPE_FULL	60	/* TYP : no|acct_name */
#define FIN_RECORD_FIELD_MAXLEN_INFO		256
#define FIN_RECORD_FIELD_MAXLEN_STATUS		2	/* single char */
#define FIN_RECORD_FIELD_MAXLEN_AMOUNT		32
#define FIN_RECORD_FIELD_MAXLEN_BALANCE		32

#define FIN_RECORD_CHECK_NO(record)	((record)->type_d.check_no)
#define FIN_RECORD_XFR_LINK(record)	((record)->type_d.transfer_link)
#define FIN_RECORD_IS_CLEARED(record)	((record)->cleared ? 1 : 0)

#define FIN_RECORD_IS_ATM(record)	((record)->type == FIN_RECORD_TYPE_ATM)
#define FIN_RECORD_IS_CC(record)	((record)->type == FIN_RECORD_TYPE_CC)
#define FIN_RECORD_IS_CHK(record)	((record)->type == FIN_RECORD_TYPE_CHK)
#define FIN_RECORD_IS_DEP(record)	((record)->type == FIN_RECORD_TYPE_DEP)
#define FIN_RECORD_IS_EFT(record)	((record)->type == FIN_RECORD_TYPE_EFT)
#define FIN_RECORD_IS_XFR(record)	((record)->type == FIN_RECORD_TYPE_XFR)

enum _FinRecordField
{
  FIN_RECORD_FIELD_DATE,
  FIN_RECORD_FIELD_TYPE,
  FIN_RECORD_FIELD_TYPE_DATA,
  FIN_RECORD_FIELD_TYPE_FULL,	/* TYPE + TYPE_DATA */
  FIN_RECORD_FIELD_INFO,
  FIN_RECORD_FIELD_STATUS,
  FIN_RECORD_FIELD_AMOUNT,
  FIN_RECORD_FIELD_BALANCE,
};

enum _FinRecordType
{
  FIN_RECORD_TYPE_ATM,	/* withdrawal via ATM */
  FIN_RECORD_TYPE_CC, 	/* withdrawal via cash card */
  FIN_RECORD_TYPE_CHK,	/* withdrawal via check */
  FIN_RECORD_TYPE_DEP,	/* deposit via check/cash */
  FIN_RECORD_TYPE_EFT,	/* deposit via electronic funds transfer */
  FIN_RECORD_TYPE_XFR,	/* transfer between accounts */
  FIN_RECORD_TYPE_NUM	/* number of unique record types */
};

struct _FinRecordInfo
{
  int 	  ref_count;	/* keep track of how many records use this */
  gchar * string;
};

struct _FinTransferLink
{
  FinAccount * account;	/* other account from/to which we transfered money */
  FinRecord  * record;  /* corresponding record in other account */
};

union _FinRecordTypeData
{
  gint    	    raw[2];	   /* as read from the data stream */
  gint		    check_no;
  FinTransferLink * transfer_link;
};

/*
  record->type_d.transfer_link->account
  record->type_d.check_no
*/

struct _FinRecord
{
  gint		    ref_count;	/* reference counter */

  GDate		    date;	/* date of transaction */
  FinRecordType	    type;	/* physical type of transaction */
  FinRecordTypeData type_d;	/* extra data associated with record type */

  GList		  * info;	/* FinRecordInfo list node */
  money_t	    amount;	/* amount of money represented by this record */
  money_t	    balance;	/* resulting balance up to this record */

  gint		    cleared          : 1;  /* this is the STATUS field */
  gint		    have_linked_XFR  : 1;  /* for XFR, is type_d valid? */
  gint		    have_info_string : 1;
  gint		    have_amount      : 1;  /* should amount be displayed? */
};

FinRecordInfo * fin_record_info_new      (const gchar * string);
void		fin_record_info_ref	 (FinRecordInfo * info);
void		fin_record_info_unref	 (FinRecordInfo * info);
gint		fin_record_info_sort_fcn (const FinRecordInfo * a,
					  const FinRecordInfo * b);

FinRecord * fin_record_new   (GList * info_cache);
void	    fin_record_ref   (FinRecord * record);
void	    fin_record_unref (FinRecord * record);

/* set/get info string */
void	      fin_record_set_info_string (FinRecord       * record,
					  const gchar     * string);
const gchar * fin_record_get_info_string (const FinRecord * record);

/* functions to parse & create persistent representation
 * the balance field is not read from/written to the stream
 * when parsing, balance is set equal to amount
 * the parse function should be used only on new records */
gboolean fin_record_parse_stream (FinRecord * record,
				  FILE      * stream);
gboolean fin_record_write_stream (FinRecord * record,
				  FILE      * stream);

/* return list of names for source types */
GList       * fin_record_stringize_types (gint multi_accounts);
FinRecordType fin_record_parse_type 	 (const gchar * string);
const gchar * fin_record_stringize_type	 (FinRecordType type);

/* return list of info strings
 * searches back to head of list, so works for record->info too */
GList * fin_record_info_stringize (GList * info_list);

/* return text version of record */
gchar ** fin_stringized_record_new  (const FinRecord      * record,
				     const FinRecordField * fields,
				     gint	            num_fields);
void     fin_stringized_record_free (gchar ** text);

/* set fields of record to match supplied text
 * only affects specified fields
 *
 * CANNOT parse TYPE_FULL or BALANCE
 *
 * returns  
 *   0 on success
 *  -1 if misused
 *  >0 indicating the (1-based) index of the field that could not be parsed
 *
 * NOTE:
 * if the record is of type XFR, then the record MUST be postprocessed
 * to complete the link... see fin_account_set_transfer_funds
 */
int fin_record_parse_fields (FinRecord      * record,
			     FinRecordField * fields,
			     gint	      num_fields,
			     gchar         ** text);
					
void fin_record_dump	  (FinRecord     * record,
		           FILE		 * stream);
void fin_record_info_dump (FinRecordInfo * info,
			   FILE		 * stream);

/* this function can be used to build sorted record lists 
 * sorts by date, when dates match positive amounts are listed first, 
 * sorted by info string
 */
gint fin_record_sort_fcn (const FinRecord * a,
			  const FinRecord * b);

FinRecord * fin_record_dup  (FinRecord * record);
void        fin_record_swap (FinRecord * a, 
			     FinRecord * b);

/* used by the transfer code */
void 	    fin_record_sync_link        (FinRecord * dest, const FinRecord * src);

#endif
