/******************************************************************************\
 gnofin/file.c   $Revision: 1.13 $
 Copyright (C) 1999 Darin Fisher
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#include "gnofin.h"

#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <math.h>
#include <glib.h>

#include "file.h"
#include "record.h"


/* various error strings */

#define ERROR_CORRUPT	"Invalid or corrupted data file"
#define ERROR_FORMAT	"Invalid file format"
#define ERROR_VERSION	"Unsupported file version"

static gchar error_message[256];


/* FIN ascii data file format:
 * 
 * header_line (optional, starts with '#')
 * tag
 * version
 * width height
 * colsp1 colsp2 ... colsp6
 * num_infos
 * info[1]
 * ...
 * info[num_infos]
 * num_accounts
 * ...
 * account[i].name
 * account[i].info
 * account[i].num_records
 * account[i].record[i]
 * ...
 * account[i].record[num_records]
 * ...
 */

/* version notes:
 *
 * 0.3 -- amount field stored as a float
 * 0.4 -- amount field stored as a signed integer
 */

#define FIN_DATAFILE_TAG	"FIN!"
#define FIN_MAX_LINE		1024

#define set_fail(reason) \
  G_STMT_START { \
    fin_trace("Error: %s", (reason)); \
    snprintf(error_message, sizeof(error_message), (reason)); \
    result = error_message; \
  } G_STMT_END

#define read_line(file, buf) \
  G_STMT_START { \
    if (fgets((buf), FIN_MAX_LINE, (file)) == NULL) \
      { set_fail(ERROR_CORRUPT); goto parse_error; } \
    (buf)[strlen(buf)-1]='\0'; \
  } G_STMT_END

#define read_int(file, buf, i) \
  G_STMT_START { \
    read_line((file), buf); \
    fin_trace("<read_int> read \"%s\"",buf); \
    if (sscanf(buf, "%d", (i)) != 1) \
      { set_fail(ERROR_CORRUPT); goto parse_error; } \
  } G_STMT_END


static gint
read_record (FILE      * file,
	     FinRecord * record,
	     int         major,
	     int         minor)
{
  GList * info;
  int day, month, year;
  int type, type_d1, type_d2;
  int info_index;
  int cleared;
  money_t amount;

  fin_trace("");

  g_return_val_if_fail(file, FALSE);
  g_return_val_if_fail(record, FALSE);
  g_return_val_if_fail(major == 0, FALSE);
  g_return_val_if_fail(minor == 3 || minor == 4, FALSE);

  /* can only be used with new records */
  g_return_val_if_fail(record->have_info_string == 0, FALSE);

  /* in fact, info must point to the info_cache (top of the list) 
   * the stream will reference (by index) an entry in the info cache */
  g_return_val_if_fail(record->info != NULL, FALSE);
  g_return_val_if_fail(record->info->prev == NULL, FALSE);

  fin_trace("parsing record ...");

  /* parse record */
  if (minor == 3)
  {
    float amount_f;
    int n =
    /* 		  DD MM YY TN T1 T2 II ST AM */ 
    fscanf(file, "%d %d %d %d %d %d %d %d %f",
      &day,
      &month,
      &year,
      &type,
      &type_d1,
      &type_d2,
      &info_index,
      &cleared,
      &amount_f);
    if (n != 9)
    {
      fin_trace("read only %i of 9 columns !!", n);
      goto parse_error;
    }
    amount = (money_t) rint((double) (amount_f * 100.0f));
  }
  else
  {
    int n =
    /* 		  DD MM YY TN T1 T2 II ST AM */ 
    fscanf(file, "%d %d %d %d %d %d %d %d %ld",
      &day,
      &month,
      &year,
      &type,
      &type_d1,
      &type_d2,
      &info_index,
      &cleared,
      &amount);
    if (n != 9)
    {
      fin_trace("read only %i of 9 columns !!", n);
      goto parse_error;
    }
  }

  /* lookup corresponding info in info_cache */
  {
    info = g_list_nth(record->info, info_index);
    if (info == NULL)
    {
      fin_trace("reference to non-existent info string !!");
      goto parse_error;
    }
    fin_record_info_ref(LIST_GET(FinRecordInfo, info));
  }

  /* assign record fields */
  {
    g_date_clear(&record->date, 1);
    fin_trace ("month: %d  day: %d  year: %d", month, day, year);
    g_date_set_dmy(&record->date, day, month, year);
    fin_trace ("Returned from calling g_date_set_dmy");

    if (!g_date_valid(&record->date))
      goto parse_error;
    fin_trace ("Valid date");

    record->type = type;
    record->type_d.raw[0] = type_d1;
    record->type_d.raw[1] = type_d2;
    record->info = info;
    record->have_info_string = 1;
    record->cleared = cleared;
    record->amount = amount;
    record->balance = amount;
    record->have_amount = 1;
  }
  return TRUE;

parse_error:
  return FALSE;
}

static gint
write_record (FILE      * file,
	      FinRecord * record,
	      GList     * info_strings)
{
  int info_index = 0;

  fin_trace("");

  g_return_val_if_fail(file, FALSE);
  g_return_val_if_fail(record, FALSE);

  /* require a valid info entry, must have an info string
   * or else there wouldn't be any info worth saving */
  g_return_val_if_fail(record->have_info_string, FALSE);
  g_return_val_if_fail(record->info, FALSE);

  /* locate record info in info cache 
   * simply count the number of links back to the top of the list */
  {
    FinRecordInfo * info = LIST_GET(FinRecordInfo, record->info);
    info_index = g_list_index(info_strings, info->string);
  }

  /* print record */
  {
    /* 		   DD MM YY TN T1 T2 II ST AM */ 
    fprintf(file, "%d %d %d %d %d %d %d %d %ld\n",
      g_date_day(&record->date),
      g_date_month(&record->date),
      g_date_year(&record->date),
      record->type,
      record->type_d.raw[0],
      record->type_d.raw[1],
      info_index,
      record->cleared ? 1 : 0,
      record->amount);
  }

  return TRUE;
}

static const gchar *
read_header (FILE            * file,
	     gchar	     * linebuf,
	     FinAccountSet   * set,
	     int	     * major,
	     int	     * minor)
{
  gchar * result = NULL;

  fin_trace("");

  /* read first line */
  read_line(file, linebuf);

  /* allow a shell style comment in the first line of the file */
  if(linebuf[0] == '#')
  {
    set->header_line = g_strdup(linebuf); /* store the header line */
    read_line(file, linebuf);
  }

  if (0 != strcmp(linebuf, FIN_DATAFILE_TAG))
  {
    set_fail(ERROR_FORMAT);
    goto parse_error;
  }

  read_line(file, linebuf);
  if (sscanf(linebuf, "%d.%d", major, minor) != 2)
  {
    set_fail(ERROR_FORMAT);
    goto parse_error;
  }
  if (*major != 0 || *minor < 3 || *minor > 4)
  {
    set_fail(ERROR_VERSION);
    goto parse_error;
  }

parse_error:
  return result;
}
		      
static const gchar *
read_info_cache (FILE          * file,
		 gchar	       * linebuf,
		 FinAccountSet * set)
{
  gchar * result = NULL;
  int num_strings, i;

  fin_trace("");

  read_int(file, linebuf, &num_strings);
  fin_trace("num_strings = %d", num_strings);

  for (i=0; i<num_strings; ++i)
  {
    read_line(file, linebuf);
    fin_trace("info_string = \"%s\"", linebuf);

    fin_account_set_store_info_string(set, linebuf, 0);
  }
parse_error:
  return result;
}

static const gchar *
read_records (FILE          * file,
	      gchar         * linebuf,
	      FinAccountSet * set,
	      FinAccount    * account,
	      int	      major,
	      int	      minor)
{
  const gchar * result = NULL;
  FinRecord * record = NULL;
  FinRecord * last_record = NULL;
  int num_records, i;

  fin_trace("");

  read_int(file, linebuf, &num_records);
  fin_trace("num_records = %d", num_records);

  for (i=0; i<num_records; ++i)
  {
    record = fin_record_new(set->info_cache);

    if (!read_record(file, record, major, minor))
    {
      set_fail(ERROR_CORRUPT);
      goto parse_error;
    }

    if (record->type == FIN_RECORD_TYPE_CHK)
    {
      /* this may seem strange, but it is necessary since we don't
	 know if check_no corresponds to raw[0] or raw[1] */
      int check_no = record->type_d.check_no;
      record->type_d.raw[0] = 0;
      record->type_d.raw[1] = 0;
      record->type_d.check_no = check_no;
    }

    if (last_record)
      record->balance += last_record->balance;

    fin_account_store_record(account, record, 0);

    last_record = record;
    record = NULL;
  }

  /* this is only necessary if we have non-empty accounts...
     fscanf does not read the end-of-line character */
  if (num_records > 0)
    fgets(linebuf, FIN_MAX_LINE, file);

parse_error:
  if (record)
    fin_record_unref(record);
  return result;
}

static const gchar *
read_accounts (FILE          * file,
	       gchar	     * linebuf,
	       FinAccountSet * set,
	       int	       major,
	       int	       minor)
{
  const gchar * result = NULL;
  FinAccount * account = NULL;
  gchar linebuf2[FIN_MAX_LINE];
  int num_accounts, i;

  fin_trace("");
  
  read_int(file, linebuf, &num_accounts);
  fin_trace("num_accounts = %d", num_accounts);

  for (i=0; i<num_accounts; ++i)
  {
    fin_trace("loading account [%d]", i);

    read_line(file, linebuf);  /* read account name */
    read_line(file, linebuf2); /* read account info */

    /* allocate account */
    account = fin_account_new(linebuf, linebuf2);

    /* read records */
    if ((result = read_records(file, linebuf, set, account, major, minor)) != NULL)
      goto parse_error;

    /* save account */
    fin_account_set_store_account(set, account, 0);

    account = NULL;
  }

parse_error:
  if (account)
    fin_account_unref(account);
  return result;
}

static const gchar *
link_transfers (FinAccountSet * set)
{
  gchar * result = NULL;
  GList * ac;

  fin_trace("");

  for (ac=set->accounts; ac; ac=ac->next)
  {
    FinAccount * account = LIST_GET(FinAccount, ac);
    GList * it;

    for (it=account->records; it; it=it->next)
    {
      FinRecord * record = LIST_GET(FinRecord, it);

      if (record->type == FIN_RECORD_TYPE_XFR && !record->have_linked_XFR)
      {
	FinTransferLink * link = g_new0(FinTransferLink, 1);
	GList * node;

	node = g_list_nth(set->accounts, record->type_d.raw[0]);
	if (node == NULL || node->data == NULL)
	{
	  fin_trace("XFR record linked to non-existent account !!");
	  set_fail(ERROR_CORRUPT);
	  goto parse_error;
	}
	link->account = LIST_GET(FinAccount, node); 

	node = g_list_nth(link->account->records, record->type_d.raw[1]);
	if (node == NULL || node->data == NULL)
	{
	  fin_trace("XFR record linked to non-existent record !!");
	  set_fail(ERROR_CORRUPT);
	  goto parse_error;
	}
	link->record = LIST_GET(FinRecord, node);

	record->have_linked_XFR = 1;
	record->type_d.transfer_link = link;

	/* link up the other record too.. because we can */
	link->record->type_d.transfer_link = g_new0(FinTransferLink, 1);
	link->record->type_d.transfer_link->account = account;
	link->record->type_d.transfer_link->record = record;
	link->record->have_linked_XFR = 1;

	/* increment reference count for both */
	record->ref_count++;
	link->record->ref_count++;
      }
    }
  }

parse_error:
  return result;
}

const gchar * 
fin_file_read (const gchar    * filename,
	       FinAccountSet ** accounts)
{
  const gchar * result = NULL;
  FinAccountSet * set = NULL;
  gchar linebuf[FIN_MAX_LINE];
  int major, minor;
  FILE * file;

  fin_trace("loading file: %s", filename);

  g_return_val_if_fail(filename != NULL, NULL);

  /* open file */
  file = fopen(filename, "r");
  if (file == NULL)
  {
    set_fail(strerror(errno));
    return result;
  }

  /* allocate account set */
  set = fin_account_set_new();

  /* read header */
  if ((result = read_header(file, linebuf, set, &major, &minor)) != NULL)
    goto parse_error;

  /* read info cache */
  if ((result = read_info_cache(file, linebuf, set)) != NULL)
    goto parse_error;

  /* read accounts */
  if ((result = read_accounts(file, linebuf, set, major, minor)) != NULL)
    goto parse_error;

  /* link up any transfers */
  if ((result = link_transfers(set)) != NULL)
    goto parse_error;

  fin_trace("load succeeded !!");

  *accounts = set;
  set = NULL;

parse_error:
  fclose(file);
  if (set)
    fin_account_set_unref(set);
  return result;
}

static GList *
get_used_info_strings(FinAccountSet * set)
{
  GList * info_strings = NULL;
  GList * ac;
  GList * it;

  /* list out all info_strings that are currently in use */

  for (ac=set->accounts; ac; ac=ac->next)
  {
    for (it=LIST_GET(FinAccount, ac)->records; it; it=it->next)
    {
      FinRecordInfo * info = LIST_GET(FinRecordInfo, LIST_GET(FinRecord, it)->info);

      if (g_list_find(info_strings, info->string) == NULL)
        info_strings = g_list_prepend(info_strings, info->string);
    }
  }
 
  if (info_strings)
    info_strings = g_list_sort(info_strings, (GCompareFunc) strcmp);

  return info_strings;
}

const gchar * 
fin_file_write (const gchar   * filename,
		FinAccountSet * set)
{
  const gchar * result = NULL;
  GList * it, * ac;
  GList * info_strings = NULL;
  FILE * file;

  fin_trace("saving file: %s", filename);

  g_return_val_if_fail(filename, FALSE);
  g_return_val_if_fail(set, FALSE);

  /* create file */
  file = fopen(filename, "w");
  if (file == NULL)
  {
    set_fail(strerror(errno));
    return result;
  }

  /* write header */
  if (set->header_line)
    fprintf(file, "%s\n", set->header_line);
  fprintf(file, "%s\n%s\n", FIN_DATAFILE_TAG, "0.4");

  /* write info cache */
  info_strings = get_used_info_strings(set);
  fprintf(file, "%d\n", g_list_length(info_strings));
  for (it=info_strings; it; it=it->next)
    fprintf(file, "%s\n", LIST_GET(char, it));
  
  /* write accounts */
  fprintf(file, "%d\n", g_list_length(set->accounts));
  for (ac=set->accounts; ac; ac=ac->next)
  {
    FinAccount * account = LIST_GET(FinAccount, ac);
    int i = 0;

    fprintf(file, "%s\n%s\n%d\n",
      account->name,
      account->info,
      g_list_length(account->records));

    for (it=account->records; it; it=it->next, ++i)
    {
      FinTransferLink * link = NULL;
      FinRecord * record = LIST_GET(FinRecord,it);

      fin_trace("writing record [%i]", i);

      switch (record->type)
      {
      case FIN_RECORD_TYPE_XFR:
	link = record->type_d.transfer_link; 
	record->type_d.raw[0] = g_list_index(set->accounts, link->account);
	record->type_d.raw[1] = g_list_index(link->account->records, link->record);
	break;
      case FIN_RECORD_TYPE_CHK:
	record->type_d.raw[0] = record->type_d.check_no;
	record->type_d.raw[1] = 0;
	break;
      default:
	record->type_d.raw[0] = 0;
	record->type_d.raw[1] = 0;
	break;
      }

      write_record(file, record, info_strings);

      if (record->type == FIN_RECORD_TYPE_XFR)
	record->type_d.transfer_link = link; /* restore transfer link */
    }
  }
  fclose(file);

  if (info_strings)
    g_list_free(info_strings);

  return result; 
}

gint
fin_file_exists (const gchar * path)
{
  struct stat buf;

  fin_trace("");

  g_return_val_if_fail(path, FALSE);

  return stat(path, &buf) == 0;
}

gint
fin_file_backup (const gchar * path)
{
  gchar * buf;
  gint result;

  fin_trace("");

  g_return_val_if_fail(path, -1);

  buf = g_strdup_printf("cp -f %s %s~", path, path);
  result = system(buf);
  g_free(buf);

  return result;
}
