/*
 *  apclock.c -- Lock file managing functions
 *
 *  Copyright (C) 1998-99 Brian Schau <bsc@fleggaard.dk>
 *                1998-99 Facchetti Riccardo <fizban@tin.it>
 *
 *  apcupsd.c -- Simple Daemon to catch power failure signals from a
 *               BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *            -- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick
 *                        <hedrick@astro.dyer.vanderbilt.edu>
 *  All rights reserved.
 *
 */

/*
 *                     GNU GENERAL PUBLIC LICENSE
 *                        Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *                           675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

/*
 *  Contributed by Facchetti Riccardo <fizban@tin.it>
 *
 *  Initial lock functions by BSC (see apcupsd.c)
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <signal.h>
#include <fcntl.h>

#include <apc_config.h>
#include <apc_i18n.h>
#include <apc_version.h>
#include <apc_defines.h>
#include <apc_struct.h>
#include <apc_extern.h>

int check_stale_lockfile (UPSINFO *ups)
{
	char pidbuffer[12];
	int size;
	int stalepid;
	int error;

	/*
	 * If the daemon is talking with APC_NET, the lock file is not
	 * needed.
	 */
	if (ups->cable.type == APC_NET)
		return LCKNOLOCK;

	if (ups->lockpath[0] == '\0') {
		/*
		 * If there's no lockfile configured, return error.
		 * This is a _must_. See my comment in apcconfig.c
		 */
		return LCKERROR;
	}

	if ((ups->lockfile = open(ups->lockpath, O_RDONLY)) < 0) {
		/*
		 * Cannot open the file (may be it don't exist and that's okay
		 * for us so return success).
		 */
		if (errno == ENOENT)
			return LCKSUCCESS;

		/*
		 * On any other error, return error.
		 */
		return LCKERROR;
	}

	if ((size = read(ups->lockfile, &pidbuffer, 11)) == -1) {
		/*
		 * If we can not read from file, close it and return error:
		 * the file exist but we can not check for stale.
		 */
		error = LCKERROR;
		goto out;
	}

	if (size == 0 || (sscanf(pidbuffer, "%d", &stalepid) != 1)) {
		/*
		 * If there's no data in the file or the data written is wrong
		 * we have a process that:
		 * 1 - running but failed to write the lock file
		 * 2 - not running and failed to write the lock file
		 *
		 * Anyway we assume the worst case (1) and return error.
		 */
		error = LCKERROR;
		goto out;
	}

	if (stalepid == getpid()) {
		/*
		 * We are us (may be a crash of the machine ... same pid
		 * because same boot sequence ... leave it alone and go run)
		 */
		error = LCKEXIST;
		goto out;
	}

	/*
	 * Okay, now we have a stalepid to check.
	 */
	if (kill(stalepid, 0) == -1 && errno == ESRCH) {
		/*
		 * Okay this is a stale lock:
		 * we can unlink even before closing it.
		 */
		unlink(ups->lockpath);
		error = LCKSUCCESS;
		goto out;
	}

	/*
	 * We have unfortunately found a perfectly valid lock file.
	 * Don't touch it.
	 */
	error = LCKERROR;

out:
	close(ups->lockfile);
	ups->lockfile = -1;
	return error;
}

int create_lockfile (UPSINFO *ups, int killpwr)
{
	char pidbuffer[12];
	int error;

	error = check_stale_lockfile(ups);

	switch (error) {
		case LCKNOLOCK:
			/*
			 * Lockfile not needed: return success.
			 */
		case LCKEXIST:
			/*
			 * Lockfile exist and contain our pid.
			 */
			return LCKSUCCESS;

		case LCKERROR:
			/*
			 * Lockfile exist and is not stale.
			 */
			return LCKERROR;

		case LCKSUCCESS:
			/*
			 * Lockfile does not exist _now_.
			 */
			break;
	}

	/*
	 * Now the file does not exist any more.
	 * Open it for creation and don't accept any kind of error.
	 */
	if ((ups->lockfile = open(ups->lockpath,
	                          O_CREAT|O_EXCL|O_RDWR, 0644)) < 0) {
		/*
		 * Okay there is some problem with the lock path or
		 * something like that.
		 */
		if (killpwr)
			goto out;

		return LCKERROR;
	}

	if (sprintf(pidbuffer, "%010d", getpid()) <= 0) {
		/*
		 * Problems with sprintf
		 */
		error = LCKERROR;
		goto out;
	}

	if (write(ups->lockfile, pidbuffer, strlen(pidbuffer))
	    != strlen(pidbuffer)) {
		/*
		 * Problems with write.
		 */
		error = LCKERROR;
		goto out;
	}

	/*
	 * Done it.
	 */
	error = LCKSUCCESS;

out:
	close(ups->lockfile);
	ups->lockfile = -1;
	return error;
}

void delete_lockfile (UPSINFO *ups)
{
	int error;

	if (ups->lockpath[0] != '\0') {

		error = check_stale_lockfile(ups);

		switch (error) {
			case LCKEXIST:
				/*
				 * The lock file exist and is our own:
				 * delete it.
				 */
				break;
			default:
				/*
				 * In any other case we don't own that
				 * lock file so don't touch it.
				 */
				return;
		}

		if (ups->lockfile != -1) {
			fprintf(stderr,
			        "delete_lockfile:"
			        " deleting opened lock file"
			        " (after closing it)!\n");
			close(ups->lockfile);
			ups->lockfile = -1;
		}

		unlink(ups->lockpath);

		/*
		 * Now ups->lockfile is == -1 so there's no need to
		 * blank ups->lockfile too.
		 */
	}
}
