/*
 * Module for style(GtkStyle).
 *
 * If rc-file is found, this module does only parse it. That's all.
 * If it's not found, we use hard-coded values, especially due to some colors and fixed-width fonts.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <unistd.h>
#include <gnome.h>
#include "gdiff.h"
#include "gui.h"
#include "style.h"
#include "fileview.h"


/* rc file */
#define SYS_GTKDIFFRC	"gtkdiff/gtkdiffrc"	/* $datadir/SYS_GTKDIFFRC */
#define USER_GTKDIFFRC	".gtkdiffrc"

/* Private variables */
static gboolean b_found_rcfile;

/* Private function declarations */
static void create_default_styles(void);

/* Default styles */
static GtkStyle *file1_style;
static GtkStyle *file2_style;
static GtkStyle *darea_style;
static GtkStyle *text_style;


/**
 * style_init:
 * At first, search for rc file in the current directory and home directory.
 * If found, set b_found_rcfile=TRUE not to use default styles.
 * If not found, create default styles to use them later.
 **/
void
style_init(void)
{
	char *homedir;
	char *sys_rcpath;
	char path[PATH_MAX];

	b_found_rcfile = TRUE;
	/* Current directory */
	if (access(USER_GTKDIFFRC, R_OK) == 0) {
		gtk_rc_parse(USER_GTKDIFFRC);
		return;
	}

	/* Home directory */
	homedir = getenv("HOME");
	if (homedir) {
		sprintf(path, "%s/%s", homedir, USER_GTKDIFFRC);
		if (access(path, R_OK) == 0) {
			gtk_rc_parse(path);
			return;
		}
	}

	/* System directory */
	sys_rcpath = gnome_datadir_file(SYS_GTKDIFFRC);
	if (sys_rcpath) {
		gtk_rc_parse(sys_rcpath);
		g_free(sys_rcpath);
		return;
	}
	
	/* rc file is not found, so use hard-coded default styles. */
	b_found_rcfile = FALSE;
	create_default_styles();
}

/**
 * set_style_fileviews:
 * Use hard-coded default styles for file views.
 **/
void
set_style_fileviews(GDiffFileViews *gfviews, PaneMode pmode)
{
	if (b_found_rcfile == TRUE)
		return;/* do nothing */

	if (pmode == ONE_PANE) {
		gtk_widget_set_style(GTK_WIDGET(gfviews->pane.one.overview[FIRST_FILE]), file1_style);
		gtk_widget_set_style(GTK_WIDGET(gfviews->pane.one.overview[SECOND_FILE]), file2_style);
		gtk_widget_set_style(GTK_WIDGET(gfviews->pane.one.darea), darea_style);
		gtk_widget_set_style(GTK_WIDGET(gfviews->pane.one.text), text_style);
	} else { /* TWO_PANE */
		gtk_widget_set_style(GTK_WIDGET(gfviews->pane.two.overview[FIRST_FILE]), file1_style);
		gtk_widget_set_style(GTK_WIDGET(gfviews->pane.two.overview[SECOND_FILE]), file2_style);
		gtk_widget_set_style(GTK_WIDGET(gfviews->pane.two.darea), darea_style);
		gtk_widget_set_style(GTK_WIDGET(gfviews->pane.two.text[FIRST_FILE]), text_style);
		gtk_widget_set_style(GTK_WIDGET(gfviews->pane.two.text[SECOND_FILE]), text_style);
	}
}



/* ---The followings are private functions--- */
/**
 * create_default_styles:
 * Create default styles with hard-coded values.
 * Called when gtkdiff can't find its rc file.
 **/
static void
create_default_styles(void)
{
	/* XXX performance hit? Don't care. This function is rarely executed. */
#define DEF_FIXED_WIDTH_FONT	"-adobe-courier-medium-r-normal--14-140-75-75-m-90-iso8859-1,-*-*-medium-r-normal--*-*-*-*-m-*-*-*,-*-*-medium-r-normal--*-*-*-*-c-*-*-*"
	GtkStyle *defstyle;
	GdkColormap *cmap;
	GdkColor color_file1_fg = {0, 0xffff, 0x7fff, 0x0000};/* slider's color on overview */
	GdkColor color_file2_fg = {0, 0x0000, 0xffff, 0xffff};/* slider's color on overview */
	GdkColor color_darea_fg = {0, 0x0000, 0x0000, 0xffff};
	GdkFont *old_font;
	
	/* allocate colors */
	cmap = gdk_colormap_get_system();
	gdk_colormap_alloc_color(cmap, &color_file1_fg, FALSE, TRUE);
	gdk_colormap_alloc_color(cmap, &color_file2_fg, FALSE, TRUE);
	gdk_colormap_alloc_color(cmap, &color_darea_fg, FALSE, TRUE);
	
	defstyle = gtk_widget_get_default_style();
	file1_style = gtk_style_copy(defstyle);
	file2_style = gtk_style_copy(defstyle);
	darea_style = gtk_style_copy(defstyle);
	text_style = gtk_style_copy(defstyle);
	
    file1_style->fg[GTK_STATE_NORMAL] = color_file1_fg;
    file2_style->fg[GTK_STATE_NORMAL] = color_file2_fg;
    darea_style->fg[GTK_STATE_NORMAL] = color_darea_fg;

	/* font for text widget (prefer fixed width font) */
	old_font = text_style->font;
    text_style->font = gdk_fontset_load(DEF_FIXED_WIDTH_FONT);
	if (text_style->font)
		gdk_font_unref(old_font);
	else
		text_style->font = old_font;
}

