/*
 * GUI actions module
 * See actions.h about the details.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdlib.h>
#include <unistd.h>
#if defined(HAVE_ERRNO_H)
#include <errno.h>
#else defined(HAVE_SYS_ERRNO_H)
#include <sys/errno.h>
#endif
#include <gnome.h>
#include "gdiff.h"
#include "misc.h"
#include "gui.h"
#include "actions.h"
#include "guiwin.h"
#include "dirview.h"
#include "fileview.h"
#include "guimisc.h"
#include "menu.h"
#include "hide.h"
#include "properties.h"


/* Private function declarations */
static gboolean is_file_modified(GDiffFileViews *gfviews);

/* Actions for directory view */
/**
 * act_dv_show_path:
 * Show pathes on directory view, otherwise show only file names.
 **/
void
act_dv_show_path(GDiffDirViews *gdviews, gboolean b_show_path)
{
	g_pref.dvpref.show_path = b_show_path;/* default preference */

	gdviews->pref.show_path = b_show_path;
	dirview_redisplay(gdviews);
}

/**
 * act_dv_add_rowhide:
 * Add the specified hide function to SList, and redisplay directory view.
 **/
void
act_dv_add_rowhide(GDiffDirViews *gdviews, RowHideFunc rh_func)
{
	GSList *list;

	/* check duplicate */
	for (list = gdviews->pref.row_hide_func_list; list; list = list->next) {
		RowHideFunc old_func = list->data;
		
		if (rh_func == old_func)
			return;
	}

	gdviews->pref.row_hide_func_list = g_slist_prepend(gdviews->pref.row_hide_func_list, rh_func);
	dirview_redisplay(gdviews);
}

/**
 * act_dv_remove_rowhide:
 * Remove the specified hide function from SList, and redisplay directory view.
 **/
void
act_dv_remove_rowhide(GDiffDirViews *gdviews, RowHideFunc rh_func)
{
	GSList *list;

	for (list = gdviews->pref.row_hide_func_list; list; list = list->next) {
		RowHideFunc old_func = list->data;
		
		if (rh_func == old_func) {
			gdviews->pref.row_hide_func_list = g_slist_remove(gdviews->pref.row_hide_func_list, rh_func);
			break;
		}
	}

	dirview_redisplay(gdviews);
}

/**
 * act_dv_rowhide_stat:
 * To hide(or show) the row depending on the files stat.
 **/
void
act_dv_rowhide_stat(GDiffDirViews *gdviews, FilesSpecialStatus hide_mask, gboolean mask_on)
{
	if (mask_on == TRUE) {
		gdviews->pref.row_hide_stat_mask |= hide_mask;
	} else {
		gdviews->pref.row_hide_stat_mask &= ~hide_mask;
	}
	dirview_redisplay(gdviews);
}

/**
 * act_dv_reload:
 **/
void
act_dv_reload(GDiffDirViews *gdviews)
{
	GDiffDirViews *new_dviews;
	GDiffWindow *gdwin = gdviews->gdwin;
	char fname1[PATH_MAX+1];
	char fname2[PATH_MAX+1];
	gboolean is_dir;
	Preference pref = g_pref;

	is_dir = gdviews->isdir;
	if (is_dir == TRUE) {
		strcpy(fname1, gdviews->dirname1);
		strcpy(fname2, gdviews->dirname2);
	} else { /* Comparing two files */
		GDiffFileViews *gfviews = g_slist_nth_data(gdviews->gdfile_list, 0);
		strcpy(fname1, gfviews->filename1);
		strcpy(fname2, gfviews->filename2);
	}

	gdiff_dirviews_delete(gdviews);

	new_dviews = gdiff_dirviews_new(gdwin, fname1, fname2, is_dir);
	dirview_display(new_dviews);

	/* update menu */
	pref.dvpref = new_dviews->pref;
	menu_update(new_dviews->gdwin, &pref, MENUSTAT_DIR_VIEW);
}


/* Actions for file view */
/**
 * act_fv_mode_change:
 * Change the mode(onepane or twopane) of the view.
 * If the mode of the view is different, look for the same mode view.
 * If it exists, focus it.
 * If it doesn't exist, opne a new view in the current mode.
 **/
void
act_fv_mode_change(GDiffFileViews *gfviews, PaneMode pmode)
{
	DiffFiles *dfiles = gfviews->dfiles;
	GDiffDirViews *gdviews = gfviews->gdirviews;

	if (is_file_modified(gfviews) == TRUE)
		return;

	g_pref.fvpref.pane_mode = pmode;/* default preference */
		
	if (gfviews->pref.pane_mode != pmode) {
		GSList *list;
		/* Look for opened views which is in the current mode. */
		for (list = gdviews->gdfile_list; list; list = list->next) {
			GDiffFileViews *gfviews2;
			
			gfviews2 = list->data;
			if (gfviews2->dfiles == dfiles
				&& gfviews2->pref.pane_mode == pmode) {
				gint page;
				
				g_assert(gfviews2 != gfviews);
				page = gtk_notebook_page_num(gfviews2->gdwin->notebook, gfviews2->base);
				if (!GTK_WIDGET_VISIBLE(gfviews2->base)) {
					gtk_widget_show(gfviews2->base);
				}
				gtk_notebook_set_page(gfviews2->gdwin->notebook, page);
				return;
			}
		}
		/* When not opened, make it open */	
		gfviews = gdiff_fileviews_new(gdviews->gdwin, gdviews, dfiles, pmode);
		if (pmode == ONE_PANE)
			onepane_display(gfviews);
		else
			twopane_display(gfviews);
	}
}

/**
 * act_fv_linenum_show:
 * Toggle show(hide) the line numbers for the view.
 **/
void
act_fv_linenum_show(GDiffFileViews *gfviews, gboolean to_show)
{
	Preference pref = g_pref;
	
	if (is_file_modified(gfviews) == TRUE)
		return;

	g_pref.fvpref.show_line_num = to_show;/* default preference */
	
	if (gfviews->pref.pane_mode == ONE_PANE)
		onepane_show_linenum(gfviews, to_show);
	else
		twopane_show_linenum(gfviews, to_show);

	/* update menu */
	pref.fvpref = gfviews->pref;
	menu_update(gfviews->gdwin, &pref, MENUSTAT_FILE_VIEW);
}


/**
 * act_fv_go_dirview:
 * Look for the appropriate directory view, and focus it.
 **/
void
act_fv_go_dirview(GDiffFileViews *gfviews)
{
	const GDiffWindow *gdwin = gfviews->gdwin;
	const GDiffDirViews *gdviews = gfviews->gdirviews;

	if (gdviews->isdir) {/* Comparing two directories */
		gint page;
		page = gtk_notebook_page_num(gdwin->notebook, gdviews->base);
		g_assert(page != -1);
#ifdef DEBUG		
		if (!GTK_WIDGET_VISIBLE(gdviews->base)) {
			g_print("Wrong in go_dirview_cb()\n");
		}
#endif		
		gtk_notebook_set_page(gdwin->notebook, page);
	}
}

/**
 * act_fv_toggle_textwrap:
 * Toggle the view's text widget line wrap mode.
 **/
void
act_fv_toggle_textwrap(GDiffFileViews *gfviews, gboolean to_wrap)
{
	Preference pref = g_pref;

	g_pref.fvpref.line_wrap = to_wrap;/* default preference */
	
	if (gfviews->pref.pane_mode == ONE_PANE) {
		gtk_text_set_line_wrap(gfviews->pane.one.text, to_wrap);
	} else {
		int n;
		for (n = 0; n < NUM_COMPARE_FILES; n++) 
			gtk_text_set_line_wrap(gfviews->pane.two.text[n], to_wrap);
	}
	gfviews->pref.line_wrap = to_wrap;
	
	/* update menu */
	pref.fvpref = gfviews->pref;
	menu_update(gfviews->gdwin, &pref, MENUSTAT_FILE_VIEW);
}

/**
 * act_fv_move_diff:
 **/
void
act_fv_move_diff(GDiffFileViews *gfviews, MoveDiff mv_diff)
{
	if (gfviews->pref.pane_mode == ONE_PANE)
		onepane_move_diff(gfviews, mv_diff);
	else
		twopane_move_diff(gfviews, mv_diff);
}

/**
 * act_fv_reload:
 **/
void
act_fv_reload(GDiffFileViews *gfviews)
{
	GDiffDirViews *gdviews = gfviews->gdirviews;
	GSList *list;
	DiffFiles *dfiles = gfviews->dfiles;
	const FileInfo *fi;
	char fname1[PATH_MAX+1];
	char fname2[PATH_MAX+1];
	GDiffFileViews *new_fviews;
	DiffFiles *new_dfiles;
	int i;
	int n_del_link = 0;
	GSList *del_link[2];/*XXX*/
	Preference pref = g_pref;
	
	/* Look for the file views among all opened views */
	for (list = gdviews->gdfile_list; list; list = list->next) {
		GDiffFileViews *fv;
		fv = list->data;
		if (dfiles == fv->dfiles) {
			gdiff_fileviews_delete(fv);
			del_link[n_del_link++] = list;
		}
	}
	for (i = 0; i < n_del_link; i++)
		gdviews->gdfile_list = g_slist_remove_link(gdviews->gdfile_list, del_link[i]);
	
	/* Update the back-end data */
	fi = dfiles_get_fileinfo(dfiles, FIRST_FILE, FALSE);
	strcpy(fname1, fi->fname);
	fi = dfiles_get_fileinfo(dfiles, SECOND_FILE, FALSE);
	strcpy(fname2, fi->fname);
	diff_dir_remove_files(gdviews->diffdir, dfiles);
	new_dfiles = diff_dir_add_files(gdviews->diffdir, fname1, fname2);
	run_diff(gdviews->diffdir, fname1, fname2, gdviews->pref.diff_args, new_dfiles);

	/* Update the front-end data */
	new_fviews = gdiff_fileviews_new(gdviews->gdwin, gdviews, new_dfiles, g_pref.fvpref.pane_mode);
	if (new_fviews->pref.pane_mode == ONE_PANE)
		onepane_display(new_fviews);
	else
		twopane_display(new_fviews);

	/* update menu */
	pref.fvpref = new_fviews->pref;
	menu_update(new_fviews->gdwin, &pref, MENUSTAT_FILE_VIEW);

	/* Set dirty-bit of directory view */
	gdviews->b_dirty = TRUE;
}

/**
 * act_fv_edit_file:
 **/
void
act_fv_edit_file(GDiffFileViews *gfviews, const char *filename)
{
	char *editor_path;
	char *editor;

	editor = g_getenv("EDITOR");
	if (editor == NULL) {
		g_message(_("No \"EDITOR\" environment variable\n"));
		return;
	}

	editor_path = gnome_is_program_in_path(editor);
	if (editor_path == NULL) {
		g_message(_("\"EDITOR\" environment variable %s is not executable.\n"), editor);
		return;
	} else {
		switch (fork()) {
		case 0:	/* the child */
			close_allfd_except_std();
			execl(editor_path, editor, filename, NULL);
			g_error("\"EDITOR\" environment variable \"%s\" exec error\n", editor);
			gtk_exit(1);
			break;
		case -1:	/* the error */
			g_error("edit_file_cb:fork %d\n", errno);
			break;
		default:/* the parent */
			break;
		}
		g_free(editor_path);
	}
}


/* ---The followings are private functions--- */
/**
 * is_file_modified:
 * Check the files have been modified.
 * If modified, ask a user to reload them, and if he answers yes, reload them.
 * If modified, return TRUE.
 **/
static gboolean
is_file_modified(GDiffFileViews *gfviews)
{
	FileModStatue fms;
	
	fms = dfiles_has_file_modified(gfviews->dfiles);
	if (fms == MOD_MODIFIED) {
		QDialogReply reply;
		
		reply = ask_yes_no(_("The files look modified. Will you reload the files?"));
		if (reply == QD_YES)
			act_fv_reload(gfviews);
		return TRUE;
	}
	return FALSE;
}
