/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-1999  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "xmms.h"
#include <dlfcn.h>

gchar *plugin_dir_list[] =
{
	PLUGINSUBS,
	NULL
};

extern struct InputPluginData *ip_data;
extern struct OutputPluginData *op_data;
extern struct EffectPluginData *ep_data;
extern struct GeneralPluginData *gp_data;
extern struct VisPluginData *vp_data;

void scan_plugins(char *dirname);
void add_plugin(gchar * filename);

static int d_iplist_compare(const void *a, const void *b)
{
	return strcmp(((char *) a), ((char *) b));
}

static int inputlist_compare_func(const void *a, const void *b)
{
	return strcasecmp(((InputPlugin *) a)->description, ((InputPlugin *) b)->description);
}

static int outputlist_compare_func(const void *a, const void *b)
{
	return strcasecmp(((OutputPlugin *) a)->description, ((OutputPlugin *) b)->description);
}

static int effectlist_compare_func(const void *a, const void *b)
{
	return strcasecmp(((EffectPlugin *) a)->description, ((EffectPlugin *) b)->description);
}

static int generallist_compare_func(const void *a, const void *b)
{
	return strcasecmp(((GeneralPlugin *) a)->description, ((GeneralPlugin *) b)->description);
}

static int vislist_compare_func(const void *a, const void *b)
{
	return strcasecmp(((VisPlugin *) a)->description, ((VisPlugin *) b)->description);
}

void init_plugins(void)
{
	gchar *dir, *temp, *temp2;
	GList *node, *disabled_iplugin_names = NULL;
	OutputPlugin *op;
	InputPlugin *ip;
	EffectPlugin *ep;
	gint dirsel = 0;

	if (cfg.disabled_iplugins)
	{
		temp = cfg.disabled_iplugins;
		while ((temp2 = strchr(temp, ',')) != NULL)
		{
			(*temp2) = '\0';
			disabled_iplugin_names = g_list_append(disabled_iplugin_names, g_strdup(temp));
			temp = temp2 + 1;
		}
		disabled_iplugin_names = g_list_append(disabled_iplugin_names, g_strdup(temp));
		g_free(cfg.disabled_iplugins);
		cfg.disabled_iplugins = NULL;
	}

	ip_data = g_malloc0(sizeof (struct InputPluginData));
	op_data = g_malloc0(sizeof (struct OutputPluginData));
	ep_data = g_malloc0(sizeof (struct EffectPluginData));
	gp_data = g_malloc0(sizeof (struct GeneralPluginData));
	vp_data = g_malloc0(sizeof (struct VisPluginData));

	while (plugin_dir_list[dirsel])
	{
		dir = g_strconcat(PLUGIN_DIR, "/", plugin_dir_list[dirsel++], NULL);
		scan_plugins(dir);
		g_free(dir);
	}

	op_data->output_list = g_list_sort(op_data->output_list, outputlist_compare_func);
	if (!op_data->current_output_plugin && g_list_length(op_data->output_list))
		op_data->current_output_plugin = (OutputPlugin *) op_data->output_list->data;
	ip_data->input_list = g_list_sort(ip_data->input_list, inputlist_compare_func);
	ep_data->effect_list = g_list_sort(ep_data->effect_list, effectlist_compare_func);
	if (!ep_data->current_effect_plugin && g_list_length(ep_data->effect_list))
		ep_data->current_effect_plugin = (EffectPlugin *) ep_data->effect_list->data;
	gp_data->general_list = g_list_sort(gp_data->general_list, generallist_compare_func);
	gp_data->enabled_list = NULL;
	vp_data->vis_list = g_list_sort(vp_data->vis_list, vislist_compare_func);
	vp_data->enabled_list = NULL;
	general_enable_from_stringified_list(cfg.enabled_gplugins);
	vis_enable_from_stringified_list(cfg.enabled_vplugins);
	if (cfg.enabled_gplugins)
	{
		g_free(cfg.enabled_gplugins);
		cfg.enabled_gplugins = NULL;
	}

	node = op_data->output_list;
	while (node)
	{
		op = (OutputPlugin *) node->data;
		if (!strcmp(cfg.outputplugin, op->filename))
			op_data->current_output_plugin = op;
		if (op->init)
			op->init();
		node = node->next;
	}

	node = ep_data->effect_list;
	while (node)
	{
		ep = (EffectPlugin *) node->data;
		if (!strcmp(cfg.effectplugin, ep->filename))
		{
			ep_data->current_effect_plugin = ep;
		}
		if (ep->init)
			ep->init();
		node = node->next;
	}

	node = ip_data->input_list;
	while (node)
	{
		ip = (InputPlugin *) node->data;
		temp = g_basename(ip->filename);
		if (g_list_find_custom(disabled_iplugin_names, temp, d_iplist_compare))
			disabled_iplugins = g_list_append(disabled_iplugins, ip);
		if (ip->init)
			ip->init();
		node = node->next;
	}

	node = disabled_iplugin_names;
	while (node)
	{
		g_free(node->data);
		node = node->next;
	}
	g_list_free(disabled_iplugin_names);

}

void add_plugin(gchar * filename)
{
	void *h;
	void *(*gpi) (void);

#ifdef RTLD_NOW
	if ((h = dlopen(filename, RTLD_NOW)) != NULL)
#else
	if ((h = dlopen(filename, 0)) != NULL)
#endif
	{
		if ((gpi = dlsym(h, "get_iplugin_info")) != NULL)
		{
			InputPlugin *p;

			p = (InputPlugin *) gpi();
			p->handle = h;
			p->filename = filename;
			p->add_vis = input_add_vis;
			p->get_vis_type = input_get_vis_type;
			p->add_vis_pcm = input_add_vis_pcm;
			p->set_info = playlist_set_info;
			p->set_info_text = input_set_info_text;

			ip_data->input_list = g_list_prepend(ip_data->input_list, p);
		}
		else if ((gpi = dlsym(h, "get_oplugin_info")) != NULL)
		{
			OutputPlugin *p;

			p = (OutputPlugin *) gpi();
			p->handle = h;
			p->filename = filename;
			op_data->output_list = g_list_prepend(op_data->output_list, p);
		}
		else if ((gpi = dlsym(h, "get_eplugin_info")) != NULL)
		{
			EffectPlugin *p;

			p = (EffectPlugin *) gpi();
			p->handle = h;
			p->filename = filename;
			ep_data->effect_list = g_list_prepend(ep_data->effect_list, p);
		}
		else if ((gpi = dlsym(h, "get_gplugin_info")) != NULL)
		{
			GeneralPlugin *p;

			p = (GeneralPlugin *) gpi();
			p->handle = h;
			p->filename = filename;
			p->xmms_session = ctrlsocket_get_session_id();
			gp_data->general_list = g_list_prepend(gp_data->general_list, p);
		}
		else if ((gpi = dlsym(h, "get_vplugin_info")) != NULL)
		{
			VisPlugin *p;

			p = (VisPlugin *) gpi();
			p->handle = h;
			p->filename = filename;
			p->xmms_session = ctrlsocket_get_session_id();
			p->disable_plugin = vis_disable_plugin;
			vp_data->vis_list = g_list_prepend(vp_data->vis_list, p);
		}
		else
		{
			dlclose(h);
		}
	}
	else
		fprintf(stderr, "%s\n", dlerror());
}

void scan_plugins(char *dirname)
{
	gchar *filename, *ext;
	DIR *dir;
	struct dirent *ent;
	struct stat statbuf;

	dir = opendir(dirname);
	if (dir)
	{
		while ((ent = readdir(dir)) != NULL)
		{
			filename = (gchar *) g_malloc(strlen(dirname) + strlen(ent->d_name) + 2);
			sprintf(filename, "%s/%s", dirname, ent->d_name);
			if (!stat(filename, &statbuf))
			{
				if (S_ISREG(statbuf.st_mode))
				{
					if ((ext = strrchr(ent->d_name, '.')) != NULL)
					{
						if (!strcmp(ext, ".so"))
							add_plugin(filename);
						else
							g_free(filename);
					}
					else
						g_free(filename);
				}
				else
					g_free(filename);
			}
			else
				g_free(filename);
		}
	}
}

void cleanup_plugins(void)
{
	InputPlugin *ip;
	OutputPlugin *op;
	EffectPlugin *ep;
	GeneralPlugin *gp;
	VisPlugin *vp;
	GList *node, *next;

	if (get_input_playing())
		input_stop();

	if (disabled_iplugins)
		g_list_free(disabled_iplugins);
	node = get_input_list();
	while (node)
	{
		ip = (InputPlugin *) node->data;
		dlclose(ip->handle);
		node = node->next;
	}
	if (ip_data->input_list)
		g_list_free(ip_data->input_list);
	g_free(ip_data);

	node = get_output_list();
	while (node)
	{
		op = (OutputPlugin *) node->data;
		dlclose(op->handle);
		node = node->next;
	}
	if (op_data->output_list)
		g_list_free(op_data->output_list);
	g_free(op_data);

	node = get_effect_list();
	while (node)
	{
		ep = (EffectPlugin *) node->data;
		if (ep && ep->cleanup)
		{
			ep->cleanup();
			GDK_THREADS_LEAVE();
			while(g_main_iteration(FALSE));
			GDK_THREADS_ENTER();

		}
		dlclose(ep->handle);
		node = node->next;
	}
	if (ep_data->effect_list)
		g_list_free(ep_data->effect_list);
	g_free(ep_data);

	node = get_general_enabled_list();
	while (node)
	{
		gp = (GeneralPlugin *) node->data;
		next = node->next;
		enable_general_plugin(g_list_index(gp_data->general_list, gp), FALSE);
		node = next;
	}
	if (gp_data->enabled_list)
		g_list_free(gp_data->enabled_list);

	GDK_THREADS_LEAVE();
	while(g_main_iteration(FALSE));
	GDK_THREADS_ENTER();
	
	node = get_general_list();
	while (node)
	{
		gp = (GeneralPlugin *) node->data;
		dlclose(gp->handle);
		node = node->next;
	}
	if (gp_data->general_list)
		g_list_free(gp_data->general_list);

	node = get_vis_enabled_list();
	while (node)
	{
		vp = (VisPlugin *) node->data;
		next = node->next;
		enable_vis_plugin(g_list_index(vp_data->vis_list, vp), FALSE);
		node = next;
	}
	if (vp_data->enabled_list)
		g_list_free(vp_data->enabled_list);
	
	GDK_THREADS_LEAVE();
	while(g_main_iteration(FALSE));
	GDK_THREADS_ENTER();
	
	node = get_vis_list();
	while (node)
	{
		vp = (VisPlugin *) node->data;
		dlclose(vp->handle);
		node = node->next;
	}
	if (vp_data->vis_list)
		g_list_free(vp_data->vis_list);
	g_free(vp_data);

}
