/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-1999  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "xmms.h"
#include "libxmms/configfile.h"

GtkWidget *equalizerwin;
static GtkWidget *equalizerwin_load_window = NULL, *equalizerwin_load_list;
static GtkWidget *equalizerwin_load_auto_window = NULL, *equalizerwin_load_auto_list;
static GtkWidget *equalizerwin_load_filesel, *equalizerwin_load_winamp_filesel;
static GtkWidget *equalizerwin_save_window = NULL, *equalizerwin_save_list,
         *equalizerwin_save_entry;
static GtkWidget *equalizerwin_save_auto_window = NULL, *equalizerwin_save_auto_list,
         *equalizerwin_save_auto_entry;
static GtkWidget *equalizerwin_save_filesel, *equalizerwin_save_winamp_filesel;
static GtkWidget *equalizerwin_delete_window = NULL, *equalizerwin_delete_list;
static GtkWidget *equalizerwin_delete_auto_window = NULL, *equalizerwin_delete_auto_list;

GdkPixmap *equalizerwin_bg, *equalizerwin_bg_dblsize;
GdkGC *equalizerwin_gc;

GList *equalizerwin_wlist = NULL;

GtkAccelGroup *equalizerwin_accel;

TButton *equalizerwin_on, *equalizerwin_auto;
extern TButton *mainwin_eq;
PButton *equalizerwin_presets, *equalizerwin_close, *equalizerwin_shade;
EqGraph *equalizerwin_graph;
EqSlider *equalizerwin_preamp, *equalizerwin_bands[10];
HSlider *equalizerwin_volume, *equalizerwin_balance;

GtkItemFactory *equalizerwin_presets_menu;

gboolean equalizerwin_focus = FALSE;

typedef struct
{
	gchar *name;
	gfloat preamp, bands[10];
}
EqualizerPreset;

static GList *equalizer_presets = NULL, *equalizer_auto_presets = NULL;

#define EQUALIZERWIN_PRESETS_MENU_ENTRIES 19

void equalizerwin_presets_menu_cb(gpointer cb_data, guint action, GtkWidget * w);

enum
{
	EQUALIZER_PRESETS_LOAD_PRESET, EQUALIZER_PRESETS_LOAD_AUTOPRESET, EQUALIZER_PRESETS_LOAD_DEFAULT,
	EQUALIZER_PRESETS_LOAD_ZERO,
	EQUALIZER_PRESETS_LOAD_FROM_FILE, EQUALIZER_PRESETS_LOAD_FROM_WINAMPFILE,
	EQUALIZER_PRESETS_SAVE_PRESET, EQUALIZER_PRESETS_SAVE_AUTOPRESET, EQUALIZER_PRESETS_SAVE_DEFAULT,
	EQUALIZER_PRESETS_SAVE_TO_FILE, EQUALIZER_PRESETS_SAVE_TO_WINAMPFILE,
	EQUALIZER_PRESETS_DELETE_PRESET, EQUALIZER_PRESETS_DELETE_AUTOPRESET
};

GtkItemFactoryEntry equalizerwin_presets_menu_entries[] =
{
	{"/Load", NULL, NULL, 0, "<Branch>"},
	{"/Load/Preset", NULL, equalizerwin_presets_menu_cb, EQUALIZER_PRESETS_LOAD_PRESET, "<Item>"},
	{"/Load/Auto-load preset", NULL, equalizerwin_presets_menu_cb, EQUALIZER_PRESETS_LOAD_AUTOPRESET, "<Item>"},
	{"/Load/Default", NULL, equalizerwin_presets_menu_cb, EQUALIZER_PRESETS_LOAD_DEFAULT, "<Item>"},
	{"/Load/-", NULL, NULL, 0, "<Separator>"},
	{"/Load/Zero", NULL, equalizerwin_presets_menu_cb, EQUALIZER_PRESETS_LOAD_ZERO, "<Item>"},
	{"/Load/-", NULL, NULL, 0, "<Separator>"},
	{"/Load/From file", NULL, equalizerwin_presets_menu_cb, EQUALIZER_PRESETS_LOAD_FROM_FILE, "<Item>"},
	{"/Load/From WinAMP EQF file", NULL, equalizerwin_presets_menu_cb, EQUALIZER_PRESETS_LOAD_FROM_WINAMPFILE, "<Item>"},
	{"/Save", NULL, NULL, 0, "<Branch>"},
	{"/Save/Preset", NULL, equalizerwin_presets_menu_cb, EQUALIZER_PRESETS_SAVE_PRESET, "<Item>"},
	{"/Save/Auto-load preset", NULL, equalizerwin_presets_menu_cb, EQUALIZER_PRESETS_SAVE_AUTOPRESET, "<Item>"},
	{"/Save/Default", NULL, equalizerwin_presets_menu_cb, EQUALIZER_PRESETS_SAVE_DEFAULT, "<Item>"},
	{"/Save/-", NULL, NULL, 0, "<Separator>"},
	{"/Save/To file", NULL, equalizerwin_presets_menu_cb, EQUALIZER_PRESETS_SAVE_TO_FILE, "<Item>"},
	{"/Save/To WinAMP EQF file", NULL, equalizerwin_presets_menu_cb, EQUALIZER_PRESETS_SAVE_TO_WINAMPFILE, "<Item>"},
	{"/Delete", NULL, NULL, 0, "<Branch>"},
	{"/Delete/Preset", NULL, equalizerwin_presets_menu_cb, EQUALIZER_PRESETS_DELETE_PRESET, "<Item>"},
	{"/Delete/Auto-load preset", NULL, equalizerwin_presets_menu_cb, EQUALIZER_PRESETS_DELETE_AUTOPRESET, "<Item>"},
};

extern GtkItemFactory *mainwin_general_menu;

void equalizerwin_set_doublesize(gboolean ds)
{
	gint height;
	
	if(cfg.equalizer_shaded)
		height = 14;
	else
		height = 116;

	if (ds)
	{
		gtk_widget_shape_combine_mask(equalizerwin, get_skin_pixmap((cfg.equalizer_shaded ? SKIN_MASK_EQ_SHADE_DS : SKIN_MASK_EQ_DS)), 0, 0);
		dock_resize(dock_window_list, equalizerwin, 550, height * 2);
		gdk_window_set_back_pixmap(equalizerwin->window, equalizerwin_bg_dblsize, 0);
		draw_equalizer_window(TRUE);

	}
	else
	{
		gtk_widget_shape_combine_mask(equalizerwin, get_skin_pixmap(cfg.equalizer_shaded ? SKIN_MASK_EQ_SHADE : SKIN_MASK_EQ), 0, 0);
		dock_resize(dock_window_list, equalizerwin, 275, height);
		gdk_window_set_back_pixmap(equalizerwin->window, equalizerwin_bg, 0);
		draw_equalizer_window(TRUE);
	}
}

void equalizerwin_set_shade(gboolean shaded)
{
	GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(mainwin_options_menu, "/Equalizer WindowShade Mode"))->active = shaded;

	
	if(shaded)
	{
		gtk_widget_shape_combine_mask(equalizerwin, get_skin_pixmap((EQUALIZER_DOUBLESIZE ? SKIN_MASK_EQ_SHADE_DS : SKIN_MASK_EQ_SHADE)), 0, 0);
		dock_shade(dock_window_list, equalizerwin, 14 * (EQUALIZER_DOUBLESIZE + 1));
		pbutton_set_button_data(equalizerwin_shade, -1, 3, -1, 47);
		pbutton_set_skin_index1(equalizerwin_shade, SKIN_EQ_EX);
		pbutton_set_button_data(equalizerwin_close, 11, 38, 11, 47);
		pbutton_set_skin_index(equalizerwin_close, SKIN_EQ_EX);
		show_widget(equalizerwin_volume);
		show_widget(equalizerwin_balance);
	}
	else
	{
		gtk_widget_shape_combine_mask(equalizerwin, get_skin_pixmap((EQUALIZER_DOUBLESIZE ? SKIN_MASK_EQ_DS : SKIN_MASK_EQ)), 0, 0);
		dock_shade(dock_window_list, equalizerwin, 116 * (EQUALIZER_DOUBLESIZE + 1));
		pbutton_set_button_data(equalizerwin_shade, -1, 137, -1, 38);
		pbutton_set_skin_index1(equalizerwin_shade, SKIN_EQMAIN);
		pbutton_set_button_data(equalizerwin_close, 0, 116, 0, 125);
		pbutton_set_skin_index(equalizerwin_close, SKIN_EQMAIN);
		hide_widget(equalizerwin_volume);
		hide_widget(equalizerwin_balance);
	}
	
	cfg.equalizer_shaded=shaded;
	draw_equalizer_window(TRUE);
}

void equalizerwin_shade_cb(void)
{
	equalizerwin_set_shade(!cfg.equalizer_shaded);
}

void equalizerwin_raise(void)
{
	if (cfg.equalizer_visible)
		gdk_window_raise(equalizerwin->window);
}

void equalizerwin_move(gint x, gint y)
{
	cfg.equalizer_x = x;
	cfg.equalizer_y = y;
	if (cfg.equalizer_visible)
		gdk_window_move(equalizerwin->window, x, y);
		
}

void equalizerwin_eq_changed(void)
{
	int i;

	cfg.equalizer_preamp = eqslider_get_position(equalizerwin_preamp);
	for (i = 0; i < 10; i++)
		cfg.equalizer_bands[i] = eqslider_get_position(equalizerwin_bands[i]);
	input_set_eq(cfg.equalizer_active, cfg.equalizer_preamp, cfg.equalizer_bands);
	draw_widget(equalizerwin_graph);
}

void equalizerwin_on_pushed(gboolean toggled)
{
	cfg.equalizer_active = toggled;
	equalizerwin_eq_changed();
}

void equalizerwin_presets_pushed(void)
{
	GdkModifierType modmask;
	gint x, y;

	gdk_window_get_pointer(NULL, &x, &y, &modmask);
	util_item_factory_popup(equalizerwin_presets_menu, x, y, 1, GDK_CURRENT_TIME);
}

void equalizerwin_auto_pushed(gboolean toggled)
{
	cfg.equalizer_autoload = toggled;
}

void draw_equalizer_window(gboolean force)
{
	GdkImage *img, *img2;
	GList *wl;
	Widget *w;
	gboolean redraw;

	lock_widget_list(equalizerwin_wlist);
	if (force)
	{
		gdk_draw_pixmap(equalizerwin_bg, equalizerwin_gc, get_skin_pixmap(SKIN_EQMAIN), 0, 0, 0, 0, 275, 116);
		if (equalizerwin_focus || !cfg.dim_titlebar)
		{
			if(!cfg.equalizer_shaded)
				gdk_draw_pixmap(equalizerwin_bg, equalizerwin_gc, get_skin_pixmap(SKIN_EQMAIN), 0, 134, 0, 0, 275, 14);
			else
				gdk_draw_pixmap(equalizerwin_bg, equalizerwin_gc, get_skin_pixmap(SKIN_EQ_EX), 0, 0, 0, 0, 275, 14);
		}
		else
		{
			if(!cfg.equalizer_shaded)
				gdk_draw_pixmap(equalizerwin_bg, equalizerwin_gc, get_skin_pixmap(SKIN_EQMAIN), 0, 149, 0, 0, 275, 14);
			else
				gdk_draw_pixmap(equalizerwin_bg, equalizerwin_gc, get_skin_pixmap(SKIN_EQ_EX), 0, 15, 0, 0, 275, 14);

		}
		draw_widget_list(equalizerwin_wlist, &redraw, TRUE);
	}
	else
		draw_widget_list(equalizerwin_wlist, &redraw, FALSE);
	if (force || redraw)
	{
		if (cfg.doublesize && cfg.eq_doublesize_linked)
		{
			if (force)
			{
				img = gdk_image_get(equalizerwin_bg, 0, 0, 275, 116);
				img2 = create_dblsize_image(img);
				gdk_draw_image(equalizerwin_bg_dblsize, equalizerwin_gc, img2, 0, 0, 0, 0, 550, 232);
				gdk_image_destroy(img2);
				gdk_image_destroy(img);
			}
			else
			{
				wl = equalizerwin_wlist;
				while (wl)
				{
					w = (Widget *) wl->data;
					if (w->redraw && w->visible)
					{
						img = gdk_image_get(equalizerwin_bg, w->x, w->y, w->width, w->height);
						img2 = create_dblsize_image(img);
						gdk_draw_image(equalizerwin_bg_dblsize, equalizerwin_gc, img2, 0, 0, w->x << 1, w->y << 1, w->width << 1, w->height << 1);
						gdk_image_destroy(img2);
						gdk_image_destroy(img);
						w->redraw = FALSE;
					}
					wl = wl->next;
				}
			}
		}
		else
			clear_widget_list_redraw(equalizerwin_wlist);
		gdk_window_clear(equalizerwin->window);
		gdk_flush();
	}
	unlock_widget_list(equalizerwin_wlist);
}

static gboolean inside_sensitive_widgets(gint x, gint y)
{
	return (inside_widget(x, y, equalizerwin_on) ||
		inside_widget(x, y, equalizerwin_auto) ||
		inside_widget(x, y, equalizerwin_presets) ||
		inside_widget(x, y, equalizerwin_close) ||
		inside_widget(x, y, equalizerwin_shade) ||
		inside_widget(x, y, equalizerwin_preamp) ||
		inside_widget(x, y, equalizerwin_bands[0]) ||
		inside_widget(x, y, equalizerwin_bands[1]) ||
		inside_widget(x, y, equalizerwin_bands[2]) ||
		inside_widget(x, y, equalizerwin_bands[3]) ||
		inside_widget(x, y, equalizerwin_bands[4]) ||
		inside_widget(x, y, equalizerwin_bands[5]) ||
		inside_widget(x, y, equalizerwin_bands[6]) ||
		inside_widget(x, y, equalizerwin_bands[7]) ||
		inside_widget(x, y, equalizerwin_bands[8]) ||
		inside_widget(x, y, equalizerwin_bands[9]) ||
		inside_widget(x, y, equalizerwin_volume) ||
		inside_widget(x, y, equalizerwin_balance));
}

void equalizerwin_press(GtkWidget * widget, GdkEventButton * event, gpointer callback_data)
{
	gint mx, my;
	gboolean grab = TRUE;

	mx = event->x;
	my = event->y;
	if (cfg.doublesize && cfg.eq_doublesize_linked)
	{
		event->x /= 2;
		event->y /= 2;
	}

	if (event->button == 1 && event->type == GDK_BUTTON_PRESS &&
	    ((cfg.easy_move || cfg.equalizer_shaded || event->y < 14) &&
	     !inside_sensitive_widgets(event->x, event->y)))
	{

		equalizerwin_raise();
		dock_move_press(dock_window_list, equalizerwin, event, FALSE);
	}
	else if(event->button == 1 && event->type == GDK_2BUTTON_PRESS && event->y < 14)
	{
		equalizerwin_set_shade(!cfg.equalizer_shaded);
		if(dock_is_moving(equalizerwin))
			dock_move_release(equalizerwin);
	}
	else if (event->button == 3 && !(inside_widget(event->x, event->y, equalizerwin_on) ||
			inside_widget(event->x, event->y, equalizerwin_auto)))
	{
		gint mx, my;
		GdkModifierType modmask;

		gdk_window_get_pointer(NULL, &mx, &my, &modmask);
		util_item_factory_popup(mainwin_general_menu, mx, my, 3, GDK_CURRENT_TIME);
		grab = FALSE;
	}
	else
	{
		handle_press_cb(equalizerwin_wlist, widget, event);
		draw_equalizer_window(FALSE);
	}
	if (grab)
		gdk_pointer_grab(equalizerwin->window, FALSE, GDK_BUTTON_MOTION_MASK | GDK_BUTTON_RELEASE_MASK, GDK_NONE, GDK_NONE, GDK_CURRENT_TIME);
}

void equalizerwin_motion(GtkWidget * widget, GdkEventMotion * event, gpointer callback_data)
{
	XEvent ev;

	if (cfg.doublesize && cfg.eq_doublesize_linked)
	{
		event->x /= 2;
		event->y /= 2;
	}
	if (dock_is_moving(equalizerwin))
	{
		dock_move_motion(equalizerwin, event);
	}
	else
	{
		handle_motion_cb(equalizerwin_wlist, widget, event);
		draw_main_window(FALSE);
	}
	gdk_flush();
	while (XCheckMaskEvent(GDK_DISPLAY(), ButtonMotionMask, &ev)) ;

}

void equalizerwin_release(GtkWidget * widget, GdkEventButton * event, gpointer callback_data)
{
	gdk_pointer_ungrab(GDK_CURRENT_TIME);
	gdk_flush();
	if (dock_is_moving(equalizerwin))
	{
		dock_move_release(equalizerwin);
	}
	else
	{
		handle_release_cb(equalizerwin_wlist, widget, event);
		draw_equalizer_window(FALSE);
	}
}

void equalizerwin_focus_in(GtkWidget * widget, GdkEvent * event, gpointer callback_data)
{
	equalizerwin_close->pb_allow_draw = TRUE;
	equalizerwin_shade->pb_allow_draw = TRUE;
	equalizerwin_focus = TRUE;
	draw_equalizer_window(TRUE);
}

void equalizerwin_focus_out(GtkWidget * widget, GdkEventButton * event, gpointer callback_data)
{
	equalizerwin_close->pb_allow_draw = FALSE;
	equalizerwin_shade->pb_allow_draw = FALSE;
	equalizerwin_focus = FALSE;
	draw_equalizer_window(TRUE);
}

gboolean equalizerwin_keypress(GtkWidget * w, GdkEventKey * event, gpointer data)
{
	if(!cfg.equalizer_shaded)
		return TRUE;
	
	switch(event->keyval)
	{
		case GDK_Up:
		case GDK_KP_Up:
			mainwin_set_volume_diff(2);
			break;
		case GDK_Down:
		case GDK_KP_Down:
			mainwin_set_volume_diff(-2);
			break;
		case GDK_Left:
		case GDK_KP_Left:
			mainwin_set_balance_diff(-4);
			break;
		case GDK_Right:
		case GDK_KP_Right:
			mainwin_set_balance_diff(4);
			break;
		default:
			break;
	}

	return TRUE;
}

gint equalizerwin_configure(GtkWidget * window)
{
	gdk_window_get_deskrelative_origin(window->window, &cfg.equalizer_x, &cfg.equalizer_y);
	return FALSE;
}

void equalizerwin_set_back_pixmap(void)
{
	if (cfg.doublesize && cfg.eq_doublesize_linked)
		gdk_window_set_back_pixmap(equalizerwin->window, equalizerwin_bg_dblsize, 0);
	else
		gdk_window_set_back_pixmap(equalizerwin->window, equalizerwin_bg, 0);
	gdk_window_clear(equalizerwin->window);
}

gint equalizerwin_client_event(GtkWidget *w,GdkEventClient *event, gpointer data)
{
	static GdkAtom atom_rcfiles = GDK_NONE;

	if (!atom_rcfiles)
		atom_rcfiles = gdk_atom_intern("_GTK_READ_RCFILES", FALSE);
	if(event->message_type == atom_rcfiles)
	{
		
		mainwin_set_back_pixmap();
		equalizerwin_set_back_pixmap();
		playlistwin_set_back_pixmap();
		return TRUE;
		
	}
	return FALSE;
}


int equalizerwin_delete(GtkWidget * w, gpointer data)
{
	equalizerwin_hide();
	return TRUE;
}

static GList *equalizerwin_read_presets(gchar * fname)
{
	gchar *tmp, *filename, *name;
	ConfigFile *cfgfile;
	GList *list = NULL;
	gint i, p = 0;
	EqualizerPreset *preset;

	filename = g_strdup_printf("%s/.xmms/%s", g_get_home_dir(), fname);
	if ((cfgfile = xmms_cfg_open_file(filename)) != NULL)
	{
		for (;;)
		{
			tmp = g_strdup_printf("Preset%d", p++);
			if (xmms_cfg_read_string(cfgfile, "Presets", tmp, &name))
			{
				g_free(tmp);
				preset = g_malloc(sizeof (EqualizerPreset));
				preset->name = name;
				xmms_cfg_read_float(cfgfile, name, "Preamp", &preset->preamp);
				for (i = 0; i < 10; i++)
				{
					tmp = g_strdup_printf("Band%d", i);
					xmms_cfg_read_float(cfgfile, name, tmp, &preset->bands[i]);
					g_free(tmp);
				}
				list = g_list_append(list, preset);
			}
			else
			{
				g_free(tmp);
				break;
			}
		}
		xmms_cfg_free(cfgfile);
	}
	g_free(filename);
	return list;
}

gint equalizerwin_volume_frame_cb(gint pos)
{
	if(equalizerwin_volume)
	{
		if (pos < 32)
			equalizerwin_volume->hs_knob_nx = equalizerwin_volume->hs_knob_px = 1;
		else if (pos < 63)
			equalizerwin_volume->hs_knob_nx = equalizerwin_volume->hs_knob_px = 4;
		else
			equalizerwin_volume->hs_knob_nx = equalizerwin_volume->hs_knob_px = 7;
	}
	return 1;
}

void equalizerwin_volume_motion_cb(gint pos)
{
	gint v = (gint) rint(pos * 100 / 94.0);
	mainwin_adjust_volume_motion(v);
	mainwin_set_volume_slider(v);
}

void equalizerwin_volume_release_cb(gint pos)
{
	mainwin_adjust_volume_release();
}

gint equalizerwin_balance_frame_cb(gint pos)
{
	if(equalizerwin_balance)
	{
		if(pos < 13)
			equalizerwin_balance->hs_knob_nx = equalizerwin_balance->hs_knob_px = 11;
		else if (pos < 26)
			equalizerwin_balance->hs_knob_nx = equalizerwin_balance->hs_knob_px = 14;
		else
			equalizerwin_balance->hs_knob_nx = equalizerwin_balance->hs_knob_px = 17;
	}
			
	return 1;
}

void equalizerwin_balance_motion_cb(gint pos)
{
	gint b;
	pos = MIN(pos,38); /* The skin uses a even number of pixels
			      for the balance-slider *sigh* */
	b = (gint) rint((pos - 19) * 100 / 19.0);
	mainwin_adjust_balance_motion(b);
	mainwin_set_balance_slider(b);
}

void equalizerwin_balance_release_cb(gint pos)
{
	mainwin_adjust_balance_release();
}

void equalizerwin_set_balance_slider(gint percent)
{
	hslider_set_position(equalizerwin_balance, (gint) rint((percent*19/100.0)+19));
}

void equalizerwin_set_volume_slider(gint percent)
{
	hslider_set_position(equalizerwin_volume, (gint) rint(percent*94/100.0));
}

void equalizerwin_create(void)
{
	int i;

	equalizerwin = gtk_window_new(GTK_WINDOW_DIALOG);
	dock_add_window(dock_window_list, equalizerwin);
	gtk_widget_set_app_paintable(equalizerwin, TRUE);
	gtk_window_set_policy(GTK_WINDOW(equalizerwin), FALSE, FALSE, TRUE);
	gtk_window_set_title(GTK_WINDOW(equalizerwin), "XMMS Equalizer");
	gtk_window_set_wmclass(GTK_WINDOW(equalizerwin), "XMMS_Equalizer", "xmms");
	gtk_window_set_transient_for(GTK_WINDOW(equalizerwin), GTK_WINDOW(mainwin));
	if (cfg.equalizer_x != -1 && cfg.save_window_position)
		dock_set_uposition(equalizerwin, cfg.equalizer_x, cfg.equalizer_y);
	if (cfg.doublesize && cfg.eq_doublesize_linked)
		gtk_widget_set_usize(equalizerwin, 550, (cfg.equalizer_shaded ? 28 : 232));
	else
		gtk_widget_set_usize(equalizerwin, 275, (cfg.equalizer_shaded ? 14 : 116));

	gtk_widget_set_events(equalizerwin, GDK_FOCUS_CHANGE_MASK | GDK_BUTTON_MOTION_MASK | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);
	gtk_widget_realize(equalizerwin);
	hint_set_skip_winlist(equalizerwin);
	gdk_window_set_decorations(equalizerwin->window, 0);

	equalizerwin_accel = gtk_accel_group_new();
	gtk_window_add_accel_group(GTK_WINDOW(equalizerwin), equalizerwin_accel);

	equalizerwin_gc = gdk_gc_new(equalizerwin->window);
	equalizerwin_bg = gdk_pixmap_new(equalizerwin->window, 275, 116, gdk_rgb_get_visual()->depth);
	equalizerwin_bg_dblsize = gdk_pixmap_new(equalizerwin->window, 550, 232, gdk_rgb_get_visual()->depth);
	if (cfg.doublesize && cfg.eq_doublesize_linked)
		gdk_window_set_back_pixmap(equalizerwin->window, equalizerwin_bg_dblsize, 0);
	else
		gdk_window_set_back_pixmap(equalizerwin->window, equalizerwin_bg, 0);

	gtk_signal_connect(GTK_OBJECT(equalizerwin), "delete_event", GTK_SIGNAL_FUNC(equalizerwin_delete), NULL);
	gtk_signal_connect(GTK_OBJECT(equalizerwin), "button_press_event", GTK_SIGNAL_FUNC(equalizerwin_press), NULL);
	gtk_signal_connect(GTK_OBJECT(equalizerwin), "button_release_event", GTK_SIGNAL_FUNC(equalizerwin_release), NULL);
	gtk_signal_connect(GTK_OBJECT(equalizerwin), "motion_notify_event", GTK_SIGNAL_FUNC(equalizerwin_motion), NULL);
	gtk_signal_connect(GTK_OBJECT(equalizerwin), "focus_in_event", GTK_SIGNAL_FUNC(equalizerwin_focus_in), NULL);
	gtk_signal_connect(GTK_OBJECT(equalizerwin), "focus_out_event", GTK_SIGNAL_FUNC(equalizerwin_focus_out), NULL);
	gtk_signal_connect(GTK_OBJECT(equalizerwin), "configure_event", GTK_SIGNAL_FUNC(equalizerwin_configure), NULL);
	gtk_signal_connect(GTK_OBJECT(equalizerwin), "client_event", GTK_SIGNAL_FUNC(equalizerwin_client_event), NULL);
	gtk_signal_connect(GTK_OBJECT(equalizerwin), "key-press-event", GTK_SIGNAL_FUNC(equalizerwin_keypress), NULL);

	equalizerwin_on = create_tbutton(&equalizerwin_wlist, equalizerwin_bg, equalizerwin_gc, 14, 18, 25, 12, 10, 119, 128, 119, 69, 119, 187, 119, equalizerwin_on_pushed, SKIN_EQMAIN);
	tbutton_set_toggled(equalizerwin_on, cfg.equalizer_active);
	equalizerwin_auto = create_tbutton(&equalizerwin_wlist, equalizerwin_bg, equalizerwin_gc, 39, 18, 33, 12, 35, 119, 153, 119, 94, 119, 212, 119, equalizerwin_auto_pushed, SKIN_EQMAIN);
	tbutton_set_toggled(equalizerwin_auto, cfg.equalizer_autoload);
	equalizerwin_presets = create_pbutton(&equalizerwin_wlist, equalizerwin_bg, equalizerwin_gc, 217, 18, 44, 12, 224, 164, 224, 176, equalizerwin_presets_pushed, SKIN_EQMAIN);
	equalizerwin_close = create_pbutton(&equalizerwin_wlist, equalizerwin_bg, equalizerwin_gc, 264, 3, 9, 9, 0, 116, 0, 125, equalizerwin_hide, SKIN_EQMAIN);
	equalizerwin_close->pb_allow_draw = FALSE;

	equalizerwin_shade = create_pbutton_ex(&equalizerwin_wlist, equalizerwin_bg, equalizerwin_gc, 254, 3, 9, 9, 254, 137, 1, 38, equalizerwin_shade_cb, SKIN_EQMAIN, SKIN_EQ_EX);
	equalizerwin_shade->pb_allow_draw = FALSE;

	equalizerwin_graph = create_eqgraph(&equalizerwin_wlist, equalizerwin_bg, equalizerwin_gc, 86, 17);
	equalizerwin_preamp = create_eqslider(&equalizerwin_wlist, equalizerwin_bg, equalizerwin_gc, 21, 38);
	eqslider_set_position(equalizerwin_preamp, cfg.equalizer_preamp);
	for (i = 0; i < 10; i++)
	{
		equalizerwin_bands[i] = create_eqslider(&equalizerwin_wlist, equalizerwin_bg, equalizerwin_gc, 78 + (i * 18), 38);
		eqslider_set_position(equalizerwin_bands[i], cfg.equalizer_bands[i]);
	}

	equalizerwin_volume = create_hslider(&equalizerwin_wlist, equalizerwin_bg, equalizerwin_gc, 61, 4, 97, 8, 1, 30, 1, 30, 3, 7, 4, 61, 0, 94, equalizerwin_volume_frame_cb, equalizerwin_volume_motion_cb, equalizerwin_volume_release_cb, SKIN_EQ_EX);
	equalizerwin_balance = create_hslider(&equalizerwin_wlist, equalizerwin_bg, equalizerwin_gc, 164, 4, 42, 8, 11, 30, 11, 30, 3, 7, 4, 164, 0, 39, equalizerwin_balance_frame_cb, equalizerwin_balance_motion_cb, equalizerwin_balance_release_cb, SKIN_EQ_EX);

	equalizerwin_presets_menu = gtk_item_factory_new(GTK_TYPE_MENU, "<Main>", equalizerwin_accel);
	gtk_item_factory_create_items(equalizerwin_presets_menu, EQUALIZERWIN_PRESETS_MENU_ENTRIES, equalizerwin_presets_menu_entries, NULL);
	equalizer_presets = equalizerwin_read_presets("eq.preset");
	equalizer_auto_presets = equalizerwin_read_presets("eq.auto_preset");
	if(!cfg.equalizer_shaded)
	{
		hide_widget(equalizerwin_volume);
		hide_widget(equalizerwin_balance);
	}
	else
	{
		pbutton_set_button_data(equalizerwin_shade, -1, 3, -1, 47);
		pbutton_set_skin_index1(equalizerwin_shade, SKIN_EQ_EX);
		pbutton_set_button_data(equalizerwin_close, 11, 38, 11, 47);
		pbutton_set_skin_index(equalizerwin_close, SKIN_EQ_EX);
	}

}

void equalizerwin_show(void)
{
	gtk_widget_show(equalizerwin);
	if (pposition_broken && cfg.equalizer_x != -1 && cfg.save_window_position)
		dock_set_uposition(equalizerwin, cfg.equalizer_x, cfg.equalizer_y);
	if (cfg.doublesize && cfg.eq_doublesize_linked)
		gtk_widget_set_usize(equalizerwin, 550, (cfg.equalizer_shaded ? 28 : 232));
	else
		gtk_widget_set_usize(equalizerwin, 275, (cfg.equalizer_shaded ? 14 : 116));
	gdk_flush();
	draw_equalizer_window(TRUE);
	cfg.equalizer_visible = TRUE;
	tbutton_set_toggled(mainwin_eq, TRUE);
	GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(mainwin_general_menu, "/Graphical EQ"))->active = TRUE;
	mainwin_set_always_on_top(cfg.always_on_top);
	hint_set_sticky(cfg.sticky);
	hint_set_skip_winlist(equalizerwin);
}

void equalizerwin_hide(void)
{
	gtk_widget_hide(equalizerwin);
	cfg.equalizer_visible = FALSE;
	tbutton_set_toggled(mainwin_eq, FALSE);
	GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(mainwin_general_menu, "/Graphical EQ"))->active = FALSE;
}

static EqualizerPreset *equalizerwin_find_preset(GList * list, gchar * name)
{
	GList *node = list;
	EqualizerPreset *preset;

	while (node)
	{
		preset = node->data;
		if (!strcasecmp(preset->name, name))
			return preset;
		node = g_list_next(node);
	}
	return NULL;
}

static void equalizerwin_write_preset_file(GList * list, gchar * fname)
{
	gchar *filename, *tmp;
	gint i, p;
	EqualizerPreset *preset;
	ConfigFile *cfgfile;
	GList *node;

	cfgfile = xmms_cfg_new();
	p = 0;
	node = list;
	while (node)
	{
		preset = node->data;
		tmp = g_strdup_printf("Preset%d", p++);
		xmms_cfg_write_string(cfgfile, "Presets", tmp, preset->name);
		g_free(tmp);
		xmms_cfg_write_float(cfgfile, preset->name, "Preamp", preset->preamp);
		for (i = 0; i < 10; i++)
		{
			tmp = g_strdup_printf("Band%d\n", i);
			xmms_cfg_write_float(cfgfile, preset->name, tmp, preset->bands[i]);
			g_free(tmp);
		}
		node = g_list_next(node);
	}
	filename = g_strdup_printf("%s/.xmms/%s", g_get_home_dir(), fname);
	xmms_cfg_write_file(cfgfile, filename);
	xmms_cfg_free(cfgfile);
	g_free(filename);
}

static gboolean equalizerwin_load_preset(GList * list, gchar * name)
{
	EqualizerPreset *preset;
	gint i;

	if ((preset = equalizerwin_find_preset(list, name)) != NULL)
	{
		eqslider_set_position(equalizerwin_preamp, preset->preamp);
		for (i = 0; i < 10; i++)
			eqslider_set_position(equalizerwin_bands[i], preset->bands[i]);
		equalizerwin_eq_changed();
		return TRUE;
	}
	return FALSE;
}

static GList *equalizerwin_save_preset(GList * list, gchar * name, gchar * fname)
{
	gint i;
	EqualizerPreset *preset;

	if (!(preset = equalizerwin_find_preset(list, name)))
	{
		preset = g_malloc(sizeof (EqualizerPreset));
		preset->name = g_strdup(name);
		list = g_list_append(list, preset);
	}

	preset->preamp = eqslider_get_position(equalizerwin_preamp);
	for (i = 0; i < 10; i++)
		preset->bands[i] = eqslider_get_position(equalizerwin_bands[i]);

	equalizerwin_write_preset_file(list, fname);

	return list;
}

static GList *equalizerwin_delete_preset(GList * list, gchar * name, gchar * fname)
{
	EqualizerPreset *preset;
	GList *node;

	if ((preset = equalizerwin_find_preset(list, name)) && (node = g_list_find(list, preset)))
	{
		list = g_list_remove_link(list, node);
		g_free(preset->name);
		g_free(preset);
		g_list_free_1(node);

		equalizerwin_write_preset_file(list, fname);
	}

	return list;
}

void equalizerwin_save_ok(GtkWidget * widget, gpointer data)
{
	gchar *text;

	text = gtk_entry_get_text(GTK_ENTRY(equalizerwin_save_entry));
	if (strlen(text) != 0)
		equalizer_presets = equalizerwin_save_preset(equalizer_presets, text, "eq.preset");
	gtk_widget_destroy(equalizerwin_save_window);
}

void equalizerwin_save_select(GtkWidget * widget, gint row, gint column, GdkEvent * event)
{
	gchar *text;

	gtk_clist_get_text(GTK_CLIST(equalizerwin_save_list), row, 0, &text);

	gtk_entry_set_text(GTK_ENTRY(equalizerwin_save_entry), text);
	if (event->type == GDK_2BUTTON_PRESS)
		equalizerwin_save_ok(NULL, NULL);

}

void equalizerwin_load_ok(GtkWidget * widget, gpointer data)
{
	gchar *text;

	if (GTK_CLIST(equalizerwin_load_list)->selection)
	{
		gtk_clist_get_text(GTK_CLIST(equalizerwin_load_list), (gint) GTK_CLIST(equalizerwin_load_list)->selection->data, 0, &text);
		equalizerwin_load_preset(equalizer_presets, text);
	}
	gtk_widget_destroy(equalizerwin_load_window);
}

void equalizerwin_load_select(GtkWidget * widget, gint row, gint column, GdkEvent * event)
{
	if (event->type == GDK_2BUTTON_PRESS)
		equalizerwin_load_ok(NULL, NULL);
}

void equalizerwin_delete_delete(GtkWidget * widget, gpointer data)
{
	gchar *text;
	GList *list, *next;

	list = GTK_CLIST(equalizerwin_delete_list)->selection;
	gtk_clist_freeze(GTK_CLIST(equalizerwin_delete_list));
	while (list)
	{
		next = g_list_next(list);
		gtk_clist_get_text(GTK_CLIST(equalizerwin_delete_list), (gint) list->data, 0, &text);
		equalizer_auto_presets = equalizerwin_delete_preset(equalizer_presets, text, "eq.preset");
		gtk_clist_remove(GTK_CLIST(equalizerwin_delete_list), (gint) list->data);
		list = next;
	}
	gtk_clist_thaw(GTK_CLIST(equalizerwin_delete_list));

}

void equalizerwin_save_auto_ok(GtkWidget * widget, gpointer data)
{
	gchar *text;

	text = gtk_entry_get_text(GTK_ENTRY(equalizerwin_save_auto_entry));
	if (strlen(text) != 0)
		equalizer_auto_presets = equalizerwin_save_preset(equalizer_auto_presets, text, "eq.auto_preset");
	gtk_widget_destroy(equalizerwin_save_auto_window);
}

void equalizerwin_save_auto_select(GtkWidget * widget, gint row, gint column, GdkEvent * event)
{
	gchar *text;

	gtk_clist_get_text(GTK_CLIST(equalizerwin_save_auto_list), row, 0, &text);

	gtk_entry_set_text(GTK_ENTRY(equalizerwin_save_auto_entry), text);
	if (event->type == GDK_2BUTTON_PRESS)
		equalizerwin_save_auto_ok(NULL, NULL);

}

void equalizerwin_load_auto_ok(GtkWidget * widget, gpointer data)
{
	gchar *text;

	if (GTK_CLIST(equalizerwin_load_auto_list)->selection)
	{
		gtk_clist_get_text(GTK_CLIST(equalizerwin_load_auto_list), (gint) GTK_CLIST(equalizerwin_load_auto_list)->selection->data, 0, &text);
		equalizerwin_load_preset(equalizer_auto_presets, text);
	}
	gtk_widget_destroy(equalizerwin_load_auto_window);
}

void equalizerwin_load_auto_select(GtkWidget * widget, gint row, gint column, GdkEvent * event)
{
	if (event->type == GDK_2BUTTON_PRESS)
		equalizerwin_load_auto_ok(NULL, NULL);
}

void equalizerwin_delete_auto_delete(GtkWidget * widget, gpointer data)
{
	gchar *text;
	GList *list, *next;

	list = GTK_CLIST(equalizerwin_delete_auto_list)->selection;
	gtk_clist_freeze(GTK_CLIST(equalizerwin_delete_auto_list));
	while (list)
	{
		next = g_list_next(list);
		gtk_clist_get_text(GTK_CLIST(equalizerwin_delete_auto_list), (gint) list->data, 0, &text);
		equalizer_auto_presets = equalizerwin_delete_preset(equalizer_auto_presets, text, "eq.auto_preset");
		gtk_clist_remove(GTK_CLIST(equalizerwin_delete_auto_list), (gint) list->data);
		list = next;
	}
	gtk_clist_thaw(GTK_CLIST(equalizerwin_delete_auto_list));
}

static void equalizerwin_load_filesel_ok(GtkWidget * w, GtkFileSelection * filesel)
{
	gchar *filename = gtk_file_selection_get_filename(filesel);
	gchar *text, *text2, *tmp;
	ConfigFile *cfgfile;
	gfloat val;
	gint i;

	struct stat buf;

	text = g_strdup(filename);
	while (*(strrchr(text, '/') + 1) == '\0')
		*(strrchr(text, '/')) = '\0';

	if (stat(text, &buf) == 0)
	{
		if (S_ISDIR(buf.st_mode))
		{		/* Selected directory -- don't close frequester */
			text2 = g_strdup_printf("%s/", text);
			gtk_file_selection_set_filename(GTK_FILE_SELECTION(filesel), text2);
			g_free(text2);
			return;
		}
	}

	g_free(text);

	if ((cfgfile = xmms_cfg_open_file(filename)) != NULL)
	{
		xmms_cfg_read_float(cfgfile, "Equalizer preset", "Preamp", &val);
		eqslider_set_position(equalizerwin_preamp, val);
		for (i = 0; i < 10; i++)
		{
			tmp = g_strdup_printf("Band%d", i);
			xmms_cfg_read_float(cfgfile, "Equalizer preset", tmp, &val);
			g_free(tmp);
			eqslider_set_position(equalizerwin_bands[i], val);
		}
		xmms_cfg_free(cfgfile);
		equalizerwin_eq_changed();
	}
	gtk_widget_destroy(GTK_WIDGET(filesel));
}

static void equalizerwin_load_winamp_filesel_ok(GtkWidget * w, GtkFileSelection * filesel)
{
	gchar *filename = gtk_file_selection_get_filename(filesel);
	FILE *file;
	gchar *text, *text2;
	gchar header[31];
	guchar bands[11];
	gint i;

	struct stat buf;

	text = g_strdup(filename);
	while (*(strrchr(text, '/') + 1) == '\0')
		*(strrchr(text, '/')) = '\0';

	if (stat(text, &buf) == 0)
	{
		if (S_ISDIR(buf.st_mode))
		{		/* Selected directory -- don't close frequester */
			text2 = g_strdup_printf("%s/", text);
			gtk_file_selection_set_filename(GTK_FILE_SELECTION(filesel), text2);
			g_free(text2);
			return;
		}
	}

	g_free(text);

	if ((file = fopen(filename, "rb")) != NULL)
	{
		fread(header, 1, 31, file);
		if (!strncmp(header, "Winamp EQ library file v1.1", 27))
		{
			fseek(file, 257, SEEK_CUR);	/* Skip name */
			fread(bands, 1, 11, file);
			eqslider_set_position(equalizerwin_preamp, 20.0 - ((bands[10] * 40.0) / 63.0));
			for (i = 0; i < 10; i++)
				eqslider_set_position(equalizerwin_bands[i], 20.0 - ((bands[i] * 40.0) / 64.0));
			equalizerwin_eq_changed();
		}
		fclose(file);
	}
	gtk_widget_destroy(GTK_WIDGET(filesel));
}

static void equalizerwin_save_filesel_ok(GtkWidget * w, GtkFileSelection * filesel)
{
	gchar *filename = gtk_file_selection_get_filename(filesel);
	gchar *text, *text2, *tmp;
	ConfigFile *cfgfile;
	gint i;

	struct stat buf;

	text = g_strdup(filename);
	while (*(strrchr(text, '/') + 1) == '\0')
		*(strrchr(text, '/')) = '\0';

	if (stat(text, &buf) == 0)
	{
		if (S_ISDIR(buf.st_mode))
		{		/* Selected directory -- don't close frequester */
			text2 = g_strdup_printf("%s/", text);
			gtk_file_selection_set_filename(GTK_FILE_SELECTION(filesel), text2);
			g_free(text2);
			return;
		}
	}

	g_free(text);

	cfgfile = xmms_cfg_new();
	xmms_cfg_write_float(cfgfile, "Equalizer preset", "Preamp", eqslider_get_position(equalizerwin_preamp));
	for (i = 0; i < 10; i++)
	{
		tmp = g_strdup_printf("Band%d", i);
		xmms_cfg_write_float(cfgfile, "Equalizer preset", tmp, eqslider_get_position(equalizerwin_bands[i]));
		g_free(tmp);
	}
	xmms_cfg_write_file(cfgfile, filename);
	xmms_cfg_free(cfgfile);
	gtk_widget_destroy(GTK_WIDGET(filesel));
}

static void equalizerwin_save_winamp_filesel_ok(GtkWidget * w, GtkFileSelection * filesel)
{
	gchar *filename = gtk_file_selection_get_filename(filesel), name[257];
	gchar *text, *text2;
	FILE *file;
	gint i;
	guchar bands[11];

	struct stat buf;

	text = g_strdup(filename);
	while (*(strrchr(text, '/') + 1) == '\0')
		*(strrchr(text, '/')) = '\0';

	if (stat(text, &buf) == 0)
	{
		if (S_ISDIR(buf.st_mode))
		{		/* Selected directory -- don't close frequester */
			text2 = g_strdup_printf("%s/", text);
			gtk_file_selection_set_filename(GTK_FILE_SELECTION(filesel), text2);
			g_free(text2);
			return;
		}
	}

	g_free(text);

	if ((file = fopen(filename, "wb")) != NULL)
	{
		fwrite("Winamp EQ library file v1.1\x1a!--", 1, 31, file);
		memset(name, 0, 257);
		strcpy(name, "Entry1");
		fwrite(name, 1, 257, file);
		for (i = 0; i < 10; i++)
			bands[i] = 63 - (((eqslider_get_position(equalizerwin_bands[i]) + 20) * 63) / 40);
		bands[10] = 63 - (((eqslider_get_position(equalizerwin_preamp) + 20) * 63) / 40);
		fwrite(bands, 1, 11, file);
		fclose(file);
	}

	gtk_widget_destroy(GTK_WIDGET(filesel));
}

static gint equalizerwin_list_sort_func(GtkCList * clist, gconstpointer ptr1, gconstpointer ptr2)
{
	GtkCListRow *row1 = (GtkCListRow *) ptr1;
	GtkCListRow *row2 = (GtkCListRow *) ptr2;

	return strcasecmp(GTK_CELL_TEXT(row1->cell[clist->sort_column])->text, GTK_CELL_TEXT(row2->cell[clist->sort_column])->text);
}

static void *equalizerwin_create_list_window(GList * preset_list, gchar * title, GtkWidget ** window, GtkWidget ** list, GtkSelectionMode sel_mode, GtkWidget ** entry, gchar * btn1_caption, gchar * btn2_caption, GtkSignalFunc btn1_func, GtkSignalFunc select_row_func)
{
	GtkWidget *vbox, *scrolled_window, *bbox, *btn1, *btn2;
	gchar *preset_text[] =
	{"Presets"};
	GList *node;

	*window = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_signal_connect(GTK_OBJECT(*window), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), window);
	gtk_window_set_transient_for(GTK_WINDOW(*window), GTK_WINDOW(equalizerwin));
	gtk_window_set_position(GTK_WINDOW(*window), GTK_WIN_POS_MOUSE);
	gtk_window_set_title(GTK_WINDOW(*window), title);
	
	gtk_widget_set_usize(*window, 350, 300);
	gtk_container_set_border_width(GTK_CONTAINER(*window), 10);

	vbox = gtk_vbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(*window), vbox);

	scrolled_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);

	*list = gtk_clist_new_with_titles(1, preset_text);
	if (select_row_func)
		gtk_signal_connect(GTK_OBJECT(*list), "select_row", GTK_SIGNAL_FUNC(select_row_func), NULL);
	gtk_clist_column_titles_passive(GTK_CLIST(*list));
	gtk_clist_set_selection_mode(GTK_CLIST(*list), sel_mode);

	node = preset_list;
	while (node)
	{
		gtk_clist_append(GTK_CLIST(*list), &((EqualizerPreset *) node->data)->name);
		node = node->next;
	}
	gtk_clist_set_compare_func(GTK_CLIST(*list), equalizerwin_list_sort_func);
	gtk_clist_sort(GTK_CLIST(*list));

	gtk_container_add(GTK_CONTAINER(scrolled_window), *list);
	gtk_widget_show(*list);
	gtk_widget_show(scrolled_window);

	gtk_box_pack_start(GTK_BOX(vbox), scrolled_window, TRUE, TRUE, 0);

	if (entry)
	{
		*entry = gtk_entry_new();
		gtk_signal_connect(GTK_OBJECT(*entry), "activate", GTK_SIGNAL_FUNC(btn1_func), NULL);
		gtk_box_pack_start(GTK_BOX(vbox), *entry, FALSE, FALSE, 0);
		gtk_widget_show(*entry);
	}

	bbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(bbox), 5);

	btn1 = gtk_button_new_with_label(btn1_caption);
	gtk_signal_connect(GTK_OBJECT(btn1), "clicked", GTK_SIGNAL_FUNC(btn1_func), NULL);
	GTK_WIDGET_SET_FLAGS(btn1, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(bbox), btn1, TRUE, TRUE, 0);
	gtk_widget_show(btn1);

	btn2 = gtk_button_new_with_label(btn2_caption);
	gtk_signal_connect_object(GTK_OBJECT(btn2), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(*window));
	GTK_WIDGET_SET_FLAGS(btn2, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(bbox), btn2, TRUE, TRUE, 0);
	gtk_widget_show(btn2);

	gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);
	gtk_widget_show(bbox);

	gtk_widget_grab_default(btn1);
	gtk_widget_show(vbox);
	gtk_widget_show(*window);
	return window;
}

void equalizerwin_presets_menu_cb(gpointer cb_data, guint action, GtkWidget * w)
{
	gchar *name;
	PlaylistEntry *playlist_position;
	GList *list;
	gint i;

	switch (action)
	{
		case EQUALIZER_PRESETS_LOAD_PRESET:
			if (!equalizerwin_load_window)
				equalizerwin_create_list_window(equalizer_presets, "Load preset", &equalizerwin_load_window, &equalizerwin_load_list, GTK_SELECTION_SINGLE, NULL, "Ok", "Cancel", equalizerwin_load_ok, equalizerwin_load_select);
			else
				gdk_window_raise(equalizerwin_load_window->window);
			break;
		case EQUALIZER_PRESETS_LOAD_AUTOPRESET:
			if (!equalizerwin_load_auto_window)
				equalizerwin_create_list_window(equalizer_auto_presets, "Load auto-preset", &equalizerwin_load_auto_window, &equalizerwin_load_auto_list, GTK_SELECTION_SINGLE, NULL, "Ok", "Cancel", equalizerwin_load_auto_ok, equalizerwin_load_auto_select);
			else
				gdk_window_raise(equalizerwin_load_auto_window->window);
			break;
		case EQUALIZER_PRESETS_LOAD_DEFAULT:
			equalizerwin_load_preset(equalizer_presets, "Default");
			break;
		case EQUALIZER_PRESETS_LOAD_ZERO:
			eqslider_set_position(equalizerwin_preamp, 0);
			for (i = 0; i < 10; i++)
				eqslider_set_position(equalizerwin_bands[i], 0);
			equalizerwin_eq_changed();
			break;
		case EQUALIZER_PRESETS_LOAD_FROM_FILE:
			if (!equalizerwin_load_filesel)
			{
				equalizerwin_load_filesel = gtk_file_selection_new("Load equalizer preset");
				gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(equalizerwin_load_filesel)->ok_button), "clicked", GTK_SIGNAL_FUNC(equalizerwin_load_filesel_ok), equalizerwin_load_filesel);
				gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(equalizerwin_load_filesel)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(equalizerwin_load_filesel));
				gtk_signal_connect(GTK_OBJECT(equalizerwin_load_filesel), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &equalizerwin_load_filesel);
				gtk_widget_show(equalizerwin_load_filesel);

			}
			break;
		case EQUALIZER_PRESETS_LOAD_FROM_WINAMPFILE:
			if (!equalizerwin_load_winamp_filesel)
			{
				equalizerwin_load_winamp_filesel = gtk_file_selection_new("Load equalizer preset");
				gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(equalizerwin_load_winamp_filesel)->ok_button), "clicked", GTK_SIGNAL_FUNC(equalizerwin_load_winamp_filesel_ok), equalizerwin_load_winamp_filesel);
				gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(equalizerwin_load_winamp_filesel)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(equalizerwin_load_winamp_filesel));
				gtk_signal_connect(GTK_OBJECT(equalizerwin_load_winamp_filesel), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &equalizerwin_load_winamp_filesel);
				gtk_widget_show(equalizerwin_load_winamp_filesel);

			}
			break;
		case EQUALIZER_PRESETS_SAVE_PRESET:
			if (!equalizerwin_save_window)
				equalizerwin_create_list_window(equalizer_presets, "Save preset", &equalizerwin_save_window, &equalizerwin_save_list, GTK_SELECTION_SINGLE, &equalizerwin_save_entry, "Ok", "Cancel", equalizerwin_save_ok, equalizerwin_save_select);
			else
				gdk_window_raise(equalizerwin_save_window->window);
			break;
		case EQUALIZER_PRESETS_SAVE_AUTOPRESET:
			if (!equalizerwin_save_auto_window)
				equalizerwin_create_list_window(equalizer_auto_presets, "Save auto-preset", &equalizerwin_save_auto_window, &equalizerwin_save_auto_list, GTK_SELECTION_SINGLE, &equalizerwin_save_auto_entry, "Ok", "Cancel", equalizerwin_save_auto_ok, equalizerwin_save_auto_select);
			else
				gdk_window_raise(equalizerwin_save_auto_window->window);
			if ((list = get_playlist()) != NULL)
			{
				playlist_position = g_list_nth(list, get_playlist_position())->data;
				if (playlist_position)
				{
					name = strrchr(playlist_position->filename, '/');
					if (name)
						name++;
					else
						name = playlist_position->filename;
					gtk_entry_set_text(GTK_ENTRY(equalizerwin_save_auto_entry), name);
				}
			}
			break;
		case EQUALIZER_PRESETS_SAVE_DEFAULT:
			equalizer_presets = equalizerwin_save_preset(equalizer_presets, "Default", "eq.preset");
			break;
		case EQUALIZER_PRESETS_SAVE_TO_FILE:
			if (!equalizerwin_save_filesel)
			{
				equalizerwin_save_filesel = gtk_file_selection_new("Save equalizer preset");
				gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(equalizerwin_save_filesel)->ok_button), "clicked", GTK_SIGNAL_FUNC(equalizerwin_save_filesel_ok), equalizerwin_save_filesel);
				gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(equalizerwin_save_filesel)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(equalizerwin_save_filesel));
				gtk_signal_connect(GTK_OBJECT(equalizerwin_save_filesel), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &equalizerwin_save_filesel);
				gtk_widget_show(equalizerwin_save_filesel);

			}
			break;
		case EQUALIZER_PRESETS_SAVE_TO_WINAMPFILE:
			if (!equalizerwin_save_winamp_filesel)
			{
				equalizerwin_save_winamp_filesel = gtk_file_selection_new("Save equalizer preset");
				gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(equalizerwin_save_winamp_filesel)->ok_button), "clicked", GTK_SIGNAL_FUNC(equalizerwin_save_winamp_filesel_ok), equalizerwin_save_winamp_filesel);
				gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(equalizerwin_save_winamp_filesel)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(equalizerwin_save_winamp_filesel));
				gtk_signal_connect(GTK_OBJECT(equalizerwin_save_winamp_filesel), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &equalizerwin_save_winamp_filesel);
				gtk_widget_show(equalizerwin_save_winamp_filesel);

			}
			break;
		case EQUALIZER_PRESETS_DELETE_PRESET:
			if (!equalizerwin_delete_window)
				equalizerwin_create_list_window(equalizer_presets, "Delete preset", &equalizerwin_delete_window, &equalizerwin_delete_list, GTK_SELECTION_EXTENDED, NULL, "Delete", "Close", equalizerwin_delete_delete, NULL);
			break;
		case EQUALIZER_PRESETS_DELETE_AUTOPRESET:
			if (!equalizerwin_delete_auto_window)
				equalizerwin_create_list_window(equalizer_auto_presets, "Delete auto-preset", &equalizerwin_delete_auto_window, &equalizerwin_delete_auto_list, GTK_SELECTION_EXTENDED, NULL, "Delete", "Close", equalizerwin_delete_auto_delete, NULL);
			break;
	}
}

void equalizerwin_load_auto_preset(gchar * filename)
{
	gchar *name;

	if (cfg.equalizer_autoload)
	{
		if ((name = strrchr(filename, '/')) != NULL)
			name++;
		else
			name = filename;
		if (!equalizerwin_load_preset(equalizer_auto_presets, name))
			equalizerwin_load_preset(equalizer_presets, "Default");
	}
}
