/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-1999  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "OSS.h"
#include "libxmms/util.h"
#include <errno.h>

#define NFRAGS		32

static gint fd = 0;
static gpointer buffer;
static gboolean going = FALSE, prebuffer, paused = FALSE, unpause = FALSE,
         do_pause = FALSE, remove_prebuffer = FALSE;
static gint device_buffer_used, buffer_size, prebuffer_size, blk_size;
static gint rd_index = 0, wr_index = 0;
static gint output_time_offset = 0;
static guint64 written = 0, output_bytes = 0;
static gint bps, ebps;
static gint flush;
static gint fragsize, oss_format, format, channels, frequency, efrequency, device_buffer_size;
static gint input_bps, input_format, input_frequency, input_channels;
static gchar *device_name;
static pthread_t buffer_thread;
static gboolean realtime = FALSE, select_works = TRUE;

static void oss_calc_device_buffer_used(void)
{
	audio_buf_info buf_info;
	if(paused)
		device_buffer_used = 0;
	else if (!ioctl(fd, SNDCTL_DSP_GETOSPACE, &buf_info))
		device_buffer_used = (buf_info.fragstotal * buf_info.fragsize) - buf_info.bytes;
}


gint oss_downsample(gpointer ob, guint length, guint speed, guint espeed);

static void oss_setup_format(AFormat fmt,gint rate, gint nch)
{
	format = fmt;
	frequency = rate;
	channels = nch;
	switch (fmt)
	{
		case FMT_U8:
			oss_format = AFMT_U8;
			break;
		case FMT_S8:
			oss_format = AFMT_S8;
			break;
		case FMT_U16_LE:
			oss_format = AFMT_U16_LE;
			break;
		case FMT_U16_BE:
			oss_format = AFMT_U16_BE;
			break;
		case FMT_U16_NE:
#ifdef AFMT_U16_NE
			oss_format = AFMT_U16_NE;
#else
#ifdef WORDS_BIGENDIAN
			oss_format = AFMT_U16_BE;
#else
			oss_format = AFMT_U16_LE;
#endif
#endif
			break;
		case FMT_S16_LE:
			oss_format = AFMT_S16_LE;
			break;
		case FMT_S16_BE:
			oss_format = AFMT_S16_BE;
			break;
		case FMT_S16_NE:
#ifdef AFMT_S16_NE
			oss_format = AFMT_S16_NE;
#else
#ifdef WORDS_BIGENDIAN
			oss_format = AFMT_S16_BE;
#else
			oss_format = AFMT_S16_LE;
#endif
#endif
			break;
	}

	bps = rate * nch;
	if (oss_format == AFMT_U16_BE || oss_format == AFMT_U16_LE || oss_format == AFMT_S16_BE || oss_format == AFMT_S16_LE)
		bps *= 2;
	fragsize = 0;
	while ((1L << fragsize) < bps / 25)
		fragsize++;
	fragsize--;

	device_buffer_size = ((1L << fragsize) * (NFRAGS + 1));
}
	

gint oss_get_written_time(void)
{
	if (!going)
		return 0;
	return (gint) ((written * 1000) / input_bps);
}

gint oss_get_output_time(void)
{
	
	guint64 bytes;

	if (!fd || !going)
		return 0;

	if(realtime)
		oss_calc_device_buffer_used();
	bytes = output_bytes - device_buffer_used;

	return output_time_offset + (gint) ((bytes  * 1000) / ebps);
}

gint oss_used(void)
{
	if (realtime)
		return 0;
	else
	{
		if (wr_index >= rd_index)
			return wr_index - rd_index;
		return buffer_size - (rd_index - wr_index);
	}
}

gint oss_playing(void)
{
	if(!going)
		return 0;
	if(realtime)
		oss_calc_device_buffer_used();
	if (!oss_used() && (device_buffer_used - (3 * blk_size)) <= 0)
		return FALSE;

	return TRUE;
}

gint oss_free(void)
{
	if (!realtime)
	{
		if (remove_prebuffer && prebuffer)
		{
			prebuffer = FALSE;
			remove_prebuffer = FALSE;
		}
		if (prebuffer)
			remove_prebuffer = TRUE;

		if (rd_index > wr_index)
			return (rd_index - wr_index) - device_buffer_size - 1;
		return (buffer_size - (wr_index - rd_index)) - device_buffer_size - 1;
	}
	else
		if (paused)
			return 0;
		else
			return 1000000;
}

static void oss_write_audio(gpointer data,gint length)
{
	audio_buf_info abuf_info;
	AFormat new_format;
	gint new_frequency,new_channels;
	EffectPlugin *ep;
	
	new_format = input_format;
	new_frequency = input_frequency;
	new_channels = input_channels;
	
	ep = get_current_effect_plugin();
	if(effects_enabled() && ep && ep->query_format)
	{
		ep->query_format(&new_format,&new_frequency,&new_channels);
	}
	
	if(new_format != format || new_frequency != frequency || new_channels != channels)
	{
		output_time_offset += (gint) ((output_bytes * 1000) / ebps);
		output_bytes = 0;
		oss_setup_format(new_format, new_frequency, new_channels);
		frequency = new_frequency;
		channels = new_channels;
		close(fd);
		fd = open(device_name,O_WRONLY);
		oss_set_audio_params();
	}
	if(effects_enabled() && ep && ep->mod_samples)
		length = ep->mod_samples(&data,length, input_format, input_frequency, input_channels);
	if(realtime && !ioctl(fd,SNDCTL_DSP_GETOSPACE,&abuf_info))
	{
		while(abuf_info.bytes < length)
		{
			xmms_usleep(10000);
			if(ioctl(fd,SNDCTL_DSP_GETOSPACE,&abuf_info))
				break;
		}
	}
	if(frequency == efrequency)
		output_bytes += write(fd,data,length);
	else
		output_bytes += oss_downsample(data,length,frequency,efrequency);
}

static void swap_words(guint16 *buffer, gint length)
{
	guint16 *ptr = buffer;
	gint i;
	for(i = 0; i < length; i++, ptr++)
		*ptr = ((*ptr & 0x00FF) << 8) | (*ptr >> 8);
}


#define RESAMPLE_STEREO(sample_type, shift) \
{ \
        gint i, x1, in_samples, out_samples; \
	sample_type *inptr = (sample_type *)ob, *outptr;\
	gint x, frac, delta;\
	nlen = (((length >> shift) * espeed) / speed) << shift;\
	if(big_endian && (oss_format == AFMT_S16_LE || oss_format == AFMT_U16_LE)) \
		swap_words(ob, length >> 1);\
	else if(!big_endian && (oss_format == AFMT_S16_BE || oss_format == AFMT_U16_BE)) \
		swap_words(ob, length >> 1);\
	if(nlen > nbuffer_size)\
	{\
		nbuffer = g_realloc(nbuffer, nlen);\
		nbuffer_size = nlen;\
	}\
	outptr = (sample_type *)nbuffer; \
        in_samples = length >> shift; \
        out_samples = nlen >> shift; \
	delta = (in_samples << 12) / out_samples;\
	for (x = 0, i = 0; i < out_samples; i++)\
	{\
		x1 = (x >> 12) << 12;\
		frac = x - x1;\
		*outptr++ = (sample_type) ((inptr[(x1 >> 12) << 1] * ((1<<12) - frac) + inptr[((x1 >> 12) + 1) << 1] * frac) >> 12);\
		*outptr++ = (sample_type) ((inptr[((x1 >> 12) << 1) + 1] * ((1<<12) - frac) + inptr[(((x1 >> 12) + 1) << 1) + 1] * frac) >> 12);\
		x += delta;\
	}\
	if(big_endian && (oss_format == AFMT_S16_LE || oss_format == AFMT_U16_LE)) \
		swap_words(nbuffer, nlen >> 1);\
	else if(!big_endian && (oss_format == AFMT_S16_BE || oss_format == AFMT_U16_BE)) \
		swap_words(nbuffer, nlen >> 1);\
	w = write(fd, nbuffer, nlen);		  \
}

#define RESAMPLE_MONO(sample_type, shift) \
{ \
        gint i, x1, in_samples, out_samples; \
	sample_type *inptr = (sample_type *)ob, *outptr;\
	gint x, frac, delta;\
	nlen = (((length >> shift) * espeed) / speed) << shift;\
	if(big_endian && (oss_format == AFMT_S16_LE || oss_format == AFMT_U16_LE)) \
		swap_words(ob, length >> 1);\
	else if(!big_endian && (oss_format == AFMT_S16_BE || oss_format == AFMT_U16_BE)) \
		swap_words(ob, length >> 1);\
	if(nlen > nbuffer_size)\
	{\
		nbuffer = g_realloc(nbuffer, nlen);\
		nbuffer_size = nlen;\
	}\
	outptr = (sample_type *)nbuffer; \
        in_samples = length >> shift; \
        out_samples = nlen >> shift; \
	delta = (in_samples << 12) / out_samples;\
	for (x = 0, i = 0; i < out_samples; i++)\
	{\
		x1 = (x >> 12) << 12;\
		frac = x - x1;\
		*outptr++ = (sample_type) ((inptr[x1 >> 12] * ((1<<12) - frac) + inptr[(x1 >> 12) + 1] * frac) >> 12);\
		x += delta;\
	}\
	if(big_endian && (oss_format == AFMT_S16_LE || oss_format == AFMT_U16_LE)) \
		swap_words(nbuffer, nlen >> 1);\
	else if(!big_endian && (oss_format == AFMT_S16_BE || oss_format == AFMT_U16_BE)) \
		swap_words(nbuffer, nlen >> 1);\
	w = write(fd, nbuffer, nlen);		  \
}


gint oss_downsample(gpointer ob, guint length, guint speed, guint espeed)
{
	guint nlen, w;
	static gpointer nbuffer = NULL;
	static gint nbuffer_size = 0;
#ifdef WORDS_BIGENDIAN
	gboolean big_endian = TRUE;
#else
	gboolean big_endian = FALSE;
#endif

	if (oss_format == AFMT_S16_BE || oss_format == AFMT_S16_LE)
	{
		if(channels == 2)
			RESAMPLE_STEREO(gint16, 2)
		else
			RESAMPLE_MONO(gint16, 1)
	}
	else if (oss_format == AFMT_U16_BE || oss_format == AFMT_U16_LE)
	{
		if(channels == 2)
			RESAMPLE_STEREO(guint16, 2)
		else
			RESAMPLE_MONO(guint16, 1)
	}
	else if (oss_format == AFMT_S8)
	{
		if(channels == 2)
			RESAMPLE_STEREO(gint8, 1)
		else
			RESAMPLE_MONO(gint8, 0)
	}
	else if (oss_format == AFMT_U8)
	{
		if(channels == 2)
			RESAMPLE_STEREO(guint8, 1)
		else
			RESAMPLE_MONO(guint8, 0)
	}
	return w;
}

void oss_write(gpointer ptr, gint length)
{
	gint cnt, off = 0;

	if (!realtime)
	{
		remove_prebuffer = FALSE;

		written += length;
		while (length > 0)
		{
			cnt = MIN(length, buffer_size - wr_index);
			memcpy(buffer + wr_index, ptr + off, cnt);
			wr_index = (wr_index + cnt) % buffer_size;
			length -= cnt;
			off += cnt;

		}
	}
	else
	{
		if (paused)
			return;
		oss_write_audio(ptr,length);
		written += length;
		
	}

}

void oss_close(void)
{
	going = 0;
	if (!realtime)
		pthread_join(buffer_thread, NULL);
	else
	{
		ioctl(fd, SNDCTL_DSP_RESET, 0);
		close(fd);
	}
	wr_index = 0;
	rd_index = 0;
}

void oss_flush(gint time)
{
	if (!realtime)
	{
		flush = time;
		while (flush != -1)
			xmms_usleep(10000);
	}
	else
	{
		ioctl(fd, SNDCTL_DSP_RESET, 0);
		close(fd);
		fd = open(device_name, O_WRONLY);
		oss_set_audio_params();
		output_time_offset = time;
		written = (guint64)(time / 10) * (guint64)(input_bps / 100);		
		output_bytes = 0;
	}
}

void oss_pause(short p)
{
	if (!realtime)
	{
		if (p == TRUE)
			do_pause = TRUE;
		else
			unpause = TRUE;
	}
	else
		paused = p;

}

void *oss_loop(void *arg)
{
	gint length, cnt;
	fd_set set;
	struct timeval tv;

	while (going)
	{
		if (oss_used() > prebuffer_size)
			prebuffer = FALSE;
		if (oss_used() > 0 && !paused && !prebuffer)
		{
			tv.tv_sec = 0;
			tv.tv_usec = 10000;
			FD_ZERO(&set);
			FD_SET(fd, &set);
			if(!select_works || (select(fd + 1, NULL, &set, NULL, &tv) > 0))
			{
				length = MIN(blk_size, oss_used());
				while (length > 0)
				{
					cnt = MIN(length,buffer_size-rd_index);
					oss_write_audio(buffer + rd_index, cnt);
					rd_index=(rd_index+cnt)%buffer_size;
					length-=cnt;				
				}
				if (!oss_used())
				{
					ioctl(fd, SNDCTL_DSP_POST, 0);
				}
			}
		}
		else
		{
			xmms_usleep(10000);
		}
		oss_calc_device_buffer_used();
		if (do_pause && !paused)
		{
			do_pause = FALSE;
			paused = TRUE;
			rd_index -= device_buffer_used;
			output_bytes -= device_buffer_used;
			if (rd_index < 0)
				rd_index += buffer_size;
			ioctl(fd, SNDCTL_DSP_RESET, 0);

		}
		if (unpause && paused)
		{
			unpause = FALSE;
			close(fd);
			fd = open(device_name, O_WRONLY);
			oss_set_audio_params();
			paused = FALSE;
		}

		if (flush != -1)
		{
			/*
			 * This close and open is a work around of a bug that exists in some drivers which 
			 * cause the driver to get fucked up by a reset
			 */

			ioctl(fd, SNDCTL_DSP_RESET, 0);
			close(fd);
			fd = open(device_name, O_WRONLY);
			oss_set_audio_params();
			output_time_offset = flush;
			written = (guint64)(flush / 10) * (guint64)(input_bps / 100);
			rd_index = wr_index = output_bytes = 0;
			flush = -1;
			prebuffer = TRUE;
		}

	}

	ioctl(fd, SNDCTL_DSP_RESET, 0);
	close(fd);
	g_free(buffer);
	pthread_exit(NULL);
}

void oss_set_audio_params(void)
{
	gint frag, stereo, ret;
	struct timeval tv;
	fd_set set;
	
	ioctl(fd, SNDCTL_DSP_RESET, 0);
	frag = (NFRAGS << 16) | fragsize;
	ioctl(fd, SNDCTL_DSP_SETFRAGMENT, &frag);
	ioctl(fd, SNDCTL_DSP_SETFMT, &oss_format);
	ioctl(fd, SNDCTL_DSP_SETFMT, &oss_format);
	stereo = channels - 1;
	ioctl(fd, SNDCTL_DSP_STEREO, &stereo);
	efrequency = frequency;
	ioctl(fd, SNDCTL_DSP_SPEED, &efrequency);

	if(ioctl(fd, SNDCTL_DSP_GETBLKSIZE, &blk_size) == -1)
		blk_size = 1L << fragsize;

	ebps = efrequency * channels;
	if (oss_format == AFMT_U16_BE || oss_format == AFMT_U16_LE ||
	    oss_format == AFMT_S16_BE || oss_format == AFMT_S16_LE)
		ebps *= 2;

	/* Stupid hack to find out if the driver support selects,
	   some drivers won't work properly without a select and
	   some won't work with a select :/ */
	
	tv.tv_sec = 0;
	tv.tv_usec = 50000;
	FD_ZERO(&set);
	FD_SET(fd, &set);
	ret = select(fd + 1, NULL, &set, NULL, &tv);
	if(ret > 0)
		select_works = TRUE;
	else
		select_works = FALSE;
}

gint oss_open(AFormat fmt, gint rate, gint nch)
{
	oss_setup_format(fmt,rate,nch);
	
	input_format = format;
	input_channels = channels;
	input_frequency = frequency;
	input_bps = bps;

	realtime = xmms_check_realtime_priority();
	
	if(!realtime)
	{
		buffer_size = (oss_cfg.buffer_size * input_bps) / 1000;
		if (buffer_size < 8192)
			buffer_size = 8192;
		prebuffer_size = (buffer_size * oss_cfg.prebuffer) / 100;
		if (buffer_size - prebuffer_size < 4096)
			prebuffer_size = buffer_size - 4096;

		buffer_size += device_buffer_size;
		buffer = g_malloc0(buffer_size);
	}
	flush = -1;
	prebuffer = 1;
	wr_index = rd_index = output_time_offset = written = output_bytes = 0;
	paused = FALSE;
	do_pause = FALSE;
	unpause = FALSE;
	remove_prebuffer = FALSE;

	if (oss_cfg.audio_device > 0)
		device_name = g_strdup_printf("/dev/dsp%d", oss_cfg.audio_device);
	else
		device_name = g_strdup("/dev/dsp");
	fd = open(device_name, O_WRONLY);
	if (fd == -1)
	{
		g_free(buffer);
		return 0;
	}
	oss_set_audio_params();
	going = 1;
	if (!realtime)
		pthread_create(&buffer_thread, NULL, oss_loop, NULL);
	return 1;
}
