/* Virtualdir is a collection of routines managing virtual directory entries.
 * It is used to display the contents of the last session in gnometoaster. */

#include <gtk/gtk.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "virtualdir.h"

/* uncomment for debugging */
/* #define DEBUG */

/* create a new dir */
virtualdir_dir *virtualdir_create(/* informations about the file/directory
				   * to be created */
				  char *name,
				  unsigned int mode,
				  unsigned long int filesize)
{
   virtualdir_dir *result;
   
   result=(virtualdir_dir*)malloc(sizeof(virtualdir_dir));
   /* directory is empty */
   result->entries=NULL;
   strcpy((char*)&result->name,name);
   result->mode=mode;
   result->filesize=filesize;
   
   return result;
};

/* find entry with name in directory dir */
virtualdir_dir *virtualdir_findentry(virtualdir_dir *dir,
				     gchar *name)
{
   virtualdir_dir *result=NULL;
   GList *current;
   
#ifdef DEBUG
   printf ("virtualdir_findentry: scanning for %s ... ",name);
#endif	

   for (current=dir->entries;current!=NULL;current=current->next)
     {
	if (!strcmp(name,
		    (gchar*)&(((virtualdir_dir*)current->data)->name)))
	  {
	     result=(virtualdir_dir*)current->data; 
	  };
     };
   
#ifdef DEBUG
   if (result!=NULL)
     printf ("found.\n");
   else
     printf ("not found.\n");
#endif
   return result;
};

/* find a subdirectory of dir root */
virtualdir_dir *virtualdir_finddir(virtualdir_dir *root,
				   gchar *path)
{
   gchar *subpath;
   gchar *rest;
   virtualdir_dir *entry;
   virtualdir_dir *result;

#ifdef DEBUG
   printf ("virtualdir_finddir: seeking directory %s\n",path);
#endif
   
   /* get the "leftmost" dir name of the directory tree */
   subpath=(gchar*)malloc(strlen(path)+1);
   if (path[0]=='/')
     strcpy(subpath,&path[1]);
   else
     strcpy(subpath,path);
   
   /* find the end of the "leftmost" dir name */
   rest=strchr(subpath,'/');
   /* if terminated by '/',terminate the string with 0. */
   if (rest!=NULL)
     {      
	*rest=0;
	rest++;
     }	
   /* position pointer to rest string to stringend in case
    * of no more slashes,position behind the last slash otherwise */
   else     
     rest=subpath+strlen(subpath);
#ifdef DEBUG
   printf ("virtualdir_finddir: yet to search: %s\n",
	   rest);
#endif
   
   if (strlen(subpath)>0)
     {
	/* seek the desired directory entry */   
	entry=virtualdir_findentry(root,
				   subpath);
	/* see if a directory with that name exists */
	if (entry!=NULL)
	  {
	     /* if so, check if we have any more subtrees to get in */
	     if (strlen(rest)>0)
	       {		  
		  result=virtualdir_finddir(entry,rest);
#ifdef DEBUG
		  if (result!=NULL)
		    printf ("virtualdir_finddir: got result from rec. call %s\n",			    
			    (char*)&result->name);
		  else
		    printf ("virtualdir_finddir: got invalid result from rec. call\n");
#endif
	       }
	     /* in case we don't,the entry just found is the "rightmost"
	      * dir of the tree,hence what we're searching for */
	     else	  
	       result=entry;
	  }
	/* if a directory with the desired name couldn't be found,
	 * return NULL */
	else
	  result=NULL;
     }
   else
     /* if no directory name was given,return pointer to root directory */
     result=root;

   
#ifdef DEBUG
   printf ("virtualdir_finddir: freeing temporary data\n");
#endif
   /* free our temporary string */
   free(subpath);
   
#ifdef DEBUG
   if (result!=NULL)
     printf ("virtualdir_finddir: found %s\n",(char*)&result->name);
#endif   
   /* and return what we have found */
   return result;
};

/* open a subdirectory of dir root for reading */
virtualdir_VDIR *virtualdir_open(virtualdir_dir *root,
				 gchar *path)
{
   virtualdir_VDIR *result;
   virtualdir_dir *dir;
   
   result=(virtualdir_VDIR*)malloc(sizeof(virtualdir_VDIR));
   dir=virtualdir_finddir(root,path);
   if (dir==NULL)
     {
	free(result);
	result=NULL;	
     }
   else
     *result=dir->entries;
   return result;
};

/* read a directory entry. this call's structur is similar to
 * the readdir() libc function.  it will return 
 * a pointer to the direntry struct or NULL if the end of the directory
 * has been reached */
virtualdir_dir *virtualdir_read(virtualdir_VDIR *vdir)
{
   virtualdir_dir *result;
   
   if (*vdir!=NULL)
     {	
	result=(virtualdir_dir*)((*vdir)->data);
	*vdir=(*vdir)->next;
     }
   else
     result=NULL;

   return result;
};

/* close a VDIR opened by virtualdir_open */
void virtualdir_close(virtualdir_VDIR *vdir)
{
   free(vdir);
};

/* add a direntry structure to the virtual directory
 * structure specified by root in subdir path,
 * client should be created with virtualdir_create() */
void virtualdir_addentry(virtualdir_dir *root,
			 virtualdir_dir *client,
			 char *path)
{
   virtualdir_dir *destdir;
   
   destdir=virtualdir_finddir(root,path);
   /* if desired destination could be found */   
   if (destdir!=NULL)
     {	
#ifdef DEBUG
	printf ("virtualdir_addentry: adding client to %s\n",(char*)&destdir->name);
#endif
	destdir->entries=g_list_append(destdir->entries,(gpointer)client);
     };
};

void virtualdir_deletestructure(virtualdir_dir *root)
{
   
   /* delete recursively */
   while (root->entries!=NULL)
     {
	virtualdir_deletestructure((virtualdir_dir*)(root->entries->data));
	root->entries=g_list_remove(root->entries,root->entries->data);
     };	
   free (root);
};
  
