/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>

#include "globals.h"
#include "message.h"
#include "message-window.h"
#include "util.h"
#include "smtp.h"
#include "sockets.h"
#include "status-item.h"

extern const gchar* sockread_err_msg;
extern const gchar* sockwrite_err_msg;

/**
 * examine response from server.
 *
 * If a read error occurs, a sockread_err is emitted by the StatusItem.
 * If expected is >0 and differs from the return value, an error is emitted.
 * 
 * @param item the StatusItem to whom errors are emitted
 * @param sock the socket to read from.
 * @param expected the expected numeric value.
 * @return the received numeric value.
 */
static int
smtp_get_response (StatusItem *item, PanSocket *sock, int expected)
{
	const gchar *buf = NULL;
	int retval = 0;

	if (pan_socket_getline (sock, &buf)) {
		status_item_emit_error (item, sockread_err_msg);
		return -1;
	}

	retval = atoi(buf);
	if (expected>0 && retval!=expected)
		status_item_emit_error (
			item,
			"Got unexpected response from mail server: "
			"expected %d; got %s",
			expected,
			buf);

	return retval;
}

/**
 * send a line to the server.
 *
 * If a write error occurs, a sockwrite_err is emitted from StatusItem.
 *
 * @param item the StatusItem to whom errors are emitted
 * @param sock the socket to write to.
 * @param string a string to write.
 * @return zero on success (no write errors), nonzero on failure.
 */
static gint
smtp_send_line (StatusItem *item, PanSocket *sock, const char* str)
{
	int retval = 0;
	if (Pan.mute)
	{
		g_message ("(MUTE) to smtp: [%s]", str);
		retval = 0;
	}
	else
	{
		const int retval = pan_socket_putline (sock, str);
		if (retval)
			status_item_emit_error (item, sockwrite_err_msg);
	}
	return retval;
}


/**
 * send a line to the server.
 *
 * If a write error occurs, a sockwrite_err is emitted from StatusItem.
 *
 * @param item the StatusItem to whom errors are emitted
 * @param sock the socket to write to.
 * @param fmt a printf-style format string
 * @return zero on success (no write errors), nonzero on failure.
 */
static gint
smtp_send_line_va (StatusItem *item, PanSocket *sock, const char* fmt, ...)
{
	va_list ap;
	int retval = 0;
	char *line = NULL;

	va_start (ap, fmt);
	line = g_strdup_vprintf (fmt, ap);
	va_end (ap);
	retval = smtp_send_line (item, sock, line);
	g_free (line);

	return retval;
}


/**
 * Sends a message to the specified smtp socket.
 *
 * Possible emissions:
 * * If the "From" line is empty and there's no fallback in the
 *     Pan config file, a "No Sender Specified" error is emitted.
 * * If the server doesn't know any of the recipients, a "No Known Recipients"
 *     error is emitted.
 * * If no recipients are specified, a "No Recipients!" error is emitted.  
 * * If a socket write error occurs, a sockwrite_err is emitted.
 * * If a socket read error occurs, a sockread_err is emitted.
 *
 * @param item the StatusItem through which emissions are made.
 * @param socket the smtp socket where the message is mailed.
 * @param msg the message being sent.
 * @return zero on success, nonzero on failure
 *
 * FIXME: the tight coupling with the Message UI
 * FIXME: add wrap_long_lines after it's added to prefs
 */
static int
smtp_send_message (
	StatusItem *item,
	PanSocket *sock,
	Message *msg,
	const gchar* leader)
{
	gchar *from = NULL;
	gchar *subject = NULL;
	gchar *full_to = NULL;
	gchar *reply_to = NULL;
	gchar *body = NULL;
	gboolean okay = TRUE;
	GSList *l = NULL;
	gboolean maybe_okay = FALSE;

	g_return_val_if_fail (item!=NULL, -1);
	g_return_val_if_fail (sock!=NULL, -1);
	g_return_val_if_fail (msg!=NULL, -1);

	if (!msg->rcpts) {
		status_item_emit_error(item, _("No Recipients!"));
		return -1;
	}

	status_item_emit_status (item, _("Preparing to Send Mail"));

	/**
	***  Generate headers
	**/

	subject = gtk_editable_get_chars (
		GTK_EDITABLE (msg->window->subject), 0, -1);

	body = gtk_editable_get_chars (
		GTK_EDITABLE (msg->window->body), 0, -1);

	from = gtk_editable_get_chars (
		GTK_EDITABLE (msg->window->from), 0, -1);
	if (!from)
		from = gnome_config_get_string ("/Pan/User/Email");
	if (!from) {
		status_item_emit_error(item, _("No Sender Specified!"));
		okay = FALSE;
	}

	l = msg->rcpts;
	while (l) {
		if (full_to)
			full_to = g_strconcat (full_to, (gchar*)l->data, NULL);
		else
			full_to = g_strconcat ((gchar*)l->data, NULL);
		if (l->next)
			full_to = g_strconcat (full_to, ", ", NULL);
		l = l->next;
	}

	/* Build "Reply-To: " contents */
        if (GTK_IS_ENTRY(msg->window->reply_to))
		reply_to = gtk_editable_get_chars (GTK_EDITABLE (msg->window->reply_to), 0, -1);

	/**
	*** start sending a letter
	**/

	if (okay)
		status_item_emit_status (item, _("Sending Mail"));

	if (okay) {
		const char *fmt = "MAIL FROM: <%s>\r\n";
		okay = !smtp_send_line_va (item, sock, fmt, from);
	}
	if (okay)
		okay = Pan.mute || smtp_get_response(item, sock, 250)==250;

	/**
	*** tell the server who we're sending the letter to
	**/

	maybe_okay = FALSE;
	for (l=msg->rcpts; okay && l!=NULL; l=l->next)
	{
		if (okay)
		{
			const char *fmt = "RCPT TO: <%s>\r\n";
			okay = !smtp_send_line_va (item, sock, fmt, l->data);
		}
		if (okay)
		{
			const int retval = Pan.mute ? 250 : smtp_get_response(item, sock, -1);

			/* >=1 known recepient */
			maybe_okay = retval==250;

			/* keep going even if one recipient is unknown */
			okay = retval==250 || retval==550;
		}
	}

	/* don't proceed unless we have at least one known recipient */
	if (okay) {
		okay = maybe_okay;
		if (!okay)
			status_item_emit_error(item,
				_("No Known Recipients"));
	}

	/**
	*** tell the server we're about to start sending a letter
	**/

	if (okay)
		okay = !smtp_send_line (item, sock, "DATA\r\n");
	if (okay)
		okay = Pan.mute || smtp_get_response (item, sock, 354)==354;

	/**
	*** send the message
	**/

	if (okay)
		okay = !smtp_send_line_va (item, sock, "X-Mailer: Pan %s\n", VERSION);
	if (okay)
		okay = !smtp_send_line_va (item, sock, "From: %s\n", from);
	if (okay)
		okay = !smtp_send_line_va (item, sock, "To: %s\n", full_to);
	if (okay)
		okay = !smtp_send_line_va (item, sock, "Subject: %s\n", subject);
	if (okay && reply_to)
		okay = !smtp_send_line_va (item, sock, "Reply-To: %s\n", reply_to);
	if (okay)
		okay = !smtp_send_line (item, sock, "\r\n" );
	if (okay && leader && *leader)
		okay = !smtp_send_line_va (item, sock, "%s\n\n", leader);
	if (okay) {
		int i;
		gchar **sd = g_strsplit (body, "\n", -1);
		for (i=0; sd[i]!=NULL; ++i) {
	 		if (okay) okay = !smtp_send_line (item, sock, sd[i] );
			if (okay) okay = !smtp_send_line (item, sock, "\n" );
		}
		g_strfreev (sd);
	}
	if (okay)
		okay = !smtp_send_line (item, sock, ".\r\n");
	if (okay)
		okay = Pan.mute || smtp_get_response (item, sock, 250)==250;
	if (okay)
		status_item_emit_status (item, _("Mail Sent!"));
	
	/* clean up */
	g_free (from);
	g_free (full_to);
	g_free (reply_to);
	g_free (subject);
	g_free (body);

	return okay ? 0 : -1;
}


/* begin a SMTP session */
static int
smtp_handshake (StatusItem *item, PanSocket *sock)
{
	gchar my_name[256+1];
	struct hostent *hp;

	status_item_emit_status (item, _("Handshaking with mail server"));

	gethostname (my_name, sizeof(my_name));
	hp = gethostbyname (my_name);
	strcpy (my_name, hp->h_name);

	if (smtp_send_line_va (item, sock, "HELO %s\r\n", my_name))
		return -1;
	if (!Pan.mute && smtp_get_response (item, sock, 250) != 250)
		return -1;

	return 0;
}


/* end a SMTP session */
static void
smtp_disconnect (StatusItem *item, PanSocket *sock)
{
	if (sock != NULL)
	{
		//smtp_send_line (item, sock, ".\r\n");

		smtp_send_line (item, sock, "QUIT\r\n");

		if (!Pan.mute)
			smtp_get_response (item, sock, 221);
	}
}

/* open a connection to a mail server */
static PanSocket*
smtp_get_connection (StatusItem *item)
{
	PanSocket *sock = NULL;
	gchar *mailhost = NULL;
	gint mailport = -1;

	/* get smtp address/port */
	mailhost = gnome_config_get_string ("/Pan/Mail/smtp_address=localhost");
	mailport = gnome_config_get_int ("/Pan/Mail/smtp_port=25");
	status_item_emit_status (item, _("Connecting to mail server"));

	/* get the socket... */
	sock = pan_socket_new (mailhost, mailport);
	pan_object_sink (PAN_OBJECT(sock));
	if (sock->sockfd == 0) {
		status_item_emit_error (item,
			_("Unable to open connection to mail server ``%s''"),
			mailhost);
		g_free (mailhost);
		pan_object_unref (PAN_OBJECT(sock));
		sock = NULL;
		return NULL;
	}
	g_free (mailhost);

	/* get the "ready" message from the socket... */
	if (smtp_get_response (item, sock, 220) != 220) {
		status_item_emit_error (item, _("Mail Server Not Ready"));
		pan_object_unref (PAN_OBJECT(sock));
		sock = NULL;
	}

	/* handshake */
	if (smtp_handshake (item, sock)) {
		pan_object_unref (PAN_OBJECT(sock));
		sock = NULL;
	}

	return sock;
}

/***
****
****  PUBLIC FUNCTIONS
****
***/

gint
smtp_send (StatusItem *item, Message *msg, const gchar* leader)
{
	PanSocket *sock = NULL;
	int retval;

	/* sanity checks */
	g_return_val_if_fail (item!=NULL, -1);
	g_return_val_if_fail (msg!=NULL, -1);

	/* connect to the smtp server */
	sock = smtp_get_connection (item);
	if (!sock)
		return -1;

	/* send the message */
	retval = smtp_send_message (item, sock, msg, leader);

	/* say goodbye and disconnect */
	smtp_disconnect (item, sock);
	pan_object_unref (PAN_OBJECT(sock));
	return retval;
}
