
/***

othfilter.c - a menu-based module that allows selection of
	other protocols to display
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997, 1998

This software is open source; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License in the included COPYING file for
details.
			
***/

#include <curses.h>
#include <panel.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <sys/stat.h>
#include <netinet/udp.h>
#include <netinet/in.h>
#include <linux/if_ether.h>
#include "addproto.h"
#include "dirs.h"
#include "utfdefs.h"
#include "othfilter.h"
#include "utfilter.h"
#include "menurt.h"
#include "stdwinset.h"
#include "deskman.h"
#include "attrs.h"
#include "error.h"
#include "instances.h"

void makeothfiltermenu(struct MENU *menu)
{
    initmenu(menu, 12, 15, (LINES - 11) / 2, (COLS - 30) / 2 - 5);
    additem(menu, " ^U^DP...", "User Datagram Protocol");
    additem(menu, " ^I^CMP", "Internet Control Message Protocol");
    additem(menu, " ^O^SPF", "Open Shortest Path First");
    additem(menu, " I^G^P", "Interior Gateway Protocol");
    additem(menu, " IG^M^P", "Internet Group Management Protocol");
    additem(menu, " IG^R^P", "Interior Gateway Routing Protocol");
    additem(menu, " ^A^RP", "Address Resolution Protocol");
    additem(menu, " RAR^P^ ", "Reverse Address Resolution Protocol");
    additem(menu, " ^N^on-IP", "Non-IP packets");
    additem(menu, " E^x^it menu", "Return to main menu");
}

void setothfilter(struct othpoptions *filter, unsigned int row)
{
    int aborted;
    switch (row) {
    case 1:
	udpfilterselect(&(filter->fl), &(filter->filtercode),
			filter->filename, &aborted);

	if (!aborted) {
	    if (filter->filtercode != 0)
		filter->udp = 1;
	    else
		filter->udp = 0;
	}
	break;
    case 2:
	filter->icmp = ~(filter->icmp);
	break;
    case 3:
	filter->ospf = ~(filter->ospf);
	break;
    case 4:
	filter->igp = ~(filter->igp);
	break;
    case 5:
	filter->igmp = ~(filter->igmp);
	break;
    case 6:
	filter->igrp = ~(filter->igrp);
	break;
    case 7:
	filter->arp = ~(filter->arp);
	break;
    case 8:
	filter->rarp = ~(filter->rarp);
	break;
    case 9:
	filter->nonip = ~(filter->nonip);
	break;
    }
}

void blankoutentry(WINDOW * win)
{
    wprintw(win, "        ");
}

void toggleprotodisplay(WINDOW * win, struct othpoptions *filter,
			unsigned int row)
{
    wmove(win, row, 2);
    switch (row) {
    case 1:
	if (filter->udp)
	    wprintw(win, "UDP");
	else
	    blankoutentry(win);

	break;
    case 2:
	if (filter->icmp)
	    wprintw(win, "ICMP");
	else
	    blankoutentry(win);

	break;
    case 3:
	if (filter->ospf)
	    wprintw(win, "OSPF");
	else
	    blankoutentry(win);

	break;
    case 4:
	if (filter->igp)
	    wprintw(win, "IGP");
	else
	    blankoutentry(win);

	break;
    case 5:
	if (filter->igmp)
	    wprintw(win, "IGMP");
	else
	    blankoutentry(win);

	break;
    case 6:
	if (filter->igrp)
	    wprintw(win, "IGRP");
	else
	    blankoutentry(win);

	break;
    case 7:
	if (filter->arp)
	    wprintw(win, "ARP");
	else
	    blankoutentry(win);

	break;
    case 8:
	if (filter->rarp)
	    wprintw(win, "RARP");
	else
	    blankoutentry(win);

	break;
    case 9:
	if (filter->nonip)
	    wprintw(win, "Non-IP");
	else
	    blankoutentry(win);

	break;
    }
}

int othfilterok(struct othpoptions *filter, unsigned int protocol,
		unsigned long saddr, unsigned long daddr,
		unsigned int sport, unsigned int dport)
{
    int result = 0;

    switch (protocol) {
    case IPPROTO_UDP:
	if (filter->filtercode == 0)
	    result = 0;
	else if (filter->filtercode == 1)
	    result = 1;
	else if (filter->filtercode == 2)
	    if (utfilter(&(filter->fl), saddr, daddr, sport, dport))
		result = 1;
	break;
    case IPPROTO_ICMP:
	result = filter->icmp;
	break;
    case IPPROTO_OSPFIGP:
	result = filter->ospf;
	break;
    case IPPROTO_IGP:
	result = filter->igp;
	break;
    case IPPROTO_IGMP:
	result = filter->igmp;
	break;
    case IPPROTO_IGRP:
	result = filter->igrp;
	break;
    case ETH_P_ARP:
	result = filter->arp;
	break;
    case ETH_P_RARP:
	result = filter->rarp;
	break;
    case 0:
	result = filter->nonip;
	break;
    }

    return result;
}

void othfilterselect(struct othpoptions *filter)
{
    struct MENU menu;
    WINDOW *statwin;
    PANEL *statpanel;
    int row;
    int aborted;

    statwin = newwin(11, 23, (LINES - 11) / 2, (COLS - 30) / 2 + 10);
    statpanel = new_panel(statwin);
    wattrset(statwin, BOXATTR);
    colorwin(statwin);
    box(statwin, ACS_VLINE, ACS_HLINE);
    stdwinset(statwin);
    wmove(statwin, 0, 1);
    wprintw(statwin, " Visible protocols ");
    wattrset(statwin, STDATTR);
    update_panels();
    doupdate();

    for (row = 1; row <= 11; row++)
	toggleprotodisplay(statwin, filter, row);

    makeothfiltermenu(&menu);

    row = 1;
    do {
	showmenu(&menu);
	operatemenu(&menu, &row, &aborted);
	if (row < 12) {
	    setothfilter(filter, row);
	    toggleprotodisplay(statwin, filter, row);
	}
    } while (row != 10);

    destroymenu(&menu);
    del_panel(statpanel);
    delwin(statwin);
    update_panels();
    doupdate();
}

void setodefaults(struct othpoptions *filter)
{
    bzero(filter, sizeof(struct othpoptions));
    filter->filtercode = 1;
    filter->udp = filter->icmp = filter->ospf = filter->nonip = (u_int16_t) ~ 0;
    filter->arp = filter->rarp = (u_int16_t) ~ 0;
}

void loadothfilter(struct othpoptions *filter)
{
    int pfd;
    int br;

    pfd = open(OTHFLNAME, O_RDONLY);	/* open filter record file */

    if (pfd < 0) {
	setodefaults(filter);
	return;
    }
    br = read(pfd, filter, sizeof(struct othpoptions));
    if (br < 0)
	setodefaults(filter);

    close(pfd);

    /* Reload custom UDP filter is one was previously set */

    if (filter->filtercode == 2)
	loadfilter(filter->filename, &(filter->fl), FLT_RESOLVE);
}

void saveothfilter(struct othpoptions *filter)
{
    int pfd;
    int bw;
    int resp;

    if (!facility_active(OTHFLTIDFILE))
	mark_facility(OTHFLTIDFILE, "Non-TCP filter change");
    else {
	errbox("Non-TCP data file currently in use; try again later",
	       ANYKEY_MSG, &resp);
	return;
    }

    pfd = open(OTHFLNAME, O_CREAT | O_TRUNC | O_WRONLY, S_IRUSR | S_IWUSR);
    bw = write(pfd, filter, sizeof(struct othpoptions));
    if (bw < 1)
	errbox("Unable to write non-TCP filter", ANYKEY_MSG, &resp);

    close(pfd);

    unlink(OTHFLTIDFILE);
}
