/**************************************************************************** 
**
** File: ipgrab.c
**
** Author: Mike Borella
**
** Comments: Sniffs all packets on the link and dumps the fields of
**	the data link, IP, TCP, and UDP headers.  
**
*****************************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <pcap.h>
#include "config.h"
#include "ipgrab.h"
#include "open_pcap.h"
#include "addrtoname.h"
#include "loopback.h"
#include "ethernet.h"
#include "slip.h"
#include "raw.h"
#include "error.h"

char    *pcap_cmd;         /* command string for pcap */
pcap_t  *pd;               /* pcap device descriptor */
int     cnt;               /* number of packets to read */
int     datalink;          /* data link layer type */
pcap_handler dev_prcsr;    /* ptr to func that processes packet for a device */
struct arg_t *my_args;     /* Command line arguments */

extern char version[];

/*----------------------------------------------------------------------------
 *
 * main()
 *
 *----------------------------------------------------------------------------
 */

int main(int argc, char *argv[])
{
  u_char *userdata;

  /*
   * Clear device (interface) handle, user data and packet count
   */

  userdata = NULL;
  cnt = -1;

  /*
   * Parse command line for options
   */

  my_args = parse_cl(argc, argv);
  if (my_args->c) 
    cnt = my_args->c;

  /*
   * Make stdout buffered, if necessary
   */

  if (my_args->b)
#ifdef HAVE_SETLINEBUF
    setlinebuf(stdout);
#else
    setvbuf(stdout, NULL, _IOLBF, 0);
#endif


  /* 
   * Copy filter command into a string 
   */

  pcap_cmd = copy_argv(&argv[my_args->optind]);

  /*
   * Open the pcap device for sniffing
   */

  open_pcap();

  /*
   * Get rid of root privs
   */

  setuid(getuid());

  /*
   * Initialize the protocol name and ID lookup tables
   */

  init_addrtoname();

  /*
   * Print intro stuff to stderr so output files have consistent 
   * format
   */

  fprintf(stderr, "%s %s\n", 
	  argv[0], version);
  fprintf(stderr, "Listening on device %s ", my_args->i);

  /* 
   * Decide which processing function to use based on datalink type
   */

  switch(datalink)
    {
    case DLT_NULL:
      dev_prcsr = (pcap_func_t) dump_loopback;
      fprintf(stderr,"(loopback)\n");
      break;

    case DLT_EN10MB:
      dev_prcsr = (pcap_func_t) dump_ethernet;
      fprintf(stderr, "(ethernet)\n");
      break;

    case DLT_SLIP:
      dev_prcsr = (pcap_func_t) dump_slip;
      fprintf(stderr, "(slip)\n");
      break;

#ifdef DLT_RAW /* Not supported in some arch or older pcap versions */
    case DLT_RAW:
      dev_prcsr = (pcap_func_t) dump_raw;
      fprintf(stderr, "(raw)\n");
      break;
#endif

    default:
      GWF_error_fatal("\n%s cannot handle data link type %d", argv[0], 
		      datalink);
    }

  /*
   * Read all packets on the device.  Continue until cnt packets read 
   */
  
  if (pcap_loop(pd, cnt, dev_prcsr, userdata) < 0)
    GWF_error_fatal("pcap_loop: %s", pcap_geterr(pd));
  
  /*
   * XXX Get stats here
   */

  
  /*
   * Close the pcap device
   */

  pcap_close(pd);

  exit(0);
}


/*----------------------------------------------------------------------------
 *
 * copy_argv()
 *
 * Copy arg vector into a new buffer, concatenating arguments with spaces.
 * Lifted from tcpdump.
 *
 *----------------------------------------------------------------------------
 */

char *copy_argv(char **argv)
{
  char **p;
  u_int len = 0;
  char *buf;
  char *src, *dst;
  
  p = argv;
  if (*p == 0) return 0;

  while (*p)
    len += strlen(*p++) + 1;

  buf = (char *) malloc (len);
  if (buf == NULL) 
    GWF_error_system("copy_argv: malloc() failed");
      
  p = argv;
  dst = buf;
  while ((src = *p++) != NULL) 
    {
      while ((*dst++ = *src++) != '\0');
      dst[-1] = ' ';
    }
  dst[-1] = '\0';
  
  return buf;
}
