/*
 *  Copyright (C) 1999 Bruno Pires Marinho
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include "gtm.h"
#include "file-data.h"
#include "dialogs.h"
#include "download-info.h"
#include "main-window.h"

/* Structure that keeps the widgets containing the current properties */
typedef struct 
{
    GtkWidget *pbox;                            /* GNOME Property Box */
    GtkWidget *url;                             /* Entry */
    GtkWidget *disable_proxy;                   /* Toggle button */
    GtkWidget *disable_auto_dl;                 /* Toggle button */
} PropDialogData;


/* Callback functions used to update the current properties */
static void
apply_prop (GtkWidget *widget, gint page_num, PropDialogData *prop)
{
    FileData *file_data;
    gchar *url, *filename;
    GtkWidget *error;

    /* If the page is not valid leave function */
    if (page_num == -1)
        return;

    /* The information of the file we are changing properties */
    file_data = (FileData *) 
        gtk_clist_get_row_data (GTK_CLIST (dl_file_list), 
                                GTK_CLIST (dl_file_list)->focus_row);

    /* This function is called for each page changed so we need to apply prefs
     * for each page individually */
    switch (page_num) {
        case 0: /* Download */
            /* Get the URL data and make sure it's right */
            url = g_strdup (gtk_entry_get_text (GTK_ENTRY (prop->url)));
            if (strncmp (url, "http://", 7) 
                && strncmp (url, "ftp://", 6)) {
                error = gnome_message_box_new (
                    _ ("Invalid URL"), GNOME_MESSAGE_BOX_ERROR, 
                    GNOME_STOCK_BUTTON_OK, NULL);
                gnome_dialog_set_parent (GNOME_DIALOG (error), 
                                         GTK_WINDOW (prop->pbox));
                gnome_dialog_run_and_close (GNOME_DIALOG (error));
                g_free (url);
                gtk_entry_set_text (GTK_ENTRY (prop->url), file_data->url);
            } else {
                /* Get download filename data and check it is the same */
                filename = &url[strlen (url)];
                while (*filename != '/' && filename != url)
                    filename--;
                filename++;
                if (strcmp (file_data->filename, filename) != 0) {
                    error = gnome_message_box_new (
                        _ ("Old URL filename must be the same on the new URL"),
                        GNOME_MESSAGE_BOX_ERROR, GNOME_STOCK_BUTTON_OK, NULL);
                    gnome_dialog_set_parent (GNOME_DIALOG (error), 
                                             GTK_WINDOW (prop->pbox));
                    gnome_dialog_run_and_close (GNOME_DIALOG (error));
                    g_free (url);
                    gtk_entry_set_text (GTK_ENTRY (prop->url), file_data->url);
                } else {
                    file_data_set_url (file_data, url);
                    g_free (url);
                }
            }
            file_data_set_disable_auto_dl (file_data,
                                           gtk_toggle_button_get_active (
                                               GTK_TOGGLE_BUTTON (
                                                   prop->disable_auto_dl)));
            break;

        case 1: /* Proxy */
            file_data_set_disable_proxy (file_data,
                                         gtk_toggle_button_get_active (
                                             GTK_TOGGLE_BUTTON (
                                                 prop->disable_proxy)));
            break;            
    }
    download_info_update (file_data);
}

static void
destroy_propbox (GtkWidget *widget, PropDialogData **prop)
{
    gtk_widget_destroy ((*prop)->pbox);
    g_free (*prop);
    *prop = NULL;
}

static void
prop_modified_cb (GtkWidget *widget, PropDialogData *prop)
{
    /* Tell to property box that data changed */
    gnome_property_box_changed (GNOME_PROPERTY_BOX (prop->pbox));
}


/* Function that updates the information of the properties box */
static void
dialog_prop_update (PropDialogData *prop, FileData *file_data)
{
    if (file_data != NULL) {
	gtk_toggle_button_set_active 
	    (GTK_TOGGLE_BUTTON (prop->disable_proxy),
	     file_data->disable_proxy);
	if (gtm_pref.use_proxy)
	    gtk_widget_set_sensitive (prop->disable_proxy, TRUE);
	else
	    gtk_widget_set_sensitive (prop->disable_proxy, FALSE);
        gtk_entry_set_text (GTK_ENTRY (prop->url), file_data->url);
        gtk_widget_set_sensitive (prop->url, TRUE);
	gtk_toggle_button_set_active 
	    (GTK_TOGGLE_BUTTON (prop->disable_auto_dl),
	     file_data->disable_auto_dl);
        gtk_widget_set_sensitive (prop->disable_auto_dl, TRUE);
    } else {
	gtk_widget_set_sensitive (prop->disable_proxy, FALSE);
	gtk_widget_set_sensitive (prop->url, FALSE);
	gtk_widget_set_sensitive (prop->disable_auto_dl, FALSE);
    }

    /* If an update was made we need to disable the OK and APPLY buttons */
    gnome_property_box_set_modified (GNOME_PROPERTY_BOX (prop->pbox), FALSE);
}


/* Functions to create notebook page and set it's contents to the program
 * current preferences */
static GtkWidget *
create_proxy_page (PropDialogData *prop)
{
    GtkWidget *vbox;

    /* Create vbox for page lines */
    vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD);

    /* Disable proxy */
    prop->disable_proxy = 
	gtk_check_button_new_with_label (_ ("Disable proxy"));
    gtk_box_pack_start (GTK_BOX (vbox), prop->disable_proxy, 
                        FALSE, FALSE, 0);

    /* Connect the signal that acknowledges the change on the widget to 
     * function that turns on the OK and APPLY buttons */
    gtk_signal_connect (GTK_OBJECT (prop->disable_proxy), "toggled", 
			GTK_SIGNAL_FUNC (prop_modified_cb), 
			(gpointer) prop);

    return vbox;
}

static GtkWidget *
create_download_page (PropDialogData *prop)
{
    GtkWidget *vbox, *hbox, *label;

    /* Create vbox for page lines */
    vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD);

    /* URL */
    hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    label = gtk_label_new (_ ("URL:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    prop->url = gtk_entry_new ();
    gtk_box_pack_start (GTK_BOX (hbox), prop->url, TRUE, TRUE, 0);

    /* Disable auto download */
    prop->disable_auto_dl = 
	gtk_check_button_new_with_label (_ ("Disable auto download"));
    gtk_box_pack_start (GTK_BOX (vbox), prop->disable_auto_dl, 
                        FALSE, FALSE, 0);

    /* Connect the signal that acknowledges the change on the widget to 
     * function that turns on the OK and APPLY buttons */
    gtk_signal_connect (GTK_OBJECT (prop->url), "changed", 
			GTK_SIGNAL_FUNC (prop_modified_cb), 
			(gpointer) prop);
    gtk_signal_connect (GTK_OBJECT (prop->disable_auto_dl), "toggled", 
			GTK_SIGNAL_FUNC (prop_modified_cb), 
			(gpointer) prop);

    return vbox;
}


/* Function to create and run the properties dialog box */
void
dialog_prop (gboolean open_dialog, FileData *file_data)
{
    GtkWidget *label;
    static GnomeHelpMenuEntry help_entry = { "gtm", "index.html#PROP" };
    static PropDialogData *prop_dialog_data;

    /* Only one properties dialog box is allowed */
    if (prop_dialog_data != NULL) {
	/* Already open dialog box */
	gdk_window_raise (GTK_WIDGET (
	    GNOME_DIALOG (prop_dialog_data->pbox))->window);
	if (open_dialog == FALSE)
	    /* Update the open dialog box */
	    dialog_prop_update (prop_dialog_data, file_data);
	return;
    }

    if (open_dialog == FALSE)
	/* It's an update if open so leave because it's not open */
	return;

    /* Create the gnome property box and set the common signals */
    prop_dialog_data = g_new (PropDialogData, 1);
    prop_dialog_data->pbox = gnome_property_box_new ();
    gtk_window_set_title (GTK_WINDOW (prop_dialog_data->pbox), 
			  _ ("GTM - Properties"));
    gnome_dialog_set_parent (GNOME_DIALOG (prop_dialog_data->pbox), 
			     GTK_WINDOW (gtm_app));
    gtk_signal_connect (GTK_OBJECT (prop_dialog_data->pbox), "destroy",
			GTK_SIGNAL_FUNC (destroy_propbox), 
			(gpointer) &prop_dialog_data);
    gtk_signal_connect (GTK_OBJECT (prop_dialog_data->pbox), "apply",
			GTK_SIGNAL_FUNC (apply_prop), 
			(gpointer) prop_dialog_data);
    gtk_signal_connect (GTK_OBJECT (prop_dialog_data->pbox), "help",
			GTK_SIGNAL_FUNC (gnome_help_pbox_goto), &help_entry);

    /* Create the notebook pages */
    label = gtk_label_new (_ ("Download"));
    gtk_notebook_append_page (
	GTK_NOTEBOOK (GNOME_PROPERTY_BOX (prop_dialog_data->pbox)->notebook), 
	create_download_page (prop_dialog_data), label);
    label = gtk_label_new (_ ("Proxy"));
    gtk_notebook_append_page (
	GTK_NOTEBOOK (GNOME_PROPERTY_BOX (prop_dialog_data->pbox)->notebook), 
	create_proxy_page (prop_dialog_data), label);

    /* We are ready to show it all */
    gtk_widget_show_all (prop_dialog_data->pbox);

    /* Update the properties dialog box information */
    dialog_prop_update (prop_dialog_data, file_data);
}
