/*****************************************************************************/
/*  view_dialog.c - view dialog box and ftp routines                         */
/*  Copyright (C) 1998-1999 Brian Masney <masneyb@seul.org>                  */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "ftp.h"

static void delete_file (GtkWidget *widget, char *fle);
static struct viewedit_data *fork_process (char *proc, char *filename, char *remote_filename, int viewedit, int del_file);
static struct ftp_file_data *curfle;

void view_dialog (gpointer data) {
   struct ftp_transfer_data *newtdata;
   struct ftp_window_data *wdata;
   GList *templist;
   char *tempstr;
   
   wdata = (struct ftp_window_data *) data;
   if (!check_status (_("View"), wdata, 1, 1, 1)) return;

   if ((templist = get_next_selected_filename (wdata->hdata->files)) == NULL) {
      ftp_log (gftp_logging_misc, NULL, _("Internal gFTP Error: Could not find a selected file. This is probably a bug. Please email masneyb@seul.org about it\n"));
      return;
   }
   curfle = (struct ftp_file_data *) templist->data;

   if (curfle->isdir) {
      ftp_log (gftp_logging_misc, NULL, _("View: %s is a directory. Cannot view it.\n"), curfle->file);
      return;
   }
   
   if (wdata->local == 1) view_file (curfle->file, 1, 0, 1, NULL);
   else {
      tempstr = make_temp_filename (strchr (curfle->file, '.'));
      newtdata = transfer_one_file (tempstr, curfle->file, 1);
      g_free (tempstr);
      ((struct ftp_file_data *) newtdata->hdata->files->data)->done_view = 1;
      add_file_transfer (newtdata);
   }
}
/*****************************************************************************/
void edit_dialog (gpointer data) {
   struct ftp_transfer_data *newtdata;
   struct ftp_window_data *wdata;
   GList *templist;
   char *tempstr;
   
   wdata = (struct ftp_window_data *) data;
   if (!check_status (_("Edit"), wdata, 1, 1, 1)) return;

   if (*edit_program == '\0') {
      ftp_log (gftp_logging_misc, NULL, _("Edit: You must specify an editor in the options dialog\n"));
      return;
   }

   if ((templist = get_next_selected_filename(wdata->hdata->files)) == NULL) {
      ftp_log (gftp_logging_misc, NULL, _("Internal gFTP Error: Could not find a selected file. This is probably a bug. Please email masneyb@seul.org about it\n"));
      return;
   }
   curfle = (struct ftp_file_data *) templist->data;

   if (curfle->isdir) {
      ftp_log (gftp_logging_misc, NULL, _("Edit: %s is a directory. Cannot edit it.\n"), curfle->file);
      return;
   }
   
   if (wdata->local == 1) view_file (curfle->file, 0, 0, 1, NULL);
   else {
      tempstr = make_temp_filename (strchr (curfle->file, '.'));
      newtdata = transfer_one_file (tempstr, curfle->file, 1);
      g_free (tempstr);
      ((struct ftp_file_data *) newtdata->hdata->files->data)->done_edit = 1;
      add_file_transfer (newtdata);
   }
}
/*****************************************************************************/
void view_file (char *filename, int viewedit, int del_file, int start_pos, char *remote_filename) {
   GtkWidget *dialog, *view, *table, *tempwid;
   struct viewedit_data *newproc;
   char tempstr[MAXSTR], *fle;
   struct pix_ext *tempext;
   int stlen;
   FILE *fd;

   stlen = strlen (filename);
   tempext = registered_exts;
   while (tempext != NULL) {
      if (stlen >= tempext->stlen &&
         strcmp (&filename[stlen-tempext->stlen], tempext->ext) == 0) {

         if (*tempext->view_program == '\0') break;
         ftp_log (gftp_logging_misc, NULL, _("Opening %s with %s\n"), filename, tempext->view_program);
         fork_process (tempext->view_program, filename, remote_filename, viewedit, del_file);
         return;
      }
      tempext = tempext->next;
   }
   if (viewedit && *view_program != '\0') {
      /* Open the file with the default file viewer */
      fork_process (view_program, filename, remote_filename, viewedit, del_file);
      return;
   }
   else if (!viewedit) {
      /* Open the file with the default file editor */
      newproc = fork_process (edit_program, filename, remote_filename, viewedit, del_file);
      stat (filename, &newproc->st);
      return;
   }

   if ((fd = fopen (filename, "r")) == NULL) {
      ftp_log (gftp_logging_misc, NULL, _("View: Cannot open file %s: %s\n"), filename, g_strerror (errno));
      return;
   }
   ftp_log (gftp_logging_misc, NULL, _("Viewing file %s\n"), filename);

   dialog = gtk_dialog_new ();
   gtk_window_set_title (GTK_WINDOW (dialog), filename);
   gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), 5);
   gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)->action_area), 5);
   gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 5);
   gtk_box_set_homogeneous (GTK_BOX (GTK_DIALOG (dialog)->action_area), TRUE);
   gtk_signal_connect_object (GTK_OBJECT (dialog), "delete_event", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));

   table = gtk_table_new (1, 2, FALSE);
   gtk_widget_set_usize (table, 500, 400);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), table, TRUE, TRUE, 0);

   view = gtk_text_new (NULL, NULL);
   gtk_text_set_editable (GTK_TEXT (view), FALSE);
   gtk_text_set_word_wrap (GTK_TEXT (view), TRUE);
   gtk_table_attach (GTK_TABLE (table), view, 0, 1, 0, 1,
   	GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 0);
   gtk_widget_show (view);

   tempwid = gtk_vscrollbar_new (GTK_TEXT (view)->vadj);
   gtk_table_attach (GTK_TABLE (table), tempwid, 1, 2, 0, 1,
   	GTK_FILL, GTK_EXPAND | GTK_FILL | GTK_SHRINK, 0, 0);
   gtk_widget_show (tempwid);
   gtk_widget_show (table);

   tempwid = gtk_button_new_with_label (_("  Close  "));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, FALSE, FALSE, 0);
   if (del_file) {
      fle = g_malloc (strlen (filename) + 1);
      strcpy (fle, filename);
      gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (delete_file), (gpointer) fle);
   }
   gtk_signal_connect_object (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
   gtk_widget_show (tempwid);

   memset (tempstr, 0, sizeof (tempstr));
   while (fread (tempstr, 1, sizeof (tempstr) - 1, fd)) {
      gtk_text_insert (GTK_TEXT (view), NULL, NULL, NULL, tempstr, strlen (tempstr));
      memset (tempstr, 0, sizeof (tempstr));
   }
   fclose (fd);
   gtk_widget_show (dialog);
   if (!start_pos) {
      gtk_adjustment_set_value (GTK_TEXT(view)->vadj, 
      	GTK_TEXT (view)->vadj->upper - GTK_TEXT (view)->vadj->page_size);
   }
}
/*****************************************************************************/
static struct viewedit_data *fork_process (char *proc, char *filename, char *remote_filename, int viewedit, int del_file) {
   struct viewedit_data *newproc;
   char *pos, *endpos, **argv;
   pid_t ret;
   int n;
   
   argv = NULL;
   n = 0;
   pos = proc;
   while ((endpos = strchr (pos, ' ')) != NULL) {
      *endpos = '\0';
      n++;
      argv = g_realloc (argv, n * sizeof (char *));
      argv[n - 1] = g_malloc (strlen (pos) + 1);
      strcpy (argv[n - 1], pos);
      *endpos = ' ';
      pos = endpos + 1;
   }
   argv = g_realloc (argv, (n+3) * sizeof (char *));
   argv[n] = g_malloc (strlen (pos) + 1);
   strcpy (argv[n], pos);
   argv[n + 1] = g_malloc (strlen (filename) + 1);
   strcpy (argv[n + 1], filename);
   argv[n + 2] = NULL;
   
   if ((endpos = strchr (proc, ' ')) != NULL) *endpos = '\0';
   newproc = NULL;
   switch ((ret = fork ())) {
      case 0:
         execvp (proc, argv);
         _exit (1);
      case -1:
         for (n=0; argv[n] != NULL; n++) g_free (argv[n]);
         ftp_log (gftp_logging_misc, NULL, _("View: Cannot fork another process: %s\n"), g_strerror (errno));
         break;
      default:
         if (endpos != NULL) *endpos = ' ';
         ftp_log (gftp_logging_misc, NULL, _("Running program: %s %s\n"), proc, filename);
         newproc = g_malloc0 (sizeof (struct viewedit_data));
         newproc->pid = ret;
         newproc->argv = argv;
         newproc->filename = g_malloc (strlen (filename) + 1);
         strcpy (newproc->filename, filename);
         if (remote_filename != NULL) {
            newproc->remote_filename = g_malloc (strlen (remote_filename) + 1);
            strcpy (newproc->remote_filename, remote_filename);
         }
         newproc->view = viewedit;
         if (del_file) newproc->rm = 1;
         viewedit_processes = g_list_append (viewedit_processes, newproc);
   }
   return (newproc);
}
/*****************************************************************************/
static void delete_file (GtkWidget *widget, char *fle) {
   unlink (fle);
   g_free (fle);
}
/*****************************************************************************/

