/* Bezerk
 * Copyright (C) 1998 Tony Gale.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <gtk/gtk.h>

#include "bezerk.h"
#include "menu.h"
#include "callback.h"
#include "reply.h"
#include "prefs.h"
#include "servers.h"
#include "persona.h"
#include "dialogs.h"
#include "irc.h"
#include "util.h"
#include "debug.h"
#include "version.h"

#include "copy.xpm"
#include "paste.xpm"
#include "connect.xpm"

#ifdef HAVE_TCL
#include <tcl.h>
#include "tcl_script.h"

Tcl_Interp *interp;
#endif

char *conn_status_msg[] = {
  "Connected",
  "Not Registered",
  "ZzzZZzz"
};

/* ------------------------------------------------------------
 * reset_usize
 *
 *  Reset the usize of a widget to 0,0
 *
 * Arguments:
 *	"data" - The widget object
 *
 * Results:
 *	FALSE, to prevent further calls as an idle function
 *
 * Side effects:
 *	
 * ----------------------------------------------------------- */

static gint reset_usize(gpointer data)
{ 
  gtk_widget_set_usize(GTK_WIDGET(data), 0, 0);
  return FALSE;
}

/* ----------------------------------------------------------- */

Toolbar *create_toolbar (BezWindow *window)
{
  Toolbar *toolbar;
  GtkWidget *button;
  GList *cbitems = NULL;

  toolbar = g_malloc(sizeof(Toolbar));
  toolbar->toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
  gtk_toolbar_append_space (GTK_TOOLBAR (toolbar->toolbar));

  gtk_toolbar_append_item (GTK_TOOLBAR (toolbar->toolbar),
			   "Copy", "Copy to clipboard", "Toolbar/copy",
			   create_pixmap (copy_xpm, window->channel.window->window,
					  &window->channel.window->style->bg[GTK_STATE_NORMAL]),
			   (GtkSignalFunc) toolbar_copy_callback, window);

  button = gtk_toolbar_append_item (GTK_TOOLBAR (toolbar->toolbar),
			   "Paste", "Paste from clipboard", "Toolbar/paste",
			   create_pixmap (paste_xpm, window->channel.window->window,
					  &window->channel.window->style->bg[GTK_STATE_NORMAL]),
			   (GtkSignalFunc) toolbar_paste_callback, window);

  gtk_signal_connect (GTK_OBJECT(button), "selection_received",
                           GTK_SIGNAL_FUNC (toolbar_selection_received), window);

  gtk_toolbar_append_space (GTK_TOOLBAR (toolbar->toolbar));

  gtk_toolbar_append_item (GTK_TOOLBAR (toolbar->toolbar),
			   "Connect", "Connect to server", "Toolbar/connect",
			   create_pixmap (connect_xpm, window->channel.window->window,
					  &window->channel.window->style->bg[GTK_STATE_NORMAL]),
			   (GtkSignalFunc) toolbar_connect_callback, window);
  gtk_toolbar_append_space (GTK_TOOLBAR (toolbar->toolbar));

  cbitems = servers_get_fav_list();

  toolbar->combo = gtk_combo_new ();
  gtk_toolbar_append_widget (GTK_TOOLBAR (toolbar->toolbar), toolbar->combo,
			     "Favourite servers", "Toolbar/combo");
  gtk_combo_disable_activate (GTK_COMBO (toolbar->combo));
  gtk_signal_connect (GTK_OBJECT (GTK_COMBO(toolbar->combo)->entry), "activate",
		      (GtkSignalFunc) toolbar_connect_callback, window);    
  gtk_combo_set_popdown_strings (GTK_COMBO (toolbar->combo), cbitems);
  gtk_widget_show (toolbar->combo);

  gtk_widget_show (toolbar->toolbar);

  return(toolbar);
}
/* ------------------------------------------------------------
 * create_channel_window
 *
 *  Create a new toplevel window 
 *
 * Arguments:
 *	"title" - the title string for the window
 *      "top_level" - boolean indicating whether the window is a top level window
 *
 * Results:
 *	an BezWindow object containing the window information
 *
 * Side effects:
 *	the new window is shown on the screen
 *
 * ----------------------------------------------------------- */

BezWindow *create_channel_window(char *title, gint top_level)
{
  BezWindow *new_window;
  GtkWidget *main_vbox;
  GtkWidget *main_hbox;
  GtkWidget *sub_vbox;
  GtkWidget *sub_hbox;
  GtkWidget *list_vbox;
  GtkWidget *vscrollbar;
  GtkStyle *style;
  
  static char *ch_title[] = { "Channels" };
  static char *nick_title[] = { "Members" };
  static char *notify_title[] = { "Notify" };

  new_window = g_malloc(sizeof(BezWindow));
  new_window->channel.type = BEZ_CHANNEL;
  new_window->channel.current_channel = NULL;
  new_window->channel.numlines = 0;
  new_window->channel.recall_commands=NULL;
  new_window->channel.recall_current=NULL;
  new_window->channel.incomplete_command=NULL;
  new_window->channel.selected_row=-1;
  new_window->channel.connection=NULL;

  new_window->channel.window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_widget_realize(new_window->channel.window);
  if (top_level) {
    gtk_signal_connect(GTK_OBJECT(new_window->channel.window), "destroy", 
		       GTK_SIGNAL_FUNC(destroy_main), 
		       new_window);
  } else {
    gtk_signal_connect (GTK_OBJECT(new_window->channel.window), "destroy",
			GTK_SIGNAL_FUNC(destroy_channel_window),
			new_window);
  }

  gtk_window_set_title(GTK_WINDOW(new_window->channel.window), title);
  gtk_widget_set_usize(GTK_WIDGET(new_window->channel.window),
		       ( gdk_screen_width() > 640 ? 650 : 400 ),
		       ( gdk_screen_height() > 400 ? 600 : 300 ) );
  gtk_idle_add(reset_usize, new_window->channel.window);
  
  main_vbox = gtk_vbox_new(FALSE, 1);
  gtk_container_border_width(GTK_CONTAINER(main_vbox), 1);
  gtk_container_add(GTK_CONTAINER(new_window->channel.window), main_vbox);
  gtk_widget_show(main_vbox);

  if (top_level) {
    new_window->channel.menu_bar = menubar_create();
/*     gtk_window_add_accelerator_table(GTK_WINDOW(new_window->channel.window), accel); */
    gtk_box_pack_start(GTK_BOX(main_vbox), new_window->channel.menu_bar->subfactory[0]->widget,
		       FALSE, TRUE, 0);
    gtk_widget_show(new_window->channel.menu_bar->subfactory[0]->widget);
    menu_set_right_justify(new_window->channel.menu_bar, "<Main>/Help");
/*     menus_set_sensitive("<Main>/File/Disconnect", FALSE); */

    new_window->channel.toolbar = create_toolbar(new_window);
    gtk_box_pack_start (GTK_BOX (main_vbox), new_window->channel.toolbar->toolbar, FALSE, TRUE, 0);
  }

  main_hbox = gtk_hbox_new(FALSE,1);
  gtk_box_pack_start (GTK_BOX (main_vbox), main_hbox, TRUE, TRUE, 0);
  gtk_widget_show(main_hbox);

  if (top_level) {
    list_vbox = gtk_vbox_new(FALSE, 1);
    gtk_widget_set_usize(GTK_WIDGET(list_vbox), 75, 0);
    gtk_box_pack_start (GTK_BOX (main_hbox), list_vbox, FALSE, TRUE, 0);
    gtk_widget_show (list_vbox);

    new_window->channel.ch_clist = gtk_clist_new_with_titles(1, ch_title);
    GTK_WIDGET_UNSET_FLAGS(new_window->channel.ch_clist, GTK_CAN_FOCUS);
    gtk_clist_column_titles_passive (GTK_CLIST(new_window->channel.ch_clist));
    /*   gtk_clist_set_column_width (GTK_CLIST (new_window->channel.ch_clist), 0, 150); */
    gtk_signal_connect (GTK_OBJECT (new_window->channel.ch_clist),
			"select_row",
			(GtkSignalFunc) channel_clist_callback,
			new_window);
        
    gtk_clist_set_selection_mode (GTK_CLIST (new_window->channel.ch_clist), GTK_SELECTION_BROWSE);
    gtk_clist_set_policy (GTK_CLIST (new_window->channel.ch_clist),
			  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_clist_set_column_justification (GTK_CLIST (new_window->channel.ch_clist),
					0, GTK_JUSTIFY_LEFT);     
    gtk_box_pack_start(GTK_BOX(list_vbox), new_window->channel.ch_clist, TRUE, TRUE, 0);
    gtk_widget_show (new_window->channel.ch_clist);

    new_window->channel.notify_clist = gtk_clist_new_with_titles(1, notify_title);
    GTK_WIDGET_UNSET_FLAGS(new_window->channel.notify_clist, GTK_CAN_FOCUS);
    gtk_clist_column_titles_passive (GTK_CLIST(new_window->channel.notify_clist));
    /*   gtk_clist_set_column_width (GTK_CLIST (new_window->channel.notify_clist), 0, 150); */
   
    gtk_clist_set_selection_mode (GTK_CLIST (new_window->channel.notify_clist), GTK_SELECTION_BROWSE);
    gtk_clist_set_policy (GTK_CLIST (new_window->channel.notify_clist), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_clist_set_column_justification (GTK_CLIST (new_window->channel.notify_clist), 0, GTK_JUSTIFY_CENTER);     
    gtk_box_pack_start(GTK_BOX(list_vbox), new_window->channel.notify_clist, TRUE, TRUE, 0);
    gtk_widget_show (new_window->channel.notify_clist);
  } else {
    new_window->channel.ch_clist = new_window->channel.nick_clist = new_window->channel.notify_clist = NULL;
  }

  sub_vbox = gtk_vbox_new(FALSE,1);
  gtk_box_pack_start(GTK_BOX(main_hbox), sub_vbox, TRUE, TRUE, 0);
  gtk_widget_show(sub_vbox);

  sub_hbox = gtk_hbox_new(FALSE,1);
  gtk_box_pack_start(GTK_BOX(sub_vbox), sub_hbox, TRUE, TRUE, 0);
  gtk_widget_show(sub_hbox);

  new_window->channel.text = gtk_text_new(NULL, NULL);
  GTK_WIDGET_UNSET_FLAGS(new_window->channel.text, GTK_CAN_FOCUS);
  gtk_text_set_word_wrap(GTK_TEXT(new_window->channel.text), TRUE);
  gtk_text_set_editable(GTK_TEXT(new_window->channel.text), FALSE);
  gtk_box_pack_start(GTK_BOX(sub_hbox), new_window->channel.text, TRUE, TRUE, 0);
  gtk_widget_show (new_window->channel.text);

  style = gtk_style_new();
  style->fg[GTK_STATE_NORMAL] = style->white;
  style->text[GTK_STATE_NORMAL] = style->white;
  style->base[GTK_STATE_NORMAL] = style->black;
  gtk_widget_set_style(new_window->channel.text, style);

  vscrollbar = gtk_vscrollbar_new (GTK_TEXT(new_window->channel.text)->vadj);
  GTK_WIDGET_UNSET_FLAGS(vscrollbar, GTK_CAN_FOCUS);
  gtk_range_set_update_policy (GTK_RANGE (vscrollbar),
			       GTK_UPDATE_CONTINUOUS);  
  gtk_box_pack_start(GTK_BOX(sub_hbox), vscrollbar, FALSE, TRUE, 0);
  gtk_widget_show (vscrollbar);

  new_window->channel.nick_clist = gtk_clist_new_with_titles(1, nick_title);
  GTK_WIDGET_UNSET_FLAGS(new_window->channel.nick_clist, GTK_CAN_FOCUS);
  gtk_signal_connect (GTK_OBJECT (new_window->channel.nick_clist),
		      "select_row",
		      (GtkSignalFunc) nick_clist_callback,
		      new_window);

  gtk_clist_column_titles_passive (GTK_CLIST(new_window->channel.nick_clist));
  gtk_clist_set_policy (GTK_CLIST (new_window->channel.nick_clist), 
			GTK_POLICY_ALWAYS, GTK_POLICY_AUTOMATIC);
  gtk_clist_set_column_justification (GTK_CLIST (new_window->channel.nick_clist), 0,
				      GTK_JUSTIFY_LEFT);
  gtk_widget_set_usize(GTK_WIDGET(new_window->channel.nick_clist), 110, 150); 
  gtk_clist_set_selection_mode (GTK_CLIST (new_window->channel.nick_clist), GTK_SELECTION_BROWSE);
  gtk_box_pack_start(GTK_BOX(sub_hbox), new_window->channel.nick_clist, FALSE, TRUE, 0);
  gtk_widget_show (new_window->channel.nick_clist);

  new_window->channel.entry = gtk_entry_new_with_max_length(470);
  gtk_signal_connect( GTK_OBJECT(new_window->channel.entry), "activate", 
		      GTK_SIGNAL_FUNC(console_entry_callback), 
		      new_window);
  gtk_signal_connect( GTK_OBJECT(new_window->channel.entry), "key_press_event",
                      GTK_SIGNAL_FUNC(entry_key_press), 
		      new_window); 

  gtk_box_pack_start (GTK_BOX (sub_vbox), new_window->channel.entry, FALSE, TRUE, 0);
  gtk_widget_grab_focus(new_window->channel.entry);
  gtk_widget_show (new_window->channel.entry);

  sub_hbox = gtk_hbox_new(FALSE,1);
  gtk_box_pack_start(GTK_BOX(main_vbox), sub_hbox, FALSE, TRUE, 0);
  gtk_widget_show(sub_hbox);
 
  new_window->channel.user_status = gtk_statusbar_new();
  gtk_box_pack_start (GTK_BOX (sub_hbox), new_window->channel.user_status, TRUE, TRUE, 0);
  gtk_widget_show (new_window->channel.user_status);

  new_window->channel.channel_status = gtk_statusbar_new();
  gtk_box_pack_start (GTK_BOX (sub_hbox), new_window->channel.channel_status, TRUE, TRUE, 0);
  gtk_widget_show (new_window->channel.channel_status);

  gtk_widget_show(new_window->channel.window);

  return(new_window);
}

/* ------------------------------------------------------------
 * create_message_window
 *
 *  Create a new toplevel window 
 *
 * Arguments:
 *	"type" - the type of message window
 *      "nick" - the nick of the person we are talking with
 *
 * Results:
 *	an BezWindow object containing the window information
 *
 * Side effects:
 *	the new window is shown on the screen
 *
 * ----------------------------------------------------------- */

BezWindow *create_message_window(BezMessageType type, char *nick)
{
  BezWindow *window;
  GtkWidget *main_vbox;
  GtkWidget *sub_hbox;
  GtkWidget *vscrollbar;
  GtkStyle *style;
  char buff[BUFFLEN];

  window = g_malloc(sizeof(BezWindow));
  window->message.type = BEZ_MESSAGE;
  window->message.message_type = type;
  window->message.nick = g_strdup(nick);
  window->message.numlines = 0;
  window->message.recall_commands=NULL;
  window->message.recall_current=NULL;
  window->message.incomplete_command=NULL;
  window->message.connection=NULL;

  window->message.window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_signal_connect (GTK_OBJECT(window->message.window), "destroy",
		      GTK_SIGNAL_FUNC(destroy_message_window),
		      window);

  switch (type)
    {
    case DCC_CHAT:
      g_snprintf(buff, BUFFLEN, "%s [DCC CHAT]", nick);
      break;
    case IRC_MSG:
      g_snprintf(buff, BUFFLEN, "%s [IRC MSG]", nick);
      break;
    }
  gtk_window_set_title(GTK_WINDOW(window->message.window), buff);
  gtk_widget_set_usize(GTK_WIDGET(window->message.window),
		       ( gdk_screen_width() > 640 ? 650 : 400 ),
		       ( gdk_screen_height() > 400 ? 600 : 300 ) );
  gtk_idle_add(reset_usize, window->message.window);

  main_vbox = gtk_vbox_new(FALSE, 1);
  gtk_container_border_width(GTK_CONTAINER(main_vbox), 1);
  gtk_container_add(GTK_CONTAINER(window->message.window), main_vbox);
  gtk_widget_show(main_vbox);

  sub_hbox = gtk_hbox_new(FALSE,1);
  gtk_box_pack_start(GTK_BOX(main_vbox), sub_hbox, TRUE, TRUE, 0);
  gtk_widget_show(sub_hbox);

  window->message.text = gtk_text_new(NULL, NULL);
  GTK_WIDGET_UNSET_FLAGS(window->message.text, GTK_CAN_FOCUS);
  gtk_text_set_word_wrap(GTK_TEXT(window->message.text), TRUE);
  gtk_text_set_editable(GTK_TEXT(window->message.text), FALSE);
  gtk_box_pack_start(GTK_BOX(sub_hbox), window->message.text, TRUE, TRUE, 0);
  gtk_widget_show (window->message.text);

  style = gtk_style_new();
  style->fg[GTK_STATE_NORMAL] = style->white;
  style->text[GTK_STATE_NORMAL] = style->white;
  style->base[GTK_STATE_NORMAL] = style->black;
  gtk_widget_set_style(window->message.text, style);
  
  vscrollbar = gtk_vscrollbar_new (GTK_TEXT(window->message.text)->vadj);
  GTK_WIDGET_UNSET_FLAGS(vscrollbar, GTK_CAN_FOCUS);
  gtk_range_set_update_policy (GTK_RANGE (vscrollbar),
			       GTK_UPDATE_CONTINUOUS);  
  gtk_box_pack_start(GTK_BOX(sub_hbox), vscrollbar, FALSE, TRUE, 0);
  gtk_widget_show (vscrollbar);

  window->message.entry = gtk_entry_new_with_max_length(470);
  gtk_signal_connect( GTK_OBJECT(window->message.entry), "activate", 
		      GTK_SIGNAL_FUNC(console_entry_callback), 
		      window);
  gtk_signal_connect( GTK_OBJECT(window->message.entry), "key_press_event",
                      GTK_SIGNAL_FUNC(entry_key_press), 
 		      window); 
  gtk_box_pack_start (GTK_BOX (main_vbox), window->message.entry, FALSE, TRUE, 0);
  gtk_widget_grab_focus(window->message.entry);
  gtk_widget_show (window->message.entry);

  window->message.status = gtk_statusbar_new();
  gtk_box_pack_start (GTK_BOX (main_vbox), window->message.status, FALSE, TRUE, 0);
  gtk_widget_show (window->message.status);

  gtk_widget_show(window->message.window);

  return(window);
}

/* ------------------------------------------------------------
 * bez_add_persona_ok
 *
 *  Callback for adding an initial persona
 *
 * Arguments:
 *	"persona_strs" - array of the created persona values
 *      "data" - instance data (unused)
 *
 * Results:
 *
 * Side effects:
 *	
 * ----------------------------------------------------------- */

void bez_add_persona_ok(gchar **persona_strs, void *data)
{
  persona_add(g_strdup(persona_strs[0]), g_strdup(persona_strs[1]), g_strdup(persona_strs[2]));
  gtk_main_quit();
}

/* ------------------------------------------------------------
 * main
 *
 *  Program entry point. Initialise the components and hand off control
 *
 * Arguments:
 *	"argc", "argv"
 *
 * Results:
 *	program exit status
 *
 * Side effects:
 *	
 * ----------------------------------------------------------- */

int main(int argc, char *argv[])
{
  gint timer_tag;
  gchar title[BUFFLEN];
#ifdef HAVE_TCL
  int init_val;
#endif  /* HAVE_TCL  */
  gchar *version;
  BezChannelWindow *window;

  gtk_init(&argc, &argv);
/*   gtk_rc_parse("bezerkrc"); */

#ifdef HAVE_TCL
  /* tcl interpreter - davidw  */
  interp = Tcl_CreateInterp();
  if (interp == NULL) {
    fputs("cannot create tcl interpreter", stderr);
    exit (1);
  }

  register_tcl(interp);
  init_val = Tcl_EvalFile(interp, "bezerk.tcl");
  if(init_val == TCL_ERROR) {
    fputs( interp->result, stderr);
    exit(1);
  }
#endif  /* HAVE_TCL  */
 
  if (!preference_init()) {
    preference_save();
  }

  if ( !(version=preference_get("bezerk.version")) ) {
    preference_set("bezerk.version", BEZERK_VERSION);
    preference_save();
  }
  servers_init();
  irc_init();
  channel_init();
  aliases_init(); 

  if (!persona_init()) {
    persona_edit_dialog(NULL, NULL, bez_add_persona_ok, NULL);
    gtk_main();
    persona_save();
  }

  g_snprintf(title, BUFFLEN, "Bezerk %s by Tony Gale [Trog]", BEZERK_VERSION);
  window = BEZ_CHANNEL_WINDOW(create_channel_window(title, TRUE));
  if (!version) {
    gtk_text_freeze( GTK_TEXT(window->text) );
    gtk_text_insert(GTK_TEXT(window->text), NULL, NULL, NULL, "Welcome to Bezerk ", -1);
    gtk_text_insert(GTK_TEXT(window->text), NULL, NULL, NULL, BEZERK_VERSION, -1);
    gtk_text_insert(GTK_TEXT(window->text), NULL, NULL, NULL, " by Tony Gale [Trog]", -1);
    gtk_text_insert(GTK_TEXT(window->text), NULL, NULL, NULL, "\nNew Features Include:\n", -1);
    gtk_text_insert(GTK_TEXT(window->text), NULL, NULL, NULL, "  o Multi-server support\n", -1);
    gtk_text_insert(GTK_TEXT(window->text), NULL, NULL, NULL, "  o Initial Toolbar\n", -1);
    gtk_text_insert(GTK_TEXT(window->text), NULL, NULL, NULL, "  o New Preference dialogs\n", -1);
    gtk_text_insert(GTK_TEXT(window->text), NULL, NULL, NULL, "  o Personae (incomplete)\n", -1);
    window->numlines = 6;
    gtk_text_thaw( GTK_TEXT(window->text) );
  }
  timer_tag = gtk_timeout_add (20000, lag_check_callback, NULL);

  gtk_main();
  
  return(0);
}
