/* AMCL - A simple Mud CLient
 * Copyright (C) 1998 Robin Ericsson <lobbin@lobbin.ml.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>

#include "amcl.h"

#ifdef HAVE_TELNET_H
#include <telnet.h>
#endif
#ifdef HAVE_ARPA_TELNET_H
#include <arpa/telnet.h>
#endif

/*
 * Local functions
 */
void alias_button_add     (GtkWidget *button, gpointer data);
void alias_button_delete  (GtkWidget *button, gpointer data);
void alias_selection_made (GtkWidget *clist, gint row, gint column,
                           GdkEventButton *event, gpointer data);

typedef enum { NORM, ESC, SQUARE, PARMS } STATE;



/*
 * Global Variables
 */
GdkColor  *foreground;
GtkWidget *textalias;
GtkWidget *textreplace;
GtkWidget *alias_window;
gint      alias_selected_row    = -1;
gint      alias_selected_column = -1;
static gint parms[10], nparms;
bool      BOLD = FALSE;

static void cons_escm()
{
    int i, p;

    for ( i = 0; i < nparms; i++ )
    {
        switch ( p = parms[i])
        {
        case 0: /* none */
            foreground = &color_white;
            BOLD       = FALSE;
            break;
        case 1:/* bold */
            BOLD = TRUE;
            break;
        case 4: /* underscore */
        case 5: /* blink */
        case 7: /* inverse */
            break;

        case 30:
            if ( BOLD )
                foreground = &color_grey;
            else
                foreground = &color_black;
            break;

        case 31:
            if ( BOLD )
                foreground = &color_orange;
            else
                foreground = &color_red;
            break;

        case 32:
            if ( BOLD )
                foreground = &color_lightgreen;
            else
                foreground = &color_green;
            break;

        case 33:
            if ( BOLD )
                foreground = &color_yellow;
            else
                foreground = &color_brown;
            break;

        case 34:
            if ( BOLD )
                foreground = &color_lightblue;
            else
                foreground = &color_blue;
            break;

        case 35:
            if ( BOLD )
                foreground = &color_pink;
            else
                foreground = &color_purple;
            break;

        case 36:
            if ( BOLD )
                foreground = &color_lightcyan;
            else
                foreground = &color_cyan;
            break;

        case 37:
            foreground = &color_white;
            break;

        default:
            break;
        }
    }
}

void alias_selection_made (GtkWidget *clist, gint row, gint column,
                           GdkEventButton *event, gpointer data)
{
    gchar *text;
    
    alias_selected_row    = row;
    alias_selected_column = column;

    gtk_clist_get_text ((GtkCList*) data, row, 0, &text);
    gtk_entry_set_text (GTK_ENTRY (textalias), text);
    gtk_clist_get_text ((GtkCList*) data, row, 1, &text);
    gtk_entry_set_text (GTK_ENTRY (textreplace), text);
    
    return;
}

void alias_button_add (GtkWidget *button, gpointer data)
{
    gchar *text[2];
    gint   i;
    ALIAS_DATA *alias;

    text[0]   = gtk_entry_get_text (GTK_ENTRY (textalias  ));
    text[1]   = gtk_entry_get_text (GTK_ENTRY (textreplace));

    if ( text[0][0] == '\0' || text[1][0] == '\0' )
    {
        popup_window ("Please complete the alias first.");
        return;
    }

    for ( i = 0 ; i < strlen (text[0]) ; i++ )
    {
        if ( isspace (text[0][i]) )
        {
            popup_window ("I can't make an alias of that.");
            return;
        }
    }

    if ( strlen (text[0]) > 15)
    {
        popup_window ("Alias to big.");
        return;
    }
    
    if ( strlen (text[1]) > 80)
    {
        popup_window ("Replace to big.");
        return;
    }

    for ( alias = alias_list; alias != NULL; alias = alias->next )
    {
        if ( alias->alias && !strcmp (text[0], alias->alias) )
        {
            popup_window ("Can't alias an existing alias.");
            return;
        }
    }

    gtk_clist_append ((GtkCList *) data, text);
    add_alias (text[0], text[1]);

    return;
}

void alias_button_delete (GtkWidget *button, gpointer data)
{
    ALIAS_DATA *alias;
    gchar *word;
    
    if ( alias_selected_row == -1 )
    {
        popup_window ("No selection made.");
        return;
    }

    gtk_clist_get_text ((GtkCList*) data, alias_selected_row, 0, &word);

    if ( !strcmp (alias_list->alias, word ) )
    {
        ALIAS_DATA *temp_alias;

        temp_alias = alias_list;
        alias_list = alias_list->next;
        g_free (temp_alias->alias  );
        g_free (temp_alias->replace);
        g_free (temp_alias);
    }
    else
    {
        for ( alias = alias_list; alias != NULL; alias = alias->next )
        {
            ALIAS_DATA *temp_alias;

            temp_alias = alias->next;

            if ( alias->next && !strcmp (alias->next->alias, word) )
            {
                if ( alias->next->next )
                    alias = alias->next->next;
                else
                    alias = NULL;
                g_free (temp_alias->alias  );
                g_free (temp_alias->replace);
                g_free (temp_alias);
            }
        }
    }

    gtk_clist_remove ((GtkCList*) data, alias_selected_row);
    alias_selected_row = -1;

    return;
}

void alias_close_window ()
{
    gtk_widget_set_sensitive (menu_option_alias, TRUE);
    gtk_widget_destroy (alias_window);
}

void window_alias (GtkWidget *widget, gpointer data)
{
    GtkWidget *vbox;
    GtkWidget *hbox;
    GtkWidget *hbox2;
    GtkWidget *hbox3;
    GtkWidget *clist;
    GtkWidget *button_add;
    GtkWidget *button_quit;
    GtkWidget *button_delete;
    GtkWidget *button_save;
    GtkWidget *label;
    GtkWidget *separator;

    gchar     *titles[2] = { "Alias", "Replacement" };

    gtk_widget_set_sensitive (menu_option_alias, FALSE);

    alias_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (alias_window), "Amcl Alias Center");
    gtk_signal_connect_object (GTK_OBJECT (alias_window), "destroy",
                               GTK_SIGNAL_FUNC(alias_close_window), NULL );
    gtk_widget_set_usize (alias_window,450,320);

    vbox = gtk_vbox_new (FALSE, 5);
    gtk_container_border_width (GTK_CONTAINER (vbox), 5);
    gtk_container_add (GTK_CONTAINER (alias_window), vbox);
    gtk_widget_show (vbox         );

    clist = gtk_clist_new_with_titles (2, titles);
    gtk_signal_connect_object (GTK_OBJECT (clist), "select_row",
                               GTK_SIGNAL_FUNC (alias_selection_made),
                               (gpointer) clist);
    gtk_clist_set_border (GTK_CLIST (clist), GTK_SHADOW_OUT);
    gtk_clist_set_policy (GTK_CLIST (clist), GTK_POLICY_AUTOMATIC,
                          GTK_POLICY_AUTOMATIC);
    gtk_clist_set_column_width (GTK_CLIST (clist), 0, 100);
    gtk_clist_set_column_width (GTK_CLIST (clist), 1, 250);
    gtk_clist_set_column_justification (GTK_CLIST (clist), 0, GTK_JUSTIFY_LEFT);
    gtk_clist_set_column_justification (GTK_CLIST (clist), 1, GTK_JUSTIFY_LEFT);

    gtk_clist_column_titles_show (GTK_CLIST (clist));
    gtk_box_pack_start (GTK_BOX (vbox), clist, TRUE, TRUE, 0);
    gtk_widget_show (clist);

    hbox3 = gtk_hbox_new (TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), hbox3, FALSE, FALSE, 0);
    gtk_widget_show (hbox3);

    label = gtk_label_new ("Alias");
    gtk_box_pack_start (GTK_BOX (hbox3), label, FALSE, TRUE, 0);
    gtk_widget_show (label);
    label = gtk_label_new ("Replacement");
    gtk_box_pack_start (GTK_BOX (hbox3), label, FALSE, TRUE, 0);
    gtk_widget_show (label);

    hbox2 = gtk_hbox_new (TRUE, 15);
    gtk_box_pack_start (GTK_BOX (vbox), hbox2, FALSE, FALSE, 0);
    gtk_widget_show (hbox2);

    textalias   = gtk_entry_new ();
    textreplace = gtk_entry_new ();
    gtk_box_pack_start (GTK_BOX (hbox2), textalias,   FALSE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hbox2), textreplace, FALSE, TRUE, 0);
    gtk_widget_show (textalias  );
    gtk_widget_show (textreplace);
    
    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (vbox), separator, FALSE, TRUE, 5);
    gtk_widget_show (separator);

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    gtk_widget_show (hbox);

    button_add    = gtk_button_new_with_label ("  add   ");
    button_quit   = gtk_button_new_with_label (" close  ");
    button_delete = gtk_button_new_with_label (" delete ");
    button_save   = gtk_button_new_with_label ("  save  ");
        gtk_signal_connect_object (GTK_OBJECT (button_add), "clicked",
                               GTK_SIGNAL_FUNC (alias_button_add),
                               (gpointer) clist);
    gtk_signal_connect_object (GTK_OBJECT (button_delete), "clicked",
                               GTK_SIGNAL_FUNC (alias_button_delete),
                               (gpointer) clist);
    gtk_signal_connect_object (GTK_OBJECT (button_save), "clicked",
                               GTK_SIGNAL_FUNC (save_aliases),
                               (gpointer) clist);
    gtk_signal_connect_object (GTK_OBJECT (button_quit), "clicked",
                               GTK_SIGNAL_FUNC (alias_close_window),
                               NULL);
    gtk_box_pack_start (GTK_BOX (hbox), button_add,    TRUE, TRUE, 15);
    gtk_box_pack_start (GTK_BOX (hbox), button_delete, TRUE, TRUE, 15);
    gtk_box_pack_start (GTK_BOX (hbox), button_save,   TRUE, TRUE, 15);
    gtk_box_pack_start (GTK_BOX (hbox), button_quit,   TRUE, TRUE, 15);

    gtk_widget_show (button_add   );
    gtk_widget_show (button_quit  );
    gtk_widget_show (button_delete);
    gtk_widget_show (button_save  );

    insert_aliases  (clist        );
    gtk_widget_show (alias_window );

    return;
}

void popup_window (const gchar *message)
{
    GtkWidget *window;
    GtkWidget *label;
    GtkWidget *button;
    GtkWidget *box;
    GtkWidget *separator;

    gchar       buf[2048];

    window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (window), "Amcl Popup Message");

    box = gtk_vbox_new (FALSE, 5);
    gtk_container_border_width (GTK_CONTAINER (box), 5);
    gtk_container_add (GTK_CONTAINER (window), box);

    snprintf ( buf, 2048, " %s ", message);
    label = gtk_label_new (buf);
    gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 5);
    gtk_widget_show (label);

    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (box), separator, TRUE, TRUE, 0);
    gtk_widget_show (separator);
    
    button = gtk_button_new_with_label (" OK ");
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
                        GTK_SIGNAL_FUNC (close_window), window);
    gtk_box_pack_start (GTK_BOX (box), button, TRUE, TRUE, 5);
    gtk_widget_show (button);
    
    gtk_widget_show (box   );
    gtk_widget_show (window);
}

void textfield_add (gchar *message, gint colortype)
{
    gchar *start, c;
    gint  len;
    static STATE state = NORM;

    if ( message[0] == '\0' )
    {
        return;
    }

    switch (colortype)
    {
    case MESSAGE_SENT:
        gtk_text_insert (GTK_TEXT (text_field), font_normal, &color_yellow,
                         NULL, message, strlen (message));
        break;
    case MESSAGE_ERR:
        gtk_text_insert (GTK_TEXT (text_field), font_normal, &color_green,
                         NULL, message, strlen (message));
        break;
    case MESSAGE_ANSI:
        if ( !strstr (message, "\033[") )
        {
            gtk_text_insert (GTK_TEXT (text_field), font_normal,
                             foreground, NULL, message, -1);
            gtk_text_insert (GTK_TEXT (text_field), NULL, NULL, NULL," ", 1 );
            gtk_text_backward_delete (GTK_TEXT (text_field), 1);

            break;
        }
        
        /*while ( *message != '\0' || message == NULL )
        {
            if ( *message == '\033' || *message =='\e')
            {
                while ( !done )
                {
                    switch ( *message )
                    {
                    case '[':
                        found_1 = TRUE;
                        break;
                        
                    case ';':
                        found_2 = TRUE;
                        break;
                        
                    case 'm':
                        found_3 = TRUE;
                        break;
                    }
                    if ( found_1 && !found_2 && !found_3)
                    {
                        message++;

                        if ( *message == 'm' )
                        {
                            *//* This is color reset *//*
                            foreground = &color_white;
                            found_2 = TRUE;
                            found_3 = TRUE;
                        }
                        else
                        {
                            found_2 = TRUE;
                            *ptr = *message;
                            bold = atoi(ptr);

                            message++; message++;
                            ptr[0] = *message;
                            message++;

                            if ( *message != 'm' )
                            {
                                ptr[1] = *message;
                            }
                            color = atoi(ptr);

                            if ( bold == 0 )
                                *//* wonder if these colors are right *//*
                                switch (color)
                                {
                                case 30: foreground = &color_black;   break;
                                case 31: foreground = &color_red;     break;
                                case 32: foreground = &color_green;   break;
                                case 33: foreground = &color_brown;   break;
                                case 34: foreground = &color_blue;    break;
                                case 35: foreground = &color_purple;  break;
                                case 36: foreground = &color_cyan;    break;
                                case 37: foreground = &color_white;   break;
                                }
                            else if ( bold == 1 )
                            {
                                switch (color)
                                {
                                case 30: foreground = &color_grey;       break;
                                case 31: foreground = &color_orange;     break;
                                case 32: foreground = &color_lightgreen; break;
                                case 33: foreground = &color_yellow;     break;
                                case 34: foreground = &color_lightblue;  break;
                                case 35: foreground = &color_pink;       break;
                                case 36: foreground = &color_lightcyan;  break;
                                case 37: foreground = &color_white;      break;
                                }
                            }

                        }
                    }
                    message++;

                    if ((found_1 == TRUE) && (found_2 == TRUE) && (found_3 == TRUE))
                        done = TRUE;
                }

                found_1 = FALSE;
                found_2 = FALSE;
                found_3 = FALSE;
                done    = FALSE;
                ptr[0]  = ptr[1] = '\0';
            }
            else
            {
                if ( *message != '\r' )
                {
                    gtk_text_insert (GTK_TEXT (text_field), display_font,
                                     foreground, NULL, message, 1);
                }
                message++;
            }
            }*/

        while ( (c = *message) )
        {
            switch ( state )
            {
            case NORM:
                if ( c >= ' ' )
                {
                    start = message; len = 0;
                    while ( *message && *message >= ' ')
                    {
                        message++;
                        len++;
                    }
                    gtk_text_insert (GTK_TEXT (text_field), font_normal,
                                     foreground, NULL, start, len );
                }

                if ( !(c = *message))
                    break;

                if ( c != '\033' )
                {
                    gtk_text_insert (GTK_TEXT (text_field), font_normal,
                                     foreground, NULL, &c, 1);
                    ++message;
                    break;
                }

                state = ESC;
                c = *++message;

            case ESC:
                if ( c != '[' )
                {
                    state = NORM;
                    break;
                }

                state = SQUARE;
                c = *++message;

            case SQUARE:
                nparms = 0;

                while ( c && (isdigit (c) || c == ';'))
                {
                    if ( isdigit (c))
                    {
                        if ( nparms < 10 )
                        {
                            parms[nparms] = atoi(message);
                            nparms++;
                        }
                        while ( isdigit(c))
                            c = *++message;

                        if ( c == ';' )
                            c = *++message;
                    }
                    else if ( c == ';' )
                    {
                        if ( nparms < 10 )
                        {
                            parms[nparms] = 0;
                            nparms++;
                        }
                        c = *++message;
                    }
                }
                if ( nparms == 0 )
                    parms[nparms++] = 0;

                state = PARMS;

            case PARMS:
                switch (c)
                {
                case 'A':
                    ++message;
                    break;
                case 'B':
                    ++message;
                    break;
                case 'C':
                    ++message;
                    break;
                case 'D':
                    ++message;
                    break;
                case 'H':
                    ++message;
                    break;
                case 'J':
                    ++message;
                    break;
                case 'K':
                    ++message;
                    break;
                case 'm':
                    cons_escm();
                    ++message;
                    break;
                default:
                    break;
                }
                state = NORM;
            }
        }

        break;
        
    case MESSAGE_NORMAL:
    case MESSAGE_NONE:
    default:
        gtk_text_insert (GTK_TEXT (text_field), font_normal, &color_white,
                         NULL, message, strlen (message));
        break;
    }

    /*
     * A hack to make it scroll, doesn't seem to need this when we use a
     * background color, then it scrolls by it self...
     */
    //gtk_text_insert (GTK_TEXT (text_field), NULL, NULL, NULL,
    //                 " ", 1 );
    //gtk_text_backward_delete (GTK_TEXT (text_field), 1);
}
