/*
 * FontSel.c -- Example how to use a ComboBox within a font selector
 * 
 * compile and link with:
 * $ cc -DFUNCPROTO FontSel.c ComboBox.o -o FontSel \
 *   -lXm -lXt -lX11
 * then run:
 * $ ./FontSel
 * 
 * (c) 1994 Harald Albrecht
 * Institut fuer Geometrie und Praktische Mathematik
 * RWTH Aachen, Germany
 * albrecht@igpm.rwth-aachen.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING for more details);
 * if not, write to the Free Software Foundation, Inc., 675 Mass Ave, 
 * Cambridge, MA 02139, USA.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/Separator.h>
#include <Xm/Frame.h>
#include <Xm/PushB.h>
#include <Xm/ToggleB.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>
#include <Xm/TextF.h>
#include <Xm/List.h>
#include "ComboBox.h"

#ifdef USE_BUTTONFACELIB
#include "../../ButtonFaceLib/pushbuttons.h"
#endif

static char *FallbackResources[] = {
    "*.background:					gray75",
    "*.fontList:					-*-helvetica-bold-r-normal-*-*-120-*-*-*-*-*-*",
    "*.fontnamelabel.labelString:			Family:",
    "*.fontweightlabel.labelString:			Style:",
    "*.fontsizelabel.labelString:			Size:",
    "*.okay.labelString:				Ok",
    "*.XmPushButton.width:				70",
    "*.cancel.labelString:				Cancel",
    "*.help.labelString:				Help",
    "*.scalabletoggle.labelString:			Show scalable fonts only",
    "*.scalabletoggle.selectColor:			yellow",
    "*.isofontstoggle.labelString:			Show LATIN1 (ISO8859-1) fonts only", 
    "*.isofontstoggle.selectColor:			yellow",
    "*.fonttype.labelString:				?",
    "*.xlfdlabel.labelString:				?", 
    "*.previewwindow.background:			white",
    "*.previewwindow.labelString:			*\\n*\\n*\\n*\\n*\\n*", 
/*    "*.previewwindow.height:				60",*/
    NULL
};


Widget TopLevel, MainForm;
Widget FontnameLabel, FontweightLabel, FontsizeLabel;
Widget FontnameList, FontweightList, FontsizeList;
Widget ToggleFrame, ToggleRow, ScalableToggle, IsoFontsToggle;
Widget FontTypeLabel;
Widget XLFDLabel, FontPreviewWindow, FontPreviewFrame;
Widget Sepp;
Widget Okay, Cancel, Help;

char         **FontNames = NULL;
char         **FontNamesOrig = NULL;
int          MaxFonts, FontCount;

XmFontList   OrigFontList;
XmFontList   PreviewFontList;
XFontStruct  *PreviewFontStruct = NULL;

Boolean      PendingUpdate = False;
Boolean      DontUseTimer  = False;
XtIntervalId TimerID;


void UpdatePreview();

/* --------------------------------------------------------------------
 * Fordere eine Liste aller verfuegbaren Fonts an. Dabei wird beachtet, 
 * ob wahlweise nur solche Zeichensaetze aufgezaehlt werden sollen, die
 * bestimmte Kriterien erfuellen:
 * 1. Font ist skalierbar
 * 2. Font hat Zeichenkodierung (Zeichenanordnung) nach ISO 8859-1
 * 
 * Ergebnis:
 *   keines. Setzt FontNames, FontNamesOrig, MaxFonts und FontCount.
 */
int CompareFontNames(const void *p1, const void *p2)
{
    return strcmp(*(char **) p1, *(char **) p2);
} /* CompareFontNames */

void GetListOfFonts()
{
    char mask[256];
    Boolean OnlyScalable;
    Boolean OnlyLATIN1;
    
    int i;
    char **NewList;
    
    MaxFonts = 1000;
    
    if ( FontNamesOrig ) XFreeFontNames(FontNamesOrig);
    if ( FontNames )     XtFree((char *) FontNames);
    FontNames = NULL;
    
    XtVaGetValues(ScalableToggle, XmNset, &OnlyScalable, NULL);
    XtVaGetValues(IsoFontsToggle, XmNset, &OnlyLATIN1, NULL);
    sprintf(mask, "-*-*-*-*-*-*-*-%s-*-*-*-*-%s", 
            OnlyScalable ? "0" : "*", 
	    OnlyLATIN1 ? "iso8859-1" : "*-*");
    
    for ( ;; ) {
        FontNamesOrig = XListFonts(XtDisplay(TopLevel), 
                                   mask, MaxFonts, &FontCount);
        if ( (FontNamesOrig == NULL) || (FontCount < MaxFonts) ) break;
        XFreeFontNames(FontNamesOrig);
        MaxFonts *= 2;
    }
    /*
     * Ja isset denn moeschlich?! Sortiere niemals die Original-
     * tabelle, ansonsten core-dump. Das liegt daran, dass XListFonts
     * noch direkt VOR den Namen des allerersten Fonts in der Liste
     * zusaetzliche Informationen ablegt. Warum steht das eigentlich
     * nicht in der Dokumentation?!
     */    
    if ( FontCount ) {
	FontNames = (char **) XtMalloc(sizeof(char **) * FontCount);
	memcpy(FontNames, FontNamesOrig, sizeof(char **) * FontCount);
	qsort(FontNames, FontCount, sizeof(char *), CompareFontNames);
    }
} /* GetListOfFonts */


/* --------------------------------------------------------------------
 * Liefert den Namen der Zeichensatzkodierung zurueck, aber nur dann, 
 * wenn es keine Standard-ISO-Kodierung ist.
 * 
 * Parameter:
 *   pStr	    Vollstaendiger Fontname im XLFD-Format.
 * 
 * Ergebnis:
 *   Zeiger auf die Kodierung oder NULL, falls im ISO 8859-1 Format.
 */
char *GetEncoding(char *pStr)
{
    char *pHyphen, *pEncoding;
    int  count;
    
    pHyphen = pStr - 1; pEncoding = NULL;
    for ( count = 0; count <= 12; ++count ) {
	pHyphen = strchr(pHyphen + 1, '-');
	if ( pHyphen == NULL ) return NULL;
    }
    pEncoding = pHyphen + 1;
    return strcmp(pEncoding, "iso8859-1") == 0 ? NULL : pEncoding;
} /* GetEncoding */


/* --------------------------------------------------------------------
 * Fuelle die erste ComboBox mit den Namen aller verfuegbaren Font-
 * Familien sowie deren Hersteller. Der Herstellername wird einfach 
 * hinten an den Familiennamen des Zeichensatzes in Klammern angefuegt.
 * Ausserdem wird noch die jeweilige Zeichensatzkodierung beruecksich-
 * tigt, sofern es sich nicht um ISO8859-1 handelt.
 */
void ListFontnames()
{
    int      i, StrLen;
    char     *pLastFontName, *pHyphen1, *pHyphen2, *pFontName;
    char     *pEncoding, *pLastEncoding;
    char     *OldInput;
    XmString Item;
    
    OldInput = XmComboBoxGetString(FontnameList);
    XmComboBoxDeleteAllItems(FontnameList);
    XmComboBoxClearItemSelection(FontnameList);
    if ( FontCount == 0 ) {
	XtFree(OldInput);
	return;
    }
    
    pLastFontName = XtNewString("");
    pLastEncoding = NULL;
    for ( i = 0; i < FontCount; ++i ) {
    /*
     * Es kommen ueberhaupt nur sog. "well-formed font names" in die
     * engere Auswahl. Alles, was nicht mit einem Strich beginnt, 
     * fliegt heraus!
     */
    	if ( *FontNames[i] != '-' ) continue;
        pHyphen1 = strchr(FontNames[i] + 1, '-');
        if ( pHyphen1 == NULL ) continue;
        pHyphen2 = strchr(pHyphen1 + 1, '-');
        if ( pHyphen2 == NULL ) continue;
        pFontName = FontNames[i] + 1;
        StrLen    = pHyphen2 - pFontName;
    /*
     * Wenn wir diesen Namen noch nicht in der Liste haben, dann
     * muessen wir ihn aufnehmen. Da die Namensliste der verfueg-
     * baren Fonts alphabetisch sortiert ist, faellt diese Aktion
     * denkbar einfach aus: alle Zeichensaetze mit gleichem Her-
     * steller und aus gleicher Fontfamilie stehen direkt hinter-
     * einander in der Liste.
     */
	pEncoding = GetEncoding(FontNames[i]);
        if ( (strncmp(pFontName, pLastFontName, StrLen) != 0) ||
	     (pEncoding == NULL) || (pLastEncoding == NULL) ||
	     (strcmp(pEncoding, pLastEncoding) !=0) ) {
            char *pName;
            
	    /*
	     * Hier legen wir uns zuerst eine Kopie an, mit deren
	     * Hilfe wir spaeter leicht vergleichen koennen, ob nun
	     * ein neuer Font in der Liste auftaucht.
	     */
            XtFree(pLastFontName);
            pLastFontName = (char *) XtCalloc(StrLen + 1, 1);
            strncpy(pLastFontName, pFontName, StrLen);
            
	    pLastEncoding = pEncoding;
	    if ( pEncoding ) {
		StrLen += strlen(pEncoding) + 2;
	    }
	    /*
	     * Nun auch noch einen Eintrag fuer die ComboBox zu-
	     * sammenbasteln.
	     */
            pName = (char *) XtCalloc(StrLen + 1 + 3, 1);
            strncpy(pName, pHyphen1 + 1, pHyphen2 - pHyphen1 - 1);
            if ( (pFontName != pHyphen1) || pEncoding ) {
		strcat(pName, " (");
		if ( pFontName != pHyphen1 )
		    strncpy(pName + (pHyphen2 - pHyphen1) + 1, 
			      FontNames[i] + 1,
			      pHyphen1 - FontNames[i] - 1);
		if ( pEncoding ) {
		    strcat(pName, ", "); strcat(pName, pEncoding);
		}
		strcat(pName, ")");
            }
            Item = XmStringCreate(pName, XmFONTLIST_DEFAULT_TAG);
            if ( !XmComboBoxItemExists(FontnameList, Item) )
		XmComboBoxAddItem(FontnameList, Item, 0);
            XmStringFree(Item);
            XtFree(pName);
        }
    }
    XtFree(pLastFontName);
    
    /*
     * Abschliessend noch eine Vorgabe treffen. Existierte noch eine
     * alte Eingabe, dann versuchen wir zuerst, diese zu nehmen, 
     * ansonsten halt den erstbesten Zeichensatz.
     */
    if ( *OldInput ) {
	Item = XmStringCreate(OldInput, XmFONTLIST_DEFAULT_TAG);
	if ( XmComboBoxItemExists(FontnameList, Item) )
	    XmComboBoxSelectItem(FontnameList, Item, True);
	else
	    XmComboBoxSelectPos(FontnameList, 1, True);
	XmStringFree(Item);
    } else
	XmComboBoxSelectPos(FontnameList, 1, True);
    XtFree(OldInput);
} /* ListFontnames */

/* --------------------------------------------------------------------
 * Entferne alle Leerfelder am Anfang und am Ende einer Zeichenkette.
 * Dabei wird ein neuer Zeiger zurueckgeliefert, die auf das erste
 * Zeichen zeigt, welches kein Leerfeld mehr ist. Die Leerfelder am
 * Ende des Textes sind durch \0 ersetzt worden.
 */
char *RemoveSpaces(char *pStr)
{
    char *pEnd;
    int  len;
    
    if ( pStr == NULL ) return NULL;
    
    while ( *pStr == ' ' )
	++pStr;
    len = strlen(pStr);
    if ( len ) {
	pEnd = pStr + strlen(pStr) - 1;
	while ( (pEnd >= pStr) && (*pEnd == ' ') )
	    *pEnd-- = 0;
    }
    return pStr;
} /* RemoveSpaces */

/* --------------------------------------------------------------------
 * Anhand der Eingabe in der ersten ComboBox (fuer die Font-Family)
 * bastele den Anfang des XLFD zusammen.
 * 
 * Ergebnis:
 *   Zeiger auf den Beginn des XLFD (die ersten beiden Felder).
 */
char *GetFontname()
{
    static char Fontname[256];
           char *UserInput, *OpenBracket, *ClosingBracket, *pEncoding;
    
    UserInput = XmComboBoxGetString(FontnameList);
    if ( *UserInput == 0 ) {
	XtFree(UserInput);
	return "-*-*-";
    }
    
    UserInput = RemoveSpaces(UserInput);
    strcpy(Fontname, "-");
    OpenBracket = strchr(UserInput, '(');
    if ( OpenBracket == NULL ) {
	/*
	 * Es existiert nur der Familienname des Fonts, aber kein
	 * Hinweis auf den Urheber.
	 */
	strcat(Fontname, "-");
	strcat(Fontname, UserInput);
	strcat(Fontname, "-");
    } else {
	/*
	 * Es existieren sowohl der Familienname des Fonts als auch
	 * der Urhebername.
	 */
	ClosingBracket = strchr(OpenBracket, ')');
	if ( ClosingBracket == NULL ) {
	    strcat(Fontname, UserInput);
	    strcat(Fontname, "-*-");
	} else {
	    pEncoding = strchr(OpenBracket, ',');
	    if ( pEncoding ) ClosingBracket = pEncoding;
	    *ClosingBracket = 0;
	    *OpenBracket    = 0;
	    UserInput = RemoveSpaces(UserInput);
	    strcat(Fontname, RemoveSpaces(OpenBracket + 1));
	    strcat(Fontname, "-");
	    strcat(Fontname, UserInput);
	    strcat(Fontname, "-");
	}
    }
    XtFree(UserInput);
    return Fontname;
} /* GetFontname */

/* --------------------------------------------------------------------
 * Anhand der Eingabe in der ersten ComboBox das Encoding ermitteln.
 * Ist dort nichts angegeben, so wird von der ISO 8859-1 Kodierung
 * ausgegangen.
 */
char *GetFontEncoding()
{
    static char Encoding[256];
           char *UserInput, *OpenBracket, *ClosingBracket, *pEncoding;
    
    UserInput = XmComboBoxGetString(FontnameList);
    if ( *UserInput == 0 ) {
	XtFree(UserInput);
	return "iso8859-1";
    }

    Encoding[0] = 0;
    OpenBracket = strchr(UserInput, '(');
    if ( OpenBracket ) {
	ClosingBracket = strchr(OpenBracket, ')');
	if ( ClosingBracket == NULL )
	    strcpy(Encoding, "iso8859-1");
	else {
	    pEncoding = strchr(OpenBracket, ',');
	    if ( pEncoding == NULL )
		strcpy(Encoding, "iso8859-1");
	    else {
		OpenBracket = pEncoding + 1;
		*ClosingBracket = 0;
		strcpy(Encoding, RemoveSpaces(OpenBracket));
	    }
	}
    }
    XtFree(UserInput);
    return Encoding;
} /* GetFontEncoding */

/* --------------------------------------------------------------------
 * Hole den aktuell ausgewaehlten Zeichensatzstil und forme ihn in ein
 * fuer einen XLFD geeignetes Format um.
 */
char *GetFontWeightSlant()
{
    static char WeightSlant[256];
           char *UserInput, *pHyphen, *pStr, *pEnd;
	   char *pWeight, *pSlant;
    
    UserInput = XmComboBoxGetString(FontweightList);
    if ( *UserInput == 0 ) {
	XtFree(UserInput);
	return "*-*-";
    }
    
    pWeight = "*"; pSlant = "r";
    WeightSlant[0] = 0; pStr = UserInput;
    for ( ;; ) {
	pHyphen = strchr(pStr, ',');
	if ( pHyphen != NULL ) *pHyphen = 0;
	pStr = RemoveSpaces(pStr);
	
	if      ( strcmp(pStr, "italic") == 0 )
	    pSlant = "i";
	else if ( strcmp(pStr, "oblique") == 0 )
	    pSlant = "o";
	else if ( strcmp(pStr, "reverse italic") == 0 )
	    pSlant = "ri";
	else if ( strcmp(pStr, "reverse oblique") == 0 )
	    pSlant = "ro";
	else if ( strcmp(pStr, "other") == 0 )
	    pSlant = "ot";
	else
	    pWeight = pStr;
	
	if ( pHyphen == NULL ) break;
	pStr = pHyphen + 1;
    }
    
    strcpy(WeightSlant, pWeight);
    strcat(WeightSlant, "-");
    strcat(WeightSlant, pSlant);
    strcat(WeightSlant, "-");
    
    XtFree(UserInput);
    return WeightSlant;
} /* GetFontWeightSlant */

/* --------------------------------------------------------------------
 * Suche nach dem ersten Font, der mit dem angegebenen Praefix be-
 * ginnt. (Praefix, wie von GetFontname() ermittelt.)
 */
int FindFont(char *prefix)
{
    int len, index, left, right, result;
    
    len = strlen(prefix);
    
    left = 0; right = FontCount;
    do {
	index = (left + right) / 2;
	result = strncmp(prefix, FontNames[index], len);
	if      ( result < 0 ) right = index - 1;
	else if ( result > 0 ) left  = index + 1;
    } while ( (result != 0) && (left <= right) );
    if ( result != 0 ) return -1;
    while ( index >= 0 )
	if ( strncmp(prefix, FontNames[index], len) != 0 ) {
	    ++index; break;
	} else
	    --index;
    return index < 0 ? 0 : index;
} /* FindFont */

/* --------------------------------------------------------------------
 *
 */
Boolean CompareEncoding(char *pItem, char *pEncoding)
{
    int EncodingLen, ItemLen;
    
    EncodingLen = strlen(pEncoding);
    ItemLen     = strlen(pItem);
    if ( EncodingLen > ItemLen ) return False;
    return strcmp(pItem + ItemLen - EncodingLen, pEncoding) == 0 ?
	True : False;
} /* CompareEncoding */

/* --------------------------------------------------------------------
 * Fuelle die zweite ComboBox mit einer Liste aller fuer die momentan
 * ausgewaehlte Font-Familie verfuegbaren Zeichensatzstile.
 */
void ListFontStyles(char *Prefix, char *pEncoding, int index)
{
    int      len;
    char     *pStr, *pHyphen, *OldInput;
    char     Item[256];
    XmString ListItem, Text;
    
    OldInput = XmComboBoxGetString(FontweightList);
    XmComboBoxDeleteAllItems(FontweightList);
    XmComboBoxClearItemSelection(FontweightList);
    if ( index < 0 ) {
	XtFree(OldInput);
	return;
    }
    
    len = strlen(Prefix);
    while ( (index < FontCount) &&
            (strncmp(Prefix, FontNames[index], len) == 0) ) {
	
	if  ( !CompareEncoding(FontNames[index], pEncoding) ) {
	    ++index; continue;
	}
	
	pStr = FontNames[index] + len;
	pHyphen = strchr(pStr, '-');
	if ( pHyphen == NULL ) continue;
	
	if ( pHyphen == pStr )
	    strcpy(Item, "(none)");
	else {
	    *pHyphen = 0;
	    strcpy(Item, pStr);
	    *pHyphen = '-';
	}
	
	pStr = pHyphen + 1;
	pHyphen = strchr(pStr, '-');
	if ( pHyphen == NULL ) continue;
	
	if ( pHyphen != pStr ) {
	    *pHyphen = 0;
	    if      ( strcmp(pStr, "r") == 0 )
		;
	    else if ( strcmp(pStr, "o") == 0 )
		strcat(Item, ", oblique");
	    else if ( strcmp(pStr, "i") == 0 )
		strcat(Item, ", italic");
	    else if ( strcmp(pStr, "ro") == 0 )
		strcat(Item, ", reverse oblique");
	    else if ( strcmp(pStr, "ri") == 0 )
		strcat(Item, ", reverse italic");
	    else if ( strcmp(pStr, "ot") == 0 )
		strcat(Item, ", other");
	    else {
		strcat(Item, ", ");
		strcat(Item, pStr);
	    }
	    *pHyphen = '-';
	}
	
	ListItem = XmStringCreate(Item, XmFONTLIST_DEFAULT_TAG);
	if ( !XmComboBoxItemExists(FontweightList, ListItem) )
	    XmComboBoxAddItem(FontweightList, ListItem, 0);
	XmStringFree(ListItem);
	++index;
    }
    /*
     * Abschliessend versuchen wir noch, eine sinnvolle Vorgabe
     * einzustellen.
     */
    if ( *OldInput ) {
	Text = XmStringCreate(OldInput, XmFONTLIST_DEFAULT_TAG);
	if ( XmComboBoxItemExists(FontweightList, Text) )
	    XmComboBoxSelectItem(FontweightList, Text, True);
	XmStringFree(Text);
    }
    if ( XmComboBoxGetSelectedPos(FontweightList) == 0 ) {
	Text = XmStringCreate("medium", XmFONTLIST_DEFAULT_TAG);
	if ( XmComboBoxItemExists(FontweightList, Text) )
	    XmComboBoxSelectItem(FontweightList, Text, True);
	else
	    XmComboBoxSelectPos(FontweightList, 1, True);
	XmStringFree(Text);
    }
    XtFree(OldInput);
} /* ListFontStyles */


char *DefaultFontSizes[] = {
    "6", "8", "9", "10", "12", "14", "16", "20", "24", "30", "36", 
    "42", "48", "64",  
    NULL
};


void InsertFontSize(char *Item)
{
    XmString      ListItem;
    XmStringTable Items;
    int           ItemCount, index, len1, len2, result;
    char          *pOctets;

    ListItem = XmStringCreate(Item, XmFONTLIST_DEFAULT_TAG);
    if ( !XmComboBoxItemExists(FontsizeList, ListItem) )
        XmComboBoxAddItem(FontsizeList, ListItem, 0);
    XmStringFree(ListItem);

/*    
    len2 = strlen(Item);
    XtVaGetValues(FontsizeList, XmNitems,     &Items, 
	                        XmNitemCount, &ItemCount, NULL);
    for ( index = 1; index <= ItemCount; ++index ) {
	if ( !XmStringGetLtoR(Items[index - 1], XmFONTLIST_DEFAULT_TAG, 
			&pOctets) ) continue;
	len1 = strlen(pOctets);
	if ( (len2 < len1) ||
	     ((len2 == len1) && 
	      ((result = strcmp(Item, pOctets)) <= 0)) ) {
	    XtFree((char *) pOctets);
	    if ( result == 0 ) return;
	    break;
	}
	XtFree((char *) pOctets);
    }
    
    ListItem = XmStringCreate(Item, XmFONTLIST_DEFAULT_TAG);
    XmComboBoxAddItem(FontsizeList, ListItem, index);
    XmStringFree(ListItem);
*/
} /* InsertFontSize */

void SortCB(Widget w, XtPointer pClientData, XtPointer pCallData)
{
    XmComboBoxSortingPtr pData = (XmComboBoxSortingPtr) pCallData;
    char                 *pOctets;
    int                  len1, len2;

    switch ( pData->operation ) {
        case XmOP_INIT:
            XmStringGetLtoR(pData->item, XmFONTLIST_DEFAULT_TAG,
                            (char **) &(pData->item_internal));
            break;
        case XmOP_DONE:
            XtFree((char *) pData->item_internal);
            break;
        case XmOP_COMPARE:
	    XmStringGetLtoR(pData->item, XmFONTLIST_DEFAULT_TAG, 
			    &pOctets);
	    len1 = strlen(pOctets); 
	    len2 = strlen((char *) pData->item_internal);
	    if ( len2 < len1 ) 
	        pData->result = -1;
	    else if ( len2 == len1 )
	        pData->result = strcmp((char *) pData->item_internal, pOctets);
	    else
	        pData->result = 1;
	    XtFree((char *) pOctets);
            break;
    }
} /* SortCB */

/* --------------------------------------------------------------------
 * 
 *   Prefix	    -Foundry-Family-
 *   FullPrefix	    -Foundry-Family-weight-slant-
 */
void ListFontSizes(char *Prefix, char *FullPrefix, char *pEncoding, 
                   int index)
{
    XmString ListItem, Text;
    int      i, lenPrefix, lenFullPrefix;
    Boolean  Scalable;
    char     *OldInput;
    
    OldInput = XmComboBoxGetString(FontsizeList);
    if ( *OldInput == 0 ) {
	XtFree(OldInput);
	OldInput = XtNewString("12");
    }
    
    DontUseTimer = True;
    XmComboBoxDeleteAllItems(FontsizeList);
    XmComboBoxClearItemSelection(FontsizeList);
    
    Scalable      = False;
    lenPrefix     = strlen(Prefix);
    lenFullPrefix = strlen(FullPrefix);
    
    while ( (index < FontCount) &&
            (strncmp(Prefix, FontNames[index], lenPrefix) == 0) ) {
	
	if  ( !CompareEncoding(FontNames[index], pEncoding) ) {
	    ++index; continue;
	}
	
	if ( strncmp(FullPrefix, FontNames[index], lenFullPrefix) == 0 ) {
	    char *pHyphen, *pStr;
	    int  count;
	    
	    pHyphen = FontNames[index] - 1;
	    for ( count = 0; count <= 7; ++count ) {
		pHyphen = strchr(pHyphen + 1, '-');
		if ( pHyphen == NULL ) break;
	    }
	    if ( pHyphen != NULL ) {
		pStr = pHyphen + 1;
		pHyphen = strchr(pHyphen + 1, '-');
		if ( pHyphen != NULL ) {
		    char backup;
		    
		    if ( pStr + 1 == pHyphen )
			Scalable = True;
		    else {
			backup = *--pHyphen; *pHyphen = 0;
			InsertFontSize(pStr);
			*pHyphen = backup;
		    }
		}
	    }
	}
	
	++index;
    }    
    
    /*
     * Falls darunter ein frei skalierbarer Font war, gib' als
     * Zugabe noch ein paar Standardwerte in die Liste.
     */
    if ( Scalable ) {
	for ( i = 0; DefaultFontSizes[i] != NULL; ++i )
	    InsertFontSize(DefaultFontSizes[i]);
	Text = XmStringCreate(
		"This is a scalable font.", XmFONTLIST_DEFAULT_TAG);
    } else
	Text = XmStringCreate(
		"This is a fixed font.", XmFONTLIST_DEFAULT_TAG);
    XtVaSetValues(FontTypeLabel, XmNlabelString, Text, NULL);
    XmStringFree(Text);
    
    if ( Scalable ) {
	XmComboBoxSetString(FontsizeList, OldInput);
	UpdatePreview();
    } else {
	ListItem = XmStringCreate(OldInput, XmFONTLIST_DEFAULT_TAG);
	if ( XmComboBoxItemExists(FontsizeList, ListItem) )
	    XmComboBoxSelectItem(FontsizeList, ListItem, True);
	else {
	    XmStringFree(ListItem);
	    ListItem = XmStringCreate("12", XmFONTLIST_DEFAULT_TAG);
	    if ( XmComboBoxItemExists(FontsizeList, ListItem) )
		XmComboBoxSelectItem(FontsizeList, ListItem, True);
	    else
		XmComboBoxSelectPos(FontsizeList, 1, True);
	}
	XmStringFree(ListItem);
    }
    XtFree(OldInput);
    DontUseTimer = False;
} /* ListFontSizes */


char *GetXLFD()
{
    static char XLFD[1024];
           char *pInput;
    
    strcpy(XLFD, GetFontname());
    strcat(XLFD, GetFontWeightSlant());
    strcat(XLFD, "*-*-*-");
    pInput = XmComboBoxGetString(FontsizeList);
    if ( *pInput ) {
	char *pStr;
	
	pStr = RemoveSpaces(pInput);
	strcat(XLFD, pStr);
	strcat(XLFD, "0");
    } else
	strcat(XLFD, "*");
    XtFree(pInput);
    strcat(XLFD, "-*-*-*-*-");
    strcat(XLFD, GetFontEncoding());
    return XLFD;
} /* GetXLFD */


void UpdatePreview()
{
    char     *XLFD;
    XmString Name;
    
    XLFD = GetXLFD();
    Name = XmStringCreate(XLFD, XmFONTLIST_DEFAULT_TAG);
    XtVaSetValues(XLFDLabel, XmNlabelString, Name, NULL);
    XmStringFree(Name);
    
    if ( PreviewFontStruct ) {
	XtVaSetValues(FontPreviewWindow, XmNfontList, OrigFontList, NULL);
	XmFontListFree(PreviewFontList);
	XFreeFont(XtDisplay(FontPreviewWindow), PreviewFontStruct);
	PreviewFontStruct = NULL;
    }
    
    PreviewFontStruct = XLoadQueryFont(XtDisplay(FontPreviewWindow), XLFD);
    if ( PreviewFontStruct ) {
	PreviewFontList = XmFontListCreate(PreviewFontStruct, 
                                           XmSTRING_DEFAULT_CHARSET);
	XtVaSetValues(FontPreviewWindow, XmNfontList, PreviewFontList, NULL);
    }
    
    Name = XmStringCreate("AaBbCcYyZz", XmFONTLIST_DEFAULT_TAG);
    XtVaSetValues(FontPreviewWindow, XmNlabelString, Name, NULL);
    XmStringFree(Name);
} /* UpdatePreview */


void FontnameListSelectionCB(Widget w, XtPointer pClientData, 
                             XtPointer pCallData)
{
    char *pFontname, *pEncoding;
    int  i;
    
    pFontname    = GetFontname();
    pEncoding    = GetFontEncoding();
    i = FindFont(pFontname);
    ListFontStyles(pFontname, pEncoding, i);
} /* FontnameListSelectionCB */

void FontweightListSelectionCB(Widget w, XtPointer pClientData, 
                               XtPointer pCallData)
{
    char *pFontname, *pEncoding, *pWeightSlant;
    int  i;
    char Prefix[256];
    
    pFontname    = GetFontname();
    pEncoding    = GetFontEncoding();
    i = FindFont(pFontname);
    pWeightSlant = GetFontWeightSlant();
    strcpy(Prefix, pFontname); strcat(Prefix, pWeightSlant);
    ListFontSizes(pFontname, Prefix, pEncoding, i);
} /* FontweightListSelectionCB */

void FontsizeListSelectionCB(Widget w, XtPointer pClientData, 
                             XtPointer pCallData)
{
    UpdatePreview();
} /* FontsizeListSelectionCB */


void UpdateProc(XtPointer ClientData, XtIntervalId *ID)
{
    PendingUpdate = False;
    UpdatePreview();
}

void FontsizeValueChangeCB(Widget w, XtPointer pClientData, 
                           XtPointer pCallData)
{
    XmTextVerifyPtr pData = (XmTextVerifyPtr) pCallData;
    XmTextBlock     pNewText = pData->text;
    int             i;
    
    for ( i = 0; i < pNewText->length; ++i )
        if ( ((pNewText->ptr)[i] < '0') ||
	     ((pNewText->ptr)[i] > '9') ) {
	    pData->doit = False;
	    return;
	}
    
    if ( PendingUpdate ) XtRemoveTimeOut(TimerID);
    if ( !DontUseTimer ) {
	TimerID = XtAppAddTimeOut(XtWidgetToApplicationContext(w), 
                              1000, 
			      (XtTimerCallbackProc) UpdateProc, 
			      NULL);
	PendingUpdate = True;
    }
}

void LatinOnlyCB(Widget w, XtPointer pClientData, XtPointer pCallData)
{
    GetListOfFonts();
    ListFontnames();
} /* LatinOnlyCB */

void QuitCB(Widget w, XtPointer pClientData, XtPointer pCallData)
{
    puts(GetXLFD());
    exit(0);
} /* QuitCB */

void CancelCB(Widget w, XtPointer pClientData, XtPointer pCallData)
{
    exit(0);
} /* CancelCB */

static void CreateDialog(Widget Form)
{
    FontnameList = XtVaCreateManagedWidget(
    	"fontnamelist", xmComboBoxWidgetClass, Form,
	XmNresizable,			False, 
    	XmNcolumns,			30,
    	XmNsorted,			True,
    	XmNeditable,			False,
    	XmNselectionPolicy,		XmBROWSE_SELECT,
    	XmNstaticList,			True,
    	XmNvisibleItemCount,		10,
    	XmNleftAttachment,		XmATTACH_FORM,
    	XmNleftOffset,			8,
    	NULL);
    FontnameLabel = XtVaCreateManagedWidget(
    	"fontnamelabel", xmLabelWidgetClass, Form,
    	XmNleftAttachment,		XmATTACH_FORM,
    	XmNleftOffset,			8,
    	XmNtopAttachment,		XmATTACH_FORM,
    	XmNtopOffset,			8,
    	XmNalignment,			XmALIGNMENT_BEGINNING,
    	NULL);
    XtVaSetValues(FontnameList,
    	XmNtopAttachment,		XmATTACH_WIDGET,
    	XmNtopWidget,			FontnameLabel,
    	NULL);
    
    FontweightList = XtVaCreateManagedWidget(
    	"fontweightlist", xmComboBoxWidgetClass, Form,
	XmNresizable,			False, 
    	XmNcolumns,			15,
    	XmNeditable,			False,
    	XmNselectionPolicy,		XmBROWSE_SELECT,
    	XmNstaticList,			True,
    	XmNvisibleItemCount,		10,
    	XmNleftAttachment,		XmATTACH_WIDGET,
    	XmNleftWidget,			FontnameList,
    	XmNleftOffset,			8,
    	XmNtopAttachment,		XmATTACH_OPPOSITE_WIDGET,
    	XmNtopWidget,			FontnameList,
    	NULL);
    FontweightLabel = XtVaCreateManagedWidget(
    	"fontweightlabel", xmLabelWidgetClass, Form,
    	XmNleftAttachment,		XmATTACH_WIDGET,
    	XmNleftWidget,			FontnameList,
    	XmNleftOffset,			8,
    	XmNtopAttachment,		XmATTACH_FORM,
    	XmNtopOffset,			8,
    	XmNalignment,			XmALIGNMENT_BEGINNING,
    	NULL);
    
    FontsizeList = XtVaCreateManagedWidget(
    	"fontsizelist", xmComboBoxWidgetClass, Form,
	XmNresizable,			False, 
    	XmNcolumns,			9,
    	XmNeditable,			True,
    	XmNsorted,			True,
	XmNmaxLength,			3, 
    	XmNselectionPolicy,		XmBROWSE_SELECT,
    	XmNstaticList,			True,
    	XmNvisibleItemCount,		10,
    	XmNleftAttachment,		XmATTACH_WIDGET,
    	XmNleftWidget,			FontweightList,
    	XmNleftOffset,			8,
    	XmNtopAttachment,		XmATTACH_OPPOSITE_WIDGET,
    	XmNtopWidget,			FontweightList,
    	NULL);
    XtAddCallback(FontsizeList, XmNsortingCallback, (XtCallbackProc) SortCB, NULL);
    FontsizeLabel = XtVaCreateManagedWidget(
    	"fontsizelabel", xmLabelWidgetClass, Form,
    	XmNleftAttachment,		XmATTACH_WIDGET,
    	XmNleftWidget,			FontweightList,
    	XmNleftOffset,			8,
    	XmNtopAttachment,		XmATTACH_FORM,
    	XmNtopOffset,			8,
    	XmNalignment,			XmALIGNMENT_BEGINNING,
    	NULL);

    Sepp = XtVaCreateManagedWidget(
    	"sepp", xmSeparatorWidgetClass, Form,
    	XmNorientation,			XmVERTICAL,
    	XmNtopAttachment,		XmATTACH_FORM,
    	XmNbottomAttachment,		XmATTACH_FORM,
    	XmNleftAttachment,		XmATTACH_WIDGET,
    	XmNleftWidget,			FontsizeList,
    	XmNleftOffset,			8,
    	NULL);

#ifdef USE_BUTTONFACELIB
    Okay = XmCreateStandardPushButton(Form, "okay", XmPUSHBUTTON_OK, NULL, 0);
    XtManageChild(Okay);
#else
    Okay = XtVaCreateManagedWidget("okay", xmPushButtonWidgetClass, 
                                   Form, NULL);
#endif
    XtVaSetValues(Okay,
    	XmNleftAttachment,		XmATTACH_WIDGET,
    	XmNleftWidget,			Sepp,
    	XmNleftOffset,			8,
    	XmNrightAttachment,		XmATTACH_FORM,
    	XmNrightOffset,			8,
    	XmNtopAttachment,		XmATTACH_FORM,
    	XmNtopOffset,			8,
    	NULL);
    XtAddCallback(Okay, XmNactivateCallback, (XtCallbackProc) QuitCB, NULL);
#ifdef USE_BUTTONFACELIB
    Cancel = XmCreateStandardPushButton(Form, "cancel", XmPUSHBUTTON_CANCEL, NULL, 0);
    XtManageChild(Cancel);
#else
    Cancel = XtVaCreateManagedWidget("cancel", xmPushButtonWidgetClass, 
                                     Form, NULL);
#endif
    XtVaSetValues(Cancel,
    	XmNleftAttachment,		XmATTACH_WIDGET,
    	XmNleftWidget,			Sepp,
    	XmNleftOffset,			8,
    	XmNrightAttachment,		XmATTACH_FORM,
    	XmNrightOffset,			8,
    	XmNtopAttachment,		XmATTACH_WIDGET,
    	XmNtopWidget,			Okay,
    	XmNtopOffset,			8,
    	NULL);
    XtAddCallback(Cancel, XmNactivateCallback, (XtCallbackProc) CancelCB, NULL);
#ifdef USE_BUTTONFACELIB
    Help = XmCreateStandardPushButton(Form, "Help", XmPUSHBUTTON_HELP, NULL, 0);
    XtManageChild(Help);
#else
    Help = XtVaCreateManagedWidget("help", xmPushButtonWidgetClass, 
                                    Form, NULL);
#endif
    XtVaSetValues(Help,
    	XmNleftAttachment,		XmATTACH_WIDGET,
    	XmNleftWidget,			Sepp,
    	XmNleftOffset,			8,
    	XmNrightAttachment,		XmATTACH_FORM,
    	XmNrightOffset,			8,
    	XmNtopAttachment,		XmATTACH_WIDGET,
    	XmNtopWidget,			Cancel,
    	XmNtopOffset,			8,
    	XmNsensitive,			False,
    	NULL);

    XtVaSetValues(Form, 
	XmNdefaultButton,		Okay, 
	XmNcancelButton,		Cancel, 
	NULL);

    ToggleFrame = XtVaCreateManagedWidget(
    	"toggleframe", xmFrameWidgetClass, Form,
	XmNleftAttachment,		XmATTACH_FORM, 
	XmNleftOffset,			8, 
    	XmNtopAttachment,		XmATTACH_WIDGET,
    	XmNtopWidget,			FontnameList,
    	XmNtopOffset,			8,
	XmNrightAttachment,		XmATTACH_WIDGET, 
	XmNrightWidget,			Sepp, 
	XmNrightOffset,			8, 
	NULL);

    ToggleRow = XtVaCreateManagedWidget(
    	"togglerow", xmRowColumnWidgetClass, ToggleFrame, 
	XmNorientation,			XmVERTICAL, 
	NULL);

    ScalableToggle = XtVaCreateManagedWidget(
    	"scalabletoggle", xmToggleButtonWidgetClass, ToggleRow,
	XmNindicatorType,		XmN_OF_MANY, 
    	XmNalignment,			XmALIGNMENT_BEGINNING,
        NULL);
    XtAddCallback(ScalableToggle, XmNvalueChangedCallback, 
                  (XtCallbackProc) LatinOnlyCB, NULL);

    IsoFontsToggle = XtVaCreateManagedWidget(
    	"isofontstoggle", xmToggleButtonWidgetClass, ToggleRow,
	XmNindicatorType,		XmN_OF_MANY, 
    	XmNalignment,			XmALIGNMENT_BEGINNING,
        NULL);
    XtAddCallback(IsoFontsToggle, XmNvalueChangedCallback, 
                  (XtCallbackProc) LatinOnlyCB, NULL);

    XLFDLabel = XtVaCreateManagedWidget(
    	"xlfdlabel", xmLabelWidgetClass, Form, 
    	XmNleftAttachment,		XmATTACH_FORM,
    	XmNleftOffset,			8,
    	XmNtopAttachment,		XmATTACH_WIDGET,
    	XmNtopWidget,			ToggleFrame,
    	XmNtopOffset,			8,
	XmNrightAttachment,		XmATTACH_WIDGET, 
	XmNrightWidget,			Sepp, 
	XmNrightOffset,			8, 
    	XmNalignment,			XmALIGNMENT_BEGINNING,
    	NULL);

    FontTypeLabel = XtVaCreateManagedWidget(
    	"fonttype", xmLabelWidgetClass, Form,
	XmNresizable,			False, 
    	XmNleftAttachment,		XmATTACH_FORM,
    	XmNleftOffset,			8,
    	XmNtopAttachment,		XmATTACH_WIDGET,
    	XmNtopWidget,			XLFDLabel,
    	XmNtopOffset,			8,
	XmNrightAttachment,		XmATTACH_WIDGET, 
	XmNrightWidget,			Sepp, 
	XmNrightOffset,			8, 
    	XmNalignment,			XmALIGNMENT_BEGINNING,
	NULL);

    FontPreviewFrame = XtVaCreateManagedWidget(
    	"previewframe", xmFrameWidgetClass, Form,
    	XmNrightAttachment,		XmATTACH_WIDGET,
    	XmNrightWidget,			Sepp,
    	XmNrightOffset,			8,
    	XmNtopAttachment,		XmATTACH_WIDGET,
    	XmNtopWidget,			FontTypeLabel,
    	XmNtopOffset,			8,
    	XmNbottomAttachment,		XmATTACH_FORM,
    	XmNbottomOffset,		8,
    	XmNleftAttachment,		XmATTACH_OPPOSITE_WIDGET,
    	XmNleftWidget,			FontTypeLabel,
    	XmNshadowThickness,		2,
    	XmNshadowType,			XmSHADOW_IN,
    	NULL);
    FontPreviewWindow = XtVaCreateManagedWidget(
    	"previewwindow", xmLabelWidgetClass, FontPreviewFrame,
    	NULL);
    XtVaGetValues(FontPreviewWindow, 
                  XmNfontList, &OrigFontList, NULL);

    XtAddCallback(FontnameList, XmNselectionCallback, 
                  (XtCallbackProc) FontnameListSelectionCB, NULL);
    XtAddCallback(FontweightList, XmNselectionCallback, 
                  (XtCallbackProc) FontweightListSelectionCB, NULL);
    XtAddCallback(FontsizeList, XmNselectionCallback, 
                  (XtCallbackProc) FontsizeListSelectionCB, NULL);
    XtAddCallback(FontsizeList, XmNmodifyVerifyCallback, 
                  (XtCallbackProc) FontsizeValueChangeCB, NULL);
} /* CreateDialog */


void TraverseTree(Widget w, Pixel Background)
{
    WidgetList Children;
    int        NumChildren, i;
    
    if ( XtIsSubclass(w, compositeWidgetClass) ) {
        XtVaGetValues(w, XmNchildren, &Children, 
                         XmNnumChildren, &NumChildren, NULL);
        for ( i = 0; i < NumChildren; ++i )
            TraverseTree(Children[i], Background);
    } else if ( XtIsSubclass(w, xmListWidgetClass) ||
                XtIsSubclass(w, xmTextWidgetClass) ||
                XtIsSubclass(w, xmTextFieldWidgetClass) ) {
        XtVaSetValues(w, XmNbackground, Background, NULL);
    }
}

void SetInputBackground(Widget Tree)
{
    XrmValue from, to;
    Pixel Background;
    
    from.addr = "white"; from.size = strlen((char *) from.addr) + 1;
    to.addr   = NULL;
    XtConvert(Tree, XmRString, &from, XmRPixel, &to);
    Background = *((Pixel *) to.addr);
    TraverseTree(Tree, Background);
}


int main(int argc, String argv[])
{
    XtAppContext     AppContext;

#if (XmREVISION > 1)
    XtSetLanguageProc(NULL, NULL, NULL); 
#endif
    TopLevel = XtAppInitialize(&AppContext, "XFontSelDemo", NULL, 0, 
#if (XmREVISION > 1)
        		       &argc, 
#else
                               (Cardinal *) &argc,
#endif
        		       argv, FallbackResources, NULL, 0);
 
    MainForm = XtVaCreateManagedWidget("mainform", xmFormWidgetClass, 
                                       TopLevel, NULL);
    CreateDialog(MainForm);
    SetInputBackground(MainForm);
    
    XtRealizeWidget(TopLevel);
    
    GetListOfFonts();
    ListFontnames();
    
    XtAppMainLoop(AppContext);
    return 0; /* Never will reach this */
} /* main */

/* End of FontSel.c */
