/****************************************************************/
/* file analysis.c

ARIBAS interpreter for Arithmetic
Copyright (C) 1996 O.Forster

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

Address of the author

	Otto Forster
	Math. Institut der LMU
	Theresienstr. 39
	D-80333 Muenchen, Germany

Email	forster@rz.mathematik.uni-muenchen.de
*/
/****************************************************************/
/*
** analysis.c
** transcendental functions
**
** date of last change
** 95-02-21
** 97-04-14:	changed inipi
*/

#include "common.h"

PUBLIC void inianalys	_((void));
PUBLIC int lognum	_((int prec, numdata *nptr, word2 *hilf));
PUBLIC int expnum	_((int prec, numdata *nptr, word2 *hilf));

/*--------------------------------------------------------*/
PRIVATE truc expsym, logsym, sqrtsym, sinsym, cossym;
PRIVATE truc tansym, atansym, atan2sym, asinsym, acossym;
PRIVATE truc pisym;

PRIVATE truc inipi	_((int prec));
PRIVATE int Gget1flt	_((truc symb, numdata *nptr));
PRIVATE truc Fsqrt	_((void));
PRIVATE truc Fexp	_((void));
PRIVATE truc Flog	_((void));
PRIVATE truc Fsin	_((void));
PRIVATE truc Fcos	_((void));
PRIVATE truc Ftan	_((void));
PRIVATE truc Gtrig	_((truc symb));
PRIVATE truc Fatan	_((void));
PRIVATE truc Fatan2	_((void));
PRIVATE truc Fasin	_((void));
PRIVATE truc Facos	_((void));
PRIVATE truc Garcus	_((truc symb));

PRIVATE int atannum	_((int prec, numdata *nptr1, numdata *nptr2,
			   word2 *hilf));
PRIVATE int atanprep	_((int prec, numdata *nptr1, numdata *nptr2,
			   word2 *x, int *segptr));
PRIVATE int trignum	_((int prec, numdata *nptr, word2 *hilf, truc symb));
PRIVATE int expovfl	_((numdata *nptr, word2 *hilf));
PRIVATE long redmod	_((int prec, numdata *nptr, word2 *modul, int modlen,
			   word2 *hilf));
PRIVATE int exp0	_((int prec, word2 *x, int n, word2 *z, word2 *hilf));
PRIVATE int exp0aux	_((word2 *x, int n, unsigned a, int k, word2 *temp));
PRIVATE int exp1aux	_((word2 *x, int n, unsigned a, int k, word2 *temp));
PRIVATE int sin0	_((int prec, word2 *x, int n, word2 *z, word2 *hilf));
PRIVATE int cos0	_((int prec, word2 *x, int n, word2 *z, word2 *hilf));
PRIVATE int log0	_((int prec, word2 *x, int n, word2 *z, word2 *hilf));
PRIVATE unsigned log1_16  _((unsigned x));
PRIVATE int atan0	_((int prec, word2 *x, int n, word2 *z, word2 *hilf));

PRIVATE int curfltprec;

PRIVATE word2 LOG2DAT[]	 =	/* log(2) */
{14, 0x8A0D, 0xB62D, 0x7298, 0x4326, 0x40F3, 0xF6AF, 0x03F2, 0xB398, 0xC9E3,
     0x79AB, 0xD1CF, 0x17F7, 0xB172};
PRIVATE word2 PI4THDAT[] =	 /* pi/4 */
{14, 0x020B, 0xCC74, 0x8A67, 0x4E08, 0x2902, 0x1CD1, 0x80DC, 0x628B, 0xC4C6,
     0xC234, 0x2168, 0xDAA2, 0xC90F};
PRIVATE word2 ATAN4DAT[] =	 /* atan(1/4) */
{14, 0xDE8E, 0xE0DA, 0xE22C, 0xEA40, 0x6A9F, 0x85F9, 0x7DE8, 0xE7BD, 0x5B71,
     0xBAC5, 0x5901, 0xEBF2, 0x3EB6};
#define LOG2(prec)	LOG2DAT + (*LOG2DAT - (prec))
#define PI4TH(prec)	PI4THDAT + (*PI4THDAT - (prec))
#define ATAN4(prec)	ATAN4DAT + (*ATAN4DAT - (prec))

/*----------------------------------------------------------------*/
PUBLIC void inianalys()
{
	expsym	  = newsymsig("exp",   sFBINARY, (truc)Fexp, s_rr);
	sqrtsym	  = newsymsig("sqrt",  sFBINARY, (truc)Fsqrt,s_rr);
	sinsym	  = newsymsig("sin",   sFBINARY, (truc)Fsin, s_rr);
	cossym	  = newsymsig("cos",   sFBINARY, (truc)Fcos, s_rr);
	tansym	  = newsymsig("tan",   sFBINARY, (truc)Ftan, s_rr);
	logsym	  = newsymsig("log",   sFBINARY, (truc)Flog, s_rr);
	atansym	  = newsymsig("arctan",sFBINARY, (truc)Fatan,s_rr);
	atan2sym  = newsymsig("arctan2",sFBINARY, (truc)Fatan2,s_rrr);
	asinsym	  = newsymsig("arcsin",sFBINARY, (truc)Fasin,s_rr);
	acossym	  = newsymsig("arccos",sFBINARY, (truc)Facos,s_rr);

	pisym	  = newsym("pi",   sSCONSTANT, inipi(FLT4PREC));
}
/*------------------------------------------------------------------*/
PRIVATE truc inipi(prec)
int prec;
{
	numdata acc;

	acc.sign = 0;
	acc.digits = PI4TH(prec);
	acc.len = prec;
	acc.expo = -(prec<<4) + 2;    /*   (pi/4)*4   */
	return(mk0float(&acc));
}
/*----------------------------------------------------------------*/
/*
** Setzt nptr->digits = AriBuf 
** und holt float aus argStkPtr nach nptr;
** setzt curfltprec und gibt prec = curfltprec + 1 zurueck.
** Im Fehlerfall Rueckgabewert = ERROR
*/
PRIVATE int Gget1flt(symb,nptr)
truc symb;
numdata *nptr;
{
	int prec, type;

	type = chknum(symb,argStkPtr);
	if(type == aERROR)
		return(aERROR);

	curfltprec = deffltprec();
	prec = curfltprec + 1;
	nptr->digits = AriBuf;
	getnumtrunc(prec,argStkPtr,nptr);
	return(prec);
}
/*----------------------------------------------------------------*/
PRIVATE truc Fsqrt()
{
	numdata acc;
	word2 *hilf, *x;
	long m;
	int prec;
	int sh, len, rlen;

	prec = Gget1flt(sqrtsym,&acc);
	if(prec == aERROR)
		return(brkerr());
	if(acc.sign) {
		error(sqrtsym,err_p0num,*argStkPtr);
		return(brkerr());
	}
	if((len = acc.len)) {
		sh = (curfltprec << 5) + 8;
		sh -= bitlen(*(AriBuf + len - 1)) + ((len - 1) << 4) - 1;
		len = shiftarr(AriBuf,len,sh);
		m = acc.expo - sh;
		if(m & 1) {
			len = shlarr(AriBuf,len,1);
			m -= 1;
		}
		x = AriScratch;
		hilf = AriScratch + aribufSize;
		cpyarr(AriBuf,len,x);
		acc.len = bigsqrt(x,len,AriBuf,&rlen,hilf);
		acc.expo = (m >> 1);
	}
	return(mkfloat(curfltprec,&acc));
}
/*----------------------------------------------------------------*/
PRIVATE truc Fexp()
{
	numdata acc;
	int prec;
	int ret;

	prec = Gget1flt(expsym,&acc);
	if(prec == aERROR)
		return(brkerr());
	ret = expnum(prec,&acc,AriScratch);
	if(ret == aERROR) {
		error(expsym,err_ovfl,voidsym);
		return(brkerr());
	}
	return(mkfloat(curfltprec,&acc));
}
/*----------------------------------------------------------------*/
PRIVATE truc Flog()
{
	numdata acc;
	int prec;
	int ret;

	prec = Gget1flt(logsym,&acc);
	if(prec == aERROR)
		return(brkerr());
	ret = lognum(prec,&acc,AriScratch);
	if(ret == aERROR) {
		error(logsym,err_pnum,*argStkPtr);
		return(brkerr());
	}
	return(mkfloat(curfltprec,&acc));
}
/*----------------------------------------------------------------*/
PRIVATE truc Fsin()
{
	return(Gtrig(sinsym));
}
/*----------------------------------------------------------------*/
PRIVATE truc Fcos()
{
	return(Gtrig(cossym));
}
/*----------------------------------------------------------------*/
PRIVATE truc Ftan()
{
	return(Gtrig(tansym));
}
/*----------------------------------------------------------------*/
PRIVATE truc Gtrig(symb)
truc symb;
{
	numdata acc, acc2;
	int prec;
	int ret;

	prec = Gget1flt(symb,&acc);
	if(prec == aERROR)
		return(brkerr());

	if(symb == sinsym || symb == cossym)
		ret = trignum(prec,&acc,AriScratch,symb);
	else {	/* symb == tansym */
		acc2.digits = AriScratch + aribufSize;
		cpynumdat(&acc,&acc2);
		ret = trignum(prec,&acc2,AriScratch,cossym);
		ret = trignum(prec,&acc,AriScratch,sinsym);
		ret = divtrunc(prec,&acc,&acc2,AriScratch);
	}
	if(ret == aERROR) {
		error(symb,err_ovfl,voidsym);
		return(brkerr());
	}
	return(mkfloat(curfltprec,&acc));
}
/*----------------------------------------------------------------*/
PRIVATE truc Fatan()
{
	truc res;

	ARGpush(constone);
	res = Fatan2();
	ARGpop();
	return(res);
}
/*----------------------------------------------------------------*/
PRIVATE truc Fatan2()
{
	numdata acc1, acc2;
	word2 *hilf;
	int type, prec;
	int ret;

	type = chknums(atan2sym,argStkPtr-1,2);
	if(type == aERROR)
		return(brkerr());
	acc1.digits = AriBuf;
	acc2.digits = AriScratch;
	hilf = AriScratch + aribufSize;
	curfltprec = fltprec(type);
	prec = curfltprec + 1;

	getnumtrunc(prec,argStkPtr-1,&acc1);
	getnumtrunc(prec,argStkPtr,&acc2);
	ret = atannum(prec,&acc1,&acc2,hilf);
	if(ret == aERROR) {
		error(atansym,err_ovfl,voidsym);
		return(brkerr());
	}
	return(mkfloat(curfltprec,&acc1));
}
/*----------------------------------------------------------------*/
PRIVATE truc Fasin()
{
	return(Garcus(asinsym));
}
/*----------------------------------------------------------------*/
PRIVATE truc Facos()
{
	return(Garcus(acossym));
}
/*----------------------------------------------------------------*/
PRIVATE truc Garcus(symb)
truc symb;
{
	numdata acc1, acc2;
	word2 *x, *y, *z, *hilf;
	int prec, prec2, ret, cmp, rlen;
	int n, m;

	prec = Gget1flt(symb,&acc1);
	if(prec == aERROR)
		return(brkerr());
	prec2 = prec + prec;

	x = acc1.digits;
	acc2.digits = y = AriScratch;
	z = AriScratch + aribufSize;
	hilf = z + prec2 + 2;
	setarr(z,prec2,0);
	z[prec2] = 1;

	n = alignfix(prec,&acc1);
	if(n == aERROR || (cmp = cmparr(x,n,z+prec,prec+1)) > 0) {
		error(symb,err_range,*argStkPtr);
		return(brkerr());
	}
	if(cmp == 0)		/* abs(x) = 1 */
		int2numdat(0,&acc2);
	else if(n == 0)		/* x = 0 */
		int2numdat(1,&acc2);
	else {
		m = multbig(x,n,x,n,y,hilf);
		m = sub1arr(y,m,z,prec2+1);	/* z = 1 - x*x */
		m = bigsqrt(y,m,z,&rlen,hilf);
		cpyarr(z,m,y);			/* y = sqrt(1 - x*x) */
		acc2.len = m;
		acc2.sign = 0;
		acc2.expo = -(prec<<4);
	}
	if(symb == asinsym)
		ret = atannum(prec,&acc1,&acc2,hilf);
	else {
		ret = atannum(prec,&acc2,&acc1,hilf);
		cpynumdat(&acc2,&acc1);
	}
	if(ret == aERROR) {
		error(symb,err_ovfl,voidsym);
		return(brkerr());
	}
	return(mkfloat(curfltprec,&acc1));
}
/*----------------------------------------------------------------*/
/*
** Ersetzt nptr1 durch den Arcus-Tangens des Quotienten
** aus nptr1 und nptr2
*/
PRIVATE int atannum(prec,nptr1,nptr2,hilf)
int prec;
numdata *nptr1, *nptr2;
word2 *hilf;
{
	word2 *x, *z;
	int seg, s, sign, m, n;

	x = hilf;
	hilf += prec << 1;
	z = nptr1->digits;

	n = atanprep(prec,nptr1,nptr2,x,&seg);
	if(n < 0)
		return(n);	/* aERROR */
	m = atan0(prec,x,n,z,hilf);
	nptr1->sign = sign = (seg < 0 ? MINUSBYTE : 0);
	if(sign)
		seg = -seg - 2;
	s = ((seg + 2) >> 1) & 0xFFFE;
	if(s) {
		n = multarr(PI4TH(prec),prec,s,hilf);
		if(seg & 2) {
			m = sub1arr(z,m,hilf,n);
		}
		else
			m = addarr(z,m,hilf,n);
	}
	nptr1->len = m;
	nptr1->expo = -(prec << 4);
	return(m);
}
/*----------------------------------------------------------------*/
/*
** Falls Zahl nptr2 groesser als Zahl in nptr1,
** wird nptr1 durch nptr2 dividiert,
** andernfalls wird nptr2 durch nptr1 dividiert.
** Ist len der Rueckgabewert so erhaelt man
**	Ergebnis = (x,len) * (2**16)**(-prec)
** Platz x muss genuegend lang sein
** Arbeitet destruktiv auf nptr1 und nptr2 !!!
*/
PRIVATE int atanprep(prec,nptr1,nptr2,x,segp)
int prec;
numdata *nptr1, *nptr2;
word2 *x;
int *segp;
{
	numdata *temp;
	long diff, sh;
	int n, m, sign1, sign2;
	int cmp;

	sign1 = nptr1->sign;
	sign2 = nptr2->sign;
	n = alignfloat(prec+1,nptr1);
	m = alignfloat(prec+1,nptr2);
	if(!n) {
		if(!m)
			return(aERROR);
		else {
			*segp = (sign2 ? 8 : 0);
			return(0);
		}
	}
	else if(!m) {
		*segp = (sign1 ? -4 : 4);
		return(0);
	}
	if(!sign1)
		*segp = (sign2 ? 4 : 0);
	else
		*segp = (sign2 ? -8 : -4);
	if(sign1 != sign2) {
		temp = nptr1;
		nptr1 = nptr2;
		nptr2 = temp;
	}

	diff = nptr1->expo - nptr2->expo;
	if(diff == 0)
		cmp = (nptr1->digits[prec] > nptr2->digits[prec]);
	else
		cmp = (diff > 0);

	if(cmp) {	       /* nptr1 groesser */
		*segp += 2;
		temp = nptr1;
		nptr1 = nptr2;
		nptr2 = temp;
	}
	n = divtrunc(prec,nptr1,nptr2,x);
	cpyarr(nptr1->digits,n,x);
	sh = (prec << 4) + nptr1->expo;
	n = lshiftarr(x,n,sh);
	return(n);
}
/*----------------------------------------------------------------*/
PRIVATE int trignum(prec,nptr,hilf,symb)
int prec;
numdata *nptr;
word2 *hilf;
truc symb;
{
	word2 *x;
	int m, n;

	m = redmod(prec,nptr,PI4TH(prec),prec,hilf);
	if(m & 1) {
		nptr->len =
		sub1arr(nptr->digits,nptr->len,PI4TH(prec),prec);
	}
	x = hilf;
	hilf += prec;
	n = nptr->len;
	cpyarr(nptr->digits,n,x);
	if(symb == cossym)
		m += 2;
	if((m+1) & 2)
		nptr->len = cos0(prec,x,n,nptr->digits,hilf);
	else
		nptr->len = sin0(prec,x,n,nptr->digits,hilf);
	nptr->sign = (m & 4 ? MINUSBYTE : 0);
	nptr->expo = -(prec << 4);
	return(nptr->len);
}
/*----------------------------------------------------------------*/
/*
** Die durch nptr dargestellte Zahl z wird ersetzt durch exp(z)
** Bei overflow wird aERROR zurueckgegeben
*/
PUBLIC int expnum(prec,nptr,hilf)
int prec;
numdata *nptr;
word2 *hilf;
{
	word2 *x;
	long m;
	int ovfl, n;

	ovfl = expovfl(nptr,hilf);
	if(ovfl > 0)
		return(aERROR);
	else if(ovfl < 0) {
		int2numdat(0,nptr);
		return(0);
	}
	m = redmod(prec+1,nptr,LOG2(prec+1),prec+1,hilf);
	x = hilf;
	hilf += prec;
	n = nptr->len - 1;
	cpyarr(nptr->digits+1,n,x);
	nptr->len = exp0(prec,x,n,nptr->digits,hilf);
	nptr->sign = 0;
	nptr->expo = m - (prec << 4);
	return(nptr->len);
}
/*----------------------------------------------------------------*/
PRIVATE int expovfl(nptr,hilf)
numdata *nptr;
word2 *hilf;
{
	int n;

	if(nptr->expo <= 1) {
		cpyarr(nptr->digits,nptr->len,hilf);
		n = lshiftarr(hilf,nptr->len,nptr->expo);
		if(n == 0 || (n <= 2 && big2long(hilf,n) < exprange))
			return(0);
	}
	return(nptr->sign ? -1 : 1);
}
/*----------------------------------------------------------------*/
/*
** Die durch nptr gegebene Zahl wird destruktiv dargestellt als
**   ((nptr->digits,nptr->len) + ret*(modul,modlen)) * (2**16)**(-prec),
** wobei ret der Rueckgabewert ist.
** Die Zahl (nptr->digits,nptr->len) ist nicht negativ und < (2**16)**prec
** ret kann auch negativ sein
*/
PRIVATE long redmod(prec,nptr,modul,modlen,hilf)
int prec, modlen;
numdata *nptr;
word2 *modul, *hilf;
{
	word2 *x, *quot;
	word4 u;
	long ret;
	int n, len, rlen;

	x = nptr->digits;
	len = lshiftarr(x,nptr->len,(prec << 4) + nptr->expo);
	quot = hilf + prec + 1;
	n = divbig(x,len,modul,modlen,quot,&rlen,hilf);
	if(n <= 2)
		u = big2long(quot,n);
	if(n > 2 || u >= 0x80000000) {
		error(scratch("redmod"),err_ovfl,voidsym);
		return(LONGERROR);
	}
	else
		ret = u;
	if(nptr->sign) {
		ret = -ret;
		if(rlen) {
			rlen = sub1arr(x,rlen,modul,modlen);
			ret--;
		}
	}
	nptr->len = rlen;
	return(ret);
}
/*----------------------------------------------------------------*/
PUBLIC int lognum(prec,nptr,hilf)
int prec;
numdata *nptr;
word2 *hilf;
{
	word2 *x, *z;
	word2 aa[2];
	word4 u;
	long expo;
	int m, n, len;

	if(nptr->sign || nptr->len == 0)
		return(aERROR);
	x = nptr->digits;
	z = hilf;
	hilf += prec + 2;

	normfloat(prec,nptr);
	n = shlarr(x,prec,1);
	expo = nptr->expo + (prec << 4) - 1;
	len = log0(prec,x,n,z,hilf);
	cpyarr(z,len,x);
	if(expo) {
		u = (expo > 0 ? expo : -expo);
		m = long2big(u,aa);
		n = multbig(LOG2(prec),prec,aa,m,z,hilf);
		if(expo > 0) {
			len = addarr(x,len,z,n);
			nptr->sign = 0;
		}
		else if(cmparr(x,len,z,n) >= 0) {
			len = subarr(x,len,z,n);
			nptr->sign = 0;
		}
		else {
			len = sub1arr(x,len,z,n);
			nptr->sign = MINUSBYTE;
		}
	}
	if(len == 0)
		int2numdat(0,nptr);
	else {
		nptr->len = len;
		nptr->expo = -(prec << 4);
	}
	return(len);
}
/*----------------------------------------------------------------*/
/*
** Berechnet die Exponentialfunktion von (x,n) * (2**16)**(-prec)
** Ist len der Rueckgabewert, so erhaelt man
**	Resultat = (z,len) * (2**16)**(-prec)
** Es wird vorausgesetzt, dass n <= prec ist
** Platz hilf muss mindestens prec + 2 lang sein
** Platz z muss mindestens prec + 1 lang sein
*/

PRIVATE int exp0(prec,x,n,z,hilf)
int prec, n;
word2 *x, *z, *hilf;
{
	int len;

	setarr(z,prec,0);
	z[prec] = 1;
	len = prec + 1;
	while(--n >= 0)
		len = exp0aux(z,len,x[n],prec-n,hilf);
	return(len);
}
/*----------------------------------------------------------------*/
/*
** Multipliziert (x,n) mit exp(a * (2**16)**(-k))
** Platz temp muss mindestens n + 2 lang sein
** Arbeitet destruktiv auf x !!!
*/
PRIVATE int exp0aux(x,n,a,k,temp)
word2 *x, *temp;
unsigned a;
int n, k;
{
	int i, m;
	word2 rest;

	if(a == 0)
		return(n);
	temp++;
	cpyarr(x,n,temp);
	m = n;
	for(i=1; m>k; i++) {
		m = multarr(temp+k-1,m-k+1,a,temp-1) - 1;
		m = divarr(temp,m,i,&rest);
		n = addarr(x,n,temp,m);
	}
	return(n);
}
/*----------------------------------------------------------------*/
/*
** Dividiert (x,n) durch exp(a * (2**16)**(-k))
** Platz temp muss mindestens n + 2 lang sein
** Arbeitet destruktiv auf x !!!
*/
PRIVATE int exp1aux(x,n,a,k,temp)
word2 *x, *temp;
unsigned a;
int n, k;
{
	int i, m;
	word2 rest;

	if(a == 0)
		return(n);
	temp++;
	cpyarr(x,n,temp);
	m = n;
	for(i=1; m>k; i++) {
		m = multarr(temp+k-1,m-k+1,a,temp-1) - 1;
		m = divarr(temp,m,i,&rest);
		n = (i&1 ? subarr(x,n,temp,m) : addarr(x,n,temp,m));
	}
	return(n);
}
/*----------------------------------------------------------------*/
/*
** Berechnet die Funktion sin(x) von (x,n) * (2**16)**(-prec)
** Ist len der Rueckgabewert so erhaelt man
**	Resultat = (z,len) * (2**16)**(-prec)
*/
PRIVATE int sin0(prec,x,n,z,hilf)
int prec, n;
word2 *x, *z, *hilf;
{
	word2 *temp, *temp1, *x2;
	unsigned i;
	int len, m, m2;

	m = (prec + 1) << 1;
	temp = hilf + m;
	temp1 = temp + m;
	x2 = temp1 + m;
	cpyarr(x,n,temp);
	m = n;
	cpyarr(temp,m,z);
	len = m;
	m2 = multfix(prec,x,n,x,n,x2,hilf);
	for(i=2; m>0; i+=2) {
		m = multfix(prec,x2,m2,temp,m,temp1,hilf);
		cpyarr(temp1,m,temp);
		m = divarr(temp,m,i*(i+1),hilf);
		if(i & 2)
			len = subarr(z,len,temp,m);
		else
			len = addarr(z,len,temp,m);
	}
	return(len);
}
/*----------------------------------------------------------------*/
/*
** Berechnet die Funktion cos(x) von (x,n) * (2**16)**(-prec)
** Ist len der Rueckgabewert so erhaelt man
**	Resultat = (z,len) * (2**16)**(-prec)
*/
PRIVATE int cos0(prec,x,n,z,hilf)
int prec, n;
word2 *x, *z, *hilf;
{
	word2 *temp, *temp1, *x2;
	int len, m, m2;
	unsigned i;

	m = (prec + 1) << 1;
	temp = hilf + m;
	temp1 = temp + m;
	x2 = temp1 + m;
	for(i=0; i<prec; i++)
		z[i] = 0;
	z[prec] = 1;
	len = prec + 1;
	m2 = multfix(prec,x,n,x,n,x2,hilf);
	cpyarr(x2,m2,temp);
	m = m2;
	for(i=1; ; i+=2) {
		m = divarr(temp,m,i*(i+1),hilf);
		if(i & 2)
			len = addarr(z,len,temp,m);
		else
			len = subarr(z,len,temp,m);
		if(m <= 1)
			break;
		m = multfix(prec,x2,m2,temp,m,temp1,hilf);
		cpyarr(temp1,m,temp);
	}
	return(len);
}
/*----------------------------------------------------------------*/
/*
** Berechnet die Funktion log(x) von (x,n) * (2**16)**(-prec)
** Ist len der Rueckgabewert so erhaelt man
**	Resultat = (z,len) * (2**16)**(-prec)
** Es wird vorausgesetzt, dass (2**16)**prec <= (x,n) < 2*(2**16)**prec
** Platz hilf muss prec + 2 lang sein
** Arbeitet destruktiv auf x !!!
*/
PRIVATE int log0(prec,x,n,z,hilf)
int prec, n;
word2 *x, *z, *hilf;
{
	unsigned u;
	int k, len;

	setarr(z,prec,0);

	k = prec - 1;
	u = log1_16(x[k]);
	len = 0;

	while(1) {
		n = exp1aux(x,n,u,prec-k,hilf);
		while(n <= prec) {
			u--;
			n = exp0aux(x,n,1,prec-k,hilf);
		}
		len = incarr(z+k,len,u);
		u = x[k];
		if(u == 0)
			u = x[k-1];
		else if(u == 1)
			u = 0xFFFF;
		else
			continue;
		if(--k < 0)
			break;
		if(len)
			len++;
	}
	while(len > 0 && z[len-1] == 0)
		len--;
	return(len);
}
/*----------------------------------------------------------------*/
/*
** berechnet 2**16 * log(1 + x*(2**-16))
** gerundet auf ganze Zahl
*/
PRIVATE unsigned log1_16(x)
unsigned x;
{
static word4 logtab[] = {0x49A58844,0x108598B5,0x04081596,0x01008055,
			 0x00400801,0x00100080,0x00040008,0x00010000};
	/*  log(2**m/(2**m - 1)) * 2**32 fuer m = 2,4,...,16 */

	word4 *logptr;
	word4 xx,d,z;
	int m;

	logptr = logtab;
	z = 0;
	xx = x;
	xx <<= 16;
	m = 1;
	while(m < 16) {
		d = xx;
		d >>= 1;
		d += 0x80000000;
		d >>= m;
		if(xx >= d) {
			xx -= d;
			z += *logptr;
		}
		else {
			m += 2;
			logptr++;
		}
	}
	return(z >> 16);
}
/*----------------------------------------------------------------*/
/*
** Berechnet die Funktion atan(x) von (x,n) * (2**16)**(-prec)
** Ist len der Rueckgabewert so erhaelt man
**	Resultat = (z,len) * (2**16)**(-prec)
** Es wird vorausgesetzt, dass (x,n) < (1/2)*(2**16)**prec
** Platz hilf muss 8*(prec+1) lang sein
*/
PRIVATE int atan0(prec,x,n,z,hilf)
int prec, n;
word2 *x, *z, *hilf;
{
	word2 *u, *v, *y, *temp, *temp1, *x2, *arctan;
	int i, k, m, m1, m2, len;

	i = (prec+1) << 1;
	u = temp = hilf;
	v = temp1 = temp + i;
	y = x2 = temp1 + i;
	hilf = x2 + i;

	len = 0;				/* z = 0 */
	setarr(u,prec-1,0);
	u[prec-1] = 0x4000;			/* u = 1/4 */
	while(cmparr(x,n,u,prec) >= 0) {	/* x = (x-u)/(1+u*x) */
		arctan = ATAN4(prec);
		cpyarr(x,n,v);
		k = shrarr(v,n,2);
		setarr(v+k,prec-k,0);
		v[prec] = 1;
		n = subarr(x,n,u,prec);
		n = divfix(prec,x,n,v,prec+1,y,hilf);
		cpyarr(y,n,x);
		len = addarr(z,len,arctan,prec);
	}
	len = addarr(z,len,x,n);
	cpyarr(x,n,temp);
	m = n;
	m2 = multfix(prec,x,n,x,n,x2,hilf);
	for(i=3; m>1; i+=2) {
		m = multfix(prec,x2,m2,temp,m,temp1,hilf);
		cpyarr(temp1,m,temp);
		m1 = divarr(temp1,m,i,hilf);
		if(i & 2)
			len = subarr(z,len,temp1,m1);
		else
			len = addarr(z,len,temp1,m1);
	}
	return(len);
}
/******************************************************************/
