## This module handles the configuration file for Sympa.

package Conf;

use Log;
use Language;

require Exporter;
use Carp;

@ISA = qw(Exporter);
@EXPORT = qw(%Conf);

my @valid_options = qw(
		       avg bounce_warn_rate bounce_halt_rate
		       clean_delay_queue clean_delay_queueauth clean_delay_queuemod 
		       cookie create_list db_host db_name db_passwd db_type db_user 
		       default_list_priority edit_list email etc
		       global_remind home host lang listmaster log_socket_type 
		       max_size maxsmtp msgcat nrcpt owner_priority pidfile queue 
		       queueauth queuebounce queuedigest queueexpire queuemod queueoutgoing
		       loop_command_max loop_command_sampling_delay loop_command_decrease_factor
		       remind_return_path request_priority sendmail sleep 
		       sort sympa_priority syslog umask welcome_return_path wwsympa_url
                       ca_path openssl
);
my %valid_options = ();
map { $valid_options{$_}++; } @valid_options;

my %Default_Conf = 
    ('home'    => '--DIR--/expl',
     'etc'     => '--DIR--/etc',
     'ca_path' => '--DIR--/etc/ssl.crt',
     'umask'   => '027',
     'syslog'  => 'LOCAL1',
     'nrcpt'   => 25,
     'avg'     => 10,
     'maxsmtp' => 20,
     'sendmail'=> '/usr/sbin/sendmail',
     'openssl' => '',
     'host'    => undef,
     'email'   => 'sympa',
     'pidfile' => '--DIR--/sympa.pid',
     'msgcat'  => '--DIR--/nls',
     'sort'    => 'fr,ca,be,ch,uk,edu,*,com',
     'queue'   => undef,
     'queuedigest'=> '--DIR--/spool/digest',
     'queuemod'   => '--DIR--/spool/moderation',
     'queueexpire'=> '--DIR--/spool/expire',
     'queueauth'  => '--DIR--/spool/auth',
     'queueoutgoing'  => '--DIR--/spool/outgoing',
     'queuebounce'  => undef,     
     'sleep'      => 5,
     'clean_delay_queue'    => 1,
     'clean_delay_queuemod' => 10,
     'clean_delay_queueauth' => 3,
     'log_socket_type'      => 'unix',
     'remind_return_path' => 'owner',
     'welcome_return_path' => 'owner',
     'db_type' => '',
     'db_name' => 'sympa',
     'db_host' => '',
     'db_user' => '', 
     'db_passwd'  => '',
     'listmaster' => undef,
     'default_list_priority' => 5,
     'sympa_priority' => 1,
     'request_priority' => 0,
     'owner_priority' => 9,
     'lang' => 'us',
     'max_size' => 5242880,
     'edit_list' => 'owner',
     'create_list' => 'listmaster',
     'global_remind' => 'listmaster',
     'wwsympa_url' => undef,
     'bounce_warn_rate' => '30',
     'bounce_halt_rate' => '50',
     'cookie' => undef,
     'loop_command_max' => 200,
     'loop_command_sampling_delay' => 3600,
     'loop_command_decrease_factor' => 0.5
   );
   
%Conf = ();

## Loads and parses the configuration file. Reports errors if any.
sub load {
    my $config = shift;
    my $line_num = 0;
    my $config_err = 0;
    my($i, %o);
    
    ## Open the configuration file or return and read the lines.
    unless (open(IN, $config)) {
	printf STDERR  "Unable to open %s: %s\n", $config, $!;
	return undef;
    }
    while (<IN>) {
	$line_num++;
	next if (/^\s*$/o || /^[\#\;]/o);
	if (/^(\S+)\s+(\S+|\`.*\`)\s*$/io) {
	    my($keyword, $value) = ($1, $2);
	    ##  'tri' is a synonime for 'sort' (for compatibily with old versions)
	    $keyword = 'sort' if ($keyword eq 'tri');
	    ## Special case: `command`
	    if ($value =~ /^\`(.*)\`$/) {
		$value = qx/$1/;
		chomp($value);
	    }
	    $o{$keyword} = [ $value, $line_num ];
	}else {
	    printf STDERR Msg(1, 3, "Malformed line %d: %s"), $config, $_;
	    $config_err++;
	}
    }
    close(IN);

    ## Defaults
    unless (defined $o{'wwsympa_url'}) {
	$o{'wwsympa_url'}[0] = "http://$o{'host'}[0]/wws";
    }    

    ## Check if we have unknown values.
    foreach $i (sort keys %o) {
	next if ($valid_options{$i});
	printf STDERR  "Line %d, unknown field: %s in sympa.conf\n", $o{$i}[1], $i;
	$config_err++;
    }
    ## Do we have all required values ?
    foreach $i (keys %valid_options) {
	unless (defined $o{$i} or defined $Default_Conf{$i}) {
	    printf "Required field not found in sympa.conf: %s\n", $i;
	    $config_err++;
	    next;
	}
	$Conf{$i} = $o{$i}[0] || $Default_Conf{$i};
    }
    
    my $p = 1;
    foreach (split(/,/, $Conf{'sort'})) {
	$Conf{'poids'}{$_} = $p++;
    }
    $Conf{'poids'}{'*'} = $p if ! $Conf{'poids'}{'*'};
    
    if ($config_err) {
	return undef;
    }

    @{$Conf{'listmasters'}} = split(/,/, $Conf{'listmaster'});

    $Conf{'sympa'} = "$Conf{'email'}\@$Conf{'host'}";
    $Conf{'request'} = "$Conf{'email'}-request\@$Conf{'host'}";
    
    return 1;
}

## Check a few files
sub checkfiles {
    my $config_err = 0;
    
    unless (-x $Conf{'sendmail'}) {
	do_log('err', "File %s does not exist or is not executable", $Conf{'sendmail'});
	$config_err++;
    }

    foreach my $qdir ('queue','queuedigest','queuemod','queueexpire','queueauth','queueoutgoing','queuebounce') {
	unless (-d $Conf{$qdir} && -r $Conf{$qdir} && -w $Conf{$qdir} && -x $Conf{$qdir}) {
	    do_log('err', "Spool directory %s does not exist or has incorrect permissions", $Conf{$qdir});
	    $config_err++;
	} 
    }

    return undef if ($config_err);
    return 1;
}

## Packages must return true.
1;
