/*
** Copyright 1998 - 1999 Double Precision, Inc.
** See COPYING for distribution information.
*/

#include	"config.h"
#include	<stdio.h>
#include	"soxwrap/soxwrap.h"
#include	<ctype.h>
#include	<stdlib.h>
#include	<string.h>
#include	<errno.h>
#include	<sys/types.h>
#if TIME_WITH_SYS_TIME
#include	<sys/time.h>
#include	<time.h>
#else
#if HAVE_SYS_TIME_H
#include	<sys/time.h>
#else
#include	<time.h>
#endif
#endif
#include	<netinet/in.h>
#include	<arpa/inet.h>
#if	HAVE_UNISTD_H
#include	<unistd.h>
#endif

#include	"rfc1035_res.h"
#include	"rfc1035.h"


static const char rcsid[]="$Id: rfc1035.c,v 1.5 2000/02/15 05:15:55 mrsam Exp $";

void rfc1035_init_timeout(struct rfc1035_res *res, unsigned s, unsigned n)
{
	res->rfc1035_timeout_initial=s;
	res->rfc1035_timeout_backoff=n;
}

void rfc1035_init_ns(struct rfc1035_res *res, struct in_addr *a, unsigned n)
{
unsigned i;
unsigned j;
struct sockaddr_in sin;
struct timeval	tv;
struct timezone tz;

#if 0
time_t	t;

	/* Do a half-assed job of randomizing the list */

	time(&t);
	j=(unsigned)t;
	j= n ? j % n:1;
#else
	j=0;
#endif

	gettimeofday(&tv, &tz);
	res->id=(unsigned) (tv.tv_sec ^ tv.tv_usec ^ getpid());

	sin.sin_family=AF_INET;
	sin.sin_port=htons(53);

	for (i=0; i == 0 || (i<n && i<MAXNS); i++)
	{
		if (n == 0)
			sin.sin_addr.s_addr=rfc1035_aton("127.0.0.1");
		else
			sin.sin_addr=a[(j+i)%n];
		res->nameservers[i]=sin;
		res->nameserversp[i]=
			(struct sockaddr *)(res->nameservers+i);
	}
	res->rfc1035_nnameservers=i;

}

int rfc1035_init_defaultdomain(struct rfc1035_res *res, const char *p)
{
char	*q;

	if (res->rfc1035_defaultdomain)
		free(res->rfc1035_defaultdomain);

	if ((res->rfc1035_defaultdomain=malloc(strlen(p)+1)) == 0)
		return (-1);

	strcpy(res->rfc1035_defaultdomain, p);
	for (q=res->rfc1035_defaultdomain; *q; q++)
		if (isspace((int)(unsigned char)*q))
		{
			*q=0;
			break;
		}

	return (0);
}

int rfc1035_init_resolv(struct rfc1035_res *res)
{
FILE	*fp=fopen("/etc/resolv.conf", "r");
char rfc1035_buf[512];
struct in_addr ns[MAXNS];
int nns=0;

	if (!fp)	return (-1);
	while (fgets(rfc1035_buf, sizeof(rfc1035_buf), fp))
	{
	char	*p;

		for (p=rfc1035_buf; *p; p++)
			*p=tolower(*p);
		for (p=rfc1035_buf; *p; p++)
			if (isspace((int)(unsigned char)*p))	break;
		if (*p)	*p++=0;

		if (strcmp(rfc1035_buf, "domain") == 0)
		{
			while (p && isspace((int)(unsigned char)*p))
				++p;
			rfc1035_init_defaultdomain(res, p);
			continue;
		}

		if (strcmp(rfc1035_buf, "nameserver"))	continue;
		while (*p && isspace((int)(unsigned char)*p))	p++;
		if (nns < MAXNS)
		{
		char	*q;

			for (q=p; *q && !isspace((int)(unsigned char)*q); q++)
				;
			*q=0;
			ns[nns++].s_addr=rfc1035_aton(p);
		}
	}
	fclose(fp);
	rfc1035_init_ns(res, ns, nns);
	return (0);
}

/************/

struct compresslist {
	struct compresslist *next;
	unsigned offset;
	const char *ptr;
	} ;

static int mkpacketq(void (*)(const char *, unsigned, void *), void *,
		unsigned *,
		const struct rfc1035_query *,
		unsigned,
		const char *,
		struct compresslist *,
		struct rfc1035_res *);

int rfc1035_mkquery(struct rfc1035_res *res,	/* resolver */
			unsigned opcode,	/* opcode */
			int options,		/* various options */
			const struct rfc1035_query *questions,
			unsigned nquestions,
			void (*func)(const char *, unsigned, void *), void *arg)
{
struct {
	unsigned char idhi, idlo;
	unsigned char infohi, infolo;
	unsigned char qhi, qlo;
	unsigned char ahi, alo;
	unsigned char nhi, nlo;
	unsigned char auhi, aulo;
	} header;
unsigned cnt;

	header.idhi= res->id >> 8;
	header.idlo= res->id;
	++res->id;

	header.infohi= (opcode << 4) & 0x70;
	if (options & RFC1035_RESOLVE_RECURSIVE)
		header.infohi |= 1;
	header.infolo=0;
	header.qhi=nquestions >> 8;
	header.qlo=nquestions;
	header.ahi=0;
	header.alo=0;
	header.nhi=0;
	header.nlo=0;
	header.auhi=0;
	header.aulo=0;
	(*func)( (const char *)&header, sizeof(header), arg);
	cnt=sizeof(header);
	if (nquestions)
		if (mkpacketq(func, arg, &cnt, questions, nquestions,
			questions->name, 0, res))	return (-1);
	return (0);
}

int rfc1035_hostnamecmp(const char *p, const char *q)
{
	while (*p || *q)
	{
		if (*p == '.' || *q == '.' )
		{
			if ( (*p && *p != '.') || (*q && *q != '.'))
				return (1);
			while (*p == '.')	++p;
			while (*q == '.')	++q;
			continue;
		}
		if (!*p || !*q)	return (1);
		if ( toupper((int)(unsigned char)*p) !=
			toupper((int)(unsigned char)*q))	return (1);
		++p;
		++q;
	}
	return (0);
}

static struct compresslist *search(struct compresslist *cp, const char *name)
{
	for ( ; cp; cp=cp->next)
	{
		if (rfc1035_hostnamecmp(name, cp->ptr) == 0 &&
			(cp->offset & 0x3FFF) == cp->offset)
			return (cp);
			/* Packet compression uses the two high bits */
	}
	return (0);
}

static int mkpacketq_full(void (*)(const char *, unsigned, void *),
		void *,
		unsigned *,
		const struct rfc1035_query *,
		unsigned,
		const char *,
		struct compresslist *, struct rfc1035_res *);

static int mkpacketq(void (*func)(const char *, unsigned, void *), void *arg,
		unsigned *cnt,
		const struct rfc1035_query *qp,
		unsigned nqp,
		const char *nameptr,
		struct compresslist *comp_list,
		struct rfc1035_res *res)
{
char	*buf;
int	rc;


	if (!res->rfc1035_defaultdomain || strchr(nameptr, '.'))
		return (mkpacketq_full(func, arg, cnt, qp, nqp, nameptr,
			comp_list, res));

	/* Append default domain */

	if ((buf=malloc(strlen(nameptr)+
		strlen(res->rfc1035_defaultdomain)+2)) == 0)
		return (-1);

	strcat(strcat(strcpy(buf, nameptr), "."),
		res->rfc1035_defaultdomain);

	rc=mkpacketq_full(func, arg, cnt, qp, nqp, buf, comp_list, res);
	free(buf);
	return (rc);
}

static int mkpacketq_full(void (*func)(const char *, unsigned, void *),
		void *arg,
		unsigned *cnt,
		const struct rfc1035_query *qp,
		unsigned nqp,
		const char *nameptr,
		struct compresslist *comp_list,
		struct rfc1035_res *res)
{
unsigned llen;
struct	compresslist *cp;

	while (nameptr && *nameptr == '.')
		++nameptr;

	if (!nameptr || !*nameptr)
	{
	struct {
		unsigned char padtail;
		unsigned char qtypehi, qtypelo;
		unsigned char qclasshi, qclasslo;
		} qtail;

		qtail.padtail=0;
		qtail.qtypehi=qp->qtype >> 8;
		qtail.qtypelo=qp->qtype;
		qtail.qclasshi=qp->qclass >> 8;
		qtail.qclasslo=qp->qclass;

		(*func)((const char *)&qtail, sizeof(qtail), arg);
		++qp;
		--nqp;
		*cnt += sizeof(qtail);
		if (nqp)
			return (mkpacketq(func, arg, cnt,
				qp, nqp, qp->name, comp_list, res));
		return (0);
	}

	for (llen=0; nameptr[llen] && nameptr[llen] != '.'; llen++)
		;
	cp=search(comp_list, nameptr);
	if (cp)
	{
	struct {
		unsigned char ptrhi, ptrlo;
		unsigned char qtypehi, qtypelo;
		unsigned char qclasshi, qclasslo;
		} qtail;

		qtail.ptrhi= (cp->offset >> 8) | 0xC0;
		qtail.ptrlo= cp->offset;
		qtail.qtypehi=qp->qtype >> 8;
		qtail.qtypelo=qp->qtype;
		qtail.qclasshi=qp->qclass >> 8;
		qtail.qclasslo=qp->qclass;

		(*func)( (const char *)&qtail, sizeof(qtail), arg);
		++qp;
		--nqp;
		*cnt += sizeof(qtail);

		if (nqp)
			return (mkpacketq(func, arg, cnt,
				qp, nqp, qp->name, comp_list, res));
	}
	else
	{
	unsigned n=llen;
	unsigned char c;
	struct compresslist newc;

		if (n > 63)	return (-1);

		newc.next=comp_list;
		newc.offset= *cnt;
		newc.ptr=nameptr;

		c=(unsigned char)n;
		(*func)( &c, 1, arg);
		(*func)( nameptr, c, arg);
		*cnt += 1+c;
		return (mkpacketq_full(func, arg, cnt,
				qp, nqp, nameptr+llen, &newc, res));
	}
	return (0);
}

/*******************************************************/

unsigned rfc1035_init_nscount(struct rfc1035_res *res)
{
	if (res->rfc1035_nnameservers <= 0)
		rfc1035_init_resolv(res);
	return (res->rfc1035_nnameservers);
}

const struct sockaddr *const *rfc1035_init_nsget(struct rfc1035_res *res)
{
	if (res->rfc1035_nnameservers <= 0)
		rfc1035_init_resolv(res);
	return (res->nameserversp);
}

int rfc1035_wait_reply(int fd, unsigned nsecs)
{
fd_set	fds;
struct	timeval tv;
int	n;

	FD_ZERO(&fds);
	FD_SET(fd, &fds);
	tv.tv_sec=nsecs;
	tv.tv_usec=0;
	while ((n=sox_select(fd+1, &fds, 0, 0, &tv)) < 0)
	{
		if (errno != EINTR)
			break;
	}

	if (n > 0 && FD_ISSET(fd, &fds))
		return (0);
	errno=ETIMEDOUT;
	return (-1);
}

int rfc1035_wait_query(int fd, unsigned nsecs)
{
fd_set	fds;
struct	timeval tv;
int	n;

	FD_ZERO(&fds);
	FD_SET(fd, &fds);
	tv.tv_sec=nsecs;
	tv.tv_usec=0;
	while ((n=sox_select(fd+1, 0, &fds, 0, &tv)) < 0)
	{
		if (errno != EINTR)
			break;
	}

	if (n > 0 && FD_ISSET(fd, &fds))
		return (0);
	errno=ETIMEDOUT;
	return (-1);
}

void rfc1035_ntoa(struct in_addr in, char *buf)
{
union	{
	unsigned char addr[4];
	RFC1035_UINT32 n;
	} u;

int	i;
char	*p;
char	pbuf[4];

	u.n=in.s_addr;
	for (i=0; i<4; i++)
	{
		if (i)	*buf++='.';
		p=pbuf+3;
		*p=0;
		do
		{
			*--p = '0' + (u.addr[i] % 10);
		} while ( (u.addr[i] /= 10) != 0);
		while (*p)
			*buf++ = *p++;
	}
	*buf=0;
}

static RFC1035_UINT32 doaton(const char *p)
{
RFC1035_UINT32 octets[4];
int	i;
int	base;

	for (i=0; i<4; i++)
	{
		if (i > 0)
		{
			if (*p == '\0')	break;
			if (*p != '.')	return ( -1 );
			++p;
		}
		if (*p < '0' || *p > '9')	return (-1);
		octets[i]=0;
		base=10;
		if (*p == '0')	base=8;

		while (*p >= '0' && *p < '0'+base)
			octets[i] = octets[i] * base + (*p++ - '0');
	}
	if (*p)	return ( -1 );
	if (i == 1)	return (octets[0]);
	if (octets[0] > 255)	return ( -1 );
	octets[0] <<= 24;

	if (i == 2)
	{
		if (octets[1] > 0x00FFFFFF)	return ( -1 );
		return (octets[0] | octets[1]);
	}

	if (octets[1] > 255)	return ( -1 );
	octets[1] <<= 16;
	if (i == 3)
	{
		if (octets[2] > 0x0000FFFF)	return ( -1 );
		return (octets[0] | octets[1] | octets[2]);
	}
	if (octets[2] > 255 || octets[3] > 255)	return ( -1 );
	return (octets[0] | octets[1] | (octets[2] << 8) | octets[3]);
}

RFC1035_UINT32 rfc1035_aton(const char *p)
{
RFC1035_UINT32 n=doaton(p);
union	{
	unsigned char addr[4];
	RFC1035_UINT32 n;
	} u;

	u.addr[3]=n;
	u.addr[2]=n >> 8;
	u.addr[1]=n >> 16;
	u.addr[0]=n >> 24;
	return (u.n);
}
