#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>

#include <Y2/Y.h>
#include <Y2/Ylib.h>

#include "../include/string.h"
#include "../include/disk.h"


#ifndef DEF_CON_ARG
# define DEF_CON_ARG    "127.0.0.1:9433"
#endif            


int interrupted;
int verbose;
int allow_change_mode;


/*
 *	Handle interupt signals.
 */
void HandleSignal(int s)
{
	switch(s)
	{
	  case SIGINT:
	    signal(SIGINT, HandleSignal);
	    interrupted = 1;
	    break;

	  case SIGTERM:
            signal(SIGTERM, HandleSignal);
            interrupted = 1;
            break; 

          case SIGPIPE:
            signal(SIGPIPE, HandleSignal);
            interrupted = 1;
            break;
	}

	return;
}


/*
 *	Print help.
 */
void PrintUsage()
{
	printf("\
Usage: yplay [options] <file1> [file2] ...\n\
\n\
    [options] can be any of the following:\n\
\n\
        -m                           Allow change Audio mode as needed\n\
                                     (may cause sound objects played by\n\
                                     other applications to be killed).\n\
        -r                           Repeat untill interrupt (ctrl+c).\n\
        -vol <left> <right>          Specify volume coefficients.\n\
        -v                           Print verbose status messages.\n\
        --recorder <address:port>    Specify which YIFF server to connect to.\n\
\n\
    <file1> [file2] ... are a list of files to be played. Their paths must\n\
    be relative to the server, not your current directory on your computer.\n\
\n"
	);


	return;
}



int main(int argc, char *argv[])
{
	int i, n, playing;
	long cycle;

	char *con_arg = NULL;

	int repeat = 0;
	char **file = NULL;
	int total_files = 0;

	YConnection *con = NULL;
	YEvent event;
	YEventSoundObjectAttributes au_stats;
	YID play_id = YIDNULL;
	Coefficient	vol_left = 1,
			vol_right = 1;

	char tmp_path[1024];

	interrupted = 0;
	verbose = 0;
	allow_change_mode = 0;


	signal(SIGINT, HandleSignal);
        signal(SIGPIPE, HandleSignal);
        signal(SIGTERM, HandleSignal);



	if(argc < 2)
	{
	    PrintUsage();

            free(con_arg);
            con_arg = NULL;  

            return(0);
	}

	/* Parse arguments. */
	for(i = 1; i < argc; i++)
	{
	    if(argv[i] == NULL)
		continue;

	    /* Help. */
	    if(strcasepfx(argv[i], "--h") ||
               strcasepfx(argv[i], "-h") ||
               !strcmp(argv[i], "?")
	    )
	    {
		PrintUsage();

                free(con_arg);
                con_arg = NULL;

		return(0);
	    }


	    /* Connect address. */
	    else if(strcasepfx(argv[i], "--rec") ||
                    strcasepfx(argv[i], "-rec")
	    )
	    {
		i++;
		if(i < argc)
		{
		    free(con_arg);
		    con_arg = StringCopyAlloc(argv[i]);
		}
		else
		{
		    fprintf(stderr,
			"%s: Requires argument.\n",
			argv[i - 1]
		    );
		}
	    }
	    /* Repeat. */
            else if(!strcasecmp(argv[i], "-r"))
            {
                repeat = 1;
            }
            /* Change mode. */
            else if(!strcasecmp(argv[i], "-m"))
            {
                allow_change_mode = 1;
            }
	    /* Volume. */
	    else if(!strcasecmp(argv[i], "-vol"))
            {
		i++;
                if(i < argc)
                {
                    vol_left = atof(argv[i]);
                }
                else
                {
                    fprintf(stderr,
                        "%s: Requires 2 arguments.\n",
                        argv[i - 1]
                    );
                }

                i++;
                if(i < argc)
                {
                    vol_right = atof(argv[i]);
                }
                else
                {
                    fprintf(stderr,
                        "%s: Requires 2 arguments.\n",
                        argv[i - 1]
                    );
                }
            }
            /* Verbose. */
            else if(!strcasecmp(argv[i], "-v"))
            {
                verbose = 1;
            }
	    /* All else assume argument to be a file. */
	    else
	    {
		if(total_files < 0)
		    total_files = 0;

		n = total_files;
		total_files++;

		file = (char **)realloc(
		    file,
		    total_files * sizeof(char *)
		);
		if(file == NULL)
		{
		    total_files = 0;
		    continue;
		}

		file[n] = StringCopyAlloc(argv[i]);
	    }
	}


	/*
	 *	Connect to YIFF server.
	 */
	con = YOpenConnection(
	    NULL,		/* No start argument. */
	    con_arg
	);
	if(con == NULL)
	{
	    printf("%s: Cannot connect to YIFF server.\n",
		(con_arg == NULL) ? DEF_CON_ARG : con_arg
	    );

	    StringFreeArray(file, total_files);
            file = NULL;
            total_files = 0;

            free(con_arg);
            con_arg = NULL;

	    return(-1);
	}



	/*
	 *	Begin playing.
	 */
	do
	{
	    /* Go through each file. */
	    for(i = 0; i < total_files; i++)
	    {
		if(file[i] == NULL)
		    continue;

		/* Get sound object attributes. */
                if(YGetSoundObjectAttributes(
                    con,
                    file[i],
                    &au_stats
                ))
		{
                   fprintf( 
                        stderr,
 "%s: Cannot play.\n",
                        file[i]
                    );
		    continue;
		}

		/* Change audio mode as needed. */
		if((au_stats.format == SndObjTypeDSP) &&
                   allow_change_mode
		)
		{
		    cycle = YCalculateCycle(
			con,
			au_stats.sample_rate,	/* Sample rate. */
			au_stats.channels,	/* Channels. */
			au_stats.sample_size,	/* Bits. */
			4096			/* Fragment size. */
		    );
		    if(cycle < 100)
			cycle = 100;

		    YSetAudioModeValues(
			con,
			au_stats.sample_size,	/* Bits. */
			au_stats.channels,	/* Channels. */
			au_stats.sample_rate,	/* Sample rate. */
			0,			/* Play/record. */
			1,			/* Allow fragmenting. */
			2,			/* Num fragments. */
			4096
		    );
		    YSyncAll(con, True);
		    YSetCycle(con, cycle);
		}

		/* Begin playing this sound object. */
		play_id = YStartPlaySoundObject(
		    con,
		    file[i],
		    0,		/* Start position. */
		    vol_left, vol_right,
		    1		/* One repeat. */
		);
		if(play_id == YIDNULL)
		{
		    fprintf(
			stderr,
 "%s: Cannot play.\n",
			file[i]
		    );
		    continue;
		}

		/* Print info about this sound object as needed. */
                if(verbose)
                {
                    strncpy(
                        tmp_path,
                        file[i],
                        1024
                    );
                    tmp_path[1024 - 1] = '\0';
                    StringShortenFL(tmp_path, 24);
                        
                    switch(au_stats.format)
                    {
                      case SndObjTypeDSP:
                        fprintf(stdout,
"ID: %i  Type: DSP  SmpRate: %i Hz  Bits: %i  Ch: %i  %s\n",
                            play_id,
                            au_stats.sample_rate,
                            au_stats.sample_size,
                            au_stats.channels,
                            tmp_path
                        );
                        break;
                     
                      case SndObjTypeMIDI:
                        fprintf(stdout,
 "ID: %i  Type: MIDI  %s\n",
                            play_id,
                            tmp_path
                        );
                        break;
                        
                      default:
                        fprintf(stdout,
 "ID: %i  Type: *Unknown*  %s\n",
                            play_id,
                            tmp_path
                        );
                    }
                }


		/* Wait untill audio is done playing. */
		playing = 1;
		while(playing && !interrupted)
		{
		    if(YGetNextEvent(
			con,
			&event,
			False		/* Nonblocking. */
		    ) > 0)
		    {
	                switch(event.type)
		        {
		          case YSoundObjectKill:
			    if(event.kill.yid == play_id)
			    {
			        playing = 0;
			    }
			    else
			    {
			        fprintf(
				    stderr,
         "Warning: Unmanaged play ID %ld has stopped.\n",
				    event.kill.yid
			        );
			    }
			    break;

			  case YDisconnect:
			  case YShutdown:
                            StringFreeArray(file, total_files);
			    file = NULL;
			    total_files = 0;

			    interrupted = 1;	/* Consider disconnect as interrupt. */
			    playing = 0;
			    break;
			}
		    }

		    usleep(8000);
		}	/* Wait untill audio is done playing. */

		/* If interupted during play, then stop playing. */
		if(interrupted)
		{
		    if(verbose)
		        printf("Playback interrupted.\n");

		    repeat = 0;
		    break;
		}
	    }
	}
	while(repeat);


	/*
	 *	Disconnect from YIFF server
	 */
	YCloseConnection(con, False);
	con = NULL;

	free(con_arg);
	con_arg = NULL;

	/* Free all file names (if any left). */
        StringFreeArray(file, total_files);
        file = NULL;
        total_files = 0;


	return(0);
}
