/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    5635-34 Springhouse Dr.
    Pleasanton, CA 94588 (USA)
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_sysevents.c,v 1.21 1999/07/16 16:00:20 slouken Exp $";
#endif

/* Handle the event stream, converting X11 events into SDL events */

#include <stdio.h>
#include <setjmp.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#ifdef __SVR4
#include <X11/Sunkeysym.h>
#endif
#include <sys/time.h>

#include "SDL.h"
#include "SDL_sysevents.h"
#include "SDL_video_c.h"
#include "SDL_events_c.h"
#include "SDL_syswm.h"
#include "SDL_lowvideo.h"


/* The translation tables from an X11 keysym to a SDL keysym */
static SDLKey ODD_keymap[256];
static SDLKey MISC_keymap[256];
static SDL_keysym *TranslateKey(XKeyEvent *xkey, SDL_keysym *keysym);

/* Check to see if this is a repeated key.
   (idea shamelessly lifted from GII -- thanks guys! :)
 */
static int KeyRepeat(XEvent *event)
{
	XEvent peekevent;
	int repeated;

	repeated = 0;
	if ( XPending(SDL_Display) ) {
		XPeekEvent(SDL_Display, &peekevent);
		if ( (peekevent.type == KeyPress) &&
		     (peekevent.xkey.keycode == event->xkey.keycode) &&
		     (peekevent.xkey.time == event->xkey.time) ) {
			repeated = 1;
			XNextEvent(SDL_Display, &peekevent);
		}
	}
	return(repeated);
}

static int SDL_DispatchEvent(void)
{
	int posted;
	XEvent xevent;

	XNextEvent(SDL_Display, &xevent);

	posted = 0;
	switch (xevent.type) {

	    /* Gaining mouse coverage? */
	    case EnterNotify: {
		posted = SDL_PrivateAppActive(1, SDL_APPMOUSEFOCUS);
	    }
	    break;

	    /* Gaining input focus? */
	    case FocusIn: {
		posted = SDL_PrivateAppActive(1, SDL_APPINPUTFOCUS);
	    }
	    break;

	    /* Losing mouse coverage? */
	    case LeaveNotify: {
		posted = SDL_PrivateAppActive(0, SDL_APPMOUSEFOCUS);
	    }
	    break;

	    /* Losing input focus? */
	    case FocusOut: {
		posted = SDL_PrivateAppActive(0, SDL_APPINPUTFOCUS);
	    }
	    break;

	    /* Mouse motion? */
	    case MotionNotify: {
		if ( SDL_VideoSurface ) {
			if (SDL_VideoSurface->flags&SDL_FULLSCREEN) {
				posted = SDL_PrivateMouseMotion(0, 1,
						xevent.xmotion.x_root,
						xevent.xmotion.y_root);
			} else {
				posted = SDL_PrivateMouseMotion(0, 0,
						xevent.xmotion.x,
						xevent.xmotion.y);
			}
		}
	    }
	    break;

	    /* Mouse button press? */
	    case ButtonPress: {
		posted = SDL_PrivateMouseButton(SDL_PRESSED, 
					xevent.xbutton.button, 0, 0);
#ifdef DGA_BROKEN_VIEWPORT
		SDL_FixViewPort();
#endif
	    }
	    break;

	    /* Mouse button release? */
	    case ButtonRelease: {
		posted = SDL_PrivateMouseButton(SDL_RELEASED, 
					xevent.xbutton.button, 0, 0);
	    }
	    break;

	    /* Key press? */
	    case KeyPress: {
		SDL_keysym keysym;
		posted = SDL_PrivateKeyboard(SDL_PRESSED,
					TranslateKey(&xevent.xkey, &keysym));
	    }
	    break;

	    /* Key release? */
	    case KeyRelease: {
		SDL_keysym keysym;

		/* Check to see if this is a repeated key */
		if ( ! KeyRepeat(&xevent) ) {
			posted = SDL_PrivateKeyboard(SDL_RELEASED, 
					TranslateKey(&xevent.xkey, &keysym));
		}
	    }
	    break;

	    /* Have we been iconified? */
	    case UnmapNotify: {
		posted = SDL_PrivateAppActive(0,
					(SDL_APPACTIVE|SDL_APPMOUSEFOCUS));
	    }
	    break;

	    /* Have we been restored? */
	    case MapNotify: {
		posted = SDL_PrivateAppActive(1,
					(SDL_APPACTIVE|SDL_APPMOUSEFOCUS));
	    }
	    break;

	    /* Have we been requested to quit? */
	    case ClientMessage: {
		if ( (xevent.xclient.format == 32) &&
		     (xevent.xclient.data.l[0] == SDL_DELWIN) )
		{
			posted = SDL_PrivateQuit();
		}
	    }
	    break;

	    /* Do we need to refresh ourselves? */
	    case Expose: {
		if ( xevent.xexpose.count == 0 ) {
			SDL_UpdateRect(SDL_PublicSurface, 0, 0, 0, 0);
		}
	    }
	    break;

	    default: {
		/* Only post the event if we're watching for it */
		if ( SDL_ProcessEvents[SDL_SYSWMEVENT] == SDL_ENABLE ) {
			SDL_SysWMmsg wmmsg;

			SDL_VERSION(&wmmsg.version);
			wmmsg.subsystem = SDL_SYSWM_X11;
			wmmsg.event.xevent = xevent;
			posted = SDL_PrivateSysWMEvent(&wmmsg);
		}
	    }
	    break;
	}
	return(posted);
}

/* Ack!  XPending() actually performs a blocking read if no events available */
static int SDL_X11_Pending(void)
{
	/* Flush the display connection and look to see if events are queued */
	XFlush(SDL_Display);
	if ( XEventsQueued(SDL_Display, QueuedAlready) ) {
		return(1);
	}

	/* More drastic measures are required -- see if X is ready to talk */
	{
		static struct timeval zero_time;	/* static == 0 */
		int x11_fd;
		fd_set fdset;

		x11_fd = ConnectionNumber(SDL_Display);
		FD_ZERO(&fdset);
		FD_SET(x11_fd, &fdset);
		if ( select(x11_fd+1, &fdset, NULL, NULL, &zero_time) == 1 ) {
			return(XPending(SDL_Display));
		}
	}

	/* Oh well, nothing is ready .. */
	return(0);
}

void SDL_SYS_PumpEvents(void)
{
	/* Keep processing pending events */
	while ( SDL_X11_Pending() ) {
		SDL_DispatchEvent();
	}
}

void SDL_InitOSKeymap(void)
{
	int i;

	/* Odd keys used in international keyboards */
	for ( i=0; i<SDL_TABLESIZE(ODD_keymap); ++i )
		ODD_keymap[i] = SDLK_UNKNOWN;

	/* These X keysyms have 0xFE as the high byte */
	ODD_keymap[XK_dead_circumflex&0xFF] = SDLK_CARET;

	/* Map the miscellaneous keys */
	for ( i=0; i<SDL_TABLESIZE(MISC_keymap); ++i )
		MISC_keymap[i] = SDLK_UNKNOWN;

	/* These X keysyms have 0xFF as the high byte */
	MISC_keymap[XK_BackSpace&0xFF] = SDLK_BACKSPACE;
	MISC_keymap[XK_Tab&0xFF] = SDLK_TAB;
	MISC_keymap[XK_Clear&0xFF] = SDLK_CLEAR;
	MISC_keymap[XK_Return&0xFF] = SDLK_RETURN;
	MISC_keymap[XK_Pause&0xFF] = SDLK_PAUSE;
	MISC_keymap[XK_Escape&0xFF] = SDLK_ESCAPE;
	MISC_keymap[XK_Delete&0xFF] = SDLK_DELETE;

	MISC_keymap[XK_KP_0&0xFF] = SDLK_KP0;		/* Keypad 0-9 */
	MISC_keymap[XK_KP_1&0xFF] = SDLK_KP1;
	MISC_keymap[XK_KP_2&0xFF] = SDLK_KP2;
	MISC_keymap[XK_KP_3&0xFF] = SDLK_KP3;
	MISC_keymap[XK_KP_4&0xFF] = SDLK_KP4;
	MISC_keymap[XK_KP_5&0xFF] = SDLK_KP5;
	MISC_keymap[XK_KP_6&0xFF] = SDLK_KP6;
	MISC_keymap[XK_KP_7&0xFF] = SDLK_KP7;
	MISC_keymap[XK_KP_8&0xFF] = SDLK_KP8;
	MISC_keymap[XK_KP_9&0xFF] = SDLK_KP9;
	MISC_keymap[XK_KP_Insert&0xFF] = SDLK_KP0;
	MISC_keymap[XK_KP_End&0xFF] = SDLK_KP1;	
	MISC_keymap[XK_KP_Down&0xFF] = SDLK_KP2;
	MISC_keymap[XK_KP_Page_Down&0xFF] = SDLK_KP3;
	MISC_keymap[XK_KP_Left&0xFF] = SDLK_KP4;
	MISC_keymap[XK_KP_Begin&0xFF] = SDLK_KP5;
	MISC_keymap[XK_KP_Right&0xFF] = SDLK_KP6;
	MISC_keymap[XK_KP_Home&0xFF] = SDLK_KP7;
	MISC_keymap[XK_KP_Up&0xFF] = SDLK_KP8;
	MISC_keymap[XK_KP_Page_Up&0xFF] = SDLK_KP9;
	MISC_keymap[XK_KP_Delete&0xFF] = SDLK_KP_PERIOD;
	MISC_keymap[XK_KP_Decimal&0xFF] = SDLK_KP_PERIOD;
	MISC_keymap[XK_KP_Divide&0xFF] = SDLK_KP_DIVIDE;
	MISC_keymap[XK_KP_Multiply&0xFF] = SDLK_KP_MULTIPLY;
	MISC_keymap[XK_KP_Subtract&0xFF] = SDLK_KP_MINUS;
	MISC_keymap[XK_KP_Add&0xFF] = SDLK_KP_PLUS;
	MISC_keymap[XK_KP_Enter&0xFF] = SDLK_KP_ENTER;
	MISC_keymap[XK_KP_Equal&0xFF] = SDLK_KP_EQUALS;

	MISC_keymap[XK_Up&0xFF] = SDLK_UP;
	MISC_keymap[XK_Down&0xFF] = SDLK_DOWN;
	MISC_keymap[XK_Right&0xFF] = SDLK_RIGHT;
	MISC_keymap[XK_Left&0xFF] = SDLK_LEFT;
	MISC_keymap[XK_Insert&0xFF] = SDLK_INSERT;
	MISC_keymap[XK_Home&0xFF] = SDLK_HOME;
	MISC_keymap[XK_End&0xFF] = SDLK_END;
	MISC_keymap[XK_Page_Up&0xFF] = SDLK_PAGEUP;
	MISC_keymap[XK_Page_Down&0xFF] = SDLK_PAGEDOWN;

	MISC_keymap[XK_F1&0xFF] = SDLK_F1;
	MISC_keymap[XK_F2&0xFF] = SDLK_F2;
	MISC_keymap[XK_F3&0xFF] = SDLK_F3;
	MISC_keymap[XK_F4&0xFF] = SDLK_F4;
	MISC_keymap[XK_F5&0xFF] = SDLK_F5;
	MISC_keymap[XK_F6&0xFF] = SDLK_F6;
	MISC_keymap[XK_F7&0xFF] = SDLK_F7;
	MISC_keymap[XK_F8&0xFF] = SDLK_F8;
	MISC_keymap[XK_F9&0xFF] = SDLK_F9;
	MISC_keymap[XK_F10&0xFF] = SDLK_F10;
	MISC_keymap[XK_F11&0xFF] = SDLK_F11;
	MISC_keymap[XK_F12&0xFF] = SDLK_F12;
	MISC_keymap[XK_F13&0xFF] = SDLK_F13;
	MISC_keymap[XK_F14&0xFF] = SDLK_F14;
	MISC_keymap[XK_F15&0xFF] = SDLK_F15;

	MISC_keymap[XK_Num_Lock&0xFF] = SDLK_NUMLOCK;
	MISC_keymap[XK_Caps_Lock&0xFF] = SDLK_CAPSLOCK;
	MISC_keymap[XK_Scroll_Lock&0xFF] = SDLK_SCROLLOCK;
	MISC_keymap[XK_Shift_R&0xFF] = SDLK_RSHIFT;
	MISC_keymap[XK_Shift_L&0xFF] = SDLK_LSHIFT;
	MISC_keymap[XK_Control_R&0xFF] = SDLK_RCTRL;
	MISC_keymap[XK_Control_L&0xFF] = SDLK_LCTRL;
	MISC_keymap[XK_Alt_R&0xFF] = SDLK_RALT;
	MISC_keymap[XK_Alt_L&0xFF] = SDLK_LALT;
	MISC_keymap[XK_Meta_R&0xFF] = SDLK_RMETA;
	MISC_keymap[XK_Meta_L&0xFF] = SDLK_LMETA;
	MISC_keymap[XK_Super_L&0xFF] = SDLK_LSUPER; /* Left "Windows" */
	MISC_keymap[XK_Super_R&0xFF] = SDLK_RSUPER; /* Right "Windows */
	MISC_keymap[XK_Mode_switch&0xFF] = SDLK_MODE; /* "Alt Gr" key */

	MISC_keymap[XK_Help&0xFF] = SDLK_HELP;
	MISC_keymap[XK_Print&0xFF] = SDLK_PRINT;
	MISC_keymap[XK_Sys_Req&0xFF] = SDLK_SYSREQ;
	MISC_keymap[XK_Break&0xFF] = SDLK_BREAK;
	MISC_keymap[XK_Menu&0xFF] = SDLK_MENU;
	MISC_keymap[XK_Hyper_R&0xFF] = SDLK_MENU;   /* Windows "Menu" key */
}

static SDL_keysym *TranslateKey(XKeyEvent *xkey, SDL_keysym *keysym)
{
	KeySym xsym;

	/* Get the raw keyboard scancode */
	keysym->scancode = xkey->keycode;
	xsym = XLookupKeysym(xkey, 0);
#ifdef DEBUG_KEYS
	fprintf(stderr, "Translating key 0x%.4x (%d)\n", xsym, xkey->keycode);
#endif
	/* Get the translated SDL virtual keysym */
	keysym->sym = SDLK_UNKNOWN;
	if ( xsym ) {
		switch (xsym>>8) {
			case 0x1005FF:
#ifdef SunXK_F36
				if ( xsym == SunXK_F36 )
					keysym->sym = SDLK_F11;
#endif
#ifdef SunXK_F37
				if ( xsym == SunXK_F37 )
					keysym->sym = SDLK_F12;
#endif
				break;
			case 0x00:	/* Latin 1 */
			case 0x01:	/* Latin 2 */
			case 0x02:	/* Latin 3 */
			case 0x03:	/* Latin 4 */
			case 0x04:	/* Katakana */
			case 0x05:	/* Arabic */
			case 0x06:	/* Cyrillic */
			case 0x07:	/* Greek */
			case 0x08:	/* Technical */
			case 0x0A:	/* Publishing */
			case 0x0C:	/* Hebrew */
			case 0x0D:	/* Thai */
				keysym->sym = (SDLKey)(xsym&0xFF);
				/* Map capital letter syms to lowercase */
				if ((keysym->sym >= 'A')&&(keysym->sym <= 'Z'))
					keysym->sym += ('a'-'A');
				break;
			case 0xFE:
				keysym->sym = ODD_keymap[xsym&0xFF];
				break;
			case 0xFF:
				keysym->sym = MISC_keymap[xsym&0xFF];
				break;
			default:
				fprintf(stderr,
			"X11: Unknown xsym, sym = 0x%.4x\n", xsym);
				break;
		}
	} else {
		/* X11 doesn't know how to translate the key! */
		switch (xkey->keycode) {
			/* Caution:
			   These keycodes are from the Microsoft Keyboard
			 */
			case 115:
				keysym->sym = SDLK_LSUPER;
				break;
			case 116:
				keysym->sym = SDLK_RSUPER;
				break;
			case 117:
				keysym->sym = SDLK_MENU;
				break;
			default:
				fprintf(stderr,
			"X11: Unknown key, keycode = %d\n", xkey->keycode);
				break;
		}
	}
	keysym->mod = KMOD_NONE;

	/* If UNICODE is on, get the UNICODE value for the key */
	if ( SDL_TranslateUNICODE ) {
		static XComposeStatus state;
		/* Until we handle the IM protocol, use XLookupString() */
		unsigned char keybuf[1];
		if ( XLookupString(xkey, (char *)keybuf, sizeof(keybuf),
							NULL, &state) ) {
			keysym->unicode = keybuf[0];
		}
	} else {
		keysym->unicode = 0;
	}
	return(keysym);
}
