/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    5635-34 Springhouse Dr.
    Pleasanton, CA 94588 (USA)
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_sysvideo.h,v 1.10 1999/05/12 06:21:16 slouken Exp $";
#endif

#include "SDL_video.h"
#include "SDL_syswm.h"

/*
 * These function structures exist in SDL_video.c and can be filled in
 * by SDL_SYS_VideoInit() if they are implemented in the native driver.
 */

/* A collection of functions for window manager communication */
extern struct WMcaps {
	/* Toggle between windowed and fullscreen mode */
	SDL_Surface *(*FullScreen)(SDL_Surface *screen, int toggle);

	/* Set the window icon image */
	void (*SetIcon)(SDL_Surface *screen, SDL_Surface *icon, Uint8 *mask);

	/* Set the title and icon text */
	void (*SetCaption)(SDL_Surface *screen, const char *title, const char *icon);

	/* Get some platform dependent window information */
	int (*GetWMInfo)(SDL_SysWMinfo *info);
} SDL_WMCaps;

/* A collection of functions for hardware accelerated surfaces */
extern struct HWcaps {
	/* Information about the video hardware */
	SDL_VideoInfo info;

	/* Allocates a surface in video memory */
	int (*AllocHWSurface)(SDL_Surface *surface);

	/* Sets the hardware accelerated blit function, if any, based
	   on the current flags of the surface (colorkey, alpha, etc.)
	 */
	int (*CheckHWBlit)(SDL_Surface *src, SDL_Surface *dst);

	/* Fills surface with the given color */
	int (*FillHWRect)(SDL_Surface *src, SDL_Rect *rect, Uint32 value);

	/* Sets video mem colorkey and accelerated blit function */
	int (*SetHWColorKey)(SDL_Surface *surface, Uint32 key);

	/* Sets per surface hardware alpha value */
	int (*SetHWAlpha)(SDL_Surface *surface, Uint8 value);

	/* Returns a readable/writable surface */
	int (*LockHWSurface)(SDL_Surface *surface);
	void (*UnlockHWSurface)(SDL_Surface *surface);

	/* Performs hardware flipping */
	int (*FlipHWSurface)(SDL_Surface *surface);

	/* Frees a previously allocated video surface */
	void (*FreeHWSurface)(SDL_Surface *surface);
} SDL_HWCaps;

/*
 * This structure should be defined by any native video implementation
 * It is a member of SDL_Surface available for use by hardware acceleration.
 */
struct private_hwdata;

/*
 * These functions should be implemented by any native video subsytem
 */

/* Initialize the native video subsystem, filling 'vformat' with the 
   "best" display pixel format, returning 0 or -1 if there's an error.
 */
extern int SDL_SYS_VideoInit(SDL_PixelFormat *vformat);

/* List the available video modes for the given pixel format, sorted
   from largest to smallest.
 */
extern SDL_Rect **SDL_SYS_ListModes(SDL_Surface *screen,
					SDL_PixelFormat *format, Uint32 flags);

/* Set the requested video mode, returning a surface which will be set to
   the SDL_VideoSurface.  The width and height will already be verified by
   SDL_ListModes(), and the video subsystem is free to set the mode to a
   supported bit depth different from the one specified -- the desired bpp
   will be emulated with a shadow surface if necessary.  If a new mode is
   returned, this function should take care of cleaning up the current mode.
*/
extern SDL_Surface *SDL_SYS_SetVideoMode(SDL_Surface *current,
				int width, int height, int bpp, Uint32 flags);

/* Sets the system palette with 
   SDL_Palette *palette = screen->format->palette;
	'palette->colors[firstcolor] ... palette->colors[ncolors-1]'
   and returns 1 if all entries were set properly, or 0 otherwise, with
   the palette colors set to the actual colors that were set.
*/
extern int SDL_SYS_SetColors(SDL_Surface *screen, int firstcolor, int ncolors);

/* This pointer should exist in the native video subsystem and should
   point to an appropriate update function for the current video mode
 */
extern void (*SDL_SYS_UpdateRects)(SDL_Surface *screen,
					int numrects, SDL_Rect *rects);

/* Reverse the effects SDL_SYS_VideoInit() -- called if SDL_SYS_VideoInit()
   fails or if the application is shutting down the video subsystem.
*/
extern void SDL_SYS_VideoQuit(SDL_Surface *screen);

/* Performs any cleanup necessary to close the native video subsystem,
   after all surfaces have been released and SDL is ready to quit.
   This function will normally be empty.
*/
extern void SDL_SYS_FinalQuit(void);

