/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    5635-34 Springhouse Dr.
    Pleasanton, CA 94588 (USA)
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_audio.c,v 1.22 1999/08/02 15:52:48 slouken Exp $";
#endif

/* Allow access to a raw mixing buffer */
#ifdef FORK_HACK
#include <sys/types.h>
#include <errno.h>
#include <unistd.h>
#endif
#include <stdio.h>
#include <signal.h>
#include <string.h>

#include "SDL_audio.h"
#include "SDL_mutex.h"
#include "SDL_thread.h"
#include "SDL_timer.h"
#include "SDL_error.h"
#include "SDL_audio_c.h"
#include "SDL_audiomem.h"
#include "SDL_sysaudio.h"

/* The current audio specification (shared with audio thread) */
SDL_AudioSpec audio_spec;

/* An audio conversion block for audio format emulation */
SDL_AudioCVT convert;

#ifdef FORK_HACK

#define ALERT_SIG	SIGUSR2

void *shared_mem;
int *_audioenabled;
int *_audiopaused;
#define SDL_AudioEnabled	(*_audioenabled)
#define SDL_AudioPaused		(*_audiopaused)

int audio_pid = -1;
int audio_pipe[2];
SDL_mutex *alert_sem = NULL;

void feed_audio(int sig)
{
	Uint8 *buf[1];

	signal(sig, feed_audio);
	if ( read(audio_pipe[0], buf, sizeof(*buf)) == sizeof(*buf) ) {
		(*audio_spec.callback)(audio_spec.userdata,
						buf[0], audio_spec.size);
		SDL_mutexV(alert_sem);
	}
}
#else
int SDL_AudioEnabled;
int SDL_AudioPaused;
#endif

/* Fake audio buffer for when the audio hardware is busy */
static Uint8 *fake_stream = NULL;

/* A semaphore for locking the mixing buffers */
static SDL_mutex *mixer_lock = NULL;

/* A thread to feed the audio device */
static SDL_Thread *audio_thread = NULL;

/* The general mixing thread function */
static int RunAudio(void *unused)
{
	Uint8 *stream;
	int    stream_len;
	void  *udata;
	void (*fill)(void *userdata,Uint8 *stream, int len);
	int    silence;
#ifdef FORK_HACK
	Uint8 *buf[1];
	pid_t  ppid = getppid();
#endif

	/* Set up the mixing function */
	fill  = audio_spec.callback;
	udata = audio_spec.userdata;
	if ( convert.needed ) {
		if ( convert.src_format == AUDIO_U8 ) {
			silence = 0x80;
		} else {
			silence = 0;
		}
		stream_len = convert.len;
	} else {
		silence = audio_spec.silence;
		stream_len = audio_spec.size;
	}
	stream = fake_stream;

	/* Loop, filling the audio buffers */
	while ( SDL_AudioEnabled ) {

		/* Wait for new current buffer to finish playing */
		if ( stream == fake_stream ) {
			SDL_Delay((audio_spec.samples*1000)/audio_spec.freq);
		} else {
			SDL_SYS_WaitAudio();
		}

		/* Fill the current buffer with sound */
		if ( convert.needed ) {
			stream = convert.buf;
		} else {
			stream = SDL_SYS_GetAudioBuf();
			if ( stream == NULL ) {
				stream = fake_stream;
			}
		}
		memset(stream, silence, stream_len);

		if ( ! SDL_AudioPaused ) {
			SDL_LockAudio();
#ifdef FORK_HACK
			buf[0] = stream;
			write(audio_pipe[1], buf, sizeof(*buf)); 
			kill(ppid, ALERT_SIG);
			SDL_mutexP(alert_sem);
#else
			(*fill)(udata, stream, stream_len);
#endif
			SDL_UnlockAudio();
		}

		/* Convert the audio if necessary */
		if ( convert.needed ) {
			SDL_ConvertAudio(&convert);
			stream = SDL_SYS_GetAudioBuf();
			if ( stream == NULL ) {
				stream = fake_stream;
			}
			memcpy(stream, convert.buf, convert.len_cvt);
		}

		/* Ready current buffer for play and change current buffer */
		if ( stream != fake_stream ) {
			SDL_SYS_PlayAudio();
		}
	}
	return(0);
}

int SDL_OpenAudio(SDL_AudioSpec *desired, SDL_AudioSpec *obtained)
{
	int audio_opened;

	/* Verify some parameters */
	if ( desired->callback == NULL ) {
		SDL_SetError("SDL_OpenAudio() passed a NULL callback");
		return(-1);
	}
	switch ( desired->channels ) {
	    case 1:	/* Mono */
	    case 2:	/* Stereo */
		break;
	    default:
		SDL_SetError("1 (mono) and 2 (stereo) channels supported");
		return(-1);
	}

#ifdef FORK_HACK
	if ( shared_mem != NULL ) {
		SDL_SetError("Audio is already playing");
		return(-1);
	}

	/* Allocate memory shared with audio thread */
	shared_mem = SDL_AllocAudioMem(2*sizeof(int));
	if ( shared_mem == NULL ) {
		SDL_SetError("Couldn't allocated shared memory");
		return(-1);
	}
	_audioenabled = (int *)(((char *)shared_mem)+0*sizeof(int));
	_audiopaused = (int *)(((char *)shared_mem)+1*sizeof(int));

	/* Allocate signal semaphore */
	alert_sem = SDL_CreateMutex();
	if ( alert_sem == NULL ) {
		SDL_CloseAudio();
		SDL_SetError("Couldn't create alert semaphore");
		return(-1);
	}
	SDL_mutexP(alert_sem);
#else
	if ( SDL_AudioEnabled ) {
		SDL_SetError("Audio is already playing");
		return(-1);
	}
#endif

	/* Create a semaphore for locking the sound buffers */
	mixer_lock = SDL_CreateMutex();
	if ( mixer_lock == NULL ) {
		SDL_SetError("Couldn't create mixer lock");
		SDL_CloseAudio();
		return(-1);
	}

	/* Calculate the silence and size of the audio specification */
	SDL_CalculateAudioSpec(desired);

	/* Open the audio subsystem */
	memcpy(&audio_spec, desired, sizeof(audio_spec));
	convert.needed = 0;
	SDL_AudioEnabled = 1;
	SDL_AudioPaused  = 1;
	audio_opened = SDL_SYS_OpenAudio(&audio_spec);
	if ( audio_opened < 0 ) {
		SDL_CloseAudio();
		return(-1);
	}

	/* If the audio driver changes the buffer size, accept it */
	if ( audio_spec.samples != desired->samples ) {
		desired->samples = audio_spec.samples;
		SDL_CalculateAudioSpec(desired);
	}

	/* Allocate a fake audio memory buffer */
	fake_stream = SDL_AllocAudioMem(audio_spec.size);
	if ( fake_stream == NULL ) {
		SDL_CloseAudio();
		SDL_OutOfMemory();
		return(-1);
	}

	/* See if we need to do any conversion */
	if ( memcmp(desired, &audio_spec, sizeof(audio_spec)) == 0 ) {
		/* Just copy over the desired audio specification */
		if ( obtained != NULL ) {
			memcpy(obtained, &audio_spec, sizeof(audio_spec));
		}
	} else {
		/* Copy over the audio specification if possible */
		if ( obtained != NULL ) {
			memcpy(obtained, &audio_spec, sizeof(audio_spec));
		} else {
			/* Build an audio conversion block */
			if ( SDL_BuildAudioCVT(&convert,
				desired->format, desired->channels,
							desired->freq,
				audio_spec.format, audio_spec.channels,
							audio_spec.freq) < 0 ) {
				SDL_CloseAudio();
				return(-1);
			}
			if ( convert.needed ) {
				convert.len = desired->size;
				convert.buf = (Uint8 *)SDL_AllocAudioMem(
						convert.len*convert.len_mult);
				if ( convert.buf == NULL ) {
					SDL_CloseAudio();
					SDL_OutOfMemory();
					return(-1);
				}
			}
		}
	}

	/* Start the audio thread if necessary */
	switch (audio_opened) {
		case  0:
			/* Start the audio thread */
#ifdef FORK_HACK
			if ( pipe(audio_pipe) < 0 ) {
				SDL_CloseAudio();
				SDL_SetError("pipe() error");
				return(-1);
			}
			signal(ALERT_SIG, feed_audio);
			
			switch (audio_pid=fork()) {
				case -1: {
					SDL_CloseAudio();
					SDL_SetError("fork() error");
					return(-1);
				}
				case 0: {
					int ignore_list[] = {
						SIGINT, SIGQUIT, 0
					};
					int default_list[] = {
						SIGSEGV, SIGILL, SIGPIPE,
						SIGTERM, 0
					};
					int i;

					close(audio_pipe[0]);
					for ( i=0; ignore_list[i]; ++i ) {
						signal(ignore_list[i], SIG_IGN);
					}
					for ( i=0; default_list[i]; ++i ) {
						signal(default_list[i], _exit);
					}
					RunAudio(NULL);
					/* Don't run atexit() routines */
					_exit(0);
				}
				default: {
					close(audio_pipe[1]);
					break;
				}
			}
#else
			audio_thread = SDL_CreateThread(RunAudio, NULL);
			if ( audio_thread == NULL ) {
				SDL_CloseAudio();
				SDL_SetError("Couldn't create audio thread");
				return(-1);
			}
#endif
			break;

		default:
			/* The audio is now playing */
			break;
	}
	return(0);
}
void SDL_PauseAudio (int pause_on)
{
	SDL_AudioPaused = pause_on;
}
void SDL_LockAudio (void)
{
	/* Obtain a lock on the mixing buffers */
	SDL_mutexP(mixer_lock);
}
void SDL_UnlockAudio (void)
{
	/* Release lock on the mixing buffers */
	SDL_mutexV(mixer_lock);
}
void SDL_CloseAudio (void)
{
	if ( SDL_AudioEnabled ) {
		SDL_AudioEnabled = 0;
#ifdef FORK_HACK
		if ( audio_pid > 0 ) {
			while ( errno != ECHILD ) {
				waitpid(audio_pid, NULL, 0);
			}
			audio_pid = -1;
		}
		if ( alert_sem != NULL ) {
			SDL_DestroyMutex(alert_sem);
			alert_sem = NULL;
		}
#else
		if ( audio_thread != NULL ) {
			SDL_WaitThread(audio_thread, NULL);
			audio_thread = NULL;
		}
#endif
		if ( mixer_lock != NULL ) {
			SDL_DestroyMutex(mixer_lock);
			mixer_lock = NULL;
		}
		if ( fake_stream != NULL ) {
			SDL_FreeAudioMem(fake_stream);
			fake_stream = NULL;
		}
		if ( convert.needed ) {
			SDL_FreeAudioMem(convert.buf);
		}
		SDL_SYS_CloseAudio();
	}
#ifdef FORK_HACK
	shared_mem = NULL;	/* Cleaned up on exit() */
#endif
}

#define NUM_FORMATS	6
static int format_idx;
static int format_idx_sub;
static Uint16 format_list[NUM_FORMATS][NUM_FORMATS] = {
 { AUDIO_U8, AUDIO_S8, AUDIO_S16LSB, AUDIO_S16MSB, AUDIO_U16LSB, AUDIO_U16MSB },
 { AUDIO_S8, AUDIO_U8, AUDIO_S16LSB, AUDIO_S16MSB, AUDIO_U16LSB, AUDIO_U16MSB },
 { AUDIO_S16LSB, AUDIO_S16MSB, AUDIO_U16LSB, AUDIO_U16MSB, AUDIO_U8, AUDIO_S8 },
 { AUDIO_S16MSB, AUDIO_S16LSB, AUDIO_U16MSB, AUDIO_U16LSB, AUDIO_U8, AUDIO_S8 },
 { AUDIO_U16LSB, AUDIO_U16MSB, AUDIO_S16LSB, AUDIO_S16MSB, AUDIO_U8, AUDIO_S8 },
 { AUDIO_U16MSB, AUDIO_U16LSB, AUDIO_S16MSB, AUDIO_S16LSB, AUDIO_U8, AUDIO_S8 },
};

Uint16 SDL_FirstAudioFormat(Uint16 format)
{
	for ( format_idx=0; format_idx < NUM_FORMATS; ++format_idx ) {
		if ( format_list[format_idx][0] == format ) {
			break;
		}
	}
	format_idx_sub = 0;
	return(SDL_NextAudioFormat());
}

Uint16 SDL_NextAudioFormat(void)
{
	if ( (format_idx == NUM_FORMATS) || (format_idx_sub == NUM_FORMATS) ) {
		return(0);
	}
	return(format_list[format_idx][format_idx_sub++]);
}

void SDL_CalculateAudioSpec(SDL_AudioSpec *spec)
{
	switch (spec->format) {
		case AUDIO_U8:
			spec->silence = 0x80;
			break;
		default:
			spec->silence = 0x00;
			break;
	}
	spec->size = (spec->format&0xFF)/8;
	spec->size *= spec->channels;
	spec->size *= spec->samples;
}
