// ---------------------------------------------------------------------------
// - Object.cpp                                                              -
// - standard object library - base object implementation                    -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "Method.hpp"
#include "Exception.hpp"

namespace aleph {
  // create a new object and initialize the reference count

  Object::Object (void) {
    d_rcount = 0;
  }

  // return an object class name

  const String Object::repr (Object* object) {
    if (object == nilp) return "nil";
    return object->repr ();
  }

  // operate this object with another one

  Object* Object::oper (t_oper type, Object* object) {
    throw Exception ("operator-error", "invalid operator call for object",
		     this->repr ());
  }

  // create or set a const object to this object

  Object* Object::cdef (Interp* interp, Nameset* nset, Object* object) {
    throw Exception ("cdef-error", "invalid call to cdef method for object",
		     this->repr ());
  }

  // create or set a const object to this object by name

  Object* Object::cdef (Interp* interp, Nameset* nset, const String& name, 
			Object* object) {
    throw Exception ("cdef-error", "invalid call to cdef method for object",
		     this->repr ());
  }

  // create or set an object to this object

  Object* Object::vdef (Interp* interp, Nameset* nset, Object* object) {
    throw Exception ("vdef-error", "invalid call to vdef method for object",
		     this->repr ());
  }

  // create or set an object to this object by name

  Object* Object::vdef (Interp* interp, Nameset* nset, const String& name, 
			Object* object) {
    throw Exception ("vdef-error", "invalid call to vdef method for object",
		     this->repr ());
  }

  // evaluate an object in the current nameset

  Object* Object::eval (Interp* interp, Nameset* nset) {
    return this;
  }

  // evaluate an object in the current nameset by name

  Object* Object::eval (Interp* interp, Nameset* nset, const String& name) {
    // dispatch standard methods
    if (name == "=")    return new Method (name,this);
    if (name == "repr") return new Method (name,this);

    // not good - nothing found
    throw Exception ("eval-error", "invalid call to eval method with member",
		     name);
  }

  // apply an object with a set of arguments

  Object* Object::apply (Interp* interp, Nameset* nset, Cons* args) {
    throw Exception ("apply-error", "invalid call to apply method with object",
		     Object::repr (this));
  }

  // apply an object by name with a set of arguments

  Object* Object::apply (Interp* interp, Nameset* nset, const String& name,
			 Cons* args) {
    if (name == "=") {
      Cons::check (args, 1);
      Object* obj = args->getcar ();
      Object* val = (obj == nilp) ? nilp : obj->eval (interp, nset);
      return this->vdef (interp, nset, val);
    }
    if (name == "repr") {
      Cons::check (args, 0);
      return new String (this->repr ());
    }
    // no way - error
    throw Exception ("apply-error", "invalid call to apply method with object",
		     Object::repr (this));
  }

  // apply an object by quark with a set of arguments

  Object* Object::apply (Interp* interp, Nameset* nset, const long quark,
			 Cons* args) {
    throw Exception ("apply-error", "invalid call to apply method with object",
		     Object::repr (this));
  }

  // apply an object by object with a set of arguments

  Object* Object::apply (Interp* interp, Nameset* nset, Object* object,
			 Cons* args) {
    throw Exception ("apply-error", "invalid call to apply method with object",
		     Object::repr (this));
  }

  // apply an object by name with a vector of arguments

  Object* Object::apply (Interp* interp, Nameset* nset, const String& name,
			 Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // dispatch 0 arguments
    if ((name == "repr") && (argc == 0)) {
      return new String (this->repr ());
    }
    // dispatch 1 argument
    if ((name == "=") && (argc == 1)) {
      Object* obj = argv->get (0);
      return this->vdef (interp, nset, obj);
    }
    // no way - error
    throw Exception ("apply-error", "invalid call to apply method with object",
		     Object::repr (this));
  }
}
