// ---------------------------------------------------------------------------
// - InputFile.cpp                                                           -
// - standard object library - input file class implementation               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "cio.hxx"
#include "Vector.hpp"
#include "Method.hpp"
#include "Integer.hpp"
#include "Boolean.hpp"
#include "InputFile.hpp"
#include "Exception.hpp"

namespace aleph {

  // create a new file input stream given its name.

  InputFile::InputFile (const String& name) {
    d_name = name;
    d_eof  = false;

    // try to open the file
    char* fname = name.toChar ();
    if ((d_sid = c_openr (fname)) == -1) {
      delete [] fname;
      throw Exception ("open-error","cannot open file",name);
    }
    delete [] fname;
    // reset the input buffer
    d_buffer.reset ();
  }

  // close and destroy this input file
  InputFile::~InputFile (void) {
    if (d_sid >= 0) c_close (d_sid);
  }

  // return the class name

  String InputFile::repr (void) const {
    return "InputFile";
  }

  // read one character from the input stream.

  char InputFile::read (void) {
    // check the pushback buffer first
    if (d_buffer.length () != 0)
      return d_buffer.read ();
    
    // check if we are at the end of file
    if (d_eof == true) return eofc;
  
    // read the next character on the stream
    char byte  = nilc;
    long count = 0;
    if ((count = c_read (d_sid, &byte,1)) < 0) 
      throw Exception ("read-error", "interrupt during read operation");
    // check for eof
    if (count == 0) {
      d_eof = true;
      return eofc;
    }
    return byte;
  }

  // return true if the eof flag is set

  bool InputFile::iseof (void) const {
    return d_eof;
  }

  // return the file name associated with this stream

  String InputFile::getFileName (void) const {
    return d_name;
  }

  // close this input file and mark the eof marker
  
  void InputFile::close (void) {
    c_close (d_sid);
    d_sid = -1;
    d_eof = true;
  }

  // set the input stream at a certain position

  void InputFile::lseek (const long pos) {
    c_lseek (d_sid,pos);
  }

  // return the input file size

  long InputFile::length (void) const {
    return c_fsize (d_sid);
  }

  // create a new input file in a generic way

  Object* InputFile::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc != 1) 
      throw Exception ("argument-error", 
		       "invalid arguments with with input file"); 

    // try to map the file name
    String name = argv->getstring (0);
    return new InputFile (name);
  }

  // evaluate this input file with a member name

  Object* InputFile::eval (Interp* interp, Nameset* nset, const String& name) {
    return new Method (name, this);
  }

  // apply this input file with a method name

  Object* InputFile::apply (Interp* interp, Nameset* nset, const String& name,
			    Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (interp, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if ((name == "valid-p") && (argc == 0)) {
      delete argv;
      return new Boolean (!iseof ());
    }
    if ((name == "eof-p") && (argc == 0)) {
      delete argv;
      return new Boolean (iseof ());
    }
    if ((name == "close") && (argc == 0)) {
      delete argv;
      close ();
      return nilp;
    }
    if ((name == "get-name") && (argc == 0)) {
      delete argv;
      return new String (d_name);
    }
    if ((name == "length") && (argc == 0)) {
      delete argv;
      return new Integer (length ());
    }

    // dispatch 1 argument
    if ((name == "lseek") && (argc == 1)) {
      long val = argv->getint (0);
      lseek (val);
      delete argv;
      return nilp;
    }
    
    // input method call
    Object* result = nilp;
    try {
      result =  Input::apply (interp, nset, name, argv);
    } catch (...) {
      delete argv;
      throw;
    }
    return result;
  }
}
