// ---------------------------------------------------------------------------
// - Input.cpp                                                               -
// - standard object library - input stream class implementation             -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Input.hpp"
#include "Character.hpp"
#include "Exception.hpp"

namespace aleph {

  // read a line in this input stream

  String Input::readln (void) {
    // create a buffer to accumulate characters
    Buffer buffer;
    // read the character in the buffer
    char c = nilc;
    while ((c = read ()) != eofc) {
      if (c == '\n') return buffer.toString ();
      buffer.add (c);
    }
    pushback (eofc);
    return buffer.toString ();
  }

  // pushback a character

  void Input::pushback (const char value) {
    d_buffer.pushback (value);
  }

  // pushback a string on this stream
  
  void Input::pushback (const String& value) {
    d_buffer.pushback (value);
  }

  // apply this input class with a method name

  Object* Input::apply (Interp* interp, Nameset* nset, const String& name,
			Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if ((name == "read") && (argc == 0)) {
      Character* result = new Character (read ());
      return result;
    }
    if ((name == "readln") && (argc == 0)) {
      String* result = new String (readln ());
      return result;
    }

    // dispatch one argument
    if ((name == "pushback") && (argc == 1)) {
      Object* obj = argv->get (0);
      // check for a character
      Character* c = dynamic_cast <Character*> (obj);
      if (c != nilp) {
	pushback (c->toCharacter ());
	return nilp;
      }
      String* s = dynamic_cast <String*> (obj);
      if (s != nilp) {
	pushback (*s);
	return nilp;
      }
      throw Exception ("type-error", "invalid object with pushback method",
		       Object::repr (obj));
    }
    // apply with object
    return Object::apply (interp, nset, name, argv);
  }
}
