/* Zgv v3.3 - GIF, JPEG and PBM/PGM/PPM viewer, for VGA PCs running Linux.
 * Copyright (C) 1993-1999 Russell Marks. See README for license details.
 *
 * vgadisp.c - vga specific display routines.
 */


#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/file.h>
#include <vga.h>
#include <vgagl.h>
#include <vgamouse.h>
#include "zgv.h"
#include "magic.h"
#include "readgif.h"
#include "readjpeg.h"
#include "readpnm.h"
#include "readbmp.h"
#include "readpng.h"
#include "readtga.h"
#include "readpcx.h"
#include "readxvpic.h"
#include "readmrf.h"
#include "readxbm.h"
#include "readxpm.h"
#include "readnbkey.h"
#include "helppage.h"
#include "3deffects.h"
#include "rc_config.h"
#include "rcfile.h"
#include "mousecur.h"
#include "rbmenu.h"
#include "modesel.h"
#include "vgadisp.h"


/* zgv.c initialises these */
int curvgamode;
int zoom,virtual;
int vkludge;
int pixelsize;		/* in bytes - 1 for 8-bit, 3 for 24-bit */
int pic_incr;		/* after view, 0=stay, -1=prev, 1=next */

/* for right-buttom menu */
static int rb_ignore_first_left_click=0;
static unsigned char *rb_save=NULL;
static int rb_save_width,rb_save_height;

char *viewerhelp[]=
  {
  "? (question mark)\\this help page",
  "/\\show video mode selection help",
  " ",
  "v\\toggle smoothing in virtual/zoom modes",
  "comma/dot\\contrast down/up",
  "less-than/greater-than\\brightness down/up",
  "asterisk\\reset contrast and brightness",
  "z\\toggle zoomed mode",
  "r\\rotate clockwise 90 degrees",
  "m\\mirror",
  "f\\flip",
  "d\\double size of picture",
  "D\\halve size of picture",
  "Esc or x\\exit the viewer",
  " ",
  "hjkl or qaop keys scroll around the picture",
  "HJKL or QAOP (also the cursor keys) scroll in bigger steps",
  ""	/* end */
  };

char *viewermodeshelp[]=
  {
  "Generic VGA modes:",
  " ",
  "4 - 640x480x4\\5 - 320x200x8\\6 - 320x240x8",
  "7 - 320x400x8\\8 - 360x480x8\\",
  " ",
  "SVGA modes (try these first):",
  " ",
  "F1 - 640x480x8\\F2 - 800x600x8\\F3 - 1024x768x8",
  "F4 - 1280x1024x8\\F5 - 320x200x15\\F6 - 320x200x16",
  "F7 - 320x200x24\\F8 - 640x480x15\\F9 - 640x480x16",
  "F10 - 640x480x24\\SF1 - 800x600x15\\SF2 - 800x600x16",
  "SF3 - 800x600x24\\SF4 - 1024x768x15\\SF5 - 1024x768x16",
  "SF6 - 1024x768x24\\SF7 - 1280x1024x15\\SF8 - 1280x1024x16",
  "SF9 - 1280x1024x24\\Tab-F1 - 1152x864x8\\Tab-F2 - 1152x864x15",
  "Tab-F3 - 1152x864x16\\Tab-F4 - 1152x864x24\\Tab-F5 - 1600x1200x8",
  "Tab-F6 - 1600x1200x15\\Tab-F7 - 1600x1200x16\\Tab-F8 - 1600x1200x24",
  ""
  };


struct rbm_data_tag viewer_menu_data[]=
  {
  /* 31 chars max for label text */
  {1, "Exit viewer",		RK_ESC		},
  
  /* stuff in draw_rb_menu which sets the active field of these
   * mode buttons assumes they are in *exactly* this order/position.
   * So don't change anything before the "Normal" button without
   * thinking twice... or more. :-)
   *
   * modes less than 480 pixels high are omitted here since
   * the mouse menu is disabled on those.
   */
  {1, "360x480x8",		'8'		},
  {1, "640x480x4",		'4'		},
  {1, "640x480x8",		RK_F1		},
  {1, "800x600x8",		RK_F2		},
  {1, "1024x768x8",		RK_F3		},
  {1, "640x480x15",		RK_F8		},
  {1, "640x480x16",		RK_F9		},
  {1, "640x480x24",		RK_F10		},
  {1, "800x600x15",		RK_SHIFT_F1	},
  {1, "800x600x16",		RK_SHIFT_F2	},
  {1, "800x600x24",		RK_SHIFT_F3	},
  {1, "1024x768x15",		RK_SHIFT_F4	},
  {1, "1024x768x16",		RK_SHIFT_F5	},
  {1, "1024x768x24",		RK_SHIFT_F6	},
  {1, "Next file",		RK_ENTER	},
  {1, " ...tag first",		' '		},
  /* no gap thing here, due to split between columns */
  
  {1, "Normal size",		'n'		},
  {1, "Zoom on/off",		'z'		},
  {1, " ...smooth on/off",	'v'		},
  {1, "Scale up 1",		's'		},
  {1, "Scale down 1",		'S'		},
  {1, "Scale up x2",		'd'		},
  {1, "Scale down x2",		'D'		},
  {1, " ...smooth on/off",	'i'		},
  {1, "Normal orient'n",	128+'n'		},
  {1, "Save orient'n",		128+'s'		},
  {1, "Use last orient'n",	128+'o'		},
  {1, "Rotate c/w",		'r'		},
  {1, "Rotate anti-c/w",	'R'		},
  {1, "Mirror",			'm'		},
  {1, "Flip",			'f'		},
  /* this next one seems bizarre, but it's to the right of "next file" */
  {1, "Previous file",		127		},
  {1, "4-bit grey/colour",	'c'		},
  
  {0,"",0}
  };


unsigned char vkcache[32768],vkcache_valid[32768];


#define MOVSTP 10    /* for q,a,o,p panning */
#define BIGSTP 100   /* for Q,A,O,P panning (i.e. with shift) */

#define FIX_TO_EIGHT_BIT	1
#define FIX_TO_HIGHCOLOUR	2

byte *theimage;   /* quite slow if passed as parameter */
int width,height,numcols; /* same here */
byte *image_palette;

double contrast=1.0;  /* note that double contrast is in fact 2 :-) */
int brightness=0;
int scrnwide,scrnhigh,scrnpixelsize,scrncols;
int scaling=1,interp=0,inextpix=1;

int palr[256],palg[256],palb[256];
byte palr64[256],palg64[256],palb64[256], palt[256];
int palrgb[768];
unsigned char pal32_no_bc[768];

unsigned char dither16_greylookup[256];
unsigned char dither16_rgb[768];

int loading_file_type;
int repeat_sig,first_repeat;
int saved_px,saved_py;		/* for persistance in cfg.repeat_timer mode */

/* grey values for rgb - NTSC uses these (adding up to 1000) */
static int grey_red=299,grey_green=587,grey_blue=114;

int sgres;


/* HERE BE A BIG COMMENT ABOUT PICTURE ORIENTATION. ABANDON HOPE ALL
 * YE WHO ENTER HERE. AND STOP TALKING OLDE-WORLDE WHILE YOU'RE AT IT.
 *	-- Death in Terry Pratchett's "Moving Picture Viewers" ;-)
 *
 * The picture-orientation stuff confuses the hell out of me, anyway,
 * and I wrote it. Probably not a good sign. :-)
 *
 * Errr, moving swiftly on, there are eight possible orientations
 * (0 is the original image):
 *                             _____     _____ 
 *    _______     _______     |    a|   |    b|
 *   |a      |   |b      |    |     |   |     |
 *   |   0   |   |   1   |    |  4  |   |  5  |
 *   |______b|   |______a|    |b____|   |a____|
 *    _______     _______      _____     _____ 
 *   |      b|   |      a|    |b    |   |a    |
 *   |   2   |   |   3   |    |     |   |     |
 *   |a______|   |b______|    |  6  |   |  7  |
 *                            |____a|   |____b|
 *
 * That gives us these changes in orientation state for each of the
 * orientation-changing operations (rotate, mirror, flip):
 *
 * 		rot-cw	rot-acw	mirror	flip
 * 0 to...	4	5	3	2
 * 1 to...	5	4	2	3
 * 2 to...	7	6	1	0
 * 3 to...	6	7	0	1
 * 4 to...	1	0	7	6
 * 5 to...	0	1	6	7
 * 6 to...	2	3	5	4
 * 7 to...	3	2	4	5
 */

int orient_state_rot_cw[8] ={4,5,7,6,1,0,2,3};
int orient_state_rot_acw[8]={5,4,6,7,0,1,3,2};
int orient_state_mirror[8] ={3,2,1,0,7,6,5,4};
int orient_state_flip[8]   ={2,3,0,1,6,7,4,5};

int orient_override=0;		/* override orientation (used by Alt-s) */
int orient_override_state=0;


#define GET15BITCOLOUR(r,g,b) ((((r)&0xf8)<<7)|(((g)&0xf8)<<2)|((b)>>3))
#define GET16BITCOLOUR(r,g,b) ((((r)&0xf8)<<8)|(((g)&0xfc)<<3)|((b)>>3))
#define GET32BITCOLOUR(r,g,b) (b|(g<<8)|(r<<16))


/* indirection needed to allow high-colour brightness/contrast */
typedef void (*eventuallyfunc)(byte *,int,int,int);
eventuallyfunc eventuallydrawscansegment=NULL;
int doing_hicol_bc=0;	/* non-zero if doing hi-col b/c in current redraw */



/* prototypes */
int readpicture(char *giffn, hffunc howfarfunc, int show_dont_tell, int quick);
int has_highcolour_mode(void);
void fix_to_similar_mode(int modetype);
void aborted_file_cleanup(void);
int is_this_file_jpeg(void);
void makerealpal(void);
void filterpal(byte *palette);
int dimmer(int a);
int contrastup(int cp);
int get_mode_width(int vm);
int get_mode_height(int vm);
int get_mode_pixelbytes(int vm);
int get_mode_numcols(int vm);
void setmode_or_clear(int newmode);
void init_vkludge_cache(void);
void graphicson(void);
void graphicsoff(void);
int mode_is_usable_now(int modenum);
int showgif(byte *palette);
static void draw_rb_menu(void);
static int rb_find_closest(int r,int g,int b,int not1,int not2,int not3);
static void undraw_rb_menu(void);
static int rb_menu_event(int *keyp);
void repeat_sighandler(int foo);
void setpalvec(int start,int num,int *pal);
void redrawgif(int px, int py, int npx, int npy);
void eventuallydrawscansegment_without_bc(byte *ptr, int x, int y, int len);
void eventuallydrawscansegment_with_bc(byte *ptr, int x, int y, int len);
void dither16scansegment(unsigned char *ptr,int x,int y,int len);
int getvpix(int px, int py, int x, int y);
void drawzoomedgif(void);
void fx_mirror(void);
void fx_flip(void);
void fx_rot(void);
int closest(int r, int g, int b);
void samecentre(int *ppx, int *ppy, int newscale, int oldpx, int oldpy, int oldscale);
void show_dimensions(int px, int py, int scaling);
void orient_change_state(int from,int to,int clear_if_rot);



/* to read a picture only, set show_dont_tell=0.
 * If you're only reading the picture, not showing it, then:
 * 1. picture is returned in global vars. 'theimage' and 'image_palette'
 * 2. you have to set pixelsize=whatever and restore it afterwards.
 * 3. you have to free() both theimage and image_palette when finished.
 * The `quick' arg sets whether or not to ask the jpeg decoder for a
 * rough image. This is useful when doing thumbnails, where we don't
 * need the accuracy, but you shouldn't use it elsewhere.
 */
int readpicture(char *giffn,hffunc howfarfunc,int show_dont_tell,int quick)
{
int result=0;
PICINFO ginfo;
byte *palette=NULL;

first_repeat=1;

do
  {
  ginfo.numcols=256;	/* only changes for GIF files */
  theimage=NULL;
  if(!first_repeat) howfarfunc=NULL;
  
  switch(loading_file_type=magic_ident(giffn))
    {
    case _IS_GIF:
      /* part of the New Strategy (tm); always use 8-bit for GIFs.
       * fix the mode to a similar 8-bit one if possible.
       */
      if(show_dont_tell) fix_to_similar_mode(FIX_TO_EIGHT_BIT);
      pixelsize=1;
      result=readgif(giffn,&theimage,&palette,howfarfunc,cfg.errignore);
      getgifinfo(&ginfo);
      height=ginfo.height; width=ginfo.width;
      if(cfg.errignore && result==_PICERR_CORRUPT) result=_PIC_OK;
      break;
  
    case _IS_JPEG:
      /* don't *always* use 24-bit for JPEGs... check for 15/16/24-bit
       * video modes before doing so, as well as the config.
       */
      if(show_dont_tell)
        {
        pixelsize=has_highcolour_mode()?3:1;
        if(pixelsize==3 && cfg.jpeg24bit==0)
          pixelsize=1;
        }
  
      theimage=NULL;
      result=read_JPEG_file(giffn,howfarfunc,&palette, quick);
      
      /* if error, then palette has already been nuked, so only
       * need to deal with the image.
       */
      if(theimage!=NULL && result!=_PIC_OK) { free(theimage); theimage=NULL; }
        
      break;
      
#ifdef PNG_SUPPORT
    case _IS_PNG:
      if(show_dont_tell)
        {
        pixelsize=has_highcolour_mode()?3:1;
        if(pixelsize==3 && cfg.jpeg24bit==0)
          pixelsize=1;
        }
  
      theimage=NULL;
      result=read_png_file(giffn,howfarfunc,&palette);
      
      /* if error, then palette has already been nuked, so only
       * need to deal with the image.
       */
      if(theimage!=NULL && result!=_PIC_OK)
        if(cfg.errignore)
          result=_PIC_OK;
        else
          { free(theimage); theimage=NULL; }
        
      break;
#endif /* PNG_SUPPORT */
      
    case _IS_PNM: case _IS_BMP: case _IS_TGA: case _IS_PCX:
    case _IS_XVPIC: case _IS_MRF:
    case _IS_XBM: case _IS_XPM: case _IS_TIFF:
      if(show_dont_tell) pixelsize=has_highcolour_mode()?3:1;
      switch(loading_file_type)
        {
        case _IS_PNM:
          result=read_pnm_file(giffn,howfarfunc,&theimage,&palette,&pixelsize,
                  &ginfo); break;
        case _IS_TIFF:
          {
          static char buf[32+256];
          FILE *tmp;
          int tmpc;
          
          sprintf(buf,"tifftopnm 2>/dev/null <%s",giffn);
          if((tmp=popen(buf,"r"))==NULL)
            result=_PICERR_NOMEM;	/* usually this will mean no memory */
          else
            {
            /* popen's return value isn't useful for telling whether
             * tifftopnm is there or not. It only tells us whether the
             * *shell* was run correctly, not whether the shell managed
             * to find tifftopnm. So, here we read the first byte to check
             * for EOF, ungetc'ing it if all is well.
             */
            if((tmpc=fgetc(tmp))==EOF)
              {
              pclose(tmp);
              result=_PICERR_BADTIFFRUN;
              }
            else
              {
              ungetc(tmpc,tmp);
              result=read_pnm_main(tmp,howfarfunc,&theimage,&palette,
              		&pixelsize,&ginfo,pclose);
              }
            }
          }
          break;
        case _IS_BMP:
          result=read_bmp_file(giffn,howfarfunc,&theimage,&palette,&pixelsize,
                  &ginfo); break;
        case _IS_TGA:
          result=read_tga_file(giffn,howfarfunc,&theimage,&palette,&pixelsize,
                  &ginfo); break;
	case _IS_PCX:
          result=read_pcx_file(giffn,howfarfunc,&theimage,&palette,&pixelsize,
                  &ginfo); break;
	case _IS_XVPIC:
          result=read_xvpic(giffn,howfarfunc,&theimage,&palette,&pixelsize,
                  &ginfo); break;
	case _IS_MRF:
          result=read_mrf_file(giffn,howfarfunc,&theimage,&palette,&pixelsize,
                  &ginfo); break;
	case _IS_XBM:
          result=read_xbm_file(giffn,howfarfunc,&theimage,&palette,&pixelsize,
                  &ginfo); break;
	case _IS_XPM:
          result=read_xpm_file(giffn,howfarfunc,&theimage,&palette,&pixelsize,
                  &ginfo); break;
        }
      
      width=ginfo.width;
      height=ginfo.height;
      if(result!=_PIC_OK)
        {
        if(theimage!=NULL) { free(theimage); theimage=NULL; }
        if(palette!=NULL)  { free(palette);  palette=NULL; }
        }
      else
        if(loading_file_type==_IS_TGA && tga_need_flip)
          fx_flip();	/* right way up! */
      break;

    
    /* if they voted for "none of the above"... */
    default:
      return(_PICERR_BADMAGIC);
    }
  
  
  if(show_dont_tell)
    {
    if(pixelsize==3)
      fix_to_similar_mode(FIX_TO_HIGHCOLOUR);
    else
      fix_to_similar_mode(FIX_TO_EIGHT_BIT);
  
    if(result==_PIC_OK)
      {
      numcols=ginfo.numcols;
      sgres=showgif(palette);
      free(theimage);
      free(palette);
      }
    }
  else
    image_palette=palette;
  }
while(show_dont_tell && cfg.repeat_timer && result==_PIC_OK && sgres==0);

return(result);
}


/* modes aren't counted if they're locked out */
int has_highcolour_mode()
{
struct modedesc_tag *md_ptr;

if(cfg.hicolmodes) return(1);	/* forces positive response */

for(md_ptr=modedesc;md_ptr->mode;md_ptr++)
  if(md_ptr->bitspp>=15 &&
     vga_hasmode(md_ptr->mode) && cfg.mode_allowed[md_ptr->mode])
    return(1);

return(0);
}


/* since we've already tested if it's possible, this can't fail */
void fix_to_similar_mode(int modetype)
{
int f,newmode,numpixels,pixdiff,newnp,newdiff,mode_ok;
vga_modeinfo *vminfo;
struct modedesc_tag *md_ptr;

/* here's the plan;
 * - we try to keep the number of pixels as close as possible:
 *   as this will probably be quite noticeable, *this takes priority*.
 * - we then try to match any 8-bit modes for EIGHT_BIT, or match
 *   24/16/15-bit modes (in that order) for HIGHCOLOUR.
 * - of course, the mode has to be either 8 or 15/16/24-bit as reqd.
 *
 * no support in this for 640x480x4 (16-colour) - because that's
 * a really weird special-case mode, you have to explicitly select it
 * if you want it. However, it sticks in 640x480x4 if selected and
 * modetype is FIX_TO_EIGHT_BIT, because it'd be totally annoying
 * otherwise.
 */

if(curvgamode==G640x480x16 && modetype==FIX_TO_EIGHT_BIT) return;

newmode=-1; pixdiff=(1<<30);
numpixels=get_mode_width(curvgamode)*get_mode_height(curvgamode);

/* account for effect of 'virtual' */
if(curvgamode==G320x400x256 || curvgamode==G360x480x256)
  numpixels*=2;

for(md_ptr=modedesc;md_ptr->mode;md_ptr++)
  {
  f=md_ptr->mode;
  if(!vga_hasmode(f) || cfg.mode_allowed[f]==0 ||
     (vminfo=vga_getmodeinfo(f))==NULL)
    continue;
  
  mode_ok=0;
  if(modetype==FIX_TO_EIGHT_BIT)
    {
    if(vminfo->colors==256) mode_ok=1;
    }
  else
    {
    if(vminfo->colors>256) mode_ok=1;
    }
  
  if(mode_ok==0) continue;
  
  newnp=(vminfo->width)*(vminfo->height);
  if(f==G320x400x256 || f==G360x480x256)
    newnp*=2;   /* account for effect of 'virtual' */
  newdiff=numpixels-newnp;
  if(newdiff<0) newdiff=(-newdiff);
  
  if(newdiff<=pixdiff)
    {
    newmode=f;
    pixdiff=newdiff;
    }
  }

/* we rely on the 15/16/32/24-bit mode number ordering to sort that
 * out for us (hence the `<=' above).
 */
curvgamode=newmode;
virtual=(curvgamode==G320x400x256 || curvgamode==G360x480x256);
}


void aborted_file_cleanup()
{
switch(loading_file_type)
  {
  case _IS_GIF:
    aborted_file_gif_cleanup();
    break;
  case _IS_JPEG:
    aborted_file_jpeg_cleanup();
    break;
  case _IS_PNM: case _IS_TIFF:
    aborted_file_pnm_cleanup();
    break;
  case _IS_BMP:
    aborted_file_bmp_cleanup();
    break;
  case _IS_TGA:
    aborted_file_tga_cleanup();
    break;
#ifdef PNG_SUPPORT
  case _IS_PNG:
    aborted_file_png_cleanup();
    break;
#endif
  case _IS_XVPIC:
    fprintf(stderr,"aborted xvpic load: can't happen!\n");
    exit(1);
  case _IS_PCX:
    aborted_file_pcx_cleanup();
    break;
  case _IS_MRF:
    aborted_file_mrf_cleanup();
    break;
  case _IS_XBM:
    aborted_file_xbm_cleanup();
    break;
  case _IS_XPM:
    aborted_file_xpm_cleanup();
    break;
  }
}


void makerealpal()
{
int f;
int r,rlo,g,glo,b,blo;
int gval;

for(f=0;f<256;f++)
  {
  r=palr[f]>>2; rlo=palr[f]&3;
  g=palg[f]>>2; glo=palg[f]&3;
  b=palb[f]>>2; blo=palb[f]&3;

  if(cfg.fakecols)
    {
    gval=(rlo*grey_red+glo*grey_green+blo*grey_blue)/4;
    /* so gval is sub-value, 0<=gval<1000. (really <750... :-( )
     * possible emulated sub-values are...
     * 0, 114 (b), 299 (r), 413 (r+b), 587 (g), 701 (g+b), 886 (r+g)
     */
  
    if(gval>=886)
      r++,g++;
    else if(gval>=701)
      g++,b++;
    else if(gval>=587)
      g++;
    else if(gval>=413)
      r++,b++;
    else if(gval>=299)
      r++;
    else if(gval>=114)
      b++;

    if(r>63) r=63; if(g>63) g=63; if(b>63) b=63;
    }

  palrgb[f*3  ]=palr64[f]=r;
  palrgb[f*3+1]=palg64[f]=g;
  palrgb[f*3+2]=palb64[f]=b;
  }
}


void filterpal(byte *palette)
{
int f;

for(f=0;f<256;f++)   /* don't *really* need to know number of colours */
  {
  /* we also abuse this to get a brightness/contrast-ified index
   * to use for high-colour mode b/c. :-) (Hence palt[].)
   */
  if(cfg.bc_order_rev)
    {
    palr[f]=contrastup(dimmer(palette[    f]));
    palg[f]=contrastup(dimmer(palette[256+f]));
    palb[f]=contrastup(dimmer(palette[512+f]));
    palt[f]=contrastup(dimmer(f));
    }
  else
    {
    palr[f]=dimmer(contrastup(palette[    f]));
    palg[f]=dimmer(contrastup(palette[256+f]));
    palb[f]=dimmer(contrastup(palette[512+f]));
    palt[f]=dimmer(contrastup(f));
    }
  }

makerealpal();
}


int dimmer(int a)
{
a+=brightness;
if(a<0) a=0;
if(a>255) a=255;
return(a);
}


int contrastup(int cp)
{
float g;

g=(float)(cp);
g=128.+(g-128.)*contrast;
if(g<0.) g=0.;
if(g>255.) g=255.;
return((int)g);
}


int get_mode_width(int vm)
{
vga_modeinfo *vmi;

vmi=vga_getmodeinfo(vm);
return(vmi->width);
}

int get_mode_height(int vm)
{
vga_modeinfo *vmi;

vmi=vga_getmodeinfo(vm);
return(vmi->height);
}

int get_mode_pixelbytes(int vm)
{
vga_modeinfo *vmi;

vmi=vga_getmodeinfo(vm);
return(vmi->bytesperpixel);
}

int get_mode_numcols(int vm)
{
vga_modeinfo *vmi;

vmi=vga_getmodeinfo(vm);
return(vmi->colors);
}


void do_16col_palette()
{
int paltmp[16*3],f,r,g,b;

if(cfg.viewer16col)
  {
  /* colour */
  for(f=r=0;r<2;r++)
    for(g=0;g<2;g++)
      for(b=0;b<2;b++,f+=3)
        paltmp[f]=r*63,paltmp[f+1]=g*63,paltmp[f+2]=b*63;
  /* FWIW, zero colours 8..15. It may seem wasteful not using these,
   * but there's no way I can see to use them as part of an ordered
   * dither which doesn't look awful (and error-diffused dithering
   * is too slow).
   */
  for(;f<16*3;f+=3)
    paltmp[f]=paltmp[f+1]=paltmp[f+2]=0;
  
  /* actually, in that case, best to allocate some for right-button
   * menu colours!
   */
  f=8;
  paltmp[f*3  ]=cfg.light.r;
  paltmp[f*3+1]=cfg.light.g;
  paltmp[f*3+2]=cfg.light.b;
  f++;
  paltmp[f*3  ]=cfg.medium.r;
  paltmp[f*3+1]=cfg.medium.g;
  paltmp[f*3+2]=cfg.medium.b;
  f++;
  paltmp[f*3  ]=cfg.dark.r;
  paltmp[f*3+1]=cfg.dark.g;
  paltmp[f*3+2]=cfg.dark.b;
  f++;
  paltmp[f*3  ]=cfg.black.r;
  paltmp[f*3+1]=cfg.black.g;
  paltmp[f*3+2]=cfg.black.b;
  }
else
  {
  /* 16-grey palette */
  for(f=0;f<16;f++)
    paltmp[f*3]=paltmp[f*3+1]=paltmp[f*3+2]=(f*63)/15;
  }

vga_setpalvec(0,16,paltmp);
}


/* set new mode if needed, else just clear screen. */
void setmode_or_clear(int newmode)
{
if(vga_getcurrentmode()!=newmode)
  vga_setmode(curvgamode); 
else
  vga_clear();
}


void init_vkludge_cache(void)
{
int x;

/* invalidate any cached colours */
memset(vkcache_valid,0,sizeof(vkcache_valid));

/* the actual palette colours are fairly likely to crop up :-),
 * so index them automatically.
 */
for(x=0;x<numcols;x++)
  closest(pal32_no_bc[x*3],pal32_no_bc[x*3+1],pal32_no_bc[x*3+2]);
}


void graphicson(void)
{
msgbox_draw_ok=0;

vga_lockvc();
if((vga_hasmode(curvgamode))&&(cfg.mode_allowed[curvgamode]))
  setmode_or_clear(curvgamode);
else
  {
  /* we haven't? Aaargh!!! Ok, use 640x480 or 360x480 instead. */
  if((vga_hasmode(G640x480x256))&&(cfg.mode_allowed[G640x480x256]))
    setmode_or_clear(curvgamode=G640x480x256);
  else
    {
    if((vga_hasmode(G360x480x256))&&(cfg.mode_allowed[G360x480x256]))
      setmode_or_clear(curvgamode=G360x480x256);
    else
      /* *must* have 320x200 (see rcfile.c for more info) */
      setmode_or_clear(curvgamode=G320x200x256);
    }
  }

virtual=0;
if(curvgamode==G320x400x256 || curvgamode==G360x480x256)
  virtual=1;

if(curvgamode==G640x480x16)
  do_16col_palette();

if(pixelsize!=1) gl_setcontextvga(curvgamode);

scrnwide=get_mode_width(curvgamode);
scrnhigh=get_mode_height(curvgamode);
scrnpixelsize=get_mode_pixelbytes(curvgamode);
scrncols=get_mode_numcols(curvgamode);

if(virtual)
  scrnwide*=2,scaling=1;
if(pixelsize==1)
  setpalvec(0,256,palrgb);

vga_unlockvc();
}


void graphicsoff()
{
/* we let zgv.c deal with this now, so this is just a placeholder */
}


/* tells us if mode is ok to use in current circumstances. */
int mode_is_usable_now(int modenum)
{
if(vga_hasmode(modenum) && cfg.mode_allowed[modenum] &&
   ((pixelsize==1 && get_mode_numcols(modenum)<=256) ||
    (pixelsize==3 && get_mode_numcols(modenum)> 256)))
  return(1);
return(0);
}


/* returns 1 if ESC or 'x' was pressed, 0 if because of timeout when
 * using cfg.repeat_timer option.
 *
 * caller must restore mouse pos this saves, in appropriate mode.
 */
int showgif(byte *palette)   /* global theimage,height,width implied */
{
int opx,opy,px,py,quitshow,key,redraw;
int npx=0,npy=0;    /* equivalent to px,py in scaling>1 mode */
int f;
int mode_prefix=0;
int rb_menu_mode=0,scrolling_with_mouse=0;
int slideshow_pause=0;
static int orient_lastpicexit_state=0;	/* last picture orientation state */
int orient_current_state;		/* current picture orientation state */
int vkcache_reinit=0;

wait_for_foreground();

/* make sure we have a decent saved mouse position (initial position,
 * or (more usually) from selector).
 */
save_mouse_pos();

orient_current_state=0;		/* picture starts off at normal orientation */

/* if image has less than 256 colours, we fill in the 64 greys so that
 * using the vkludge on, say, a mono file will look much better.
 */
if(numcols<256)
  {
  for(f=numcols;f<numcols+64;f++)
    palette[f]=palette[256+f]=palette[512+f]=((f-numcols)<<2);
  /* get rid of any junk entries which vkludge might try and use */
  for(;f<256;f++)
    palette[f]=palette[256+f]=palette[512+f]=0;
  }

/* put 5-bit version of palette[] in pal32_no_bc. This is in a different
 * format, as required by closest() which uses it.
 */
for(f=0;f<256;f++)
  {
  pal32_no_bc[f*3  ]=(palette[f]>>3);
  pal32_no_bc[f*3+1]=(palette[f+256]>>3);
  pal32_no_bc[f*3+2]=(palette[f+512]>>3);
  }

/* init vkludge cache for this picture. Doesn't need updating for
 * brightness/contrast changes, and is filled in as needed.
 */
if(pixelsize==1)
  init_vkludge_cache();

pic_incr=repeat_sig=0;
if(cfg.repeat_timer && !tagview_mode)
  {
  if(cfg.repeat_timer==-1)
    repeat_sig=2;	/* make it look (nearly) instant */
  else
    {
    signal(SIGALRM,repeat_sighandler);
    alarm(cfg.repeat_timer);
    }
  }

if(tagview_mode)
  {
  signal(SIGALRM,repeat_sighandler);
  alarm(cfg.tag_timer);
  }

quitshow=0;
filterpal(palette);
if(!cfg.repeat_timer || (cfg.repeat_timer && first_repeat))
  {
  px=py=0;
  graphicson();
  first_repeat=0;
  if(cfg.revert)
    {
    scaling=1;
    interp=0;
    }
  if(!cfg.revert_orient && !orient_override)
    {
    orient_change_state(orient_current_state,orient_lastpicexit_state,0);
    orient_current_state=orient_lastpicexit_state;
    }
  }
else
  if(cfg.repeat_timer)
    {
    px=saved_px;
    py=saved_py;
    }

/* an orientation override always happens, even for repeat pics.
 * (Not least because it seems like the sort of thing which would be
 * dead useful for them. :-))
 */
if(orient_override)
  {
  orient_change_state(orient_current_state,orient_override_state,0);
  orient_current_state=orient_override_state;
  }

redrawgif(px,py,npx,npy);

while(!quitshow && (repeat_sig!=1 || rb_menu_mode || slideshow_pause))
  {
  if(rb_menu_mode) mousecur_on();
  key=wait_for_keys_or_mouse(zgv_ttyfd);
  if(rb_menu_mode) mousecur_off();
  
  opx=px; opy=py; redraw=0;
  
  if(rb_menu_mode)
    {
    /* we deal with keys ourselves... */
    if(key==RK_ESC || key=='x')
      {
      rb_menu_mode=0;
      undraw_rb_menu();
      continue;
      }
    
    /* but mouse dealt with by this. if not clicked an entry, skip
     * rest of loop.
     * (no repeat_sig==2 handling required here, as even in incredibly
     * pathological circumstances we *can't* get here with repeat_sig==2.)
     */
    if(rb_menu_event(&key))	/* returns non-zero if should stay in menu */
      continue;
    
    /* otherwise restore saved area of screen, and do action in (faked) key */
    rb_menu_mode=0;
    undraw_rb_menu();
    goto parsekey;	/* skip normal mouse stuff just in case */
    }
  
  if(has_mouse)
    {
    int mbuttons=mouse_getbutton();
    int mleft_start=is_start_click_left();
    /* important to read both, even if not using both */
    int mleft=is_end_click_left(),mright=is_end_click_right();
    
    /* if right button released, get the right-button menu. */
    if(mright)
      {
      /* only allow it if >=480 pixels high! */
      if(scrnhigh>=480)
        {
        rb_menu_mode=1; scrolling_with_mouse=0;
        draw_rb_menu();
        /* if left mouse button down, ignore first end-left-click. */
        rb_ignore_first_left_click=((mbuttons&MOUSE_LEFTBUTTON)?1:0);
        
        /* skip rest of main loop except this - we still might have to
         * sort this out (hugely unlikely, but all the same...)
         */
        if(repeat_sig==2) repeat_sig--;
        continue;
        }
      }
    
    if(scrolling_with_mouse)
      {
      /* mouse is in sensible range/location for image scroll, then;
       * read the mouse position into px/py.
       */
      px=mouse_getx();
      py=mouse_gety();
      }
    
    /* if left mouse button pressed, or currently scrolling w/mouse,
     * prepare for next event's movement.
     * (a bit nasty doing this every time, but we have to, really.)
     */
    if((mleft_start || scrolling_with_mouse) && !zoom)
      {
      int max_x,max_y;
      
      scrolling_with_mouse=1;
      max_x=width*scaling-scrnwide-1;
      max_y=height*scaling-scrnhigh-1;
      /* make sure we don't give it crazy ranges, esp. not 0..0,
       * which would probably cause a div by zero or something...
       */
      if(max_x<=0) max_x=1;
      if(max_y<=0) max_y=1;
      
      mouse_setxrange(0,max_x);
      mouse_setyrange(0,max_y);
      mouse_setposition(px,py);
      }
    
    /* if left button not down, don't move with mouse any more.
     * this is after the rest to make sure we don't miss any
     * scrolling motions if scrolling_with_mouse is currently set.
     */
    if(!(mbuttons&MOUSE_LEFTBUTTON))
      scrolling_with_mouse=0;
    
    mleft=0;	/* kludge to keep gcc -Wall happy */
    }
  
  parsekey:
  /* avoid problems with mode or scaling change etc. */
  if(key) scrolling_with_mouse=0;
  
  /* mode selection - note that F11=shift-F1 and F12=shift-F2 */
  if((key>='4' && key<='8') ||
     (key>=RK_F1 && key<=RK_F10) || (key>=RK_SHIFT_F1 && key<=RK_SHIFT_F10))
    {
    struct modedesc_tag *md_ptr;
    int mode=0,fallback_mode=0;
    
    for(md_ptr=modedesc;md_ptr->mode;md_ptr++)
      {
      if(mode_prefix!=md_ptr->is_tab_key) continue;
      
      if(key==md_ptr->viewer_key && mode_is_usable_now(md_ptr->mode))
        if(md_ptr->bytespp==4)	/* if 32-bit */
          fallback_mode=md_ptr->mode;
        else
          mode=md_ptr->mode;
      }
    
    mode_prefix=0;
    
    /* if mode is no good, try the fallback (if any) */
    if(!mode && fallback_mode)
      mode=fallback_mode;
    
    /* if we've got something valid, use it. */
    if(mode)
      {
      curvgamode=mode;
      graphicson();
      redraw=1;
      }
    }
  else	/* if not a mode-selecting key... */
    {
    if(key) mode_prefix=0;
    
    switch(key)
      {
      case 9:		/* tab */
        mode_prefix=1;
        break;
      case 'F':
        cfg.fakecols=!cfg.fakecols;
        if(pixelsize==1)
          filterpal(palette),setpalvec(0,256,palrgb);
        break;
      
      /* the way pause/resume slideshow works is quite simple -
       * pausing overrides the repeat_sig var set when the timer
       * times out, and resuming disables that override.
       */
      case 'S'-0x40:	/* pause slideshow */
        if(tagview_mode)
          slideshow_pause=1;
        break;
      case 'Q'-0x40:	/* resume slideshow */
        if(tagview_mode)
          slideshow_pause=0;
        break;
      
      case 's': case 'd':
        if(virtual) break;
        if(zoom) vga_clear();	/* cls if zoomed, in case x2 is smaller! */
        zoom=0;
        if(scaling<512)
          {
          int oldscale=scaling;
          
          scaling+=(key=='d')?scaling:1;
          if(scaling>512) scaling=512;
          samecentre(&px,&py,scaling,px,py,oldscale);
          redraw=1; /* no cls reqd. - must be bigger */
          }
        break;
      case 'S': case 'D':
        if(virtual) break;
        if(scaling>1 || zoom)
          {
          int oldscale=scaling;
          
          scaling-=(key=='D')?scaling/2:1;
          if(scaling<1) scaling=1;
          samecentre(&px,&py,scaling,px,py,oldscale);
          redraw=1;
          if(width*scaling<scrnwide || height*scaling<scrnhigh) vga_clear();
          }
        zoom=0;
        break;
      case 'n':
        if(zoom || scaling>1)
          {
          samecentre(&px,&py,1,px,py,scaling);
          scaling=1; zoom=0;
          vga_clear(); 
          }
        redraw=1; interp=0; inextpix=1;
        break;
      case RK_HOME: case 'A'-0x40:
        px=py=0; break;
      case RK_END:  case 'E'-0x40:
        px=py=1<<30; break;
      case RK_PAGE_UP: case 'U'-0x40:
        py-=scrnhigh*9/10; break;
      case RK_PAGE_DOWN: case 'V'-0x40:
        py+=scrnhigh*9/10; break;
      case '-':
        px-=scrnwide*9/10; break;
      case '=':
        px+=scrnwide*9/10; break;
      case 'v': vkludge=((vkludge==1)?0:1); redraw=1; graphicson(); break;
      case ',': case '.': case '<': case '>': case 'B': case '*':
        switch(key)
          {
          case ',':	contrast-=0.05; break;
          case '.':	contrast+=0.05; break;
          case '<':	brightness-=10; break;
          case '>':	brightness+=10; break;
          case 'B':	cfg.bc_order_rev=!cfg.bc_order_rev; break;
          default: 	brightness=0; contrast=1.0; break;
          }
        filterpal(palette);	/* make palr/g/b/t */
        if(scrnpixelsize>1)
          redraw=1;			/* redraw if non-palette-based... */
        else
          setpalvec(0,256,palrgb);	/* else change onscreen palette */
        /* 640x480 16-colour needs *both*! */
        if(curvgamode==G640x480x16) redraw=1;
        /* any palette change will invalidate the vkludge cache.
         * Set flag to say we should reinit the cache when we next redraw.
         */
        vkcache_reinit=1;
        break;
      case 'q': case 'k': py-=MOVSTP; break;
      case 'a': case 'j': py+=MOVSTP; break;
      case 'o': case 'h': px-=MOVSTP; break;
      case 'p': case 'l': px+=MOVSTP; break;
      case 'Q': case 'K': case RK_CURSOR_UP:    py-=BIGSTP; break;
      case 'A': case 'J': case RK_CURSOR_DOWN:  py+=BIGSTP; break;
      case 'O': case 'H': case RK_CURSOR_LEFT:  px-=BIGSTP; break;
      case 'P': case 'L': case RK_CURSOR_RIGHT: px+=BIGSTP; break;
      case 'm':		/* mirror */
        fx_mirror();
        orient_current_state=orient_state_mirror[orient_current_state];
        px=py=0; redraw=1;
        break;
      case 'f':		/* flip */
        fx_flip();
        orient_current_state=orient_state_flip[orient_current_state];
        px=py=0; redraw=1;
        break;
      case 'r':		/* rotate clockwise */
        fx_rot(); vga_clear();
        orient_current_state=orient_state_rot_cw[orient_current_state];
        px=py=0; redraw=1;
        break;
      case 'R':		/* rotate anti-clockwise */
        fx_rot(); fx_flip(); fx_mirror(); vga_clear();
        orient_current_state=orient_state_rot_acw[orient_current_state];
        px=py=0; redraw=1;
        break;
      case 128+'n':	/* restore original orientation (state 0) */
        orient_change_state(orient_current_state,0,1);
        orient_current_state=0;
        px=py=0; redraw=1;
        break;
      case 128+'o':	/* re-use previous pic's orientation */
        orient_change_state(orient_current_state,orient_lastpicexit_state,1);
        orient_current_state=orient_lastpicexit_state;
        px=py=0; redraw=1;
        break;
      case 128+'s':	/* save current orientation, overriding until Esc */
        orient_override_state=orient_current_state;
        orient_override=1;
        break;
      case 'z':
        zoom=(!zoom); scaling=redraw=1; px=py=0; vga_clear(); break;
      case 'i':
        interp=(!interp); redraw=1; break;
      case '1':
        inextpix=(inextpix==1)?2:1;
        redraw=1;
        break;
      case '?':
        /* showhelp restores mouse pos */
        showhelp(zgv_ttyfd,"- KEYS FOR VIEWER -",viewerhelp);
        save_mouse_pos();
        graphicson(); redraw=1;
        break;
      case '/':
        /* showhelp restores mouse pos */
        showhelp(zgv_ttyfd,"- SELECTING VIDEO MODES -",viewermodeshelp);
        save_mouse_pos();
        /* falls through to `refresh screen' */
      case 12: case 18:     /* 12,18 = Ctrl-L, Ctrl-R */
        graphicson(); redraw=1; break;
      case 'N'-0x40:
        pic_incr=1; quitshow=1; break;
      case 'P'-0x40:
        pic_incr=-1; quitshow=1; break;
      case RK_ENTER:
	pic_incr=1; quitshow=2; break;
      case ' ':
	pic_incr=1; quitshow=3; break;
      case 127: case 8:
	pic_incr=-1; quitshow=2; break;
      case 'c': case 'C':
        if(curvgamode==G640x480x16)
          {
          cfg.viewer16col=!cfg.viewer16col;
          graphicson();		/* palette has to change */
          redraw=1;
          }
        break;
      case RK_ESC: case 'x':
        quitshow=1;
      }
    }
      
  if(!zoom)
    {
    int swidth=width*scaling,sheight=height*scaling;
    
    if(sheight<=scrnhigh)
      py=0;
    else
      if(sheight-py<scrnhigh) py=sheight-scrnhigh;
    if(swidth<=scrnwide)
      px=0;
    else
      if(swidth-px<scrnwide) px=swidth-scrnwide;
    if(px<0) px=0;
    if(py<0) py=0;
    if(scaling>1)
      {
      npx=px/scaling;
      npy=py/scaling;
      }
    }
  else
    px=py=npx=npy=0;
    
  if(redraw || opx!=px || opy!=py)
    {
    if(vkludge && vkcache_reinit)
      init_vkludge_cache();
    vkcache_reinit=0;
    redrawgif(px,py,npx,npy);
    }
  
  if(repeat_sig==2) repeat_sig--;
  }

graphicsoff();

if(cfg.selecting)
  show_dimensions(px,py,scaling);

if(cfg.repeat_timer || tagview_mode)
  {
  alarm(0);
  saved_px=px;
  saved_py=py;
  if(quitshow==1) tagview_mode=0;
  }

/* save orientation state as last-picture-exit state */
orient_lastpicexit_state=orient_current_state;

/* caller must do mouse restore */

return(quitshow);
}


/* save old contents of area to put right-button menu on, and draw it. */
static void draw_rb_menu()
{
int light,medium,dark,black;		/* for menu */
int mwhite,mblack;			/* for mouse pointer */
int f,y;

rbm_xysize(viewer_menu_data,&rb_save_width,&rb_save_height);
if(scrnpixelsize<1) scrnpixelsize=1;	/* sanity check */

if((rb_save=malloc(rb_save_width*rb_save_height*scrnpixelsize))==NULL) return;

switch(curvgamode)
  {
  /* for 16-colour and 8-bit generic-VGA, don't use vgagl.
   * though I have many modes listed here, in truth it's only
   * 640x480x4 and 360x480x8 this can possibly be called for.
   * (NB: can't use scrnwide here, as that's twice phys. width in 360x480.)
   */
  case G640x480x16:
  case G320x200x256: case G320x240x256:
  case G320x400x256: case G360x480x256:
    for(y=0;y<rb_save_height;y++)
      vga_getscansegment(rb_save+rb_save_width*y,
      		vga_getxdim()-rb_save_width,y,rb_save_width);
    break;
  
  /* otherwise use vgagl */
  default:
    /* may not be current context if 8-bit mode, so make sure. */
    gl_setcontextvga(curvgamode);
    
    gl_getbox(scrnwide-rb_save_width,0,rb_save_width,rb_save_height,rb_save);
  }

/* grey out (make non-active) any ones we shouldn't allow.
 * this is nasty, but decl of viewer_menu_data notes that there's
 * a nasty thing here, so it should be ok.
 */

f=1;
viewer_menu_data[f++].active=mode_is_usable_now(G360x480x256);
viewer_menu_data[f++].active=mode_is_usable_now(G640x480x16);
viewer_menu_data[f++].active=mode_is_usable_now(G640x480x256);
viewer_menu_data[f++].active=mode_is_usable_now(G800x600x256);
viewer_menu_data[f++].active=mode_is_usable_now(G1024x768x256);
viewer_menu_data[f++].active=mode_is_usable_now(G640x480x32K);
viewer_menu_data[f++].active=mode_is_usable_now(G640x480x64K);
viewer_menu_data[f++].active=mode_is_usable_now(G640x480x16M);
viewer_menu_data[f++].active=mode_is_usable_now(G800x600x32K);
viewer_menu_data[f++].active=mode_is_usable_now(G800x600x64K);
viewer_menu_data[f++].active=mode_is_usable_now(G800x600x16M);
viewer_menu_data[f++].active=mode_is_usable_now(G1024x768x32K);
viewer_menu_data[f++].active=mode_is_usable_now(G1024x768x64K);
viewer_menu_data[f++].active=mode_is_usable_now(G1024x768x16M);

rbm_set_active_flag(viewer_menu_data,"grey/colour",(curvgamode==G640x480x16));

/* make some colours available if needed for light/medium/dark/black
 * colours. Here's the way this works:
 * - for 15/16/24-bit modes, just use the colours.
 * - for 8-bit, find closest matches and force those to the right
 *   colour *temporarily*, i.e. while rb menu is onscreen. That means
 *   we'll need a palette `redraw' (setpalvec) when screen is restored.
 * - for 640x480 4-bit, we kludge it a bit (see below).
 */
switch(scrncols)
  {
  case 16:	/* 640x480 16-colour */
    if(cfg.viewer16col)
      {
      /* if in colour, they were reserved for us out of the 2nd 8 colours */
      light=8; medium=9; dark=10; black=11;
      mwhite=light; mblack=black;
      }
    else
      {
      /* otherwise, we use some fixed greys which match the default
       * zgv setup fairly closely.
       */
      light=9; medium=7; dark=5; black=0;
      mwhite=light; mblack=1;
      }
    break;
  
  case 256:	/* 8-bit */
    /* find closest unique non-zero matches
     * (important for mouse that light/black aren't zero)
     */
    light =rb_find_closest(cfg.light.r,cfg.light.g,cfg.light.b,
    				-1,-1,-1);
    medium=rb_find_closest(cfg.medium.r,cfg.medium.g,cfg.medium.b,
    				light,-1,-1);
    dark  =rb_find_closest(cfg.dark.r,cfg.dark.g,cfg.dark.b,
    				light,medium,-1);
    black =rb_find_closest(cfg.black.r,cfg.black.g,cfg.black.b,
    				light,medium,dark);
    mwhite=light; mblack=black;
    
    /* force colours to fit */
    vga_setpalette(light,cfg.light.r,cfg.light.g,cfg.light.b);
    vga_setpalette(medium,cfg.medium.r,cfg.medium.g,cfg.medium.b);
    vga_setpalette(dark,cfg.dark.r,cfg.dark.g,cfg.dark.b);
    vga_setpalette(black,cfg.black.r,cfg.black.g,cfg.black.b);
    break;
  
  default:	/* 15/16/24-bit */
    light =((cfg.light.r<<18)|(cfg.light.g<<10)|(cfg.light.b<<2));
    medium=((cfg.medium.r<<18)|(cfg.medium.g<<10)|(cfg.medium.b<<2));
    dark  =((cfg.dark.r<<18)|(cfg.dark.g<<10)|(cfg.dark.b<<2));
    black =((cfg.black.r<<18)|(cfg.black.g<<10)|(cfg.black.b<<2));
    /* mblack/mwhite should be in screen format */
    mblack=1;	/* black musn't be 0, unfortunately */
    switch(scrncols)
      {
      case 32768:
        mwhite=GET15BITCOLOUR(4*cfg.light.r,4*cfg.light.g,4*cfg.light.b);
        break;
      case 65536:
        mwhite=GET16BITCOLOUR(4*cfg.light.r,4*cfg.light.g,4*cfg.light.b);
        break;
      default:
        mwhite=light;	/* 24-bit is easy :-) */
      }
    break;
  }

mousecur_init(mblack,mwhite);

/* restore old mouse pos. */
restore_mouse_pos();

/* now draw the thing */
rbm_draw(viewer_menu_data,light,medium,dark,black);
}


/* hacked version of closest() for 8-bit rb menu colour lookup */
static int rb_find_closest(int r,int g,int b,int not1,int not2,int not3)
{
int rgb;
byte *pr,*pg,*pb;
byte distnum;
int xr,xg,xb,dist,distquan,f,checknumcols;

rgb=((b<<12)|(g<<6)|r);
distnum=0;
distquan=20000; /* standard arbitrary bignum */
/* if numcols=256 we do 0-255, otherwise 0-numcols+63 */
checknumcols=((numcols==256)?256:numcols+64);
for(pr=palr64,pg=palg64,pb=palb64,f=0;f<checknumcols;f++,pr++,pg++,pb++)
  {
  xr=(r-*pr);
  xg=(g-*pg);
  xb=(b-*pb);
  if((dist=xr*xr+xg*xg+xb*xb)<distquan &&
     f!=0 && f!=not1 && f!=not2 && f!=not3)
    {
    distnum=f;
    distquan=dist;
    if(dist==0) break;  /* premature exit if it can't get any better */
    }
  }
return(distnum);
}


/* restore old contents of area with right-button menu on. */
static void undraw_rb_menu()
{
int y;

/* save current mouse pos. */
save_mouse_pos();

if(rb_save==NULL) return;	/* ran out of memory, can't do much! */

switch(curvgamode)
  {
  /* again, not vgagl, and again, can't use scrnwide */
  case G640x480x16:
  case G320x200x256: case G320x240x256:
  case G320x400x256: case G360x480x256:
    for(y=0;y<rb_save_height;y++)
      vga_drawscansegment(rb_save+rb_save_width*y,
      		vga_getxdim()-rb_save_width,y,rb_save_width);
    break;
  
  /* vgagl */
  default:
    gl_putbox(scrnwide-rb_save_width,0,rb_save_width,rb_save_height,rb_save);
  }

if(scrncols==256)
  setpalvec(0,256,palrgb);	/* restore palette */

free(rb_save);
}


/* possibly have a mouse event to deal with for right-button menu.
 * uses pointer to key to fake keys to do stuff, and returns
 * 1 if we should stay in rb menu mode, else 0.
 */
static int rb_menu_event(int *keyp)
{
/* important to read both, even if not using both */
int mleft=is_end_click_left(),mright=is_end_click_right();
int key;

if(!mleft)
  return(1);

if(rb_ignore_first_left_click)
  {
  rb_ignore_first_left_click=0;
  return(1);
  }

key=mright;	/* a kludge to keep gcc -Wall quiet */

/* get faked key for viewer_menu_data, or zero if none */
*keyp=0;
key=rbm_mousepos_to_key(viewer_menu_data,mouse_getx(),mouse_gety());
if(key!=-1)	/* -1 means quit menu with no key */
  *keyp=key;

return((key==0));	/* 1 if didn't match any, else 0 */
}



void repeat_sighandler(int foo)
{
repeat_sig=1;
}


void setpalvec(int start,int num,int *pal)
{
/* only relevant for 8-bit modes */
if(pixelsize==1)
  if(curvgamode!=G640x480x16)
    vga_setpalvec(start,num,pal);
  else
    {
    int f,c;
    unsigned char *dptr=dither16_rgb+start*3;
    int *sptr=pal+start*3;
    
    for(f=start;f<start+num;f++)
      {
      if(cfg.viewer16col)
        {
        /* add contrast to compensate for dithering (kind of) */
        c=(*sptr++-32)*4/3+32;
        if(c<0) c=0; if(c>63) c=63;
        *dptr++=c;
        c=(*sptr++-32)*4/3+32;
        if(c<0) c=0; if(c>63) c=63;
        *dptr++=c;
        c=(*sptr++-32)*4/3+32;
        if(c<0) c=0; if(c>63) c=63;
        *dptr++=c;
        }
      else
        {
        c=(pal[f*3]*grey_red+pal[f*3+1]*grey_green+
           pal[f*3+2]*grey_blue)/1000;
        if(c>63) c=63;
        dither16_greylookup[f]=c;
        }
      }
    }
}


/* this routine is getting ridiculous */
void redrawgif(int px,int py,int npx,int npy)
{
int x,y,xdim;
int x_add,y_add;
byte *realline,*ptr;

/* set line-draw routine depending on whether we need to do
 * 15/16/24-bit brightness/contrast or not.
 * `doing_hicol_bc' just saves us doing this test elsewhere.
 */
eventuallydrawscansegment=eventuallydrawscansegment_without_bc;
doing_hicol_bc=0;
if(scrnpixelsize>1 && cfg.hicontrol && (brightness!=0 || contrast!=1.0))
  {
  eventuallydrawscansegment=eventuallydrawscansegment_with_bc;
  doing_hicol_bc=1;
  }

if(zoom)
  drawzoomedgif();
else
  {
  int swidth=width*scaling,sheight=height*scaling;
  
  /* draw non-zoomed pic */
  x_add=y_add=0;	/* these control the centering */
  if(cfg.centreflag)
    {
    if(swidth<scrnwide)
      x_add=(scrnwide-swidth)>>1;
    
    if(sheight<scrnhigh)
      y_add=(scrnhigh-sheight)>>1;
      
    if(virtual) x_add>>=1;
    }
  
  if(swidth-px<scrnwide) xdim=swidth-px; else xdim=scrnwide;
  if((py>=sheight)||(px>=swidth)) return;
  /* hopefully the following is fairly quick as fewer ifs... ? */
  if(virtual)		/* 320x400 or 360x480 */
    {
    if((realline=calloc(1,scrnwide))==NULL) return;
    for(y=0;(y<sheight-py)&&(y<scrnhigh);y++)
      {
      for(x=0;x<xdim;x++)
        *(realline+(x>>1))=getvpix(px,py,x,y);
      vga_drawscansegment(realline,x_add,y+y_add,xdim>>1);
      }
    free(realline);
    }
  else if(scaling>1)
    {
    /* Better grab your Joo Janta 200 Super-Chromatic Peril Sensitive
     * Sunglasses (both pairs) for this next bit...
     */
    int cdown=-1,i,pxx,pyy,pyym;
    int a1,a2,a3,a4,in_rg,in_dn,in_dr;
    int wp=width*pixelsize,sci=scaling*inextpix;
    int scaleincr=0,subpix_xpos,subpix_ypos,sxmulsiz,symulsiz,simulsiz=0;
    int sisize=0,sis2=0;
    unsigned char *ptr1,*ptr2,*ptr3,*ptr4;
    unsigned char *src,*dst,*nextline;
    
    /* this doesn't allow for 'virtual' yet! */
    if((realline=calloc(pixelsize,scrnwide))==NULL) return;
    if(pixelsize==3)
      if((nextline=calloc(pixelsize,scrnwide))==NULL) return;
    
    if(interp)
      {
      sisize=0;
      while(sisize<256) sisize+=scaling;
      scaleincr=sisize/scaling;
      simulsiz=scaleincr*sisize;
      sis2=sisize*sisize;
      }
    
    for(y=0,pyy=py;(y<sheight-py)&&(y<scrnhigh);y++,pyy++)
      {
      /* this is horribly slow... :-( */
      if(cdown<=0 || interp)
        {
        src=theimage+pixelsize*(pyy/scaling)*width;
        dst=realline;
        if(pixelsize==1)
          for(x=0;(x<swidth-px)&&(x<scrnwide);x++)
            *dst++=*(src+(px+x)/scaling);
        else if(interp==0)
          /* normal */
          for(x=0;(x<swidth-px)&&(x<scrnwide);x++)
            {
            ptr=src+((px+x)/scaling)*pixelsize;
            *dst++=*ptr++; *dst++=*ptr++; *dst++=*ptr;
            }
        else
          {
          /* interpolated */
          
          /* This has been hacked into unreadability in an attempt to get it
           * as fast as possible.
           * It's still really slow. :-(
           */

          in_rg=inextpix*3;
          in_dn=inextpix*wp;
          in_dr=in_dn+in_rg;
          pyym=pyy%scaling;
          subpix_ypos=(pyy%scaling)*scaleincr;
          subpix_xpos=(px%scaling)*scaleincr;  /* yes px not pxx */
          
          ptr1=ptr3=src+(px/scaling)*3;
          ptr2=ptr4=ptr1+in_rg;
          if(pyy<sheight-sci)
            {
            ptr3=ptr1+in_dn;
            ptr4=ptr1+in_dr;
            }
          
          symulsiz=sisize*subpix_ypos;
          sxmulsiz=sisize*subpix_xpos;
          
          for(x=0,pxx=px;x<swidth-px && x<scrnwide;x++,pxx++)
            {
            a3=symulsiz-(a4=subpix_xpos*subpix_ypos);
            a2=sxmulsiz-a4;
            a1=sis2-sxmulsiz-symulsiz+a4;

            for(i=0;i<3;i++)
              *dst++=(ptr1[i]*a1+ptr2[i]*a2+
                      ptr3[i]*a3+ptr4[i]*a4)/sis2;
              
            subpix_xpos+=scaleincr;
            sxmulsiz+=simulsiz;
            if(subpix_xpos>=sisize)
              {
              subpix_xpos=sxmulsiz=0;
              ptr1+=3; ptr2+=3; ptr3+=3; ptr4+=3;
              }

            if(pxx>=swidth-sci)
              {
              ptr2-=3;
              ptr4-=3;
              }
            }
          }
              
        cdown=(cdown==-1)?(scaling-(py%scaling)):scaling;
        }
      
      cdown--;
      
      if(scrnpixelsize==3 && !doing_hicol_bc)
        gl_putbox(x_add,y+y_add,scrnwide,1,realline);
      else
        eventuallydrawscansegment(realline,x_add,y+y_add,scrnwide*pixelsize);
      }
    free(realline);
    }
  else		/* if not scaled and not zoomed... */
    {
    switch(scrnpixelsize)
      {
      case 3:
        if(!doing_hicol_bc)
          {
          gl_putboxpart(x_add,y_add,(scrnwide<width)?scrnwide:width,
	  	(scrnhigh<height)?scrnhigh:height,width,height,theimage,px,py);
          break;
          }
        /* otherwise, FALLS THROUGH */
      
      default:
        /* for 320x200 (I think), 320x240, 320x400, 360x480 we need the
         * length to be a multiple of 8. :-( We allocate an image-width
         * line (blanked), and display via that.
         * (320x400/360x480 were dealt with earlier though; can only
         * be 320x200/320x240 if it gets here.)
         *
         * This isn't needed for svgalib >=1.3.1, but I'm leaving it in
         * for `if it ain't broke don't fix it' reasons. :-)
         */
        if(curvgamode==G320x200x256 || curvgamode==G320x240x256)
          {
          static unsigned char hack[320];
          
          memset(hack,0,320);
          for(y=0;(y<height-py)&&(y<scrnhigh);y++)
            {
            /* assume pixelsize==1, because it must be */
            memcpy(hack+x_add,theimage+((py+y)*width+px),xdim);
            vga_drawscansegment(hack,0,y+y_add,scrnwide);
            }
          }
        else
          for(y=0;(y<height-py)&&(y<scrnhigh);y++)
            eventuallydrawscansegment(theimage+pixelsize*((py+y)*width+px),
            	x_add,y+y_add,xdim*pixelsize);
        break;
      }
    }
  }
}



#define MODE_WIDTH_BUF_SIZE	16384

/* this variant should never be called with scrnpixelsize==3. */
void eventuallydrawscansegment_without_bc(byte *ptr,int x,int y,int len)
{
static unsigned short buf[MODE_WIDTH_BUF_SIZE];

switch(scrnpixelsize)
  {
  case 4:	/* 32-bit */
    {
    unsigned long *bufref=(unsigned long *)buf;
    unsigned char *ptrend=ptr+len;
    int i=0;
    
    for(;ptr<ptrend;ptr+=3,i++)
      *bufref++=GET32BITCOLOUR(ptr[2],ptr[1],*ptr);
    gl_putbox(x,y,i,1,buf);
    }
    break;
    
  case 2:	/* 15/16-bit */
    {
    unsigned short *bufref=buf;
    unsigned char *ptrend=ptr+len;
    
    if(scrncols==32768)
      for(;ptr<ptrend;ptr+=3)
        *bufref++=GET15BITCOLOUR(ptr[2],ptr[1],*ptr);
    else
      for(;ptr<ptrend;ptr+=3)
        *bufref++=GET16BITCOLOUR(ptr[2],ptr[1],*ptr);
    
    gl_putbox(x,y,bufref-buf,1,buf);
    }
    break;
    
  default:	/* 8-bit */
    if(curvgamode==G640x480x16)
      dither16scansegment(ptr,x,y,len);
    else
      vga_drawscansegment(ptr,x,y,len);
  }
}


void eventuallydrawscansegment_with_bc(byte *ptr,int x,int y,int len)
{
static unsigned short buf[MODE_WIDTH_BUF_SIZE];
unsigned char *ptrend=ptr+len;

/* if we get here, scrnpixelsize must be 2, 3, or 4. */

switch(scrnpixelsize)
  {
  case 4:	/* 32-bit */
    {
    unsigned long *bufref=(unsigned long *)buf;
    unsigned char *ptrend=ptr+len;
    int i=0;
    
    for(;ptr<ptrend;ptr+=3,i++)
      *bufref++=GET32BITCOLOUR(palt[ptr[2]],palt[ptr[1]],palt[*ptr]);
    gl_putbox(x,y,i,1,buf);
    }
    break;
  
  case 3:	/* 24-bit */
    {
    unsigned char *bufref=(unsigned char *)buf;	/* no kludges here, senor */
    
    while(ptr<ptrend)
      *bufref++=palt[*ptr++];
    
    gl_putbox(x,y,(bufref-(unsigned char *)buf)/3,1,buf);
    break;
    }
  
  case 2:	/* 15/16-bit */
    {
    unsigned short *bufref=buf;
    
    if(scrncols==32768)
      for(;ptr<ptrend;ptr+=3)
        *bufref++=GET15BITCOLOUR(palt[ptr[2]],palt[ptr[1]],palt[*ptr]);
    else
      for(;ptr<ptrend;ptr+=3)
        *bufref++=GET16BITCOLOUR(palt[ptr[2]],palt[ptr[1]],palt[*ptr]);
    
    gl_putbox(x,y,bufref-buf,1,buf);
    break;
    }
  
  default:
    fprintf(stderr,"not high-colour in ..._with_bc, can't happen!\n");
    exit(1);
  }
}


/* only called for 640x480x4-bit */
void dither16scansegment(unsigned char *ptr,int x,int y,int len)
{
static unsigned char scanline[640];
unsigned char *lptr=scanline;
int f;

if(!cfg.viewer16col)
  {
  /* greyscale, but dithered between the scales;
   * dithering slows it down more than a direct mapping would,
   * but it just looks too nasty without it.
   */
  static unsigned char ditherdata[4]=
    {
    1,4,
    3,2
    };
  unsigned char c,*dithptr=ditherdata+(y&1)*2+(x&1);
  int xm=2-(x&1);
  
  for(f=0;f<len;f++)
    {
    if((c=dither16_greylookup[ptr[f]])>=60)
      c=60;	/* saves needing a range check after :-) */
    
    *lptr++=((c>>2)+((c&3)>=*dithptr++));
    if(!(--xm))
      dithptr-=(xm=2);
    }
  }
else
  {
  /* colour dither */
  static unsigned char ditherdata[16]=
    {
    1*4,13*4, 4*4,15*4,
    9*4, 5*4,12*4, 8*4,
    3*4,15*4, 2*4,14*4,
    7*4,10*4, 6*4,11*4
    };
  unsigned char *dithptr=ditherdata+(y&3)*4+(x&3);
  unsigned char *palptr,d;
  int xm=4-(x&3);
  
  for(f=0;f<len;f++)
    {
    d=*dithptr++;
    palptr=dither16_rgb+ptr[f]*3;
    *lptr++=(((*palptr>=d)*4)|((palptr[1]>=d)*2)|(palptr[2]>=d));
    if(!(--xm))
      dithptr-=(xm=4);
    }
  }

vga_drawscansegment(scanline,x,y,len);
}


int getvpix(int px,int py,int x,int y)
{
int p1,p2;

if((vkludge)&&(pixelsize==1))
  {
  p1=*(theimage+(py+y)*width+px+x);
  if(px+x+1>=width) return(p1);
  p1*=3;
  p2=*(theimage+(py+y)*width+px+x+1)*3;
  return(closest((pal32_no_bc[p1  ]+pal32_no_bc[p2  ])>>1,
                 (pal32_no_bc[p1+1]+pal32_no_bc[p2+1])>>1,
                 (pal32_no_bc[p1+2]+pal32_no_bc[p2+2])>>1));
  }
else
  return(*(theimage+pixelsize*((py+y)*width+px+x)));
}


/* this routine is nasty writ big, but about as quick as I can manage */
void drawzoomedgif()
{
int a,b,x,yp,yw;
long y,sw,sh,lastyp;
int xoff,yoff;
int x_add,y_add;	/* for centering */
int pixwide,pixhigh;
int bigimage;
byte *rline,*sptr,*dptr;
int tmp1,tmp2,tr,tg,tb,tn;

if((rline=calloc(scrnwide*pixelsize,1))==NULL) return;

/* try landscapey */
sw=scrnwide; sh=(scrnwide*height)/width;
if(sh>scrnhigh)
  /* no, oh well portraity then */
  sh=scrnhigh,sw=(scrnhigh*width)/height;

/* so now our zoomed image will be sw x sh */

/* fix centering if needed */
x_add=y_add=0;
if(cfg.centreflag)
  {
  if(sw<scrnwide)
    x_add=((scrnwide-sw)>>1);
  
  if(sh<scrnhigh)
    y_add=((scrnhigh-sh)>>1);
  
  if(virtual) x_add>>=1;
  }

bigimage=(width>sw)?1:0;   /* 1 if image has been reduced, 0 if made bigger */
if(bigimage)
  /* it's been reduced - easy, just make 'em fit in less space */
  {
  if(virtual) sw>>=1;
  lastyp=-1;
  vga_clear();
  pixhigh=(int)(((float)height)/((float)sh)+0.5);
  pixwide=(int)(((float)width)/((float)sw)+0.5);
  pixhigh++;
  pixwide++;
  for(y=0;y<height;y++)
    {
    yp=(y*sh)/height;
    if(yp!=lastyp)
      {
      yw=y*width;
      if(vkludge)
        {
        /* we try to resample it a bit */
        for(x=0;x<width;x++,yw++)
          {
          tr=tg=tb=tn=0;
          for(b=0;(b<pixhigh)&&(y+b<height);b++)
            for(a=0;(a<pixwide)&&(x+a<width);a++)
              {
              if(scrncols<=256)
                {
                tmp2=*(theimage+yw+a+b*width)*3;
                tr+=pal32_no_bc[tmp2];
                tg+=pal32_no_bc[tmp2+1];
                tb+=pal32_no_bc[tmp2+2];
                }
              else
                {
                tb+=*(theimage+3*(yw+a+b*width));
                tg+=*(theimage+3*(yw+a+b*width)+1);
                tr+=*(theimage+3*(yw+a+b*width)+2);
                }  
              tn++;
              }
          tr/=tn; tg/=tn; tb/=tn;
          if(scrncols<=256)
            rline[(x*sw)/width]=closest(tr,tg,tb);
          else
            {
            tmp1=(x*sw)/width*pixelsize;
            rline[tmp1]=tb;
            rline[tmp1+1]=tg;
            rline[tmp1+2]=tr;
            }
          }
        }
      else	/* not vkludge... */
        for(x=0;x<width;x++,yw++)
          {
          if(scrncols<=256)
            rline[(x*sw)/width]=*(theimage+yw);
          else
            memcpy(rline+(x*sw)/width*3,theimage+yw*3,3);
          }
      eventuallydrawscansegment(rline,x_add,yp+y_add,sw*pixelsize);
      lastyp=yp;
      }
    }
  free(rline);
  }
else
  {
  /* well, we need to fill in the gaps because it's been made bigger.
   * However, it makes more sense to look at it the other way around.
   * So we don't just make small pixels bigger; instead, we go through
   * each of the pixels in the scaled-up image and copy an appropriate
   * pixel from the original (smaller) image. This is a lot simpler
   * and cleaner than the more obvious approach - and I speak from
   * experience. :-)
   */
  if(virtual)
    sw>>=1;
  for(y=0;y<sh;y++)
    {
    yoff=(y*height)/sh; 
    if(yoff>=height) yoff=height-1;
    dptr=rline;
   
    for(x=0;x<sw;x++)
      {
      xoff=(x*width)/sw;
      if(xoff>=width) xoff=width-1;
      
      if(scrncols<=256)
        /* image must be 8-bit */
        *dptr++=*(theimage+yoff*width+xoff);
      else
        {
        sptr=theimage+3*(yoff*width+xoff);
        *dptr++=*sptr; *dptr++=sptr[1]; *dptr++=sptr[2];
        }
      }
    
    /* draw the line */
    eventuallydrawscansegment(rline,x_add,y+y_add,sw*pixelsize);
    }
  free(rline);
  }
}


void fx_mirror()
{
unsigned char *src,*dst;
byte *tmp;
int x,y;

tmp=malloc(width*pixelsize);
if(tmp==NULL) return;

if(pixelsize==1)
  {
  for(y=0;y<height;y++)
    {
    dst=tmp;
    src=theimage+y*width+width-1;
    for(x=0;x<width;x++)
      *dst++=*src--;
    memcpy(src+1,tmp,width);
    }
  }
else
  {
  /* we know pixelsize must be 3 then, so take advantage of that */
  
  for(y=0;y<height;y++)
    {
    dst=tmp;
    src=theimage+3*(y*width+width-1);
    for(x=0;x<width;x++)
      {
      *dst++=*src++;
      *dst++=*src++;
      *dst++=*src;
      src-=5;
      }
    memcpy(src+3,tmp,width*3);
    }
  }

free(tmp);
}


void fx_flip()
{
unsigned char *tmp,*ptr1,*ptr2;
int y,y2,wp=width*pixelsize;

tmp=malloc(wp);
if(tmp==NULL) return;

ptr1=theimage;
ptr2=theimage+(height-1)*wp;

for(y=0,y2=height-1;y<height/2;y++,y2--,ptr1+=wp,ptr2-=wp)
  {
  memcpy(tmp,ptr1,wp);
  memcpy(ptr1,ptr2,wp);
  memcpy(ptr2,tmp,wp);
  }

free(tmp);
}


/* NB: you must do a vga_clear() after this and before redisplaying
 * the picture, otherwise the old one is left onscreen!
 */
void fx_rot()
{
unsigned char *tmp,*tmp2=NULL,*ptr1,*ptr2=NULL;
int x,y,y2,wp=width*pixelsize,hp=height*pixelsize;
int blockup=4;

tmp=malloc(wp*height);
if(pixelsize==3) tmp2=malloc(wp*blockup);
if(tmp==NULL || (pixelsize==3 && tmp2==NULL)) return;

if(pixelsize==1)
  for(y=0,y2=hp-1;y<height;y++,y2--)
    {
    ptr1=tmp+y2; ptr2=theimage+wp*y;
    for(x=0;x<width;x++,ptr1+=hp)
      *ptr1=*ptr2++;
    }
else
  {
  int hpm3=hp-3;
  
  for(y=0,y2=height-1;y<height;y++,y2--)
    {
    ptr1=tmp+y2*3;
    if(y%blockup==0)
      {
      memcpy(tmp2,theimage+wp*y,wp*((y2<blockup)?(y2+1):blockup));
      ptr2=tmp2;
      }
    for(x=0;x<width;x++,ptr1+=hpm3)
      {
      *ptr1++=*ptr2++;
      *ptr1++=*ptr2++;
      *ptr1++=*ptr2++;
      }
    }
  }
  
x=height;y=width;
width=x;height=y;

if(pixelsize==3) free(tmp2);
free(theimage);
theimage=tmp;
}


/* rgb values must be 0..31; 6 bits of the rgb count in VGA 8-bit modes,
 * so this does effectively lose a bit of detail, but since we're
 * picking from a restricted set anyway it's probably lost in the noise.
 */
int closest(int r,int g,int b)
{
int idx;
unsigned char *ptr,distnum;
int xr,xg,xb,dist,distquan,f,checknumcols;

idx=((b<<10)|(g<<5)|r);

if(vkcache_valid[idx])
  return(vkcache[idx]);

distnum=0;
distquan=(1<<30);

/* if numcols=256 we do 0-255, otherwise 0-numcols+63 */
checknumcols=((numcols==256)?256:numcols+64);

for(ptr=pal32_no_bc,f=0;f<checknumcols;f++)
  {
  xr=(r-*ptr++);
  xg=(g-*ptr++);
  xb=(b-*ptr++);
  if((dist=xr*xr+xg*xg+xb*xb)<distquan)
    {
    distnum=f;
    distquan=dist;
    if(dist==0) break;  /* premature exit if it can't get any better */
    }
  }

vkcache_valid[idx]=1;
return((int)(vkcache[idx]=distnum));
}


void samecentre(int *ppx,int *ppy,int newscale,
                int oldpx,int oldpy,int oldscale)
{
int xa,ya,sw,sh;

/* even if the centre flag is off, we still need to do this */
xa=ya=0;
sw=oldscale*width;
sh=oldscale*height;
if(sw<scrnwide) xa=(scrnwide-sw)>>1;
if(sh<scrnhigh) ya=(scrnhigh-sh)>>1;  
if(virtual) xa>>=1;

/* finds centre of old screen, and makes it centre of new one */
*ppx=(oldpx-xa+(scrnwide>>1))*newscale/oldscale;
*ppy=(oldpy-ya+(scrnhigh>>1))*newscale/oldscale;

xa=ya=0;
sw=newscale*width;
sh=newscale*height;
if(sw<scrnwide) xa=(scrnwide-sw)>>1;
if(sh<scrnhigh) ya=(scrnhigh-sh)>>1;  
if(virtual) xa>>=1;

*ppx-=(scrnwide>>1)+xa;
*ppy-=(scrnhigh>>1)+ya;
}


/* this gets run on picture exit when cfg.selecting is true.
 * the output is designed to be put straight into a pnmcut command-line.
 */
void show_dimensions(int px,int py,int scaling)
{
int x,y,w,h;

x=px/scaling; y=py/scaling;
if(width*scaling<scrnwide)  w=width;  else w=scrnwide/scaling;
if(height*scaling<scrnhigh) h=height; else h=scrnhigh/scaling;
printf("%4d %4d %4d %4d\n",x,y,w,h);
}


/* change from one orientation state to another.
 * See the big comment about this earlier on (near the start of the file)
 * if you feel a `!?' coming on.
 *
 * clear_if_rot is non-zero if it should do a vga_clear() if it needs
 * to do a rotate. See the comment above fx_rot()'s def for why.
 */
void orient_change_state(int from,int to,int clear_if_rot)
{
/* the basic idea is this:
 *
 * - if from and to are equal, return.
 * - if a single flip/mirror/rot will do it, use that.
 * - otherwise, try a rotate if we know it's needed (see below).
 * - then see if a flip/mirror does the trick.
 * - if not, it must need flip *and* mirror.
 */
int state=from;

if(from==to) return;

/* try a one-step route. */
if(orient_state_flip[state]==to)	{ fx_flip(); return; }
if(orient_state_mirror[state]==to)	{ fx_mirror(); return; }
if(orient_state_rot_cw[state]==to)
  {
  fx_rot();
  if(clear_if_rot) vga_clear();
  return;
  }

/* nope, ok then, things get complicated.
 * we can get any required rotate out of the way -
 * if it's switched from portrait to landscape or vice versa, we must
 * need one. That's if it's gone from 0..3 to 4..7 or 4..7 to 3..0.
 */
if((from<4 && to>=4) || (from>=4 && to<4))
  {
  fx_rot();
  if(clear_if_rot) vga_clear();
  state=orient_state_rot_cw[state];
  }

/* now try a flip/mirror. */
if(orient_state_flip[state]==to)	{ fx_flip(); return; }
if(orient_state_mirror[state]==to)	{ fx_mirror(); return; }

/* no? Well it must need both then. */
fx_flip();
fx_mirror();

/* sanity check */
if(orient_state_mirror[orient_state_flip[state]]!=to)
  fprintf(stderr,"can't happen - orient_change_state(%d,%d) failed!\n",
  	from,to);
}
