/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <math.h>
#include <iostream.h>
#include <stdlib.h>
#include "MathGauss.h"
#include "GlobalTrace.h"


/* 
 *  MathGauss: create 
 */
MathGauss::MathGauss (int seed) : MathRandom (seed)
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ MathGauss\n");

	init ();
}


/* 
 *  MathGauss: destructor
 */
/*
MathGauss::~MathGauss ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- MathGauss\n");
}
*/


/* 
 *  gauss: return a gaussian random between floor and ceil (default 0 <= n <= 1)
 */
double MathGauss::gauss(float floor, float ceil)
{
	// the most extreme rndGauss() values I have ever observed (even
	// when running through a few billion tries) was -3.42 and 3.42. 
	// 3.45 should savely scale the number to the -1 ... 1 range
	return (floor+ceil*(rndGauss()/3.45));
}


/* 
 *  initGauss: initialize the gaussian random number generator
 *  taken from 'The Science of Fractal Images' by Peitgen & Saupe, 
 *  1988 (page 77)
 */ 
void MathGauss::init ()
{
	s_gaussAdd = sqrt(3.0 * s_gaussNRand);
	s_gaussFac = 2 * s_gaussAdd / (s_gaussNRand * s_maxRand);

	// done via parent constructor
	//this->MathRandom::init ((unsigned int)seed); 

	if (GlobalTrace::isSet (GlobalTrace::TRACE_VERBOSE))
		cout << "InitGauss: samples=" << MathGauss::s_gaussNRand 
		     << ", add=" << s_gaussAdd << "\n";
}


/* 
 *  randGauss: return a gaussian random number with s_gaussNRand samples
 *  taken from 'The Science of Fractal Images' by Peitgen & Saupe, 
 *  1988 (page 77)
 */ 
double MathGauss::rndGauss ()
{
	float		sum=0.0;
	double		r;

	for (int i=0; i<s_gaussNRand; i++)
		sum += (this->MathRandom::rnd()*0x7FFF);
	r = (s_gaussFac * sum - s_gaussAdd);

	return (r);
}



/*
// instantiate static data members
const int MathGauss::s_gaussNRand=4;
const double MathGauss::s_maxRand=pow(2.0, 15.0)-1.0;
double MathGauss::s_gaussAdd=0.0;
double MathGauss::s_gaussFac=0.0;

int main (int argc, char *argv[])
{
	MathGauss::initGauss (10);
	for (int i=0; i<1000; i++)
		cout << MathGauss::rndGauss () << "\n";
}

*/
