/*
   drvwmf.cpp : This file is part of pstoedit
	 Backend for Windows Meta File (WMF)

   Copyright (C) 1996,1997 Jens Weber, rz47b7@PostAG.DE
   Copyright (C) 1998 Thorsten Behrens and Bjoern Petersen
   Copyright (C) 1998 Wolfgang Glunz

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/
#include "drvwmf.h"

#include I_string_h
#include I_fstream
#include I_stdio
#include I_stdlib

#include <math.h>

#include "pstoedit.h"
 
#if 0
// define this if you want to have all fonts mapped to Arial
#define MAPTOARIAL
#endif


#define USEWATCOM 1
// windows structure for standard metafile
// placeable header
const long	PLACEABLEKEY =	0x9AC6CDD7L;
const int	PLACEABLESIZE= 	22;
// see also http://premium.microsoft.com/msdn/library/techart/html/emfdcode.htm
// regarding alignment (Wo Gl)
#pragma pack(2)
struct PLACEABLEHEADER
{
    DWORD   key; 
	WORD 	hmf;
    WORD	left;
    WORD	top;
    WORD	right;
    WORD	bottom;
    WORD    inch;
    DWORD   reserved;
    WORD    checksum;
};


const int MAXPALENTRIES = 256;
const int NumColors		= 48;

const COLORREF Colors[NumColors] =
{
	0x00FFFFFFl,0x00E0E0E0l,0x00C0C0FFl,0x00C0E0FFl,0x00E0FFFFl,0x00C0FFC0l,
	0x00FFFFC0l,0x00FFC0C0l,0x00FFC0FFl,0x000000C0l,0x000040C0l,0x0000C0C0l,
	0x0000C000l,0x00C0C000l,0x00C00000l,0x00C000C0l,
	0x00C0C0C0l,0x00404040l,0x008080FFl,0x0080C0FFl,0x0080FFFFl,0x0080FF80l,
	0x00FFFF80l,0x00FF8080l,0x00FF80FFl,0x00000080l,0x00004080l,0x00008080l,
	0x00008000l,0x00808000l,0x00800000l,0x00800080l,
	0x00808080l,0x00000000l,0x000000FFl,0x000080FFl,0x0000FFFFl,0x0000FF00l,
	0x00FFFF00l,0x00FF0000l,0x00FF00FFl,0x00000040l,0x00404080l,0x00004040l,
	0x00004000l,0x00404000l,0x00400000l,0x00400040l
};


drvWMF::derivedConstructor(drvWMF):
	constructBase
{
	const char	description[]="generated by WMF/EMF backend of pstoedit";

	// do some consistency checking
	if( sizeof(PLACEABLEHEADER) != PLACEABLESIZE ) {
		errf << "WARNING: structure size mismatch. Please contact author. Expecting :" <<
				PLACEABLESIZE << " found " << sizeof(PLACEABLEHEADER) << endl;		
		errf << "sizeof(WORD)    = " << sizeof(WORD) << endl;   // MSVC:  2
		errf << "sizeof(DWORD)   = " << sizeof(DWORD) << endl;  // MSVC:  4
		errf << "sizeof(RECT)    = " << sizeof(RECT) << endl;   // MSVC: 16
		errf << "sizeof(HANDLE)  = " << sizeof(HANDLE) << endl; // MSVC:  4
		ctorOK = false;
		return;
	}
	DesktopDC	= GetDC( GetDesktopWindow() );

	// query some device info 
	const long xres		= GetDeviceCaps(DesktopDC, HORZRES);// Device-Info's
	const long yres		= GetDeviceCaps(DesktopDC, VERTRES);

	// screen's size
	rect.left	= 0;
	rect.top	= 0;
	rect.right	= xres;
	rect.bottom	= yres;

	MyLogPalette = NULL; // alloc comes later
	OldPalette   = NULL;
	tempName	 = NULL;
	enhanced = (strcmp(Pdriverdesc->symbolicname,"emf") == 0)
				||
				( ( driveroptions_p != NULL) &&
				  (	strchr(driveroptions_p,'e') != NULL ) )
				;
	if (enhanced) 
	{
		// enhanced-Metafile (.emf) for W95 or NT
		MetaDC = CreateEnhMetaFile(DesktopDC, nameOfOutputFile_p, &rect, description);
		if (!MetaDC)
		{
   			errf << "ERROR: could not open metafile: " <<
			nameOfOutputFile_p << endl;
			ctorOK = false;
			return;
		}
	} else {
		// take temp file for standard WMF, to prepend placeable header
		tempName = full_qualified_tempnam("wmftemp");

		// standard metafile for Win16
		MetaDC = CreateMetaFile(tempName);
		if(!MetaDC)
		{
   			errf << "ERROR: could not open temporary metafile: " <<	tempName << endl;
			ctorOK = false;
			return;
		}

		if( (outFile=fopen(nameOfOutputFile_p,"wb")) == 0L )
		{
			errf << "ERROR: cannot open final metafile " << nameOfOutputFile_p << endl;
			ctorOK = false;
			return;
		}
	}

	if( enhanced )
	{
		errf << "enhanced metafile created: " <<
		nameOfOutputFile_p << endl;
	}
	else
	{
		errf << "standard metafile created: " <<
		nameOfOutputFile_p << endl;
	}


	y_offset= 0;
	x_offset= 0;

	// in height and width are the highest values of the drawing
	height=0;
	width=0;
	maxstatus=0; // 0, until first value set in height and width

	// in origin_x und origin_y are the lowest values of the drawing
	origin_x=0;
	origin_y=0;
	minstatus=0; // 0, until first value set in origin_x and origin_y

	if( driveroptions_p != NULL &&
		strchr(driveroptions_p,'p') != NULL )
	{
		errf << "allocating " << (sizeof(LOGPALETTE) + sizeof(PALETTEENTRY) * MAXPALENTRIES) << endl;

		// get us a palette
		char* memspace				= new char[sizeof(LOGPALETTE) + sizeof(PALETTEENTRY) * MAXPALENTRIES];
		MyLogPalette			= (LOGPALETTE*) memspace;
		MyLogPalette->palVersion= 0x300;		// set version

		MyLogPalette->palNumEntries = NumColors;	// initial color table

		// init color values
		for( unsigned long i=2; i<NumColors; i++ )
		{
			MyLogPalette->palPalEntry[i].peRed	= GetRValue( Colors[i] );
			MyLogPalette->palPalEntry[i].peGreen= GetGValue( Colors[i] );
			MyLogPalette->palPalEntry[i].peBlue = GetBValue( Colors[i] );
			MyLogPalette->palPalEntry[i].peFlags= PC_RESERVED;
		}

		// white and black for first two colors
		MyLogPalette->palPalEntry[0].peRed	= 0xff;
		MyLogPalette->palPalEntry[0].peGreen= 0xff;
		MyLogPalette->palPalEntry[0].peBlue = 0xff;
		MyLogPalette->palPalEntry[1].peRed	= 0x0;
		MyLogPalette->palPalEntry[1].peGreen= 0x0;
		MyLogPalette->palPalEntry[1].peBlue = 0x0;

		cntPalEntries = NumColors;

		maxPalEntries=MAXPALENTRIES;

		// create and set windows palette
		ThePalette = CreatePalette(MyLogPalette);
		if(ThePalette)
		{
			OldPalette = (HPALETTE) SelectObject(MetaDC, ThePalette);
			RealizePalette(MetaDC);
		}
		else
		{
			errf << "ERROR: Can't create palette!" << endl;
		}

		palette = true;
	}
	else
	{
		palette = false;
	}

	// setup pen for drawing functions
	PenData.lopnStyle=PS_SOLID;				// solid pen
	const POINT		PenWidth = {0,0};
	PenData.lopnWidth=PenWidth;				// width of pen (white)
	PenData.lopnColor= enhanced ?
		PALETTEINDEX(1) : RGB(255,255,255);	// color of pen: black
	ColoredPen	 =0L;

	// setup brush for drawing functions
	BrushData.lbStyle=BS_SOLID;				// solid brush
	BrushData.lbColor= enhanced ?
		PALETTEINDEX(1) : RGB(255,255,255);	// color of brush (black)
	BrushData.lbHatch=NULL;					// no pattern
	ColoredBrush = 0L;

	// set default font
 	const char* const defaultfontname = "System";
  	setCurrentFontName(defaultfontname,false /* is standard font */ );
	strcpy(lastSelectedFontName,defaultfontname);
	lastSelectedFontHeight= 0;
//	lastSelectedFontAngle = 0;
	MyFont=0L;
	OldFont=0L;

	if( enhanced )
	{
		// set reasonable mapping mode (don't want distortion)
		SetMapMode(MetaDC,MM_ISOTROPIC);
	}

	// set reasonable text drawing mode
	SetBkMode(MetaDC,TRANSPARENT);

	// set Postscript-like text alignment
	SetTextAlign(MetaDC,TA_BASELINE);
}

// bool drvWMF::driverOK() const { return (MetaDC != 0); } 

drvWMF::~drvWMF() 
{
	// change origin and size of drawing:
	if( enhanced )
	{
		// change origin
		SetWindowOrgEx(MetaDC, origin_x, origin_y, NULL);

		// change size
		SetWindowExtEx(MetaDC, width - origin_x,
					   height - origin_y, NULL);
	}
	else
	{
		if( origin_x < -20000 ||
			origin_x > 20000 ||
			origin_y < -20000 ||
			origin_y > 20000 ||
			width < -20000 ||
			width > 20000 ||
			height < -20000 ||
			height > 20000		)
		{
			// issue warning, coordinate overflow possible !
			errf << "Possible coordinate overflow! " << endl;
			errf << "  Origin: " << origin_x << " , " << origin_y << endl;
			errf << "  Size: " << width-origin_x << " , "
				 << height-origin_y << endl;
		}

		// standard metafile does not care much about user space.
		// so we have to draw some dummy points at the pic's boundaries
		SetPixel(MetaDC, origin_x, origin_y, RGB(255,255,255));
		SetPixel(MetaDC, width, height, RGB(255,255,255));
	}

	// free some mem
	if( MyLogPalette )
		delete[] (char *) MyLogPalette;

	// free any allocated objects
	if(ColoredPen)
	{
		SelectObject(MetaDC,OldColoredPen);
		DeleteObject(ColoredPen);
	}

	if(ColoredBrush)
	{
		SelectObject(MetaDC,OldColoredBrush);
		DeleteObject(ColoredBrush);
	}

	if(MyFont)
	{
		SelectObject(MetaDC,OldFont);
 		DeleteObject(MyFont);
		MyFont=0L;
	}

	if (OldPalette != NULL) {
		SelectObject(MetaDC,OldPalette);
		DeleteObject(ThePalette);
	}
	ReleaseDC( GetDesktopWindow(), DesktopDC );

	// close and destroy metafile
	if( enhanced)
	{
		DeleteEnhMetaFile(CloseEnhMetaFile(MetaDC));
	}
	else
	{
		DeleteMetaFile(CloseMetaFile(MetaDC));

		// add placeable header to standard metafile
		const int BUFSIZE	=	1024;
		char				buf[BUFSIZE];
		PLACEABLEHEADER		pHd;
		FILE*				inFile;
    	WORD				checksum;
        WORD*				checksumField;

		if( (inFile=fopen(tempName,"rb")) != 0L )
		{
			if( outFile != 0L )
			{
				// setup header
				pHd.key=PLACEABLEKEY;
				pHd.hmf=0;
				pHd.left=(WORD) origin_x;
				pHd.top=(WORD) origin_y;
				pHd.right=(WORD) width;
				pHd.bottom=(WORD) height;
//				const RECT R = {origin_x,origin_y,width,height};
//				pHd.bbox = R;
				pHd.inch=12;
				pHd.reserved=0;

				// calculate checksum
				checksum = 0;
				checksumField = (WORD*)&pHd;
				for(unsigned int i = 0; i < 10; i++)
					checksum ^= checksumField[i];

				pHd.checksum = checksum;

				// write header
				if( fwrite(&pHd, sizeof(PLACEABLEHEADER) ,1,outFile) != 1 )
				{
					errf << "Can't write final metafile" << endl;
				}

				// append metafile data
				do
				{
					size_t read=fread(buf,1,BUFSIZE,inFile);

					if( read > 0 )
					{
						if( fwrite(buf,1,read,outFile) != read )
						{
							errf << "Can't write final metafile" << endl;
						}
					}
				}
				while( !feof(inFile) );

				fclose(outFile);
			}

			fclose(inFile);
		} 
		remove(tempName); 
		free(tempName); 
		tempName = NULL;
	}	
}


long drvWMF::searchPalEntry(float r, float g, float b)
{
	const BYTE red		=(BYTE)(r*255);
	const BYTE green	=(BYTE)(g*255);
	const BYTE blue		=(BYTE)(b*255);

	for(unsigned int i=0; i<cntPalEntries; i++)
	{
		if( (MyLogPalette->palPalEntry[i].peRed		== red) &&
			(MyLogPalette->palPalEntry[i].peGreen	== green) &&
			(MyLogPalette->palPalEntry[i].peBlue	== blue) )
		{
			return i;
		}
	}

	return -1;                              // Not found!
}



// the colors for pens and brushes are set here.
// they are used if new pens or brushes are created and
// selected into the DC
void drvWMF::setColor(float r, float g, float b)
{
	if( palette )
	{
		const long index = searchPalEntry(r,g,b);
		if(index < 0)
		{
			// Not found!

			if( cntPalEntries < maxPalEntries )
			{
				// we've some colors left, so set new

				MyLogPalette->palPalEntry[cntPalEntries].peRed	= (BYTE)(r*255 + .5);	// rounded byte
				MyLogPalette->palPalEntry[cntPalEntries].peGreen= (BYTE)(g*255 + .5);
				MyLogPalette->palPalEntry[cntPalEntries].peBlue = (BYTE)(b*255 + .5);

				if( SetPaletteEntries(ThePalette, 0, cntPalEntries+1, &(MyLogPalette->palPalEntry[0])) )
				{
					MyLogPalette->palNumEntries = (WORD) (cntPalEntries+1);

					// restore old palette
					SelectObject(MetaDC,OldPalette);

					if(ThePalette)
					{
						DeleteObject( ThePalette );
					}

					ThePalette = CreatePalette(MyLogPalette);
					if(!ThePalette)
					{
						errf << "ERROR: Can't create new palette!" << endl;
						return;
					}

					SelectObject(MetaDC,ThePalette);
					RealizePalette(MetaDC);

					// new drawing color is the one set (the last)
					PenData.lopnColor=PALETTEINDEX(cntPalEntries);
					BrushData.lbColor=PALETTEINDEX(cntPalEntries);

					cntPalEntries++;
				}
				else
				{
					errf << "ERROR: Can't set palette entries!" << endl;
				}
			} // end of maxPalEntries-condition
			else
			{
				errf << "WARNING: Max. number of colorentries ("<< maxPalEntries <<") exceeded!" << endl;
			}

			// get closest color
			PenData.lopnColor=PALETTERGB((BYTE)(r*255 + .5),(BYTE)(g*255 + .5),(BYTE)(b*255 + .5));
			BrushData.lbColor=PALETTERGB((BYTE)(r*255 + .5),(BYTE)(g*255 + .5),(BYTE)(b*255 + .5));
		} // end of not-found-cond.
		else
		{
			PenData.lopnColor=PALETTEINDEX(index);
			BrushData.lbColor=PALETTEINDEX(index);
		}
	}
	else
	{
		// we're not using a palette, specify in RGB values
		PenData.lopnColor=PALETTERGB((BYTE)(r*255 + .5),(BYTE)(g*255 + .5),(BYTE)(b*255 + .5));
		BrushData.lbColor=PALETTERGB((BYTE)(r*255 + .5),(BYTE)(g*255 + .5),(BYTE)(b*255 + .5));
	}


	// default is no dashing
	PenData.lopnStyle = PS_SOLID;

	switch( currentLineType() )
	{
		case dashed:
			PenData.lopnStyle = PS_DASH;
			break;

		case dotted:
			PenData.lopnStyle = PS_DOT;
			break;

		case dashdot:
			PenData.lopnStyle = PS_DASHDOT;
			break;

		case dashdotdot:
			PenData.lopnStyle = PS_DASHDOTDOT;
			break;
	}

	const POINT PenWidth ={(int)(currentLineWidth() + .5),0};	// rounded int
	PenData.lopnWidth=PenWidth;

	if(ColoredPen)
	{
		SelectObject(MetaDC,OldColoredPen);
		DeleteObject(ColoredPen);
		ColoredPen = 0L;
	}

	ColoredPen=CreatePenIndirect(&PenData);
	if(!ColoredPen)
	{
		errf << "ERROR: setColor: could not create pen !" << endl;
	}
	else
	{
		OldColoredPen = (HPEN) SelectObject(MetaDC,ColoredPen);
	}

	if(ColoredBrush)
	{
		SelectObject(MetaDC,OldColoredBrush);
		DeleteObject(ColoredBrush);
		ColoredBrush = 0L;
	}

	ColoredBrush=CreateBrushIndirect(&BrushData);
	if(!ColoredBrush)
	{
		errf << "ERROR: setColor: could not create brush !" << endl;
	}
	else
	{
		OldColoredBrush = (HBRUSH) SelectObject(MetaDC, ColoredBrush);
	}
}


int drvWMF::FetchFont(const TextInfo & textinfo, short int textHeight, short int textAngle)
{
	TheFontRec.lfHeight		= -textHeight;
	TheFontRec.lfWidth		= 0;		// optimal fit
	TheFontRec.lfEscapement = textAngle;
	TheFontRec.lfOrientation= textAngle;

	TheFontRec.lfWeight		= FW_DONTCARE;	// default: don't care

	if( strstr(textinfo.currentFontWeight.value(),"Regular") )
		TheFontRec.lfWeight = FW_NORMAL;	// other values don't work	

	if( strstr(textinfo.currentFontWeight.value(),"Medium") )
		TheFontRec.lfWeight = FW_NORMAL;	// other values don't work	

	if( strstr(textinfo.currentFontWeight.value(),"Normal") )
		TheFontRec.lfWeight = FW_NORMAL;

	if( strstr(textinfo.currentFontWeight.value(),"Thin") ||
		strstr(textinfo.currentFontName.value(),"Thin") ||
		strstr(textinfo.currentFontFullName.value(),"Thin") )
	{
		TheFontRec.lfWidth	= textHeight / 3;	// narrow font emulation (trial and error value for Arial font)
	}

	if( strstr(textinfo.currentFontWeight.value(),"Extralight") ||
		strstr(textinfo.currentFontName.value(),"Extralight") ||
		strstr(textinfo.currentFontFullName.value(),"Extralight") )
	{
		TheFontRec.lfWidth	= textHeight / 4;	// narrow font emulation (trial and error value for Arial font)
	}

	if( strstr(textinfo.currentFontWeight.value(),"Ultralight") || 
		strstr(textinfo.currentFontName.value(),"Ultralight") || 
		strstr(textinfo.currentFontFullName.value(),"Ultralight") )
	{
		TheFontRec.lfWidth	= textHeight / 4;	// narrow font emulation (trial and error value for Arial font)
	}

	if( strstr(textinfo.currentFontWeight.value(),"Light") ||
		strstr(textinfo.currentFontName.value(),"Light") ||
		strstr(textinfo.currentFontFullName.value(),"Light") ||
		strstr(textinfo.currentFontWeight.value(),"Condensed") ||
		strstr(textinfo.currentFontName.value(),"Condensed") ||
		strstr(textinfo.currentFontFullName.value(),"Condensed") )
	{
		TheFontRec.lfWidth	= textHeight / 3;	// narrow font emulation (trial and error value for Arial font)
	}

	if( strstr(textinfo.currentFontWeight.value(),"Semibold") ||
		strstr(textinfo.currentFontName.value(),"Semibold") ||
		strstr(textinfo.currentFontFullName.value(),"Semibold") )
		TheFontRec.lfWeight = FW_BOLD;	// other values don't work

	if( strstr(textinfo.currentFontWeight.value(),"Demibold") ||
		strstr(textinfo.currentFontName.value(),"Demibold") ||
		strstr(textinfo.currentFontFullName.value(),"Demibold") )
		TheFontRec.lfWeight = FW_BOLD;	// other values don't work

	if( strstr(textinfo.currentFontWeight.value(),"Bold") ||
		strstr(textinfo.currentFontName.value(),"Bold") ||
		strstr(textinfo.currentFontFullName.value(),"Bold") )
		TheFontRec.lfWeight = FW_BOLD;

	if( strstr(textinfo.currentFontWeight.value(),"Extrabold") ||
		strstr(textinfo.currentFontName.value(),"Extrabold") ||
		strstr(textinfo.currentFontFullName.value(),"Extrabold") )
		TheFontRec.lfWeight = FW_BOLD;	// other values don't work

	if( strstr(textinfo.currentFontWeight.value(),"Ultrabold") ||
		strstr(textinfo.currentFontName.value(),"Ultrabold") ||
		strstr(textinfo.currentFontFullName.value(),"Ultrabold") )
		TheFontRec.lfWeight = FW_BOLD;	// other values don't work

	if( strstr(textinfo.currentFontWeight.value(),"Heavy") ||
		strstr(textinfo.currentFontName.value(),"Heavy") ||
		strstr(textinfo.currentFontFullName.value(),"Heavy") )
		TheFontRec.lfWeight = FW_BOLD;	// other values don't work

	if( strstr(textinfo.currentFontWeight.value(),"Black") ||
		strstr(textinfo.currentFontName.value(),"Black") ||
		strstr(textinfo.currentFontFullName.value(),"Black") )
		TheFontRec.lfWeight = FW_BOLD;	// other values don't work

    if(  (strstr(textinfo.currentFontName.value(),"Italic") != NULL) ||
		 (strstr(textinfo.currentFontName.value(),"Oblique") != NULL) ||
		 (strstr(textinfo.currentFontFullName.value(),"Italic") != NULL) ||
		 (strstr(textinfo.currentFontFullName.value(),"Oblique") != NULL) )
	{
		TheFontRec.lfItalic = TRUE;
    }
	else
	{
		TheFontRec.lfItalic = 0;
    }

	TheFontRec.lfUnderline = 0;
	TheFontRec.lfStrikeOut = 0;
	TheFontRec.lfOutPrecision = OUT_DEFAULT_PRECIS;
	TheFontRec.lfClipPrecision = CLIP_DEFAULT_PRECIS;
	TheFontRec.lfQuality = PROOF_QUALITY;
	TheFontRec.lfPitchAndFamily = VARIABLE_PITCH | FF_DONTCARE;	// let every font be possible

    if(  (strstr(textinfo.currentFontFullName.value(),"Symbol") != NULL) ||
		 (strstr(textinfo.currentFontFullName.value(),"symbol") != NULL) )
	{
		TheFontRec.lfCharSet = SYMBOL_CHARSET;
		strcpy(TheFontRec.lfFaceName,"symbol");
    }
	else
	{
		TheFontRec.lfCharSet = ANSI_CHARSET;
#ifdef MAPTOARIAL
		strcpy(TheFontRec.lfFaceName,"Arial");
#else
		strcpy(TheFontRec.lfFaceName,textinfo.currentFontFamilyName.value());
#endif
    }

	if(MyFont)
	{
		SelectObject(MetaDC,OldFont);
 		DeleteObject(MyFont);
		MyFont=0L;
	}

	MyFont=CreateFontIndirect(&TheFontRec);
	if(!MyFont)
	{
		errf << "ERROR: FetchFont: could not create font !" << endl;
	}
	else
	{
		OldFont= (HFONT) SelectObject(MetaDC,MyFont);
	}

	return 0;
}


// fill point array and draw
void drvWMF::print_coords(POINT* aptlPoints, int* aptlNumPts, int type)
{
	POINT				lastStart		= {0,0};
	bool				lastWasMoveTo	= FALSE;
	bool				lastWasClosePath= FALSE;
	unsigned int		numOfPolies		= 0;
	unsigned int		numOfPts		= 0;
    	unsigned int p=0;


    	for(unsigned int n=0; n<numberOfElementsInPath() && 
				  p<2*numberOfElementsInPath() &&
				  numOfPolies<2*numberOfElementsInPath(); n++)
	{
		const basedrawingelement& elem = pathElement(n);

		aptlPoints[p].x = transX(elem.getPoint(0).x_);
		aptlPoints[p].y = transY(elem.getPoint(0).y_);

		if(elem.getType() == moveto ||
		   elem.getType() == lineto ||
		   elem.getType() == curveto )
		{
			const long lineWidth = (int)(currentLineWidth() + .5);	// rounded int

			// set drawing sizes
			if(maxstatus)
			{
				if(aptlPoints[p].x - lineWidth < origin_x)
					origin_x=aptlPoints[p].x - lineWidth;

				if(aptlPoints[p].y - lineWidth < origin_y)
					origin_y=aptlPoints[p].y - lineWidth;
			}
			else
			{
				origin_x=aptlPoints[p].x - lineWidth;
				origin_y=aptlPoints[p].y - lineWidth;
				maxstatus=1;
			}

			if(minstatus)
			{
				if(aptlPoints[p].x + lineWidth > width)
					width=aptlPoints[p].x + lineWidth;

				if(aptlPoints[p].y + lineWidth > height)
					height=aptlPoints[p].y + lineWidth;
			}
			else
			{
				width=aptlPoints[p].x + lineWidth;
				height=aptlPoints[p].y + lineWidth;
				minstatus=1;
			}
		}

		switch(elem.getType())
		{
			case moveto:
				if( type == TYPE_LINES )
				{
					if( !MoveToEx(MetaDC, aptlPoints[p].x, aptlPoints[p].y, NULL) )
					{
						errf << "ERROR: MoveTo: " << aptlPoints[p].x << "," << aptlPoints[p].y << endl;
					}
				}
				else
				{
					// ignore every path that is no area!
					if( numOfPts > 2 )
					{
						// if path not closed -> do it manually!
						if( lastWasClosePath == FALSE )
						{
							if( p>=2*numberOfElementsInPath() )
							{
								errf << "ERROR: MoveTo: Out of array mem!" << endl;
							}

							// move this point one ahead
							aptlPoints[p+1]=aptlPoints[p];

							// insert line to startpoint
							aptlPoints[p] = lastStart;
						
							p++;
							numOfPts++;
						}

						// store number of points for old subpath
						aptlNumPts[numOfPolies] = numOfPts;

						// one polygon more (for PolyPolygon)
						numOfPolies++;
					}
					else if( numOfPts == 2 )
					{
						// we have a line here, so draw it!
						if( !MoveToEx(MetaDC, aptlPoints[p-2].x, aptlPoints[p-2].y, NULL) )
						{
							errf << "ERROR: MoveTo: " << aptlPoints[p-2].x << "," << aptlPoints[p-2].y << endl;
						}

						if( !LineTo(MetaDC, aptlPoints[p-1].x, aptlPoints[p-1].y) )
						{
							errf << "ERROR: LineTo: " << aptlPoints[p-1].x << "," << aptlPoints[p-1].y << endl;
						}
					}
					// all lower numbers do not represent a line/polygon. Ignore them

					// start this subpath newly
					numOfPts = 0;

					// set flag to remove lone moveto's
					lastWasMoveTo = TRUE;

					// clear flag to connect filled segments
					lastWasClosePath = FALSE;
				}

				// save last starting position in case we've to close path
				lastStart = aptlPoints[p];

				break;

			case curveto:
			case lineto:
				if( type == TYPE_LINES )
				{
					if( !LineTo(MetaDC, aptlPoints[p].x, aptlPoints[p].y) )
					{
						errf << "ERROR: LineTo: " << aptlPoints[p].x << "," << aptlPoints[p].y << endl;
					}
				}
				else
				{
					// clear flag to remove lone moveto's
					lastWasMoveTo = FALSE;

					// clear flag to connect filled segments
					lastWasClosePath = FALSE;

					// do nothing, point is already in array
				}
				break;

			case closepath:
				if( type == TYPE_LINES )
				{
					// close the thingy
					if( !LineTo(MetaDC, lastStart.x, lastStart.y) )
					{
						errf << "ERROR: LineTo: " << lastStart.x << "," << lastStart.y << endl;
					}
				}
				else
				{
					// clear flag to remove lone moveto's
					lastWasMoveTo = FALSE;

					// set flag to connect filled segments
					lastWasClosePath = TRUE;

					// insert line to startpoint
					// (overwrite current point - closepath did not inserted points)
					aptlPoints[p] = lastStart;
				}
				break;

			default:
				errf << "\t\tFatal: unexpected case in drvwmf " << endl;
				abort();
				break;
		}

		// next free point
		p++;

		// next point in subpath
		numOfPts++;
	}

	if( type != TYPE_LINES )
	{
		// remove last moveTo, as it's lonely
		if( lastWasMoveTo != FALSE )
			numOfPts--;

		// ignore every path that is no area!
		if( numOfPts > 2 )
		{
			// if path not closed -> do it manually!
			if( lastWasClosePath == FALSE )
			{
				if( p>=2*numberOfElementsInPath() )
				{
					errf << "ERROR: MoveTo: Out of array mem!" << endl;
				}
				
				// move this point one ahead
				aptlPoints[p+1]=aptlPoints[p];
				
				// insert line to startpoint
				aptlPoints[p] = lastStart;
				
				p++;
				numOfPts++;
			}
			
			// store number of points for old subpath
			aptlNumPts[numOfPolies] = numOfPts;
			
			// one polygon more (for PolyPolygon), as either we closed it above,
			// or closepath did it and started no new
			numOfPolies++;
		}
		else if( numOfPts == 2 )
		{
			// we have a line here, so draw it!
			if( !MoveToEx(MetaDC, aptlPoints[p-2].x, aptlPoints[p-2].y, NULL) )
			{
				errf << "ERROR: MoveTo: " << aptlPoints[p-2].x << "," << aptlPoints[p-2].y << endl;
			}
			
			if( !LineTo(MetaDC, aptlPoints[p-1].x, aptlPoints[p-1].y) )
			{
				errf << "ERROR: LineTo: " << aptlPoints[p-1].x << "," << aptlPoints[p-1].y << endl;
			}
		}
		// all lower numbers do not represent a line/polygon. Ignore them

		// anything to do?
		if( numOfPolies > 0 )
		{
			if( !SetPolyFillMode(MetaDC, (currentShowType()==fill) ? WINDING : ALTERNATE) )
			{
				errf << "ERROR: could not set fill mode" << endl;
			}
			
			if( !PolyPolygon(MetaDC, aptlPoints, aptlNumPts, numOfPolies) )
			{
				DWORD ec = GetLastError();
				errf << "ERROR: Polygon could not be drawn: (" << ec <<")"<< endl;
				errf << "Number of Points: " << p << endl;
				for(unsigned int i=0; i < p; i++)
				{
					errf << aptlPoints[i].x << "," << aptlPoints[i].y << endl;
				}
			}
		}
	}
}


void drvWMF::close_page()
{
	// no function in drvwmf
}


void drvWMF::open_page()
{
	// no function in drvwmf
}


void drvWMF::show_path()
{
	POINT*	aptlPoints = NULL;
	int*	aptlNumPts = NULL;	


	// get us twice the number of elements in path,
	// as maybe every subpath is closed and consists of ONE point!
	if( (aptlPoints=new POINT[2*numberOfElementsInPath()] )==NULL)
	{
		errf << "ERROR: Cannot allocate memory for point-array" << endl;
		return;
	}

	// get us twice the number of elements in path,
	// as maybe every subpath is closed and consists of ONE point!
	if( (aptlNumPts=new int[2*numberOfElementsInPath()] )==NULL)
	{
		errf << "ERROR: Cannot allocate memory for pointNum-array" << endl;
		return;
	}

	setColor(currentR(),currentG(),currentB());

	// determine type: fill or line
	switch( currentShowType() )
	{
    	case drvbase::stroke:
			// draw lines
			print_coords(aptlPoints, aptlNumPts, TYPE_LINES);
      		break;

    	case drvbase::fill:
    	case drvbase::eofill:
			// draw filled polygon
			print_coords(aptlPoints, aptlNumPts, TYPE_FILL);
      		break;

    	default:
			// cannot happen
      		errf << "unexpected ShowType " << (int) currentShowType() ;
      		break;
	}
	
	delete aptlPoints;
	delete aptlNumPts;
};


void drvWMF::show_text(const TextInfo & textinfo)
{
	// get a close color from palette
	SetTextColor(MetaDC, PALETTERGB(textinfo.currentR,textinfo.currentG,textinfo.currentR));

	const short int textHeight	= (short int)(textinfo.currentFontSize  + .5);		// rounded int
	const short int textAngle	= (short int)(10*textinfo.currentFontAngle + .5);	// win needs 10th of degree

	if (fontchanged() ) 
//	if( lastSelectedFontHeight != textHeight ||
//		lastSelectedFontAngle != textAngle ||
//		strcmp(lastSelectedFontName,textinfo.currentFontName.value()) )
	{
		FetchFont(textinfo, textHeight, textAngle);
//		strcpy(lastSelectedFontName,textinfo.currentFontName.value());
//		lastSelectedFontHeight = textHeight;
//		lastSelectedFontAngle  = textAngle;
	}

	const int x = transX(textinfo.x);
	const int y = transY(textinfo.y);

	// set drawing sizes
	if( maxstatus )
	{
		if(x - sin(textinfo.currentFontAngle)*textHeight < origin_x)
			origin_x=x - (long)(sin(textinfo.currentFontAngle)*textHeight);

		if(y - cos(textinfo.currentFontAngle)*textHeight < origin_y)
			origin_y=y - (long)(cos(textinfo.currentFontAngle)*textHeight);
	}
	else
	{
		origin_x=x - (long)(sin(textinfo.currentFontAngle)*textHeight);
		origin_y=y - (long)(cos(textinfo.currentFontAngle)*textHeight);
		maxstatus=1;
	}

	if( minstatus)
	{
		if(x + textHeight > width)
			width=x + textHeight;		// no need to care for string extent,
										// windows handles that

		if(y + textHeight > height)
			height=y + textHeight;
	}
	else
	{
		width=x + textHeight;
		height=y + textHeight;
		minstatus=1;
	}

	TextOut(MetaDC, x, y, textinfo.thetext.value(), strlen(textinfo.thetext.value()));
}


void drvWMF::show_rectangle(const float llx, const float lly, const float urx, const float ury )
{
	RECT	localRect;

	setColor(currentR(),currentG(),currentB());

	localRect.left	= transX(llx);
	localRect.top	= transY(lly);
	localRect.right	= transX(urx);
	localRect.bottom= transY(ury);

	if( currentShowType() == drvbase::stroke )
	{
		Polyline(MetaDC,(POINT*) &localRect,2);
	}
	else
	{
		Rectangle(MetaDC,transX(llx),transY(lly),transX(urx),transY(ury));
	}
}


void drvWMF::show_image(const Image &)
{
	// not yet implemented
	errf << "show_image: sorry, not yet implemented." << endl;
}



static DriverDescriptionT<drvWMF> D_wmf("wmf","Windows metafile","wmf",
								 
		true, // backend supports subpathes
		   // if subpathes are supported, the backend must deal with
		   // sequences of the following form
		   // moveto (start of subpath)
		   // lineto (a line segment)
		   // lineto 
		   // moveto (start of a new subpath)
		   // lineto (a line segment)
		   // lineto 
		   //
		   // If this argument is set to false each subpath is drawn 
		   // individually which might not necessarily represent
		   // the original drawing.

		false, // backend does not support curves (at least for WMF - have to take least common denominator here)
		true, // backend supports elements which are filled and have edges 
		true, // backend supports text
		false, // backend supports Images (sorry, not yet...)
		DriverDescription::noopen, // we open output file ourselves
		false); // if format supports multiple pages in one file (DEFINETELY not) 

static DriverDescriptionT<drvWMF> D_emf("emf","Enhanced Windows metafile","emf",
								 
		true, // backend supports subpathes
		   // if subpathes are supported, the backend must deal with
		   // sequences of the following form
		   // moveto (start of subpath)
		   // lineto (a line segment)
		   // lineto 
		   // moveto (start of a new subpath)
		   // lineto (a line segment)
		   // lineto 
		   //
		   // If this argument is set to false each subpath is drawn 
		   // individually which might not necessarily represent
		   // the original drawing.

		false, // backend does not support curves (at least for WMF - have to take least common denominator here)
		true, // backend supports elements which are filled and have edges 
		true, // backend supports text
		false, // backend supports Images (sorry, not yet...)
		DriverDescription::noopen, // we open output file ourselves
		false); // if format supports multiple pages in one file (DEFINETELY not) 
 
