/*****************************************************************************
 *                                                                           *
 * Program:   paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Modul:     mirror.c                                                       *
 *            mirror images horizontally or/and vertically                   *
 * Author:    Andreas Tille                                                  *
 * Date:      30.09.1998                                                     *
 * Copyright: Andreas Tille, 1999; GNU Public License                        *
 *                                                                           *
 *****************************************************************************/

#include <stdlib.h>
#include "paul.h" 

static void MirrorHorizontally(PICTURE *bild)
/* mirror image horizontally (at a vertical axis)
 * --- Parameter: ---
 * PICTURE *bild : image to mirror
 * --- Return: ---
 * PICTURE *bild : image with mirrored buffer
 */
{
  register unsigned char *ap, *ep;
  unsigned char          *zp, *fzp;
  int                     linesize = bild->storepix * bild->W;
   
  g_return_if_fail( IS_PICTURE(bild));

  for ( fzp = (zp = bild->DATA) + linesize * bild->H; zp < fzp; zp += linesize )
    for ( ep = (ap = zp) + linesize - 1; ap < ep; ap++, ep-- ) {
      if ( bild->storepix == 1 ) {
        register unsigned char zw = *ep;
        *ep = *ap;
        *ap =  zw;
      } else {
        register unsigned char zw = *(ep -= 2);
        *ep = *ap;
        *ap = zw;
        zw  = *++ep;
        *ep = *++ap;
        *ap = zw;
        zw  = *++ep;
        *ep = *++ap;
        *ap = zw;
        ep -= 2;
      }
    }
  if ( bild->im ) gdk_imlib_changed_image(bild->im);
}

static void MirrorVertically(PICTURE *bild)
/* mirror image vertically (at a horizontal axis)
 * --- Parameter: ---
 * PICTURE *bild : image to mirror
 * --- Return: ---
 * PICTURE *bild : image with mirrored buffer
 */
{
  register unsigned char *ap, *ep;
  unsigned char          *buf;
  int                     linesize = bild->storepix * bild->W;
   
  g_return_if_fail( IS_PICTURE(bild));

  buf = g_malloc(linesize);
   
  for ( ep = (ap = bild->DATA) + linesize * (bild->H - 1); ap < ep; ap += linesize, ep -= linesize ) {
    memcpy(buf, ep, linesize);
    memcpy(ep,  ap, linesize);
    memcpy(ap, buf, linesize);
  }
  FREE(buf);
  if ( bild->im ) gdk_imlib_changed_image(bild->im);
}

int MakeMirror(PAUL *p)
/* mirror a set of images on vertical (at a horizontal axis) or horizontal (at a vertical axis)
 * --- Parameter: ---
 * PAUL   *p            : list of images
 * --- Return: ---
 * int     MakeMirror() : RET_ERR or RET_OK
 */
{
  PICTURE *bild;
  GList   *pl;
  char    *desc;

  g_return_val_if_fail ( IS_PAUL(p), RET_ERR );
  g_return_val_if_fail ( IS_PICTURE(BILD(p->piclist)), RET_ERR );
  
  for ( bild = BILD(pl = p->piclist); pl; bild = BILD(pl = pl->next) ) {
    if ( HMirror(p->opt->f) ) MirrorHorizontally(bild);
    if ( VMirror(p->opt->f) ) MirrorVertically(bild);
    if ( !Rotate180(p->opt->f) ) {
      desc = g_strdup_printf("Mirror %s %s", ImgFileName(bild), 
                              HMirror(p->opt->f) ? "horizontally" : "vertically");
      ImgChunksUpdate(bild, TypMirror, desc, APPMIRROR, Mirror(p->opt->f));
      FREE(desc);
    }
  }
  p->opt->f &= ~(MIRROR_H | MIRROR_V);
  return RET_OK;
}


