/*
 * image.c: Image (xpm) loading and handline functions for XTux.
 * Copyright 1999 David Lawrence (philaw@camtech.net.au)
 * Last modified July 26.
 */

#include <X11/xpm.h>
#include "header.h"
#include "win.h"
#include "tile.h"
#include "entity.h"
#include "main.h"
#include "timing.h"
#include "image.h"

extern Display *display;
extern Window win;
extern Colormap cmap;

struct image_data_struct image_data[NUM_ENTITIES] = {
  /* Name                       width hgt dir img fr fr_len */
  /* GOODIE's */
  { "tux/tux_%s_%d.xpm",           64, 64, 8, 3, 4, 150000, 0 },
  { "tux_g/tux_g_%s_%d.xpm",       64, 64, 8, 3, 4, 150000, 0 },
  { "gown/gown_%s_%d.xpm",         44, 64, 8, 3, 4, 150000, 0 },
  { "bsd/bsd_%s_%d.xpm",           64, 64, 8, 3, 4, 200000, 0 },
  { "gnu/gnu_%s_%d.xpm",           72, 64, 8, 5, 4, 200000, 0 },
  { "vi/vi_%s_%d.xpm",             80, 64, 8, 5, 4, 200000, 0 },
  /* BADDIE's */
  { "mscp/mscp_%s_%d.xpm",         64, 64, 8, 3, 4, 200000, 1 },
  { "mscp/mscp_%s_%d.xpm",         64, 64, 1, 3, 3, 200000, 2 },
  { "clippy/clippy_%s_%d.xpm",     64, 64, 8, 2, 2, 330000, 1 },
  { "clippy/clippy_%s_%d.xpm",     64, 64, 1, 2, 2, 330000, 2 },
  { "troll/troll_%s_%d.xpm",       58, 80, 8, 5, 4, 250000, 0 },
  { "flamer/flamer_%s_%d.xpm",     42, 64, 8, 2, 2, 200000, 1 },
  { "flamer/flamer_%s_%d.xpm",     42, 64, 1, 4, 4, 200000, 2 },
  { "cobol/cobol_%s_%d.xpm",       64, 64, 8, 2, 2, 200000, 1 },
  { "cobol/cobol_%s_%d.xpm",       64, 64, 1, 3, 3, 200000, 2 },
  { "bug/bug_%s_%d.xpm",           64, 64, 8, 3, 4, 150000, 0 },
  /* NEUTRAL's */
  { "bunny/bunny_%s_%d.xpm",       32, 32, 8, 3, 4, 200000, 0 },
  { "bunny/bunny_groom_%s_%d.xpm", 32, 32, 8, 2, 2, 450000, 0 },
  { "bunny/head/bunnyh_%s.xpm",    24, 24, 8, 1, 1,      0, 0 },
  { "bunny/bunny_%s_%d.xpm",       32, 32, 1, 2, 1,      0, 2 },
  { "bloodspot.xpm",               16, 16, 1, 1, 1,      0, 0 },
  /* PROJECTILEs */
  { "cd.xpm",                      24, 24, 1, 1, 1,      0, 0 },
  { "fireball/fireball_%s%d.xpm",  24, 24, 1, 3, 3, 200000, 0 },
  { "nuke/nuke_%s.xpm",            64, 64, 8, 1, 1,      0, 0 },
  /* ITEM's */
  { "items/nuke_crate.xpm",        64, 64, 1, 1, 1,      0, 0 },
  { "items/can.xpm",               48, 48, 1, 1, 1,      0, 0 },
  { "items/disk.xpm",              32, 32, 1, 1, 1,      0, 0 },
  { "items/nodoze.xpm",            48, 48, 1, 1, 1,      0, 0 },
  { "items/spoon.xpm",             16, 32, 1, 1, 1,      0, 0 }
};

/*
 * Animation frame-order for entities
 */

int frame_order[NUM_ENTITIES][4] = {
  /* Goodies */
  { 0, 1, 0, 2 }, /* Tux */
  { 0, 1, 0, 2 }, /* Tux_G */
  { 0, 1, 0, 2 }, /* Gown */
  { 0, 1, 0, 2 }, /* BSD */
  { 1, 2, 3, 4 }, /* GNU */
  { 1, 2, 3, 4 }, /* VI */
  /* Baddies */
  { 0, 1, 0, 2 }, /* MSCP */
  { 0, 1, 2, 0 }, /* MSCP_DIE */
  { 0, 1, 0, 0 }, /* Clippy */
  { 0, 1, 0, 0 }, /* CLIPPY_DIE */
  { 1, 2, 3, 4 }, /* Troll */
  { 1, 1, 1, 1 }, /* Flamer */
  { 0, 1, 2, 3 }, /* FLAMER_DIE */
  { 0, 1, 0, 0 }, /* COBOL */
  { 0, 1, 2, 0 }, /* COBOL_DIE */
  { 0, 1, 0, 2 }, /* Bug */
  /* Neutrals */
  { 0, 1, 0, 2 }, /* Bunny */
  { 0, 1, 0, 0 }, /* Bunny_g */
  { 0, 0, 0, 0 }, /* Bunny head */
  { 0, 0, 0, 0 }, /* BUNNY_DEAD */
  { 0, 0, 0, 0 }, /* Blood spot */
  /* Projectiles */
  { 0, 0, 0, 0 }, /* CD */
  { 0, 1, 2, 0 }, /* Fireball */
  { 0, 0, 0, 0 }, /* Nuke */
  /* Items */
  { 0, 0, 0, 0 }, /* nuke_crate */
  { 0, 0, 0, 0 }, /* can */
  { 0, 0, 0, 0 }, /* disk */
  { 0, 0, 0, 0 }, /* No Doze */
  { 0, 0, 0, 0 }  /* Spoon */
};

Pixmap *entity_pixmap[NUM_ENTITIES];
Pixmap real_base_pixmap[NUM_REAL_BASE_TILES];
Pixmap tech_base_pixmap[NUM_TECH_BASE_TILES];
/* 2nd dimension in object pixmap arrays are for clip masks */
Pixmap real_object_pixmap[2][NUM_REAL_OBJECT_TILES];
Pixmap tech_object_pixmap[2][NUM_REAL_OBJECT_TILES];
/* Pixmaps used for status bar and other misc purposes */ 
Pixmap misc_pixmap[NUM_MISC_PIXMAPS];


Pixmap load_image(char *xpm_name, Pixmap *mask)
{

  int status;
  Pixmap pixmap;
  XpmAttributes attribs;
  /* (From xpm.h)
   * Return ErrorStatus codes:
   * null     if full success
   * positive if partial success
   * negative if failure
   */
  char *xpm_status[] = {
    "XpmColorFailed", /* -4 */
    "XpmNoMemory",    /* -3 */
    "XpmFileInvalid", /* -2 */
    "XpmOpenFailed",  /* -1 */
    "XpmSuccess",     /*  0 */
    "XpmColorError"   /*  1 */
  };

  attribs.valuemask = XpmColormap | XpmCloseness;
  attribs.colormap = cmap;
  attribs.closeness = 1<<16;

  status = XpmReadFileToPixmap(display,win, xpm_name, &pixmap, mask, &attribs);
  
  if( status ) {

    fprintf(stderr, "Xpm \"%s\" failed to load. Status = %s\n", xpm_name,
	    (status>=-4 && status<=1)? xpm_status[status+4] : "STATUS_ERROR!");
    /* Pixmap failed to load if status < 0, we must exit */
    if( status != XpmColorError ) /* The only recoverable error message */
      die("Exiting\n", status);

  }
#if INSANELY_VERBOSE
  else
    fprintf(stderr, "loaded %s\n", xpm_name);
#endif

  return pixmap;

}


/* load images, loads them all at the moment, will fix later :) */
int load_images(void) {
  
  int ent, dir;
  int num_dir, num_img;
  int i, d;
  int size;
  char buf[256]; /* Should be long enough (famous last words..) */
  Pixmap *img, *mask;

  /*
   *  Filenames for images
   */
  char *real_base_img_name[] = {
    "grass.xpm",
    "grass1.xpm",
    "concrete.xpm",
    "plank.xpm",
    "carpet.xpm",
    "marble.xpm",
    "marble1.xpm",
    "slate.xpm",
    "crater.xpm",
    "ash.xpm",
    "ice.xpm",
    "rockice.xpm",
    "snow.xpm",
    "snowf.xpm",
    "h2o.xpm",
    "h2o1.xpm",
    "brick.xpm",
    "brick1.xpm",
    "iwallnw.xpm",
    "iwallne.xpm",
    "iwallsw.xpm",
    "iwallse.xpm",
    "iwallv.xpm",
    "iwallh.xpm",
    "iwalltn.xpm",
    "iwallts.xpm",
    "iwallte.xpm",
    "iwalltw.xpm",
    "kbench.xpm"
  };

  char *tech_base_img_name[] = {
    "lightgrey.xpm",
    "green.xpm",
    "greenbase.xpm",
    "sd_corner.xpm",
    "sd_topicbsd.xpm",
    "sd_topiccrypt.xpm",
    "sd_topicgames.xpm",
    "sd_topicgnu.xpm",
    "sd_topiclinux.xpm",
    "sd_topicms.xpm",
    "sd_topicnet.xpm",
    "sd_topicquake.xpm",
    "sd_topicscience.xpm",
    "sd_topicspam.xpm"
  };
  
  
  char *real_object_img_name[] = {
    "pic.xpm",
    "pic1.xpm",
    "pic2.xpm",
    "red.xpm",
    "rum.xpm",
    "msn.xpm",
    "bmark.xpm",
    "rubble.xpm",
    "gpathv.xpm",
    "gpathh.xpm",
    "message.xpm",
    "kend.xpm",
    "sink.xpm",
    "toaster.xpm",
    "window.xpm",
    "nofeed.xpm",
    "bath.xpm",
    "bath1.xpm",
    "bed.xpm",
    "bed1.xpm",
    "chair.xpm",
    "couch.xpm",
    "couch1.xpm",
    "drawers.xpm",
    "fridge.xpm",
    "fridge1.xpm",
    "pc.xpm",
    "sink1.xpm",
    "table.xpm",
    "toilet.xpm",
    "towels.xpm",
    "tree.xpm",
    "tree1.xpm",
    "tv.xpm",
    "desk.xpm",
    "bath_b.xpm",
    "bath1_b.xpm",
    "bed_b.xpm",
    "bed1_b.xpm",
    "chair_b.xpm",
    "couch_b.xpm",
    "couch1_b.xpm",
    "drawers_b.xpm",
    "fridge_b.xpm",
    "fridge1_b.xpm",
    "pc_b.xpm",
    "sink1_b.xpm",
    "table_b.xpm",
    "toilet_b.xpm",
    "towels_b.xpm",
    "tree_b.xpm",
    "tree1_b.xpm",
    "tv_b.xpm",
    "desk_b.xpm"
  };
  
  char *tech_object_img_name[] = {
    "gy_txt.xpm",
    "gy_txt_ln.xpm",
    "grn_txt.xpm",
    "gb_top.xpm",
    "gb_bott.xpm",
    "sd_title.xpm",
    "sd_title1.xpm"
  };
  
  char *misc_img_name[] = {
    "caff.xpm",
    "cd_pic.xpm",
    "fire_pic.xpm",
    "radioactive_pic.xpm",
    "paper.xpm"
  };

  splash_screen_print("Loading Pixmaps..");

#define IMGPATH "images/"
#define RBP "real/base/"
#define ROP "real/object/"
#define TBP "tech/base/"
#define TOP "tech/object/"

  if( chdir(IMGPATH) < 0 ) {
    fprintf(stderr, "Can't find subdirectory %s!\n", IMGPATH);
    die("Error, quitting!!\n", 1);
  }

  /* Load tile pixmaps */
  splash_screen_print("Real base tiles");
  for( i = 0; i < NUM_REAL_BASE_TILES ; i++) {
    sprintf( buf, "%s%s", RBP, real_base_img_name[i]);
    real_base_pixmap[i] = load_image( buf, NULL );
  }

  splash_screen_print("Tech base tiles");
  for( i = 0; i < NUM_TECH_BASE_TILES ; i++) {
    sprintf( buf, "%s%s", TBP, tech_base_img_name[i]);
    tech_base_pixmap[i] = load_image( buf, NULL );
  }

  splash_screen_print("Real object tiles");
  for( i = 0; i < NUM_REAL_OBJECT_TILES ; i++) {
    sprintf( buf, "%s%s", ROP, real_object_img_name[i]);
    real_object_pixmap[0][i] = load_image( buf, &real_object_pixmap[1][i]);
  }

  splash_screen_print("Tech object tiles");
  for( i = 0; i < NUM_TECH_OBJECT_TILES ; i++) {
    sprintf( buf, "%s%s", TOP, tech_object_img_name[i]);
    tech_object_pixmap[0][i] = load_image( buf, &tech_object_pixmap[1][i]);
  }   
 
  /* Load misc pixmaps */
  splash_screen_print("Misc images");
  for( i = 0; i < NUM_MISC_PIXMAPS ; i++) {
    misc_pixmap[i] = load_image( misc_img_name[i], NULL );
  }   

  /* Load entity pixmaps */
  for( ent = 0 ; ent < NUM_ENTITIES ; ent++ ) {
    sprintf(buf, "Entity %d of %d", ent, NUM_ENTITIES);
    splash_screen_print(buf);
    num_dir = image_data[ent].directions;
    num_img = image_data[ent].images;
    /* Allocate space for pixmap arrays */
    size = 2 * num_dir * num_img * sizeof(Pixmap);
    if( (entity_pixmap[ent] = (Pixmap *)malloc( size )) == NULL )
      die("Malloc() Failed!!\n", 1);

    /* Load pixmaps from disk */
    for( d = 0 ; d < image_data[ent].directions ; d++ )
      for( i = 0; i < image_data[ent].images ; i++ ) {
	img = (entity_pixmap[ent] + d*num_img + i);
	/* Masks are located in the second half of the array, ie
	   beyond num_dir*num_img */
	mask = (entity_pixmap[ent] + num_dir*num_img + (d*num_img+i));
	/* We want "die" in the filename */
	if( image_data[ent].dying_ani == 2 )
	  dir = 8; /* "die" */
	else dir = d;
	sprintf(buf,image_data[ent].name,
		(num_dir>1 || dir==8)? dirtocompass(dir): "", i);
	*img = load_image(buf, mask);
      }
  }

  if( chdir("..") < 0 )
    die("Cannot change dir to parent directory!\n", 1);


  return 1;

} /* load_images() */


/* Returns a appropriate pixmap for tile x, y on map map */
Pixmap tile_pixmap(struct map_t *map, int layer, int i, int x, int y)
{

  int tile;
  int offset;
  unsigned char *ptr;

  /* Make sure booleans are either 1 or 0 */
  layer = !!layer;
  i = !!i;

  if( layer == BASE ) {
    ptr = map->base;
    offset = NUM_GLOBAL_BASE_TILES;
  } else {
    ptr = map->object;
    offset = NUM_GLOBAL_OBJECT_TILES;
  }

  tile = *(ptr + y*map->width + x);

  /* If tile is smaller than the offset, it's not drawn */
  if( tile < offset ) {
    if( tile >= O_MSG0 && tile <= O_MSG9 )
      tile = RO_MESSAGE;
    else {
      fprintf(stderr, "Tile_pixmap: Error, tile = %d\n", tile);
      die("Exiting!\n", 1);
    }
  }

  tile -= offset;

  if( map->tileset == REAL_TILESET )
    if( layer == BASE )
      return real_base_pixmap[tile];
    else
      return real_object_pixmap[i][tile];
  else if( map->tileset == TECH_TILESET )
    if( layer == BASE )
      return tech_base_pixmap[tile];
    else
      return tech_object_pixmap[i][tile];
  else {
    fprintf(stderr, "Tile pixmap: Error, layer is %d", layer);
    die("Layer is not REAL or BASE!\n", 1);
  }

  return (Pixmap)NULL; /* never reached. */

}


Pixmap misc_image(int type)
{


  if( type < 0 || type >= NUM_MISC_PIXMAPS ) {
    fprintf(stderr, "Requested bad image (%d) in misc_pixmap!\n", type);
    die("Exiting!\n", 1);
  }

  return misc_pixmap[type];

}


/* Draws an image onto a buffer */
void draw_entity(entity *ent, Drawable dest, int x, int y)
{

  int num_dir, num_img;
  int type;
  Pixmap img, mask;

  /* Depending on mode, draw a different entity */
  /* Depending on mode, act like a different entity */
  type = ent->type;
  if( type == BUNNY && ent->mode == GROOMING )
    type++;
 /* Draw dying animation */
  else if( ent->mode == DYING && image_data[type].dying_ani == 1 )
    type++;

  /* Get image details  */
  num_dir = image_data[type].directions;
  num_img = image_data[type].images;

  /* Check so we don't get out of range. Frames are checked in calc frame */
  if( ent->dir >= num_dir )
    ent->dir = 0;

  if( ent->img_no >= num_img )
    ent->img_no = 0;

  img = *(entity_pixmap[type] + ent->dir*num_img + ent->img_no);
  mask = *(entity_pixmap[type] + num_dir*num_img
	   + ent->dir*num_img + ent->img_no);

  trans_blit(img, dest, x, y, image_data[type].width, image_data[type].height,
	     mask);

}


/******************************************************************
 * frame in the entity struct goes from 0..image_data[ent->type].frames
 * calculate frame returns the image number, which is decided in the
 * array frame_order[]
 ******************************************************************/
int calculate_frame(entity *ent)
{

  struct timeval now;
  int type;

  /* Depending on mode, act like a different entity */
  type = ent->type;
  if( type == BUNNY && ent->mode == GROOMING )
    type++;
  /* Draw dying animation */
  else if( ent->mode == DYING && image_data[type].dying_ani == 1 )
    type++;

  if( image_data[type].frames <= 1 ) {
    if( image_data[type].directions > 1 )
      ent->dir = getdir(ent->x_v, ent->y_v);
    return ent->img_no; /* No point being here */
  }

  gettimeofday( &now, NULL );

  if( time_diff( now, ent->last_frame ) >= image_data[type].frame_len ) {
    if( ++ent->frame >= image_data[type].frames ) {
      /* Animation loop has finished, Set it to dead if dying or restart */
      if( ent->mode == DYING )
	ent->mode = DEAD;
      else
	ent->frame = 0;
    }
    ent->last_frame = now;
  }

  /* Frame = 0 if Stationary, otherwise we must work out dir and new frame
   grooming bunny (BUNNY_G) is the only type that animates while stationary */
  if( type != BUNNY_G ) {
    if( !( ent->x_v || ent->y_v) && ent->mode != DYING )
      ent->frame = 0;
    else
      ent->dir = getdir(ent->x_v, ent->y_v); /* Only update dir when moving */
  }

  return frame_order[type][ent->frame];

} /* Calculate frame */
