/*
                       Weapon Fire Object Creation

	Functions:

	int WepCreate(
		long weapon_type,
		long owner,
		double birth_x,
		double birth_y,
		birth_heading,
		double power,
		double range,
		double freq
	)

	---

	Various weapon fire creation functions.   Creating a weapon
	fire object (such as a torpedo) and sending it off on its way.
	movement.c takes care of the impact and lifespan management
	for these objects.

	EventHandleWeaponFire() is the front end function that should
	be called.   Does all permission checking on object_num.
	Returns number of objects created.
 */

#include "swserv.h"



int WepCreate(
	int ocs_code,		/* Object create script code. */
        long owner,		/* MUST BE VALID!!! */
        int emission_type,
        double birth_x,
        double birth_y,
        double birth_heading,
        double power,
        double range,		/* Needed for stream style weapons. */
	double freq
)
{
	int objects_created;
	long object_num;
	xsw_object_struct *obj_ptr;
	xsw_object_struct *owner_obj_ptr;
	ocs_struct *ocs_ptr;
	xsw_object_struct *opm_ptr;

	int ocs_num;
	int opm_num;
	double work_angle;

	double tmp_x, tmp_y, tmp_z;
	double tmp_h, tmp_p, tmp_b;


	/* Cease fire impose? */
	if(sysparm.cease_fire)
	    return(0);

	/* ocs_code must be valid. */
	if(ocs_code <= OCS_TYPE_GARBAGE)
	    return(-1);

	/* emission_type must be valid. */
	if(emission_type < 0)
	{
            fprintf(stderr,
                "WepCreate(): Error: Emission type %i is invalid.\n",
                emission_type
            );
	    return(-1);
	}

	/* Owner must be valid. */
	if(DBIsObjectGarbage(owner))
	{
	    fprintf(stderr,
		"WepCreate(): Error: Owner object #%ld is invalid.\n",
		owner
	    );
	    return(-1);
	}
	else
	{
	    owner_obj_ptr = xsw_object[owner];
	}

	/* Sanitize birth_heading. */
	birth_heading = SANITIZERADIANS(birth_heading);

	/* Sanitize power. */
	if(power < 0)
	    power = 0;

	/* Sanitize range. */
	if(range < 0)
	    range = 0;

	/* Sanitize frequency. */
	if(freq < SWR_FREQ_MIN)
	    freq = SWR_FREQ_MIN;
	if(freq > SWR_FREQ_MAX)
	    freq = SWR_FREQ_MAX;


	/* ********************************************************** */

	/* Get Object Create Ccript number by Type. */
	ocs_num = OCSGetByCode(ocs_code);
	if(OCSIsGarbage(ocs_num))
	{
	    fprintf(stderr,
		"WepCreate(): Error: %s requested invalid OCS type %i.\n",
		DBGetFormalNameStr(owner),
		ocs_code
	    );
	    return(-1);
	}
	else
	{
	    /* Get ocs pointer. */
	    ocs_ptr = ocs[ocs_num];
	}

	/* coppies must be 1 or greater. */
	if(ocs_ptr->coppies < 1)
	{
	    /* Do not create anything. */
	    return(0);
	}

	/* Get work_angle. */
        work_angle = SANITIZERADIANS(ocs_ptr->heading);


	/* *********************************************************** */

	/* Get OPM number by OPM name referanced in the OCS. */
	opm_num = OPMGetByName(ocs[ocs_num]->opm_name, -1);
	if(OPMIsGarbage(opm_num))
	{
            fprintf(stderr,
		"WepCreate(): Error: %s: No such OPM.\n",
                ocs_ptr->opm_name
            );
            return(-1);
	}
	else
	{
	    /* Get object parmameter macro. */
	    opm_ptr = opm[opm_num];
	}


	/* Sanitize number of coppies. */
	if((int)ocs_ptr->coppies > MAX_OCS_COPPIES)
	    ocs_ptr->coppies = MAX_OCS_COPPIES;


	/* *********************************************************** */

	/* Begin creating weapons fire objects. */
	objects_created = 0;
	while(objects_created < (int)ocs_ptr->coppies)
	{
	    /* Create an object. */
            object_num = DBCreateObject(
                opm_ptr->imageset,
                opm_ptr->type,
                owner,
                birth_x, birth_y, 0,	/* x, y, z. */
                birth_heading, 0, 0	/* heading, pitch, bank. */
            );
	    if(DBIsObjectGarbage(object_num))
            {
                /* Failed to create object, stop creating. */
                fprintf(stderr,
                    "WepCreate(): DBCreateObject(): Could not create object.\n"
                );
                break;
            }   
	    else
	    {
		/* Increment number of objects created. */
	        objects_created++;
	    }

	    /* Must reget owner and object pointers!! */
            owner_obj_ptr = xsw_object[owner];
            obj_ptr = xsw_object[object_num];


	    /* Record tempory positions and attitudes. */
	    tmp_x = birth_x;
	    tmp_y = birth_y;
	    tmp_z = 0;

	    tmp_h = birth_heading;
	    tmp_p = 0;
            tmp_b = 0;

            /* Adjust relative position. */
	    tmp_x +=
                MuPolarRotX(
                    birth_heading + work_angle,
		    ocs_ptr->radius
		);
	    tmp_y +=
		MuPolarRotY(
		    birth_heading + work_angle,
		    ocs_ptr->radius
		);


	    /* **************************************************** */

	    /* Set parameters on new object from OPM. */
	    if(OPMModelObject(object_num, opm_num))
	    {
/* What do we do if we can't model if successfully? */
		fprintf(stderr,
    "WepCreate(): Error using OPM #%i for new object #%ld.\n",
		    opm_num, object_num
		);
	    }

	    /* Set positions. */
	    obj_ptr->x = tmp_x;
            obj_ptr->y = tmp_y;
	    obj_ptr->z = tmp_z;

            obj_ptr->heading = tmp_h;
            obj_ptr->pitch = tmp_p;
            obj_ptr->bank = tmp_b;

	    /* Set sector of owner. */
	    obj_ptr->sect_x = owner_obj_ptr->sect_x;
	    obj_ptr->sect_y = owner_obj_ptr->sect_y;
	    obj_ptr->sect_z = owner_obj_ptr->sect_z;

	    /* Set owner. */
	    obj_ptr->owner = owner;

	    /* Set empire. */
            strncpy(obj_ptr->empire,
		owner_obj_ptr->empire,
                XSW_OBJ_EMPIRE_MAX
	    );
	    obj_ptr->empire[XSW_OBJ_EMPIRE_MAX - 1] = '\0';


	    /* Set emission type dependant values. */
	    switch(emission_type)
	    {
	      /*    Stream weapons have size given as range.
	       *    They will intercept locked object of owner.
	       */
	      case WEPEMISSION_STREAM:

                obj_ptr->size = range;

		if(DBIsObjectGarbage(owner_obj_ptr->locked_on))
		{
		    obj_ptr->intercepting_object = -1;
		    obj_ptr->locked_on = -1;
		}
		else
		{
                    obj_ptr->intercepting_object = owner_obj_ptr->locked_on;
		    obj_ptr->locked_on = owner_obj_ptr->locked_on;
		}
		break;

	       /*    Projectiles have size of OPM and intercept
                *    locked object of owner.
                */
	       case WEPEMISSION_PROJECTILE:
/*  Already set by modelling.
		obj_ptr->size = opm_ptr->size;
*/
                if(DBIsObjectGarbage(owner_obj_ptr->locked_on))
		{
                    obj_ptr->intercepting_object = -1;
                    obj_ptr->locked_on = -1;
		}
                else
		{
                    obj_ptr->intercepting_object = owner_obj_ptr->locked_on;
                    obj_ptr->locked_on = owner_obj_ptr->locked_on;
		}
                break;

	      /*    Default to pulse.  They have size of OPM and do NOT
               *    intercept locked object of owner.
               */
	      default:
/*  Already set by modelling.
                obj_ptr->size = opm_ptr->size;
*/
		obj_ptr->intercepting_object = -1;

		break;
	    }


	    /* Set velocites. */
/*
	    if(owner_obj_ptr->velocity < 0.0008)
	    {
                obj_ptr->velocity_heading =
		    owner_obj_ptr->heading;
	    }
	    else
            {
                obj_ptr->velocity_heading = owner_obj_ptr->velocity_heading;
		obj_ptr->velocity = owner_obj_ptr->velocity;
            }
*/
obj_ptr->velocity_heading = owner_obj_ptr->velocity_heading;
obj_ptr->velocity = owner_obj_ptr->velocity;


	    /* Set thrust as owner's thrust. */
            obj_ptr->thrust = owner_obj_ptr->thrust;

	    /* Set power from given function input. */
            obj_ptr->power = power;
	    obj_ptr->power_max = power;

	    /* Set frequency. */
	    obj_ptr->shield_frequency = freq;

	    /* Mark the time this object was created. */
            obj_ptr->birth_time = cur_millitime;


	    /* ***************************************************** */

	    /* Send create object request to all connections. */
	    NetSendCreateObject(-1, object_num);


	    /* Modify work_angle depending on emission_type. */
	    if((emission_type == WEPEMISSION_PROJECTILE) ||
	       (emission_type == WEPEMISSION_PULSE)
	    )
	    {
	        /* Flip work_angle to negative? */
	        if(work_angle > 0)
	        {
		    work_angle *= -1;
	        }
	        /* Flip angle to positive, then decrease angle. */
	        else
	        {
		    work_angle *= -1;
		    work_angle = work_angle *
		        (double)(1 / (double)(ocs_ptr->coppies));
		}
	    }
	}



	/* Return the number of objects created. */
	return(objects_created);
}
