;;; czech-calendar.el -- Czech calendar and holidays

;; Copyright (C) 1997 Milan Zamazal

;; Authors:    David auer <xsauer@hwlab.felk.cvut.cz>
;;             Tibor imko <tibor.simko@fmph.uniba.sk>
;;             Milan Zamazal <pdm@fi.muni.cz>
;; Maintainer: Milan Zamazal <pdm@fi.muni.cz>
;; Version:    $Id: czech-calendar.el,v 3.8 1997/11/08 01:12:40 pdm Exp $
;; Keywords:   i18n, Czech, calendar, holidays
;; KnownCompatibility: 19.34, XEmacs 19.14, 19.15

;; This file is *NOT* part of GNU Emacs nor XEmacs.

;; COPYRIGHT NOTICE
;;
;; This program is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by the Free
;; Software Foundation, version 2 of the License.
;;
;; This program is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
;; or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;; for more details.
;;
;; You should have received a copy of the GNU General Public License along
;; with GNU Emacs and/or this package.  If you did not, write to the
;; Free Software Foundation, Inc., 675 Mass Ave., Cambridge, MA 02139, USA.

;;; Commentary:

;; This file defines and redefines various calendar things according to Czech
;; national habits.

;;; History:

;; So long, so very long...

;;; Code:

(require 'czech)


;; Arghh, we have to replace the whole variable definition only because of two
;; strings.  But how to do it in better way in Emacs?
;; So this is cut, paste & edit from `holidays.el'.
(defvar cz-solar-holidays
  '((if (fboundp 'atan)
	(solar-equinoxes-solstices))
    (if (progn
	  (require 'cal-dst)
	  t)
	(funcall
	 'holiday-sexp
	 calendar-daylight-savings-starts
	 (if cz-sk "Zana letn as" "Zan letn as")))
    (funcall
     'holiday-sexp
     calendar-daylight-savings-ends
     (if cz-sk "Kon letn as" "Kon letn as")))
  "Czech variation of `solar-holidays'.")


;;; *** Setting function ***

;;;###autoload
(defun cz-calendar-init ()
  "Initialize Czech calendar settings and holidays.
See variables `cz-calendar-inhibit' and `cz-calendar-all-holidays' for more
info about behaviour of this function."
  (if cz-calendar-inhibit
      ()
    ;; initialize calendar settings
    (setq european-calendar-style t
	  calendar-week-start-day 1
	  calendar-day-name-array
          (if cz-sk
	      ["nedea" "pondelok" "utorok" "streda" "tvrtok" "piatok" 
	       "sobota"]
	    ["nedle" "pondl" "ter" "steda" "tvrtek" "ptek" "sobota"])
	  calendar-month-name-array
          (if cz-sk
	      ["janur" "februr" "marec" "aprl" "mj" "jn"
	       "jl" "august" "september" "oktber" "november" "december"]
	    ["leden" "nor" "bezen" "duben" "kvten" "erven"
	     "ervenec" "srpen" "z" "jen" "listopad" "prosinec"])
	  cal-tex-24 t
	  calendar-date-display-form '(day ". " month ". " year)
	  calendar-time-display-form '(24-hours ":" minutes
						(if time-zone " (")
						time-zone
						(if time-zone ")"))
	  display-time-24hr-format t)
    ;; initialize holidays
    (setq local-holidays
	  (if cz-sk
	      '((holiday-fixed
		 1 1    "De vzniku Slovenskej republiky a Nov rok")
		(holiday-fixed
		 1 6    "Zjavenie Pna (Traja krli a vianon sviatok pravoslvnych kresanov")
		(holiday-fixed
		 2 14   "Sviatok sv. Valentna")
		(holiday-fixed
		 3 25   "De zpasu za udsk prva")
		(holiday-fixed
		 4 1    "Aprl!")
		(holiday-fixed
		 4 13   "De nespravodlivo sthanch")
		(holiday-fixed
		 5 1    "Sviatok prce")
		(holiday-fixed
		 5 4    "Vroie mrtia M. R. tefnika")
		(holiday-float
		 5 0 2  "De matiek")
		(holiday-fixed
		 5 8    "De vazstva nad faizmom (1945)")
		(holiday-fixed
		 6 7    "Vroie Memoranda nroda slovenskho (1861)")
		(holiday-fixed
		 7 5    "Sviatok svtho Cyrila a svtho Metoda")
		(holiday-fixed
		 7 17    "Vroie Deklarcie o zvrchovanosti Slovenskej republiky (1992)")
		(holiday-fixed
		 8 4     "De Matice slovenskej")
		(holiday-fixed
		 8 29     "Vroie Slovenskho nrodnho povstania")
		(holiday-fixed
		 9 1     "De stavy Slovenskej republiky")
		(holiday-fixed
		 9 15    "Sedembolestn Panna Mria")
		(holiday-fixed
		 9 19    "De prvho verejnho vystpenia Slovenskej nrodnej rady (1948)")
		(holiday-fixed
		 10 6    "De obet Dukly (1944)")
		(holiday-fixed
		 10 27   "De ernovskej tragdie (1907)")
		(holiday-fixed
		 10 30   "Vroie Deklarcie slovenskho nroda (1918)")
		(holiday-fixed
		 10 31   "De reformcie")
		(holiday-fixed
		 11 2    "Sviatok vetkch svtch")
		(holiday-fixed
		 11 17  "De boja proti totalite (1989)")
		(holiday-fixed
		 12 24  "tedr de")
		(holiday-fixed
		 12 26  "Druh vianon sviatok")
		(holiday-fixed
		 12 31  "Silvester"))
	    '((holiday-fixed
	       1 1    "Nov rok")
	      (holiday-fixed
	       2 14   "Svtek sv. Valentna")
	      (holiday-fixed
	       4 1    "Aprl!")
	      (holiday-fixed
	       5 1    "Svtek prce")
	      (holiday-float
	       5 0 2  "Den matek")
	      (holiday-fixed
	       5 5    "Prask povstn (1945)")
	      (holiday-fixed
	       5 8    "Den osvobozen od faismu (1945)")
	      (holiday-fixed
	       7 5    "Den slovanskch vrozvst Cyrila a Metodje")
	      (holiday-fixed
	       7 6    "Mistr Jan Hus (1415)")
	      (holiday-fixed
	       10 28  "Den vzniku samostatnho eskoslovenskho sttu (1918)")
	      (holiday-fixed
	       11 2   "Pamtka zesnulch")
	      (holiday-fixed
	       11 17  "Den boje student za svobodu a demokracii")
	      (holiday-fixed
	       12 24  "tdr den")
	      (holiday-fixed
	       12 26  "2. svtek vnon")
	      (holiday-fixed
	       12 31  "Silvestr"))))
    (delete '(holiday-fixed 12 25 "Christmas") christian-holidays)
    (setq christian-holidays
          (if cz-sk
	      (add-to-list 'christian-holidays
			   '(holiday-fixed 12 25 "Prv vianon sviatok"))
	    (add-to-list 'christian-holidays
			 '(holiday-fixed 12 25 "1. svtek vnon"))))
    ;; Setup new holidays
    ;; (I haven't found better way for doing this.)
    (setq calendar-holidays
	  (append
	   (if cz-calendar-all-holidays general-holidays nil)
	   local-holidays other-holidays christian-holidays hebrew-holidays
	   islamic-holidays
	   (if cz-xemacs () oriental-holidays)
	   cz-solar-holidays))))


;; Another replacing case.  Cut, paste & edit from `holidays.el' again.
;; Here we moreover slightly modify produced list, because in Czech Republic
;; there is another set of mandatory holidays.
(if cz-calendar-inhibit
    ()
  (require 'holidays)
  (defun holiday-easter-etc ()
    "List of dates related to Easter, as visible in calendar window."
    (if (and (> displayed-month 5) (not all-christian-calendar-holidays))
	nil;; Ash Wednesday, Good Friday, and Easter are not visible.
      (let* ((century (1+ (/ displayed-year 100)))
	     (shifted-epact;; Age of moon for April 5...
	      (% (+ 14 (* 11 (% displayed-year 19));;     ...by Nicaean rule
		    (-;; ...corrected for the Gregorian century rule
		     (/ (* 3 century) 4))
		    (/;; ...corrected for Metonic cycle inaccuracy.
		     (+ 5 (* 8 century)) 25)
		    (* 30 century));;              Keeps value positive.
		 30))
	     (adjusted-epact;;  Adjust for 29.5 day month.
	      (if (or (= shifted-epact 0)
		      (and (= shifted-epact 1) (< 10 (% displayed-year 19))))
		  (1+ shifted-epact)
		shifted-epact))
	     (paschal-moon;; Day after the full moon on or after March 21.
	      (- (calendar-absolute-from-gregorian
		  (list 4 19 displayed-year))
		 adjusted-epact))
	     (abs-easter (calendar-dayname-on-or-before
			  0 (+ paschal-moon 7)))
	     (mandatory
              (if cz-sk
		  (list
		   (list (calendar-gregorian-from-absolute abs-easter)
			 "Vekonon nedea")
		   (list (calendar-gregorian-from-absolute (1+ abs-easter))
			 "Vekonon pondelok"))
		(list
		 (list (calendar-gregorian-from-absolute abs-easter)
		       "Velikonon nedle")
		 (list (calendar-gregorian-from-absolute (1+ abs-easter))
		       "Velikonon pondl"))))
	     (optional
	      (list
	       (if cz-sk
		   (list (calendar-gregorian-from-absolute (- abs-easter 2))
			 "Vek piatok")
		 (list (calendar-gregorian-from-absolute (- abs-easter 2))
		       "Good Friday"))
	       (list (calendar-gregorian-from-absolute (- abs-easter 46))
		     "Ash Wednesday")
	       (list (calendar-gregorian-from-absolute (- abs-easter 63))
		     "Septuagesima Sunday")
	       (list (calendar-gregorian-from-absolute (- abs-easter 56))
		     "Sexagesima Sunday")
	       (list (calendar-gregorian-from-absolute (- abs-easter 49))
		     "Shrove Sunday")
	       (list (calendar-gregorian-from-absolute (- abs-easter 48))
		     "Shrove Monday")
	       (list (calendar-gregorian-from-absolute (- abs-easter 47))
		     "Shrove Tuesday")
	       (list (calendar-gregorian-from-absolute (- abs-easter 14))
		     "Passion Sunday")
	       (list (calendar-gregorian-from-absolute (- abs-easter 7))
		     "Palm Sunday")
	       (list (calendar-gregorian-from-absolute (- abs-easter 3))
		     "Maundy Thursday")
	       (list (calendar-gregorian-from-absolute (+ abs-easter 35))
		     "Rogation Sunday")
	       (list (calendar-gregorian-from-absolute (+ abs-easter 39))
		     "Ascension Day")
	       (list (calendar-gregorian-from-absolute (+ abs-easter 49))
		     "Pentecost (Whitsunday)")
	       (list (calendar-gregorian-from-absolute (+ abs-easter 50))
		     "Whitmonday")
	       (list (calendar-gregorian-from-absolute (+ abs-easter 56))
		     "Trinity Sunday")
	       (list (calendar-gregorian-from-absolute (+ abs-easter 60))
		     "Corpus Christi")))
	     (output-list
	      (filter-visible-calendar-holidays mandatory)))
	(if all-christian-calendar-holidays
	    (setq output-list
		  (append 
		   (filter-visible-calendar-holidays optional)
		   output-list)))
	output-list))))


;;; *** Announce ***

(provide 'czech-calendar)


;;; czech-calendar.el ends here
