/****************************************************************************
|                         Digital Audio Processor
|                         =======================
|
| Filename    : delayint.cc
|
| Revision    : 1.1
| Date        : 25/01/96
|
| Object      : DSPKitDelayInt
|
| Description : DSPKitDelayInt
|
| (c) Richard Kent 1996
|
| $Id$
|
****************************************************************************/

static char delayint_cc [] = "$Id$";

#include "delayint.h"

DSPKitDelayInt::DSPKitDelayInt ()
{
  buffer   = 0;
  modInput = 0;
}

DSPKitDelayInt::~DSPKitDelayInt ()
{
  if (buffer) delete [] buffer; 
}

long DSPKitDelayInt::setInputAndDelayTime (DSPKitProcessor *ip,double dt)
{
  return setInputAndDelayTime (ip,dt,0);
}

long DSPKitDelayInt::setInputAndDelayTime
  (DSPKitProcessor *ip,double dt,long feedback)
{
  DSPKitProcessor::setInput (ip);
  delayTime = dt;
  bufSize = ((long)(delayTime * inputSamplingRate)) * 2;
  
  if (feedback) bufSize = bufSize - 2;
  if (bufSize < 4)
  {
    bufSize = 4;
  }
  if (bufSize > 0)
  {
    if ((buffer = new double [bufSize]) == 0) return -1;
    for (long i=0; i<bufSize; i++) buffer [i] = 0;
  }

  currentSampleIndex = 0;
  lastSampleIndex    = 0;
  haveSamples        = 0;
  endOfInput         = 0;
  return 0;
}

long DSPKitDelayInt::setModInput (DSPKitProcessor *ip)
{
  modInput = ip;
  modInput->setOutput (this);
  return 1;
}

long DSPKitDelayInt::getSample (double &outputSample)
{
  double inputSample;

  if (!bufSize)
  {
    if (input->getSample (outputSample)) return 1;
    return 0;
  }

  if (endOfInput)
  {
    input->getSample (inputSample); // for feedback loops
    if (currentSampleIndex == lastSampleIndex) return 0;
    currentSampleIndex = currentSampleIndex - 1;
    while (currentSampleIndex < 0) currentSampleIndex += bufSize;
    buffer [currentSampleIndex] = 0.0;
    outputSample = getOutput (currentSampleIndex);
    return 1;
  } 

  if (input->getSample (inputSample) == 0)
  {
    endOfInput = 1;
    if (!haveSamples) return 0;
    currentSampleIndex = currentSampleIndex - 1;
    while (currentSampleIndex < 0) currentSampleIndex += bufSize;
    buffer [currentSampleIndex] = 0.0;
    outputSample = getOutput (currentSampleIndex);
    return 1;
  } 

  haveSamples = 1;
  currentSampleIndex = currentSampleIndex - 1;
  while (currentSampleIndex < 0) currentSampleIndex += bufSize;
  buffer [currentSampleIndex] = inputSample;
  lastSampleIndex = currentSampleIndex;
  outputSample = getOutput (currentSampleIndex);
  return 1;
}

double DSPKitDelayInt::getOutput(long currentSampleIndex)
{
  double modInputSample;
  
  if (!modInput || (modInput->getSample (modInputSample) == 0))
    modInputSample = 0.0;
  
  if (modInputSample < -1.0) modInputSample = -1.0;
  else if (modInputSample > 1.0) modInputSample = 1.0;
  
  double delay = (bufSize / 2) + (modInputSample * (bufSize / 2));
  
  if (delay < 0.0) delay = 0.0;
  
  long delayInt = (long) delay;
  long x1 = currentSampleIndex + delayInt;
  
  if (x1 > currentSampleIndex + bufSize - 3)
    x1 = currentSampleIndex + bufSize - 3;
  else if (x1 < currentSampleIndex + 1)
    x1 = currentSampleIndex + 1;
  
//   // Linear Interpolation
//   long x2 = x1 + 1;
//   x1 = x1 % bufSize;
//   x2 = x2 % bufSize;
//   double val =
//     buffer [x1] + (delay - delayInt) * (buffer [x2] - buffer [x1]); 
  
  // Cubic Interpolation
  double x = currentSampleIndex + delay;
  long x0 = x1 - 1;
  long x2 = x1 + 1;
  long x3 = x1 + 2;

  double f0 = buffer [x0 % bufSize];
  double f1 = buffer [x1 % bufSize];
  double f2 = buffer [x2 % bufSize];
  double f3 = buffer [x3 % bufSize];

  double df0   = f1 - f0;
  double df1   = f2 - f1;
  double df2   = f3 - f2;
  double ddf0  = df1 - df0;
  double ddf1  = df2 - df1;
  double dddf0 = ddf1 - ddf0;    

  double val =
    (f0 + ((x - x0) * df0) +
    ((x - x0) * (x - x1) * ddf0 / 2.0) +
    ((x - x0) * (x - x1) * (x - x2) * dddf0 / 6.0));  

  return val;
}

/***************************************************************************/
