/****************************************************************************
|                         Digital Audio Processor
|                         =======================
|
| Filename    : DPSample.cc
|
| Revision    : 1.1
| Date        : 25/01/96
|
| Object      : DPSample
|
| Description : DPSample class for handling audio samples
|
| (c) Richard Kent 1996
|
| $Id$
|
****************************************************************************/

static char DPSample_cc [] = "$Id$";

#include "DPSample.h"

// defines

// #define NOFORMS to compile without using xforms
// #define SNEAKYCHECK for xforms sneaky check

#define PORTNAME   "TichPort"
#define ANNONEW    "  edited with Digital Audio Processor - Richard Kent"
#define ZOOMFACTOR       1.50
#define CHECKTIME        0.20
#define LINETIME         0.50
#define DRAWTIME         1.00
#define PLAYQUEUETIME    0.50
#define RECORDQUEUETIME  1.00

// Global variables used when playing samples
long DPSampleStop;
long DPSampleRelease1;
long DPSampleRelease2;
long DPSampleInc;

#define COPYREST                       \
  if (stereoMono)                      \
  {                                    \
    if (mode == 0)                     \
    {                                  \
      for (i=0; i<newBufferLen; i++)   \
      {                                \
        temp = getFrame24 (i,1);       \
        newSamp.setFrame24 (i,1,temp); \
      }                                \
    }                                  \
    else                               \
    {                                  \
      for (i=0; i<newBufferLen; i++)   \
      {                                \
        temp = getFrame24 (i,0);       \
        newSamp.setFrame24 (i,0,temp); \
      }                                \
    }                                  \
  }                                    \
  else if (quadroMono)                 \
  {                                    \
    if (mode == 0)                     \
    {                                  \
      for (i=0; i<newBufferLen; i++)   \
      {                                \
        temp = getFrame24 (i,1);       \
        newSamp.setFrame24 (i,1,temp); \
        temp = getFrame24 (i,2);       \
        newSamp.setFrame24 (i,2,temp); \
        temp = getFrame24 (i,3);       \
        newSamp.setFrame24 (i,3,temp); \
      }                                \
    }                                  \
    else if (mode == 1)                \
    {                                  \
      for (i=0; i<newBufferLen; i++)   \
      {                                \
        temp = getFrame24 (i,0);       \
        newSamp.setFrame24 (i,0,temp); \
        temp = getFrame24 (i,2);       \
        newSamp.setFrame24 (i,2,temp); \
        temp = getFrame24 (i,3);       \
        newSamp.setFrame24 (i,3,temp); \
      }                                \
    }                                  \
    else if (mode == 3)                \
    {                                  \
      for (i=0; i<newBufferLen; i++)   \
      {                                \
        temp = getFrame24 (i,0);       \
        newSamp.setFrame24 (i,0,temp); \
        temp = getFrame24 (i,1);       \
        newSamp.setFrame24 (i,1,temp); \
        temp = getFrame24 (i,3);       \
        newSamp.setFrame24 (i,3,temp); \
      }                                \
    }                                  \
    else                               \
    {                                  \
      for (i=0; i<newBufferLen; i++)   \
      {                                \
        temp = getFrame24 (i,0);       \
        newSamp.setFrame24 (i,0,temp); \
        temp = getFrame24 (i,1);       \
        newSamp.setFrame24 (i,1,temp); \
        temp = getFrame24 (i,2);       \
        newSamp.setFrame24 (i,2,temp); \
      }                                \
    }                                  \
  }

//  Most edit functions have a mode :-
//
//  mode = 1 : Channel 1 edit
//  mode = 2 : Channel 2 edit
//  mode = 3 : All channels edit
//  mode = 4 : Channel 3 edit
//  mode = 5 : Channel 4 edit
//
//  Setup like this due to legacy (was originally only a stereo program !)

/*---------------------------------------------------------------------------
| PUBLIC METHOD DPSample::DPSample
---------------------------------------------------------------------------*/

DPSample::DPSample ()
{
//cerr << "DPSample::DPSample" << endl;
  filename   = 0;
  name       = 0;
  author     = 0;
  copyright  = 0;
  annotation = 0;
  data       = 0;
  extra      = 0;
  clear ();
}

/*---------------------------------------------------------------------------
| PUBLIC METHOD DPSample::~DPSample
---------------------------------------------------------------------------*/

DPSample::~DPSample ()
{
//cerr << "DPSample::~DPSample" << endl;
  clear ();
}

/*---------------------------------------------------------------------------
| PUBLIC METHOD DPSample::DPSample (copy construxtor)
---------------------------------------------------------------------------*/

DPSample::DPSample (DPSample &newSample)
{
//cerr << "DPSample::DPSample" << endl;
  clone (newSample);
}

/*---------------------------------------------------------------------------
| PUBLIC METHOD DPSample::operator= (assignment operator)
---------------------------------------------------------------------------*/

DPSample &DPSample::operator= (DPSample &newSample)
{
  clone (newSample);
  return *this;
}

/*---------------------------------------------------------------------------
| PUBLIC METHOD DPSample::clone
---------------------------------------------------------------------------*/

long DPSample::clone (DPSample &newSample)
{
  long error  = FALSE;
  clear ();
  
  valid       = newSample.valid;
  changed     = newSample.changed;
  format      = newSample.format;
  compression = newSample.compression;
  rate        = newSample.rate;
  width       = newSample.width;
  channels    = newSample.channels;
  frames      = newSample.frames;
  samples     = newSample.samples;
  bytes       = newSample.bytes;
  AESused     = newSample.AESused;
  
  for (long i=0; i<24; i++)
    AES [i] = newSample.AES [i];
    
  if (newSample.filename)
  {
    filename = new char [strlen(newSample.filename) + 1];
    strcpy (filename,newSample.filename);
  }
  else filename = 0;

  if (newSample.name)
  {
    name = new char [strlen(newSample.name) + 1];
    strcpy (name,newSample.name);
  }
  else name = 0;

  if (newSample.copyright)
  {
    copyright = new char [strlen(newSample.copyright) + 1];
    strcpy (copyright,newSample.copyright);
  }
  else copyright = 0;
  
  if (newSample.author)
  {
    author = new char [strlen(newSample.author) + 1];
    strcpy (author,newSample.author);
  }
  else author = 0;
  
  if (newSample.annotation)
  {
    annotation = new char [strlen(newSample.annotation) + 1];
    strcpy (annotation,newSample.annotation);
  }
  else annotation = 0;
  
  susLoopMode  = newSample.susLoopMode;
  susLoopStart = newSample.susLoopStart;
  susLoopEnd   = newSample.susLoopEnd;
  relLoopMode  = newSample.relLoopMode;
  relLoopStart = newSample.relLoopStart;
  relLoopEnd   = newSample.relLoopEnd;
  rangeValid   = newSample.rangeValid;
  rangeStart   = newSample.rangeStart;
  rangeEnd     = newSample.rangeEnd;
  displayStart = newSample.displayStart;
  displayEnd   = newSample.displayEnd;
  redraw       = newSample.redraw;
  currentPos   = newSample.currentPos;

  if (newSample.data && bytes > 0)
  {
    data = new char [bytes];
    if (data)
    {
      for (long i=0; i<bytes; i++)
        ((char *) data) [i] = ((char *) newSample.data) [i];
    }
    else
    {
      frames  = 0;
      samples = 0;
      bytes   = 0;
      error   = TRUE;
    }
  }
  else data = 0;
  
  extra = 0; // Does not copy extra across
  if (error) return ERROR;
  return NOERROR;
}

/*---------------------------------------------------------------------------
| PUBLIC METHOD DPSample::cloneSimple
---------------------------------------------------------------------------*/

long DPSample::cloneSimple (DPSample &newSample)
{
  // Just copy across strings

  long error = FALSE;

  if (!newSample.valid) return ERROR;

  delete [] filename;
  delete [] name;
  delete [] author;
  delete [] copyright;
  delete [] annotation;

  if (newSample.filename)
  {
    filename = new char [strlen(newSample.filename) + 1];
    strcpy (filename,newSample.filename);
  }
  else filename = 0;

  if (newSample.name)
  {
    name = new char [strlen(newSample.name) + 1];
    strcpy (name,newSample.name);
  }
  else name = 0;

  if (newSample.copyright)
  {
    copyright = new char [strlen(newSample.copyright) + 1];
    strcpy (copyright,newSample.copyright);
  }
  else copyright = 0;
  
  if (newSample.author)
  {
    author = new char [strlen(newSample.author) + 1];
    strcpy (author,newSample.author);
  }
  else author = 0;
  
  if (newSample.annotation)
  {
    annotation = new char [strlen(newSample.annotation) + 1];
    strcpy (annotation,newSample.annotation);
  }
  else annotation = 0;
  
  if (error) return ERROR;
  return NOERROR;
}

/*---------------------------------------------------------------------------
| PUBLIC METHOD DPSample::cloneData
---------------------------------------------------------------------------*/

long DPSample::cloneData (DPSample &newSample)
{
  // Just copy across data

  long error  = FALSE;

  if (!newSample.valid) return ERROR;
  
  changed     = TRUE;
  rate        = newSample.rate;
  width       = newSample.width;
  channels    = newSample.channels;
  frames      = newSample.frames;
  samples     = newSample.samples;
  bytes       = newSample.bytes;

  delete [] data;

  if (newSample.data && bytes > 0)
  {
    data = new char [bytes];
    if (data)
    {
      for (long i=0; i<bytes; i++)
        ((char *) data) [i] = ((char *) newSample.data) [i];
    }
    else
    {
      frames  = 0;
      samples = 0;
      bytes   = 0;
      error   = TRUE;
    }
  }
  else data = 0;
  
  // Loop, range and display might be invalid
  if (susLoopMode != AF_LOOP_MODE_NOLOOP)
    setSusLoop (susLoopStart,susLoopEnd);
  if (relLoopMode != AF_LOOP_MODE_NOLOOP)
    setRelLoop (relLoopStart,relLoopEnd);
  if (rangeValid)
    setRange (rangeStart,rangeEnd);
  setDisplay (displayStart,displayEnd);

  extra = 0; // Does not copy extra across
  if (error) return ERROR;
  return NOERROR;
}

/*---------------------------------------------------------------------------
| PUBLIC METHOD DPSample::getFrame24Interp
---------------------------------------------------------------------------*/

long DPSample::getFrame24Interp (double frame,long channel)
{
  if (frame < 0.0 || frame > (double)(frames - 1)) return 0;
  if (channels == 1)
    channel = 0;
  else if (channels == 2)
    channel = channel % 2;
  else
    channel = channel % 4;
  if (frames == 0)
  {
    return 0;
  }
  if (frames == 1)
  {
    return getFrame24 (0,channel);
  }
  if (frames == 2)
  {
    long val = (long) rint ((1.0 - frame) * getFrame24 (0,channel) + 
      frame * getFrame24 (1,channel));
    if (val > MAX23_1) val = MAX23_1;
    if (val < -MAX23) val = -MAX23;
    return val;
  }
  if (frames == 3)
  {
    double x  = frame;
    long x0   = 0;
    long x1   = 1;
    long x2   = 2;
    long f0   = getFrame24 (x0,channel);
    long f1   = getFrame24 (x1,channel);
    long f2   = getFrame24 (x2,channel);
    long df0  = f1 - f0;
    long df1  = f2 - f1;
    long ddf0 = df1 - df0;
    
    long val  = (long) rint
      (f0 + ((x - x0) * df0) + 
      ((x - x0) * (x - x1) * ddf0 / 2.0));
    if (val > MAX23_1) val = MAX23_1;
    if (val < -MAX23) val = -MAX23;
    return val;
  }

  double x   = frame;
  long x0;
  long x1;
  long x2;
  long x3;
  
  if (frame < 1.0)
    x1 = 1;
  else if (frame > (double)(frames - 3))
    x1 = frames - 3;
  else
    x1 = (long) frame;
  x0 = x1 - 1;
  x2 = x1 + 1;
  x3 = x2 + 1;
  
  long f0    = getFrame24 (x0,channel);
  long f1    = getFrame24 (x1,channel);
  long f2    = getFrame24 (x2,channel);
  long f3    = getFrame24 (x3,channel);
  long df0   = f1 - f0;
  long df1   = f2 - f1;
  long df2   = f3 - f2;
  long ddf0  = df1 - df0;
  long ddf1  = df2 - df1;
  long dddf0 = ddf1 - ddf0;    
  
  long val = (long) rint
    (f0 + ((x - x0) * df0) +
    ((x - x0) * (x - x1) * ddf0 / 2.0) +
    ((x - x0) * (x - x1) * (x - x2) * dddf0 / 6.0));  
  if (val > MAX23_1) val = MAX23_1;
  if (val < -MAX23) val = -MAX23;
  return val;
}

/*---------------------------------------------------------------------------
| PUBLIC METHOD DPSample::getFrame16Interp
---------------------------------------------------------------------------*/

short DPSample::getFrame16Interp (double frame,long channel)
{
  return ((short)((getFrame24Interp (frame,channel)) / 256));
}

/*---------------------------------------------------------------------------
| PUBLIC METHOD DPSample::getFrameDbInterp
---------------------------------------------------------------------------*/

double DPSample::getFrameDbInterp (double frame,long channel)
{
  return (((double) getFrame24Interp (frame,channel)) / MAX23);
}

/*---------------------------------------------------------------------------
| PUBLIC METHOD DPSample::loadAIFF
---------------------------------------------------------------------------*/

char *DPSample::loadAIFF (char *newFilename)
{
  int           irixFile;
  AFfilesetup   fileSetup;
  AFfilehandle  file;
  long          bits;
  long          byteFormat;
  int           license;
  int           licensevoid;
  char          *licenseString;
  long          markerCount;
  long          *markerIDs;
  long          *markerFrames;
  long          instCount;
  long          *instIDs;
  long          susLoopID;
  long          relLoopID;
  long          susStartID;
  long          relStartID;
  long          susEndID;
  long          relEndID;
  long          miscCount;
  long          *miscIDs;

//cerr << "DPSample::load" << endl;
  clear ();

  // Open requested irix file
  irixFile = open (newFilename,O_RDONLY);
  if (irixFile == -1) return "Cannot open specified file";
    
  // Check irix file format
  format = AFidentifyfd (irixFile);
  switch (format)
  {
    case AF_FILE_AIFF :
    case AF_FILE_AIFFC :
      break;
    case AF_FILE_UNSUPPORTED :
      close (irixFile);
      return "Unsupported file format";
    case AF_FILE_UNKNOWN :
    default:
      close (irixFile);
      return "Unknown file format";
  }

  close (irixFile);
  
  // Disable void handling
  AFseterrorhandler (DPSampleAudiofileError);

  // Create new audio file setup
  fileSetup = AFnewfilesetup ();  

  // Create audio file handle
  file = AFopenfile (newFilename,"r",fileSetup);
  if (file == AF_NULL_FILEHANDLE)
  {
    AFfreefilesetup (fileSetup);
    return "Cannot create audio file handle";
  }

  // Get audio track sample rate, format and bits
  rate = AFgetrate (file,AF_DEFAULT_TRACK);
  AFgetsampfmt (file,AF_DEFAULT_TRACK,&byteFormat,&bits);

  // Check sample format
  switch (byteFormat)
  {
    case AF_SAMPFMT_TWOSCOMP :
      break;
    default :
      AFclosefile (file);
      AFfreefilesetup (fileSetup);
      return "Unknown sample format";
  }

  // Check audio track no. of bits and calculate byte width (byte
  // widths allowed are 1 (char) and 2 (short) for up to 16 bits)
  if (bits > 16)
  {
    AFclosefile (file);
    AFfreefilesetup (fileSetup);
    return "Bit resolution of more than 16 not supported";
  }
  width = ((bits - 1) / 8) + 1;

  // Get number of audio track sample channels
  channels = AFgetchannels (file,AF_DEFAULT_TRACK);
  if (channels != 1 && channels != 2 && channels != 4)
  {
    AFclosefile (file);
    AFfreefilesetup (fileSetup);
    return "Only 1,2 or 4 channel AIFF files supported";
  }
  
  // Get audio track compression
  compression = AFgetcompression (file,AF_DEFAULT_TRACK);
  switch (compression)
  {
    #ifndef LINUX
    case AF_COMPRESSION_NONE :
    case AF_COMPRESSION_G722 :
    case AF_COMPRESSION_G711_ULAW :
    case AF_COMPRESSION_G711_ALAW :
      break;
    #else
    case AF_COMPRESSION_NONE :
      break;
    case AF_COMPRESSION_G722 :
    case AF_COMPRESSION_G711_ULAW :
    case AF_COMPRESSION_G711_ALAW :
      AFclosefile (file);
      AFfreefilesetup (fileSetup);
      return "Unsupported compression method";
    #endif
    case AF_COMPRESSION_AWARE_MPEG :
      license = AUchecklicense
        (AU_LICENSE_AWARE_MPEG_DECODER,&licensevoid,&licenseString);
      if (license != AU_LICENSE_OK)
      {
        AFclosefile (file);
        AFfreefilesetup (fileSetup);
        return "MPEG decoder not available";
      }
      break;
    case AF_COMPRESSION_AWARE_MULTIRATE :
      license = AUchecklicense
        (AU_LICENSE_AWARE_MULTIRATE_DECODER,&licensevoid,&licenseString);
      if (license != AU_LICENSE_OK)
      {
        AFclosefile (file);
        AFfreefilesetup (fileSetup);
        return "Multirate decoder not available";
      }
      break;
    case AF_COMPRESSION_APPLE_ACE2 :
    case AF_COMPRESSION_APPLE_ACE8 :
    case AF_COMPRESSION_APPLE_MAC3 :
    case AF_COMPRESSION_APPLE_MAC6 :
      AFclosefile (file);
      AFfreefilesetup (fileSetup);
      return "Unsupported compression method";
    case AF_COMPRESSION_UNKNOWN :
    default :
      AFclosefile (file);
      AFfreefilesetup (fileSetup);
      return "Unknown compression method";
  }
  
  // Get AES data (if any)
  AESused = AFgetaeschanneldata (file,AF_DEFAULT_TRACK,AES);
  
  // Get audio track frame count (and calculate samples and bytes)
  frames  = AFgetframecnt (file,AF_DEFAULT_TRACK);
  samples = frames * channels;
  bytes   = samples * width;
  
  // Get audio track markers
  markerCount = AFgetmarkids (file,AF_DEFAULT_TRACK,NULL);
  if (markerCount == 0)
  {
    markerIDs    = 0;
    markerFrames = 0;
  }
  else
  {
    markerIDs    = new long [markerCount];
    markerFrames = new long [markerCount];
    AFgetmarkids (file,AF_DEFAULT_TRACK,markerIDs);
    for (long i=0; i<markerCount; i++)
    {
      markerFrames [i] = AFgetmarkpos (file,AF_DEFAULT_TRACK,markerIDs [i]);
    }
  }
  
  // Get instruments
  instCount = AFgetinstids (file,NULL);
  if (instCount == 0)
  {
    susLoopMode  = AF_LOOP_MODE_NOLOOP;
    susLoopStart = 0;
    susLoopEnd   = 0;
    relLoopMode  = AF_LOOP_MODE_NOLOOP;
    relLoopStart = 0;
    relLoopEnd   = 0;
  }
  else
  {
    long i;
    instIDs = new long [instCount];
    AFgetinstids (file,instIDs);
    
    // Search for default instrument
    long defaultFound = FALSE;
    for (i=0; i<instCount && !defaultFound; i++)
    {
      if (instIDs [i] == AF_DEFAULT_INST) defaultFound = TRUE;
    }
    
    // No longer need instrument IDs
    delete [] instIDs;
    
    if (!defaultFound)
    {
      susLoopMode  = AF_LOOP_MODE_NOLOOP;
      susLoopStart = 0;
      susLoopEnd   = 0;
      relLoopMode  = AF_LOOP_MODE_NOLOOP;
      relLoopStart = 0;
      relLoopEnd   = 0;
    }
    else
    {
      // Get loop IDs, modes, start IDs and end IDs
      susLoopID = 
        AFgetinstparamlong (file,AF_DEFAULT_INST,AF_INST_SUSLOOPID);
      relLoopID = 
        AFgetinstparamlong (file,AF_DEFAULT_INST,AF_INST_RELLOOPID);
      susLoopMode = AFgetloopmode (file,AF_DEFAULT_INST,susLoopID);
      relLoopMode = AFgetloopmode (file,AF_DEFAULT_INST,relLoopID);
      susStartID  = AFgetloopstart (file,AF_DEFAULT_INST,susLoopID);
      relStartID  = AFgetloopstart (file,AF_DEFAULT_INST,relLoopID);
      susEndID    = AFgetloopend (file,AF_DEFAULT_INST,susLoopID);
      relEndID    = AFgetloopend (file,AF_DEFAULT_INST,relLoopID);
      switch (susLoopMode)
      {
        case AF_LOOP_MODE_NOLOOP :
        case AF_LOOP_MODE_FORW :
        case AF_LOOP_MODE_FORWBAKW :
          break;
        default :
          susLoopMode = AF_LOOP_MODE_NOLOOP;
          break;
      }
      switch (relLoopMode)
      {
        case AF_LOOP_MODE_NOLOOP :
        case AF_LOOP_MODE_FORW :
        case AF_LOOP_MODE_FORWBAKW :
          break;
        default :
          relLoopMode = AF_LOOP_MODE_NOLOOP;
          break;
      }

      // Calculate frame positions of loop starts and ends
      long found1=FALSE;
      long found2=FALSE;
      long found3=FALSE;
      long found4=FALSE;
      for (i=0; i<markerCount; i++)
      {
        if (markerIDs [i] == susStartID)
        {
          susLoopStart = markerFrames [i];
          found1 = TRUE;
        }
        if (markerIDs [i] == susEndID)
        {
          susLoopEnd = markerFrames [i] + 1;
          found2 = TRUE;
        }
        if (markerIDs [i] == relStartID)
        {
          relLoopStart = markerFrames [i];
          found3 = TRUE;
        }
        if (markerIDs [i] == relEndID)
        {
          relLoopEnd = markerFrames [i] + 1;
          found4 = TRUE;
        }
      }

      // If sustain loop - need found1 and found2
      if (susLoopMode != AF_LOOP_MODE_NOLOOP)
      {
        if (!(found1 && found2))
        {
          susLoopMode  = AF_LOOP_MODE_NOLOOP;
          susLoopStart = 0;
          susLoopEnd   = 0;
        }
      }
      else
      {
        susLoopStart = 0;
        susLoopEnd   = 0;
      }

      // If release loop - need found3 and found4
      if (relLoopMode != AF_LOOP_MODE_NOLOOP)
      {
        if (!(found3 && found4))
        {
          relLoopMode  = AF_LOOP_MODE_NOLOOP;
          relLoopStart = 0;
          relLoopEnd   = 0;
        }
      }
      else
      {
        relLoopStart = 0;
        relLoopEnd   = 0;
      }
    }
  }
  
  // Delete allocations (not needed again)
  delete [] markerIDs;
  delete [] markerFrames;

  // Default to no range and display whole sample
  rangeValid   = FALSE;
  rangeStart   = 0;
  rangeEnd     = 0;
  displayStart = 0;
  displayEnd   = frames;
  redraw       = 0;
  currentPos   = -1;
 
  // Read in sample data
  data = new char [bytes];
  if (!data)
  {
    AFclosefile (file);
    AFfreefilesetup (fileSetup);
    return "Unable to allocate memory for data";
  }

  long actualRead = AFreadframes (file,AF_DEFAULT_TRACK,data,frames);
  if (actualRead != frames)
  {
    delete [] data;
    data = 0;
    AFclosefile (file);
    AFfreefilesetup (fileSetup);
    return "Error occured while reading sample frames";
  }

  // Copy filename across
  delete [] filename;
  filename = new char [strlen (newFilename) + 1];
  strcpy (filename,newFilename);
  
  // Get name, author, copyright and annotation strings
  miscCount = AFgetmiscids (file,NULL);
  if (miscCount)
  {
    miscIDs = new long [miscCount];
    AFgetmiscids (file,miscIDs);
    for (long i=0; i<miscCount; i++)
    {
      long miscType;
      long miscSize;
      miscType = AFgetmisctype (file,miscIDs [i]);
      miscSize = AFgetmiscsize (file,miscIDs [i]);
      switch (miscType)
      {
        case AF_MISC_AIFF_NAME :
          delete [] name;
          name = new char [miscSize + 1];
          AFreadmisc (file,miscIDs [i],name,miscSize);
          name [miscSize] = 0;
          break;
        case AF_MISC_AIFF_AUTH :
          delete [] author;
          author = new char [miscSize + 1];
          AFreadmisc (file,miscIDs [i],author,miscSize);
          author [miscSize] = 0;
          break;
        case AF_MISC_AIFF_COPY :
          delete [] copyright;
          copyright = new char [miscSize + 1];
          AFreadmisc (file,miscIDs [i],copyright,miscSize);
          copyright [miscSize] = 0;
          break;
        case AF_MISC_AIFF_ANNO :
          delete [] annotation;
          annotation = new char [miscSize + 1];
          AFreadmisc (file,miscIDs [i],annotation,miscSize);
          annotation [miscSize] = 0;
          break;
        default :
          break;
      }
    }
    delete [] miscIDs;
  }

  // Close file and free file setup
  AFclosefile (file);
  AFfreefilesetup (fileSetup);
  
  // Set valid and return
  valid = TRUE;
  return 0;
}
  
/*---------------------------------------------------------------------------
| PUBLIC METHOD DPSample::saveAIFF
---------------------------------------------------------------------------*/

char *DPSample::saveAIFF (
  char *newFilename,
  int rangeonly,
  int copysave,
  int replaceFilename)
{
//cerr << "DPSample::save" << endl;

  AFfilesetup   fileSetup;
  AFfilehandle  file;
  int           license;
  int           licensevoid;
  char          *licenseString;
  
  if (!valid)
    return "No valid sample to save";
  if (rangeonly && !getRangeValid ())
    return "No valid range to save";
  
  // Disable error handling
  AFseterrorhandler (DPSampleAudiofileError);

  // Initialise file setup
  fileSetup = AFnewfilesetup ();
  AFinitfilefmt (fileSetup,format);
  AFinitrate (fileSetup,AF_DEFAULT_TRACK,rate);
  AFinitsampfmt (fileSetup,AF_DEFAULT_TRACK,AF_SAMPFMT_TWOSCOMP,width * 8);
  AFinitchannels (fileSetup,AF_DEFAULT_TRACK,channels);
  
  if (AESused) AFinitaeschanneldata (fileSetup,AF_DEFAULT_TRACK);

  if (format == AF_FILE_AIFF && compression != AF_COMPRESSION_NONE)
    return "Cannot use compression with AIFF samples";

  if (format == AF_FILE_AIFFC)
  {
    switch (compression)
    {
      case AF_COMPRESSION_NONE :
        break;
      #ifndef LINUX
      case AF_COMPRESSION_G722 :
        if (width != 2)
        {
          AFfreefilesetup (fileSetup);
          return "G722 encoding can only be used on 16 bit samples";
        }
        AFinitcompression (fileSetup,AF_DEFAULT_TRACK,compression);
        break;
      case AF_COMPRESSION_G711_ULAW :
        if (width != 2)
        {
          AFfreefilesetup (fileSetup);
          return "G711 ulaw encoding can only be used on 16 bit samples";
        }
        AFinitcompression (fileSetup,AF_DEFAULT_TRACK,compression);
        break;
      case AF_COMPRESSION_G711_ALAW :
        if (width != 2)
        {
          AFfreefilesetup (fileSetup);
          return "G711 alaw encoding can only be used on 16 bit samples";
        }
        AFinitcompression (fileSetup,AF_DEFAULT_TRACK,compression);
        break;
      #else
      case AF_COMPRESSION_G722 :
      case AF_COMPRESSION_G711_ULAW :
      case AF_COMPRESSION_G711_ALAW :
        AFfreefilesetup (fileSetup);
        return "Unsupported compression method";
      #endif
      case AF_COMPRESSION_AWARE_MPEG :
      case AF_COMPRESSION_AWARE_DEFAULT_MPEG_I :
      case AF_COMPRESSION_AWARE_DEFAULT_MPEG_II :
        license = AUchecklicense
          (AU_LICENSE_AWARE_MPEG_ENCODER,&licensevoid,&licenseString);
        if (license != AU_LICENSE_OK)
        {
          AFfreefilesetup (fileSetup);
          return "MPEG encoder not available";
        }
        AFinitcompression (fileSetup,AF_DEFAULT_TRACK,compression);
        break;
      case AF_COMPRESSION_AWARE_MULTIRATE :
      case AF_COMPRESSION_AWARE_DEFAULT_MULTIRATE :
      case AF_COMPRESSION_AWARE_DEFAULT_LOSSLESS :
        license = AUchecklicense
          (AU_LICENSE_AWARE_MULTIRATE_ENCODER,&licensevoid,&licenseString);
        if (license != AU_LICENSE_OK)
        {
          AFfreefilesetup (fileSetup);
          return "Multirate encoder not available";
        }
        AFinitcompression (fileSetup,AF_DEFAULT_TRACK,compression);
        break;
      default :
        AFfreefilesetup (fileSetup);
        return "Unsupported compression method";
    }
  }

  long markerIDs [] = {1,2,3,4};
  AFinitmarkids (fileSetup,AF_DEFAULT_TRACK,markerIDs,4);
  AFinitmarkname (fileSetup,AF_DEFAULT_TRACK,1,"sustain loop start");
  AFinitmarkname (fileSetup,AF_DEFAULT_TRACK,2,"sustain loop end");
  AFinitmarkname (fileSetup,AF_DEFAULT_TRACK,3,"release loop start");
  AFinitmarkname (fileSetup,AF_DEFAULT_TRACK,4,"release loop end");
  
  long instIDs [] = {AF_DEFAULT_INST};
  AFinitinstids (fileSetup,instIDs,1);
  
  long loopIDs [] = {1,2};
  AFinitloopids (fileSetup,AF_DEFAULT_TRACK,loopIDs,2);
  
  char *newName       = "";
  char *newCopyright  = "";
  char *newAuthor     = "";
  char *newAnnotation = "";
  
  if (name)       newName       = name;
  if (copyright)  newCopyright  = copyright;
  if (author)     newAuthor     = author;
  if (annotation) newAnnotation = annotation;

  char *tempAnnotation = 0;
  
  if (!strstr (newAnnotation,ANNONEW))
  {
    tempAnnotation =
      new char [strlen (newAnnotation) + strlen (ANNONEW) + 1];
    strcpy (tempAnnotation,newAnnotation);
    strcpy (tempAnnotation + strlen (newAnnotation),ANNONEW);
    newAnnotation = tempAnnotation;
  }
  
  long miscIDs [] = {1,2,3,4};

  if (copysave)
    AFinitmiscids  (fileSetup,miscIDs,4);
  else
    AFinitmiscids  (fileSetup,miscIDs,3);

  AFinitmisctype (fileSetup,1,AF_MISC_AIFF_NAME);
  AFinitmisctype (fileSetup,2,AF_MISC_AIFF_AUTH);
  AFinitmisctype (fileSetup,3,AF_MISC_AIFF_ANNO);
  if (copysave)
    AFinitmisctype (fileSetup,4,AF_MISC_AIFF_COPY);
  AFinitmiscsize (fileSetup,1,strlen (newName));
  AFinitmiscsize (fileSetup,2,strlen (newAuthor));
  AFinitmiscsize (fileSetup,3,strlen (newAnnotation));
  if (copysave)
    AFinitmiscsize (fileSetup,4,strlen (newCopyright));

  if (newFilename) file = AFopenfile (newFilename,"w",fileSetup);
  else file = AFopenfile (filename,"w",fileSetup);

  if (!file)
  {
    AFfreefilesetup (fileSetup);
    delete [] tempAnnotation;
    return "Unable to open file for writing";
  }
  
  if (newFilename && replaceFilename)
  {
    delete [] filename;
    filename = new char [strlen (newFilename) + 1];
    strcpy (filename,newFilename);
  }

  if (!rangeonly)
  {
    AFsetmarkpos (file,AF_DEFAULT_TRACK,1,susLoopStart);
    AFsetmarkpos (file,AF_DEFAULT_TRACK,2,susLoopEnd-1);
    AFsetmarkpos (file,AF_DEFAULT_TRACK,3,relLoopStart);
    AFsetmarkpos (file,AF_DEFAULT_TRACK,4,relLoopEnd-1);
  }
  else
  {
    AFsetmarkpos (file,AF_DEFAULT_TRACK,1,0);
    AFsetmarkpos (file,AF_DEFAULT_TRACK,2,0);
    AFsetmarkpos (file,AF_DEFAULT_TRACK,3,0);
    AFsetmarkpos (file,AF_DEFAULT_TRACK,4,0);
  }

  if (AESused) AFsetaeschanneldata (file,AF_DEFAULT_TRACK,AES);
  
  if (!rangeonly)
  {
    AFwriteframes (file,AF_DEFAULT_TRACK,data,frames);

    if (extra)
    {
      extraListEntry *e;
      e = extra;
      while (e)
      {
        AFwriteframes (file,AF_DEFAULT_TRACK,e->data,e->end - e->start + 1);
        e = e->next;
      }
    }
  }
  else
  {
    char *dataRangeStart;
    long dataRangeLen;
    
    dataRangeStart = (char *) data + (rangeStart * channels * width);
    dataRangeLen   = rangeEnd - rangeStart;
    
    AFwriteframes (file,AF_DEFAULT_TRACK,dataRangeStart,dataRangeLen);
  }
  
  if (!rangeonly)
  {  
    AFsetinstparamlong (file,AF_DEFAULT_INST,AF_INST_SUSLOOPID,1);
    AFsetinstparamlong (file,AF_DEFAULT_INST,AF_INST_RELLOOPID,2);
    AFsetloopmode  (file,AF_DEFAULT_INST,1,susLoopMode);
    AFsetloopmode  (file,AF_DEFAULT_INST,2,relLoopMode);
    AFsetloopstart (file,AF_DEFAULT_INST,1,1);
    AFsetloopend   (file,AF_DEFAULT_INST,1,2);
    AFsetloopstart (file,AF_DEFAULT_INST,2,3);
    AFsetloopend   (file,AF_DEFAULT_INST,2,4);
  }
  else
  {  
    AFsetinstparamlong (file,AF_DEFAULT_INST,AF_INST_SUSLOOPID,1);
    AFsetinstparamlong (file,AF_DEFAULT_INST,AF_INST_RELLOOPID,2);
    AFsetloopmode  (file,AF_DEFAULT_INST,1,AF_LOOP_MODE_NOLOOP);
    AFsetloopmode  (file,AF_DEFAULT_INST,2,AF_LOOP_MODE_NOLOOP);
    AFsetloopstart (file,AF_DEFAULT_INST,1,1);
    AFsetloopend   (file,AF_DEFAULT_INST,1,2);
    AFsetloopstart (file,AF_DEFAULT_INST,2,3);
    AFsetloopend   (file,AF_DEFAULT_INST,2,4);
  }
 
  AFwritemisc (file,1,newName,strlen (newName));
  AFwritemisc (file,2,newAuthor,strlen (newAuthor));
  AFwritemisc (file,3,newAnnotation,strlen (newAnnotation));
  if (copysave)
    AFwritemisc (file,4,newCopyright,strlen (newCopyright));
  
  AFclosefile (file);
  AFfreefilesetup (fileSetup);
  delete [] tempAnnotation;
  
  changed = FALSE;
  return 0;
}
  
/*---------------------------------------------------------------------------
| PUBLIC METHOD DPSample::play
---------------------------------------------------------------------------*/

char *DPSample::play (long playMode,FL_OBJECT *object)
{
//cerr << "DPSample::play" << endl;

  // Range     = mode 0
  // Display   = mode 1
  // All       = mode 2
  
  ALconfig audioConfig;
  ALport   audioPort;
  long     queueSize;

  if (!valid)
  {
    // No valid sample (not an error)
    return 0;
  }

  if (playMode == 0 && !rangeValid)
  {
    // No valid sample range (not an error)
    return 0;
  }

  long   longPlayRate;
  double playRate;
  
  longPlayRate = getOutputRate ();
  if (longPlayRate == AL_RATE_UNDEFINED)
    playRate = rate;
  else
    playRate = longPlayRate;

  // Disable error handler
  ALseterrorhandler (0);

  // Set up new audio configuration 
  audioConfig = ALnewconfig ();
  if (!audioConfig)
    return "Unable to create new audio configuration";
  
  // Set sample format and width
  ALsetsampfmt (audioConfig,AL_SAMPFMT_TWOSCOMP);
  if (width == 1) ALsetwidth (audioConfig,AL_SAMPLE_8);
  else ALsetwidth (audioConfig,AL_SAMPLE_16);

  // Set number of channels
  ALsetchannels (audioConfig,channels);

  // Set queue size (PLAYQUEUETIME seconds at playback rate)
  queueSize = (long) (playRate * channels * PLAYQUEUETIME);
  limitQueueSize (channels,&queueSize);

  if (ALsetqueuesize (audioConfig,queueSize))
  {
    ALfreeconfig (audioConfig);
    return "Unable to establish audio queue";
  }

  // Open new audio port with given configuration
  audioPort = ALopenport (PORTNAME,"w",audioConfig);
  if (!audioPort)
  {
    ALfreeconfig (audioConfig);
    return "Unable to open audio port";
  }
  
  // Play the sample
  long playStart;
  long playEnd;
  long susLoopIncrement = 1;
  long relLoopIncrement = 1;
  long tempSusLoopMode  = susLoopMode;
  long tempRelLoopMode  = relLoopMode;
  DPSampleInc           = 1;
  DPSampleStop          = FALSE;
  DPSampleRelease1      = FALSE;
  DPSampleRelease2      = FALSE;

  if (playMode == 0)
  {
    playStart = rangeStart;
    if (rangeStart == rangeEnd) playEnd = frames;
    else playEnd = rangeEnd;
  }
  else if (playMode == 1)
  {
    playStart = displayStart;
    playEnd = displayEnd;
  }
  else
  {
    playStart = 0;
    playEnd = frames;
  }
  
  if (playEnd <= playStart)
  {
    // No sample data to play (not an error)
    ALcloseport (audioPort);
    ALfreeconfig (audioConfig);
    return 0;
  }
  
  #ifdef NOFORMS
  #ifndef LINUX
  sigset (SIGTSTP,DPSampleReleaseCatch);
  sigset (SIGINT,DPSampleStopCatch);
  #else
  signal (SIGTSTP,DPSampleReleaseCatch);
  signal (SIGINT,DPSampleStopCatch);
  #endif
  #endif

  // No looping if playing range or display
  if (playMode == 0 || playMode == 1)
  {
    tempSusLoopMode = AF_LOOP_MODE_NOLOOP;
    tempRelLoopMode = AF_LOOP_MODE_NOLOOP;
  }
  else
  {
    // Catch special case (loops of duration zero and one)
    if (susLoopStart == susLoopEnd) tempSusLoopMode = AF_LOOP_MODE_NOLOOP;
    if (relLoopStart == relLoopEnd) tempRelLoopMode = AF_LOOP_MODE_NOLOOP;
    if (susLoopStart == susLoopEnd - 1) susLoopIncrement = 0;
    if (relLoopStart == relLoopEnd - 1) relLoopIncrement = 0;
  }
  
  long i          = playStart;
  long checkReset = (long) (playRate * CHECKTIME);
  long check      = 1;
  long lineReset  = (long) (playRate * LINETIME);
  long line       = 1;
  long drawReset  = (long) (playRate * DRAWTIME);
  long draw       = 1;
  long oldDisplay = -1;

  #ifdef SNEAKYCHECK
  XEvent newEvent;
  #endif

  while (i >= playStart && i < playEnd && !DPSampleStop)
  {
    ALwritesamps
      (audioPort,((char *) data) + (i * width * channels),channels);

    if (!DPSampleRelease1)
    {
      if (tempSusLoopMode == AF_LOOP_MODE_FORW && i == susLoopEnd - 1)
        i = susLoopStart - 1;
      else if (tempSusLoopMode == AF_LOOP_MODE_FORWBAKW)
      {
        if (i == susLoopEnd - 1) DPSampleInc = -susLoopIncrement;
        else if (i == susLoopStart) DPSampleInc = susLoopIncrement;
      }
    }
    else if (!DPSampleRelease2)
    {
      if (tempRelLoopMode == AF_LOOP_MODE_FORW && i == relLoopEnd - 1)
        i = relLoopStart - 1;
      else if (tempRelLoopMode == AF_LOOP_MODE_FORWBAKW)
      {
        if (i == relLoopEnd - 1) DPSampleInc = -relLoopIncrement;
        else if (i == relLoopStart) DPSampleInc = relLoopIncrement;
      }
    }
    
    currentPos = i;
    
    if (object)
    {
      if (!--check)
      {
        check = checkReset;
        #ifndef NOFORMS
        #ifdef SNEAKYCHECK
        if (XCheckWindowEvent (fl_display,
          playForm->playForm->window,~(long)0,&newEvent) ||
          XCheckWindowEvent (fl_display,
          mixerForm->mixerForm->window,~(long)0,&newEvent))
        {
          XPutBackEvent (fl_display,&newEvent);
          fl_check_forms ();
        }
        #else
        fl_check_forms ();
        #endif // SNEAKYCHECK
        #endif // NOFORMS
      }
      if (!--line)
      {
        line = lineReset;
        redraw = 1;
        #ifndef NOFORMS
        fl_redraw_object (object);
        #endif
        redraw = 0;
      }
      if (!--draw)
      {
        draw = drawReset;
        if (displayStart != oldDisplay)
        {
          #ifndef NOFORMS
          fl_redraw_object (mainForm->scrollBarSlider);
          updateDisplayDetails ();
          #endif
          oldDisplay = displayStart;
        }
      }
    }
    i += DPSampleInc;
  }
  
  #ifdef NOFORMS
  #ifndef LINUX
  sigset (SIGTSTP,SIG_DFL);
  sigset (SIGINT,SIG_DFL);
  #else
  signal (SIGTSTP,SIG_DFL);
  signal (SIGINT,SIG_DFL);
  #endif
  #endif

  // Wait until port cleared 
  if (!DPSampleStop)
  {
    while (ALgetfilled (audioPort))
    {
      #ifdef LINUX
      long tempBufOut [4] = {0,0,0,0};
      ALwritesamps
        (audioPort,tempBufOut,channels);
      #endif
    }
  }

  // Close the port, free the configuration and return
  ALcloseport (audioPort);
  ALfreeconfig (audioConfig);  
  currentPos = -1;
  if (object)
  {
    #ifndef NOFORMS
    redraw = 1;
    fl_redraw_object (object);
    redraw = 0;
    fl_redraw_object (mainForm->scrollBarSlider);
    updateDisplayDetails ();
    #endif
  }

  return 0;
}

/*---------------------------------------------------------------------------
| PUBLIC METHOD DPSample::record
---------------------------------------------------------------------------*/

char *DPSample::record (long recordMode,long trigger,FL_OBJECT *object)
{
  ALconfig audioConfig;
  ALport   audioPort;
  long     queueSize;
  
//cerr << "DPSample::record" << endl;

  if (!valid)
  {
    // No valid sample (not an error)
    return 0;
  }

  if (recordMode == 0 && !rangeValid)
  {
    // No valid sample range (not an error)
    return 0;
  }

  long   longRecordRate;
  double recordRate;
  
  longRecordRate = getInputRate ();
  if (longRecordRate == AL_RATE_UNDEFINED)
    recordRate = rate;
  else
    recordRate = longRecordRate;

  // Disable error handler
  ALseterrorhandler (0);

  // Set up new audio configuration 
  audioConfig = ALnewconfig ();
  if (!audioConfig)
    return "Unable to create new audio configuration";
  
  // Set sample format and width
  ALsetsampfmt (audioConfig,AL_SAMPFMT_TWOSCOMP);
  if (width == 1) ALsetwidth (audioConfig,AL_SAMPLE_8);
  else ALsetwidth (audioConfig,AL_SAMPLE_16);

  // Set number of channels
  ALsetchannels (audioConfig,channels);

  // Set queue size (RECORDQUEUETIME seconds at record rate)
  queueSize = (long) (recordRate * channels * RECORDQUEUETIME);
  limitQueueSize (channels,&queueSize);
  
  if (ALsetqueuesize (audioConfig,queueSize))
  {
    ALfreeconfig (audioConfig);
    return "Unable to establish audio queue";
  }

  // Open new audio port with given configuration
  audioPort = ALopenport (PORTNAME,"r",audioConfig);
  if (!audioPort)
  {
    ALfreeconfig (audioConfig);
    return "Unable to open audio port";
  }
  
  // Record the sample
  long recordStart;
  long recordEnd;
  DPSampleStop = FALSE;

  if (recordMode == 0)
  {
    recordStart = rangeStart;
    if (rangeStart == rangeEnd) recordEnd = frames;
    else recordEnd = rangeEnd;
  }
  else if (recordMode == 1)
  {
    recordStart = displayStart;
    recordEnd = displayEnd;
  }
  else
  {
    recordStart = 0;
    recordEnd = frames;
  }
  
  if (recordEnd <= recordStart)
  {
    // No sample data to record (not an error)
    ALcloseport (audioPort);
    ALfreeconfig (audioConfig);
    return 0;
  }
  
  #ifdef NOFORMS
  #ifndef LINUX
  sigset (SIGINT,DPSampleStopCatch);
  #else
  signal (SIGINT,DPSampleStopCatch);
  #endif
  #endif

  long i          = recordStart;
  long checkReset = (long) (recordRate * CHECKTIME);
  long check      = 1;
  long lineReset  = (long) (recordRate * LINETIME);
  long line       = 1;
  long drawReset  = (long) (recordRate * DRAWTIME);
  long draw       = 1;
  long oldDisplay = -1;
  char *tempbuf   = new char [channels * width];
  long trigval    = trigger;
  long usingExtra = FALSE;
  extraListEntry *e;

  #ifdef SNEAKYCHECK
  XEvent newEvent;
  #endif
  
  removeExtra ();
  
  if (width == 2) trigval *= 256;
  
  if (trigval >= 0)
  {
    long trigset = 0;
    while (!trigset && !DPSampleStop)
    {
      ALreadsamps (audioPort,(char *) tempbuf,channels);
      if (width == 1)
      {
        if (((signed char *) tempbuf) [0] >= trigval ||
        ((signed char *) tempbuf) [0] <= -trigval)
          trigset = 1;
        if ((channels == 2  || channels == 4) &&
        (((signed char *) tempbuf) [1] >= trigval ||
        ((signed char *) tempbuf) [1] <= -trigval))
          trigset = 1;
        if (channels == 4 &&
        (((signed char *) tempbuf) [2] >= trigval ||
        ((signed char *) tempbuf) [2] <= -trigval))
          trigset = 1;
        if (channels == 4 &&
        (((signed char *) tempbuf) [3] >= trigval ||
        ((signed char *) tempbuf) [3] <= -trigval))
          trigset = 1;
      }
      else
      {
        if (((short *) tempbuf) [0] >= trigval ||
        ((short *) tempbuf) [0] <= -trigval)
          trigset = 1;
        if ((channels == 2 || channels == 4)
        && (((short *) tempbuf) [1] >= trigval ||
        ((short *) tempbuf) [1] <= -trigval))
          trigset = 1;
        if (channels == 4
        && (((short *) tempbuf) [2] >= trigval ||
        ((short *) tempbuf) [2] <= -trigval))
          trigset = 1;
        if (channels == 4
        && (((short *) tempbuf) [3] >= trigval ||
        ((short *) tempbuf) [3] <= -trigval))
          trigset = 1;
      }
      if (object)
      {
        if (!--check)
        {
          check = checkReset;
          #ifndef NOFORMS
          #ifdef SNEAKYCHECK
          if (XCheckWindowEvent (fl_display,
            recordForm->recordForm->window,~(long)0,&newEvent) ||
            XCheckWindowEvent (fl_display,
            mixerForm->mixerForm->window,~(long)0,&newEvent))
          {
            XPutBackEvent (fl_display,&newEvent);
            fl_check_forms ();
          }
          #else
          fl_check_forms ();
          #endif // SNEAKYCHECK
          #endif // NOFORMS
        }
        if (!--line)
        {
          line = lineReset;
          redraw = 1;
          #ifndef NOFORMS
          fl_redraw_object (object);
          #endif
          redraw = 0;
        }
        if (!--draw)
        {
          draw = drawReset;
          redraw = 2;
          #ifndef NOFORMS
          fl_redraw_object (object);
          #endif
          redraw = 0;
          if (displayStart != oldDisplay)
          {
            #ifndef NOFORMS
            fl_redraw_object (mainForm->scrollBarSlider);
            updateDisplayDetails ();
            #endif
            oldDisplay = displayStart;
          }
        }
      }
    }
  }

  while (i >= recordStart && i <= recordEnd && !DPSampleStop)
  {
    if (i == recordEnd)
    {
      if (recordMode == 2 && (e = expand (1.0)))
      {
        usingExtra = TRUE;
        recordEnd = e->end + 1;
      }
      else
      {
        DPSampleStop = TRUE;
        break;
      }
    }
    
    if (!usingExtra)
    {
      ALreadsamps
        (audioPort,((char *) data) + (i * width * channels),channels);
      currentPos = i;
    }
    else
    {
      ALreadsamps
        (audioPort,((char *) e->data) +
          ((i - e->start) * width * channels),channels);
    }
    
    if (object)
    {
      if (!--check)
      {
        check = checkReset;
        #ifndef NOFORMS
        #ifdef SNEAKYFORMS
        if (XCheckWindowEvent (fl_display,
          recordForm->recordForm->window,~(long)0,&newEvent) ||
          XCheckWindowEvent (fl_display,
          mixerForm->mixerForm->window,~(long)0,&newEvent))
        {
          XPutBackEvent (fl_display,&newEvent);
          fl_check_forms ();
        }
        #else
        fl_check_forms ();
        #endif // SNEAKYCHECK
        #endif // NOFORMS
      }
      if (!--line)
      {
        line = lineReset;
        redraw = 1;
        #ifndef NOFORMS
        fl_redraw_object (object);
        #endif
        redraw = 0;
      }
      if (!--draw)
      {
        draw = drawReset;
        redraw = 2;
        #ifndef NOFORMS
        fl_redraw_object (object);
        #endif
        redraw = 0;
        if (displayStart != oldDisplay)
        {
          #ifndef NOFORMS
          fl_redraw_object (mainForm->scrollBarSlider);
          updateDisplayDetails ();
          #endif
          oldDisplay = displayStart;
        }
      }
    }
    i++;
  }
  
  #ifdef NOFORMS
  #ifndef LINUX  
  sigset (SIGINT,SIG_DFL);
  #else
  signal (SIGINT,SIG_DFL);
  #endif
  #endif

  // Close the port, free the configuration and return
  ALcloseport (audioPort);
  ALfreeconfig (audioConfig);  
  currentPos = -1;
  if (object)
  {
    #ifndef NOFORMS
    redraw = 2;
    fl_redraw_object (object);
    redraw = 0;
    fl_redraw_object (mainForm->scrollBarSlider);
    updateDisplayDetails ();
    #endif
  }
  
  delete tempbuf;

  // Adjust DC for calibration between recordStart and i
  long rangeValidBak = rangeValid;
  long rangeStartBak = rangeStart;
  long rangeEndBak   = rangeEnd;
  rangeValid = TRUE;
  rangeStart = recordStart;
  rangeEnd   = i - 1;
  #ifndef NOFORMS
  if (recordAdjustLeft != 0.0)
    adjustDC ((long) recordAdjustLeft,0);
  if ((channels == 2 || channels == 4) && recordAdjustRight != 0.0)
    adjustDC ((long) recordAdjustRight,1);
  if (channels == 4 && recordAdjustThree != 0.0)
    adjustDC ((long) recordAdjustThree,2);
  if (channels == 4 && recordAdjustFour != 0.0)
    adjustDC ((long) recordAdjustFour,3);
  #endif
  setRange (rangeStartBak,rangeEndBak);
  rangeValid = rangeValidBak;

  changed = TRUE;
  return 0;
}
  
/*---------------------------------------------------------------------------
| PUBLIC METHOD DPSample::clear
---------------------------------------------------------------------------*/

void DPSample::clear ()
{
//cerr << "DPSample::clear" << endl;

  // Note by default clear resets changed to FALSE
  // (this may not be as intended - use setChanged if necessary)
  
  delete [] filename;
  delete [] name;
  delete [] author;
  delete [] copyright;
  delete [] annotation;
  delete [] data;
  valid        = FALSE;
  changed      = FALSE;
  format       = AF_FILE_AIFF;
  compression  = AF_COMPRESSION_NONE;
  rate         = 44100.0;
  width        = 1;
  channels     = 1;
  frames       = 0;
  samples      = 0;
  bytes        = 0;
  AESused      = 0;
  for (int i=0; i<24; i++)
    AES [i] = 0;
  filename     = 0;
  name         = 0;
  copyright    = 0;
  author       = 0;
  annotation   = 0;
  susLoopMode  = AF_LOOP_MODE_NOLOOP;
  susLoopStart = 0;
  susLoopEnd   = 0;
  relLoopMode  = AF_LOOP_MODE_NOLOOP;
  relLoopStart = 0;
  relLoopEnd   = 0;
  rangeValid   = FALSE;
  rangeStart   = 0;
  rangeEnd     = 0;
  displayStart = 0;
  displayEnd   = 0;
  redraw       = 0;
  currentPos   = -1;
  data         = 0;
  removeExtra ();
}

/*---------------------------------------------------------------------------
| PUBLIC METHOD DPSample::fresh
---------------------------------------------------------------------------*/

long DPSample::fresh (
  double newRate,
  long newWidth,
  long newChannels,
  long newFrames)
{
//cerr << "DPSample::fresh" << endl;

  clear ();
  rate         = newRate;
  width        = newWidth;
  channels     = newChannels;
  frames       = newFrames;
  samples      = frames * width;
  bytes        = samples * channels;
  
  if (!frames)
  {
    displayStart = 0;
    displayEnd   = frames;
    valid        = FALSE;
    return NOERROR;
  }
  
  data = new char [bytes];
  if (data)
  {
    for (long i=0; i<bytes; i++)
      ((char *) data) [i] = 0;
    displayStart = 0;
    displayEnd   = frames;
    valid        = TRUE;
    return NOERROR;
  }
  return ERROR;
}

/*---------------------------------------------------------------------------
| Range functions
---------------------------------------------------------------------------*/

void DPSample::rangeClear ()
{
//cerr << "DPSample::rangeClear" << endl;
  rangeValid = FALSE;
  rangeStart = 0;
  rangeEnd   = 0;
}

void DPSample::rangeDisplay ()
{
//cerr << "DPSample::rangeDisplay" << endl;
  rangeValid = TRUE;
  rangeStart = displayStart;
  rangeEnd   = displayEnd;
}

void DPSample::rangeAll ()
{
//cerr << "DPSample::rangeAll" << endl;
  rangeValid = TRUE;
  rangeStart = 0;
  rangeEnd   = frames;
}

void DPSample::setRangeSusLoop ()
{
//cerr << "DPSample::setRangeSusLoop" << endl;
  if (susLoopMode != AF_LOOP_MODE_NOLOOP)
  {
    rangeValid = TRUE;
    rangeStart = susLoopStart;
    rangeEnd = susLoopEnd;
  }
}

void DPSample::setRangeRelLoop ()
{
//cerr << "DPSample::setRangeRelLoop" << endl;
  if (relLoopMode != AF_LOOP_MODE_NOLOOP)
  {
    rangeValid = TRUE;
    rangeStart = relLoopStart;
    rangeEnd = relLoopEnd;
  }
}

void DPSample::setSusLoopRange ()
{
//cerr << "DPSample::setSusLoopRange" << endl;
  if (rangeValid)
  {
    if (susLoopMode == AF_LOOP_MODE_NOLOOP)
      susLoopMode = AF_LOOP_MODE_FORW;
    susLoopStart = rangeStart;
    susLoopEnd = rangeEnd;
    changed = TRUE;
  }
}

void DPSample::setRelLoopRange ()
{
//cerr << "DPSample::setRelLoopRange" << endl;
  if (rangeValid)
  {
    if (relLoopMode == AF_LOOP_MODE_NOLOOP)
      relLoopMode = AF_LOOP_MODE_FORW;
    relLoopStart = rangeStart;
    relLoopEnd = rangeEnd;
    changed = TRUE;
  }
}

/*---------------------------------------------------------------------------
| Zoom functions
---------------------------------------------------------------------------*/

void DPSample::zoomIn ()
{
//cerr << "DPSample::zoomIn" << endl;
  long displayWidth = displayEnd - displayStart;
  long newDisplayWidth = (long) (displayWidth / ZOOMFACTOR);
  if (newDisplayWidth == displayWidth) newDisplayWidth -= 2;
  if (newDisplayWidth < 1) newDisplayWidth = 1;
  zoomWidth (newDisplayWidth);
}

void DPSample::zoomOut ()
{
//cerr << "DPSample::zoomOut" << endl;
  long displayWidth = displayEnd - displayStart;
  long newDisplayWidth = (long) (displayWidth * ZOOMFACTOR);
  if (newDisplayWidth == displayWidth) newDisplayWidth += 2;
  zoomWidth (newDisplayWidth);
}

void DPSample::zoomAll ()
{
//cerr << "DPSample::zoomAll" << endl;
  displayStart = 0;
  displayEnd = frames;
}

void DPSample::zoomRange ()
{
//cerr << "DPSample::zoomRange" << endl;
  if (rangeValid)
  {
    displayStart = rangeStart;
    displayEnd = rangeEnd;
  }
}

void DPSample::zoomWidth (long width)
{
//cerr << "DPSample::zoomWidth" << endl;
  long newDisplayStart;
  long newDisplayEnd;
  long displayWidth;
  displayWidth = displayEnd - displayStart;
  newDisplayStart = (long) (displayStart + ((displayWidth - width) / 2.0));
  newDisplayEnd   = (long) (displayStart + ((displayWidth + width) / 2.0));
  if (newDisplayStart < 0)
  {
    newDisplayEnd   -= newDisplayStart;
    newDisplayStart = 0;
    if (newDisplayEnd > frames) newDisplayEnd = frames;
  }
  else if (newDisplayEnd > frames)
  {
    newDisplayStart -= (newDisplayEnd - frames);
    newDisplayEnd = frames;
    if (newDisplayStart < 0) newDisplayStart = 0;
  }
  displayStart = newDisplayStart;
  displayEnd   = newDisplayEnd;
}    

/*---------------------------------------------------------------------------
| Scroll functions
---------------------------------------------------------------------------*/

void DPSample::scrollLstart ()
{
//cerr << "DPSample::scrollLstart" << endl;
  long displayWidth;
  displayWidth = displayEnd - displayStart;
  displayStart = 0;
  displayEnd   = displayWidth;
}

void DPSample::scrollL (long scrollFrames)
{
//cerr << "DPSample::scrollL" << endl;
  long newDisplayStart;
  long displayWidth;
  displayWidth = displayEnd - displayStart;
  newDisplayStart = displayStart - scrollFrames;
  if (newDisplayStart < 0) newDisplayStart = 0;
  displayStart = newDisplayStart;
  displayEnd   = newDisplayStart + displayWidth;
}

void DPSample::scrollR (long scrollFrames)
{
//cerr << "DPSample::scrollR" << endl;
  long newDisplayEnd;
  long displayWidth;
  displayWidth = displayEnd - displayStart;
  newDisplayEnd = displayEnd + scrollFrames;
  if (newDisplayEnd > frames) newDisplayEnd = frames;
  displayStart = newDisplayEnd - displayWidth;
  displayEnd   = newDisplayEnd;
}


void DPSample::scrollRend ()
{
//cerr << "DPSample::scrollRend" << endl;
  long displayWidth;
  displayWidth = displayEnd - displayStart;
  displayStart = frames - displayWidth;
  displayEnd   = frames;
}

/*---------------------------------------------------------------------------
| Conversion functions
---------------------------------------------------------------------------*/

void DPSample::to8bit ()
{
  if (!valid || width == 1) return;
  DPSample newSamp;
  if (!newSamp.fresh (rate,1,channels,frames)) return;
  
  long i;
  long temp;
  long stereo = (channels == 2);
  long quadro = (channels == 4);
  
  for (i=0; i<frames; i++)
  {
    temp = getFrame24 (i,0);
    newSamp.setFrame24 (i,0,temp);
    if (stereo || quadro)
    {
      temp = getFrame24 (i,1);
      newSamp.setFrame24 (i,1,temp);
    }
    if (quadro)
    {
      temp = getFrame24 (i,2);
      newSamp.setFrame24 (i,2,temp);
      temp = getFrame24 (i,3);
      newSamp.setFrame24 (i,3,temp);
    }
  }

  void *tempPointer = data;
  data = newSamp.data;
  newSamp.data = tempPointer;

  width   = 1;
  samples = frames * channels;
  bytes   = samples * width;
  changed = TRUE;
}

void DPSample::to16bit ()
{
  if (!valid || width == 2) return;
  DPSample newSamp;
  if (!newSamp.fresh (rate,2,channels,frames)) return;
  
  long i;
  long temp;
  long stereo = (channels == 2);
  long quadro = (channels == 4);
  
  for (i=0; i<frames; i++)
  {
    temp = getFrame24 (i,0);
    newSamp.setFrame24 (i,0,temp);
    if (stereo || quadro)
    {
      temp = getFrame24 (i,1);
      newSamp.setFrame24 (i,1,temp);
    }
    if (quadro)
    {
      temp = getFrame24 (i,2);
      newSamp.setFrame24 (i,2,temp);
      temp = getFrame24 (i,3);
      newSamp.setFrame24 (i,3,temp);
    }
  }

  void *tempPointer = data;
  data = newSamp.data;
  newSamp.data = tempPointer;

  width   = 2;
  samples = frames * channels;
  bytes   = samples * width;
  changed = TRUE;
}

void DPSample::toMono (int convertMode)
{
  // From Stereo
  // convertMode == 0 for left only
  // convertMode == 1 for left mixed with right (& three & four)
  // convertMode == 2 for right only

  // From Quadro
  // convertMode == Binary number 1 to 15 for channels 0 to 3
  
  if (!valid || channels == 1) return;
  DPSample newSamp;
  if (!newSamp.fresh (rate,width,1,frames)) return;

  long i;
  long temp;
  
  if (channels == 2)
  {
    // From Stereo
    for (i=0; i<frames; i++)
    {
      if (convertMode == 1)
      {
        // From stereo
        temp = getFrame24 (i,0) +  getFrame24 (i,1);
        temp = temp / 2;
      }
      else if (convertMode == 0)
        temp = getFrame24 (i,0);
      else
        temp = getFrame24 (i,1);
      newSamp.setFrame24 (i,0,temp);
    }
  }
  else
  {
    // From Quadro
    if (convertMode == 0) convertMode = 15;
    long use1;
    long use2;
    long use3;
    long use4;
    long n = 0;
    use1 = convertMode % 2;
    convertMode = convertMode / 2;
    use2 = convertMode % 2;
    convertMode = convertMode / 2;
    use3 = convertMode % 2;
    convertMode = convertMode / 2;
    use4 = convertMode % 2;
    if (use1) n++;
    if (use2) n++;
    if (use3) n++;
    if (use4) n++;
    for (i=0; i<frames; i++)
    {
      temp = 0;
      if (use1) temp += getFrame24 (i,0);
      if (use2) temp += getFrame24 (i,1);
      if (use3) temp += getFrame24 (i,2);
      if (use4) temp += getFrame24 (i,3);
      if (n) temp = temp / n;
      newSamp.setFrame24 (i,0,temp);
    }
  }

  void *tempPointer = data;
  data = newSamp.data;
  newSamp.data = tempPointer;

  channels = 1;
  samples  = frames * channels;
  bytes    = samples * width;
  changed  = TRUE;
}

void DPSample::toStereo (int convertMode)
{
  // From Quadro
  // convertMode == Binary number 1 to 255
  // for channels 0 to 3 (L) 0 to 3 (R)
  
  if (!valid || channels == 2) return;
  DPSample newSamp;
  if (!newSamp.fresh (rate,width,2,frames)) return;

  long i;
  long temp;
  
  if (channels == 1)
  {
    // From mono
    for (i=0; i<frames; i++)
    {
      temp = getFrame24 (i,0);
      newSamp.setFrame24 (i,0,temp);
      newSamp.setFrame24 (i,1,temp);
    }
  }
  else
  {
    // From Quadro
    if (convertMode == 0) convertMode = 255;
    long use1;
    long use2;
    long use3;
    long use4;
    long n;
    n = 0;
    use1 = convertMode % 2;
    convertMode = convertMode / 2;
    use2 = convertMode % 2;
    convertMode = convertMode / 2;
    use3 = convertMode % 2;
    convertMode = convertMode / 2;
    use4 = convertMode % 2;
    convertMode = convertMode / 2;
    if (use1) n++;
    if (use2) n++;
    if (use3) n++;
    if (use4) n++;
    for (i=0; i<frames; i++)
    {
      temp = 0;
      if (use1) temp += getFrame24 (i,0);
      if (use2) temp += getFrame24 (i,1);
      if (use3) temp += getFrame24 (i,2);
      if (use4) temp += getFrame24 (i,3);
      if (n) temp = temp / n;
      newSamp.setFrame24 (i,0,temp);
    }
    
    n = 0;
    use1 = convertMode % 2;
    convertMode = convertMode / 2;
    use2 = convertMode % 2;
    convertMode = convertMode / 2;
    use3 = convertMode % 2;
    convertMode = convertMode / 2;
    use4 = convertMode % 2;
    if (use1) n++;
    if (use2) n++;
    if (use3) n++;
    if (use4) n++;
    for (i=0; i<frames; i++)
    {
      temp = 0;
      if (use1) temp += getFrame24 (i,0);
      if (use2) temp += getFrame24 (i,1);
      if (use3) temp += getFrame24 (i,2);
      if (use4) temp += getFrame24 (i,3);
      if (n) temp = temp / n;
      newSamp.setFrame24 (i,1,temp);
    }
  }

  void *tempPointer = data;
  data = newSamp.data;
  newSamp.data = tempPointer;

  channels = 2;
  samples  = frames * channels;
  bytes    = samples * width;
  changed  = TRUE;
}

void DPSample::toQuadro (int convertMode)
{
  // From Stereo
  // convertMode == Binary number 1 to 255
  // for channels L & R (0) L & R (1) L & R (2) L & R (3)
  
  if (!valid || channels == 4) return;
  DPSample newSamp;
  if (!newSamp.fresh (rate,width,4,frames)) return;

  long i;
  long temp;
  
  if (channels == 1)
  {
    // From mono
    for (i=0; i<frames; i++)
    {
      temp = getFrame24 (i,0);
      newSamp.setFrame24 (i,0,temp);
      newSamp.setFrame24 (i,1,temp);
      newSamp.setFrame24 (i,2,temp);
      newSamp.setFrame24 (i,3,temp);
    }
  }
  else
  {
    // From Stereo
    if (convertMode == 0) convertMode = 255;
    long use1;
    long use2;
    long n;

    n = 0;
    use1 = convertMode % 2;
    convertMode = convertMode / 2;
    use2 = convertMode % 2;
    convertMode = convertMode / 2;
    if (use1) n++;
    if (use2) n++;
    for (i=0; i<frames; i++)
    {
      temp = 0;
      if (use1) temp += getFrame24 (i,0);
      if (use2) temp += getFrame24 (i,1);
      if (n) temp = temp / n;
      newSamp.setFrame24 (i,0,temp);
    }
    
    n = 0;
    use1 = convertMode % 2;
    convertMode = convertMode / 2;
    use2 = convertMode % 2;
    convertMode = convertMode / 2;
    if (use1) n++;
    if (use2) n++;
    for (i=0; i<frames; i++)
    {
      temp = 0;
      if (use1) temp += getFrame24 (i,0);
      if (use2) temp += getFrame24 (i,1);
      if (n) temp = temp / n;
      newSamp.setFrame24 (i,1,temp);
    }
    
    n = 0;
    use1 = convertMode % 2;
    convertMode = convertMode / 2;
    use2 = convertMode % 2;
    convertMode = convertMode / 2;
    if (use1) n++;
    if (use2) n++;
    for (i=0; i<frames; i++)
    {
      temp = 0;
      if (use1) temp += getFrame24 (i,0);
      if (use2) temp += getFrame24 (i,1);
      if (n) temp = temp / n;
      newSamp.setFrame24 (i,2,temp);
    }
    
    n = 0;
    use1 = convertMode % 2;
    convertMode = convertMode / 2;
    use2 = convertMode % 2;
    convertMode = convertMode / 2;
    if (use1) n++;
    if (use2) n++;
    for (i=0; i<frames; i++)
    {
      temp = 0;
      if (use1) temp += getFrame24 (i,0);
      if (use2) temp += getFrame24 (i,1);
      if (n) temp = temp / n;
      newSamp.setFrame24 (i,3,temp);
    }
  }

  void *tempPointer = data;
  data = newSamp.data;
  newSamp.data = tempPointer;

  channels = 4;
  samples  = frames * channels;
  bytes    = samples * width;
  changed  = TRUE;
}

/*---------------------------------------------------------------------------
| DPSample::trim ()
---------------------------------------------------------------------------*/
void DPSample::trim ()
{
  long i;
  long done;
  
  done = FALSE;
  i = frames - 1;
  
  if (channels == 1)
  {
    while (i >= 0 && !done)
    {
      if (getFrame24 (i,0) != 0) done = TRUE;
      else i--;
    }
  }
  else if (channels == 2)
  {
    while (i >= 0 && !done)
    {
      if (getFrame24 (i,0) != 0 || getFrame24 (i,1) != 0) done = TRUE;
      else i--;
    }
  }
  else
  {
    while (i >= 0 && !done)
    {
      if (getFrame24 (i,0) != 0 || getFrame24 (i,1) != 0 ||
      getFrame24 (i,2) != 0 || getFrame24 (i,3) != 0) done = TRUE;
      else i--;
    }
  }
  
  long rangeValidBak = rangeValid;
  long rangeStartBak = rangeStart;
  long rangeEndBak   = rangeEnd;
  
  rangeValid = TRUE;
  rangeStart = i+1;
  rangeEnd   = frames;
  
  del (2);
  
  setRange (rangeStartBak,rangeEndBak);
  rangeValid = rangeValidBak;
  changed = TRUE;
}

/*---------------------------------------------------------------------------
| Cut & paste functions
---------------------------------------------------------------------------*/

void DPSample::cut (DPSample *clip,int mode)
{
  if (!valid || !rangeValid) return;
  long rangeLen = rangeEnd - rangeStart;
  if (rangeLen == 0) return;
  long newBufferLen = frames - rangeLen;
  DPSample newSamp;

  int stereoMono = (channels == 2 && mode != 2);
  int quadroMono = (channels == 4 && mode != 2);
  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));

  if (stereoMono || quadroMono)
  {
    newBufferLen = frames;
    if (!newSamp.fresh (rate,width,channels,newBufferLen)) return;
    if (!clip->fresh (rate,width,1,rangeLen)) return;
  }
  else
  {
    if (!newSamp.fresh (rate,width,channels,newBufferLen)) return;
    if (!clip->fresh (rate,width,channels,rangeLen)) return;
  }  

  long i;
  long x;
  long temp;
  
  for (i=0; i<frames; i++)
  {
    if (i < rangeStart)
    {
      if (left)
      {
        temp = getFrame24 (i,0);
        newSamp.setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = getFrame24 (i,1);
        newSamp.setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = getFrame24 (i,2);
        newSamp.setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = getFrame24 (i,3);
        newSamp.setFrame24 (i,3,temp);
      }
    }
    else if (i >= rangeEnd)
    {
      x = i - rangeLen;
      if (left)
      {
        temp = getFrame24 (i,0);
        newSamp.setFrame24 (x,0,temp);
      }
      if (right)
      {
        temp = getFrame24 (i,1);
        newSamp.setFrame24 (x,1,temp);
      }
      if (three)
      {
        temp = getFrame24 (i,2);
        newSamp.setFrame24 (x,2,temp);
      }
      if (four)
      {
        temp = getFrame24 (i,3);
        newSamp.setFrame24 (x,3,temp);
      }
    }
    else
    {
      x = i - rangeStart;
      if (left)
      {
        temp = getFrame24 (i,0);
        clip->setFrame24 (x,0,temp);
      }
      if (right)
      {
        temp = getFrame24 (i,1);
        clip->setFrame24 (x,1,temp);
      }
      if (three)
      {
        temp = getFrame24 (i,2);
        clip->setFrame24 (x,2,temp);
      }
      if (four)
      {
        temp = getFrame24 (i,3);
        clip->setFrame24 (x,3,temp);
      }
    }
  }
  
  COPYREST

  void *tempPointer = data;
  data = newSamp.data;
  newSamp.data = tempPointer;

  frames  = newBufferLen;
  samples = frames * channels;
  bytes   = samples * width;
  
  // Sort out markers
  if (!stereoMono && !quadroMono)
  {
    long displayDuration = displayEnd - displayStart;
    if (displayEnd > frames) displayEnd = frames;
    displayStart = displayEnd - displayDuration;
    if (displayStart < 0) displayStart = 0;

    if (susLoopStart >= rangeEnd)        susLoopStart -= rangeLen;
    else if (susLoopStart >= rangeStart) susLoopStart  = rangeStart;
    if (susLoopEnd >= rangeEnd)          susLoopEnd   -= rangeLen;
    else if (susLoopEnd >= rangeStart)   susLoopEnd    = rangeStart;

    if (relLoopStart >= rangeEnd)        relLoopStart -= rangeLen;
    else if (relLoopStart >= rangeStart) relLoopStart  = rangeStart;
    if (relLoopEnd >= rangeEnd)          relLoopEnd   -= rangeLen;
    else if (relLoopEnd >= rangeStart)   relLoopEnd    = rangeStart;
  }

  rangeStart = 0;
  rangeEnd   = 0;
  rangeValid = FALSE;
  changed    = TRUE;
}

void DPSample::del (int mode)
{
  if (!valid || !rangeValid) return;
  long rangeLen = rangeEnd - rangeStart;
  if (rangeLen == 0) return;
  long newBufferLen = frames - rangeLen;
  DPSample newSamp;
  
  int stereoMono = (channels == 2 && mode != 2);
  int quadroMono = (channels == 4 && mode != 2);
  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));

  if (stereoMono || quadroMono)
  {
    newBufferLen = frames;
    if (!newSamp.fresh (rate,width,channels,newBufferLen)) return;
  }
  else
  {
    if (!newSamp.fresh (rate,width,channels,newBufferLen)) return;
  }  

  long i;
  long x;
  long temp;
 
  for (i=0; i<frames; i++)
  {
    if (i < rangeStart)
    {
      if (left)
      {
        temp = getFrame24 (i,0);
        newSamp.setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = getFrame24 (i,1);
        newSamp.setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = getFrame24 (i,2);
        newSamp.setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = getFrame24 (i,3);
        newSamp.setFrame24 (i,3,temp);
      }
    }
    else if (i >= rangeEnd)
    {
      x = i - rangeLen;
      if (left)
      {
        temp = getFrame24 (i,0);
        newSamp.setFrame24 (x,0,temp);
      }
      if (right)
      {
        temp = getFrame24 (i,1);
        newSamp.setFrame24 (x,1,temp);
      }
      if (three)
      {
        temp = getFrame24 (i,2);
        newSamp.setFrame24 (x,2,temp);
      }
      if (four)
      {
        temp = getFrame24 (i,3);
        newSamp.setFrame24 (x,3,temp);
      }
    }
  }
  
  COPYREST
  
  void *tempPointer = data;
  data = newSamp.data;
  newSamp.data = tempPointer;

  frames  = newBufferLen;
  samples = frames * channels;
  bytes   = samples * width;
  
  // Sort out markers
  if (!stereoMono && !quadroMono)
  {
    long displayDuration = displayEnd - displayStart;
    if (displayEnd > frames) displayEnd = frames;
    displayStart = displayEnd - displayDuration;
    if (displayStart < 0) displayStart = 0;

    if (susLoopStart >= rangeEnd)        susLoopStart -= rangeLen;
    else if (susLoopStart >= rangeStart) susLoopStart  = rangeStart;
    if (susLoopEnd >= rangeEnd)          susLoopEnd   -= rangeLen;
    else if (susLoopEnd >= rangeStart)   susLoopEnd    = rangeStart;

    if (relLoopStart >= rangeEnd)        relLoopStart -= rangeLen;
    else if (relLoopStart >= rangeStart) relLoopStart  = rangeStart;
    if (relLoopEnd >= rangeEnd)          relLoopEnd   -= rangeLen;
    else if (relLoopEnd >= rangeStart)   relLoopEnd    = rangeStart;
  }

  rangeStart = 0;
  rangeEnd   = 0;
  rangeValid = FALSE;
  changed    = TRUE;
}

void DPSample::copy (DPSample *clip,int mode)
{
  if (!valid || !rangeValid) return;
  long rangeLen = rangeEnd - rangeStart;
  if (rangeLen == 0) return;

  int stereoMono = (channels == 2 && mode != 2);
  int quadroMono = (channels == 4 && mode != 2);
  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));

  if (stereoMono || quadroMono)
  {
    if (!clip->fresh (rate,width,1,rangeLen)) return;  
  }
  else
  {
    if (!clip->fresh (rate,width,channels,rangeLen)) return;  
  }  

  long i;
  long x;
  long temp;
  
  for (i=rangeStart; i<rangeEnd; i++)
  {
    x = i - rangeStart;
    if (left)
    {
      temp = getFrame24 (i,0);
      clip->setFrame24 (x,0,temp);
    }
    if (right)
    {
      temp = getFrame24 (i,1);
      clip->setFrame24 (x,1,temp);
    }
    if (three)
    {
      temp = getFrame24 (i,2);
      clip->setFrame24 (x,2,temp);
    }
    if (four)
    {
      temp = getFrame24 (i,3);
      clip->setFrame24 (x,3,temp);
    }
  }
}

void DPSample::paste (DPSample *clip,int mode)
{
  if (!valid || !rangeValid || !clip->valid) return;
  long rangeLen = clip->frames;
  if (rangeLen == 0) return;
  long newBufferLen = frames + rangeLen;
  DPSample newSamp;
  
  int stereoMono = (channels == 2 && mode != 2);
  int quadroMono = (channels == 4 && mode != 2);
  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));

  if (stereoMono || quadroMono)
  {
    if (!newSamp.fresh (rate,width,channels,newBufferLen)) return;
  }
  else
  {
    if (!newSamp.fresh (rate,width,channels,newBufferLen)) return;
  }  

  rangeEnd = rangeStart + rangeLen;

  long i;
  long x;
  long temp;
 
  for (i=0; i<newBufferLen; i++)
  {
    if (i < rangeStart)
    {
      if (left)
      {
        temp = getFrame24 (i,0);
        newSamp.setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = getFrame24 (i,1);
        newSamp.setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = getFrame24 (i,2);
        newSamp.setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = getFrame24 (i,3);
        newSamp.setFrame24 (i,3,temp);
      }
    }
    else if (i >= rangeEnd)
    {
      x = i - rangeLen;
      if (left)
      {
        temp = getFrame24 (x,0);
        newSamp.setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = getFrame24 (x,1);
        newSamp.setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = getFrame24 (x,2);
        newSamp.setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = getFrame24 (x,3);
        newSamp.setFrame24 (i,3,temp);
      }
    }
    else
    {
      x = i - rangeStart;
      if (left)
      {
        temp = clip->getFrame24 (x,0);
        newSamp.setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = clip->getFrame24 (x,1);
        newSamp.setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = clip->getFrame24 (x,2);
        newSamp.setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = clip->getFrame24 (x,3);
        newSamp.setFrame24 (i,3,temp);
      }
    }
  }
  
  COPYREST

  void *tempPointer = data;
  data = newSamp.data;
  newSamp.data = tempPointer;

  frames  = newBufferLen;
  samples = frames * channels;
  bytes   = samples * width;
  
  // Sort out markers
  if (!stereoMono && !quadroMono)
  {
    if (susLoopStart >= rangeStart) susLoopStart += rangeLen;
    if (susLoopEnd   >= rangeStart) susLoopEnd   += rangeLen;
    if (relLoopStart >= rangeStart) relLoopStart += rangeLen;
    if (relLoopEnd   >= rangeStart) relLoopEnd   += rangeLen;
  }
  changed = TRUE;
}

void DPSample::keep (DPSample *clip,int mode)
{
  if (!valid || !rangeValid) return;
  long rangeLen = rangeEnd - rangeStart;
  if (rangeLen == 0) return;
  long newBufferLen = rangeLen;
  DPSample newSamp;
  
  int stereoMono = (channels == 2 && mode != 2);
  int quadroMono = (channels == 4 && mode != 2);
  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));

  if (stereoMono || quadroMono)
  {
    newBufferLen = frames;
    if (!newSamp.fresh (rate,width,channels,newBufferLen)) return;
    if (!clip->fresh (rate,width,1,frames - rangeLen)) return;
  }
  else
  {
    if (!newSamp.fresh (rate,width,channels,newBufferLen)) return;
    if (!clip->fresh (rate,width,channels,frames - rangeLen)) return;
  }  

  long i;
  long x;
  long temp;
  
  for (i=0; i<frames; i++)
  {
    if (i < rangeStart)
    {
      if (left)
      {
        temp = getFrame24 (i,0);
        clip->setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = getFrame24 (i,1);
        clip->setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = getFrame24 (i,2);
        clip->setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = getFrame24 (i,3);
        clip->setFrame24 (i,3,temp);
      }
    }
    else if (i >= rangeEnd)
    {
      x = i - rangeLen;
      if (left)
      {
        temp = getFrame24 (i,0);
        clip->setFrame24 (x,0,temp);
      }
      if (right)
      {
        temp = getFrame24 (i,1);
        clip->setFrame24 (x,1,temp);
      }
      if (three)
      {
        temp = getFrame24 (i,2);
        clip->setFrame24 (x,2,temp);
      }
      if (four)
      {
        temp = getFrame24 (i,3);
        clip->setFrame24 (x,3,temp);
      }
    }
    else
    {
      x = i - rangeStart;
      if (left)
      {
        temp = getFrame24 (i,0);
        newSamp.setFrame24 (x,0,temp);
      }
      if (right)
      {
        temp = getFrame24 (i,1);
        newSamp.setFrame24 (x,1,temp);
      }
      if (three)
      {
        temp = getFrame24 (i,2);
        newSamp.setFrame24 (x,2,temp);
      }
      if (four)
      {
        temp = getFrame24 (i,3);
        newSamp.setFrame24 (x,3,temp);
      }
    }
  }
  
  COPYREST
  
  void *tempPointer = data;
  data = newSamp.data;
  newSamp.data = tempPointer;

  frames  = newBufferLen;
  samples = frames * channels;
  bytes   = samples * width;
  
  // Sort out markers
  if (!stereoMono && !quadroMono)
  {
    displayStart = displayStart - rangeStart;
    displayEnd = displayEnd - rangeStart;
    if (displayStart < 0) displayStart = 0;
    if (displayEnd > frames) displayEnd = frames;
    if (displayStart > frames || displayEnd < 0)
    {
      displayStart = 0;
      displayEnd = frames;
    }

    if (susLoopStart >= rangeEnd)        susLoopStart  = frames;
    else if (susLoopStart >= rangeStart) susLoopStart -= rangeStart;
    else susLoopStart = 0;
    if (susLoopEnd >= rangeEnd)          susLoopEnd    = frames;
    else if (susLoopEnd >= rangeStart)   susLoopEnd   -= rangeStart;
    else susLoopEnd = 0;

    if (relLoopStart >= rangeEnd)        relLoopStart  = frames;
    else if (relLoopStart >= rangeStart) relLoopStart -= rangeStart;
    else relLoopStart = 0;
    if (relLoopEnd >= rangeEnd)          relLoopEnd    = frames;
    else if (relLoopEnd >= rangeStart)   relLoopEnd   -= rangeStart;
    else relLoopEnd = 0;
  }

  rangeStart = 0;
  rangeEnd   = 0;
  rangeValid = FALSE;
  changed    = TRUE;
}

void DPSample::overwriteRange (DPSample *clip,int mode)
{
  mixRange (clip,0,100,0,0,mode);
}

void DPSample::mixRange (
  DPSample *clip,
  double buffVol,
  double clipVol,
  long   mixin,
  long   mixout,
  int    mode)
{
  if (!valid || !rangeValid || !clip->valid) return;
  long rangeLen = rangeEnd - rangeStart;
  if (rangeLen == 0) return;
  
  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));
    
  long i;
  long x;
  long y;
  long temp;
  
  double clipVolume;
  double buffVolume;
  
  for (i=rangeStart; i<rangeEnd; i++)
  {
    if (i < rangeStart + mixin)
    {
      x = i - rangeStart;
      clipVolume = (x * clipVol) / (mixin * 100.0);
      buffVolume = 1.0 - ((x * (100.0 - buffVol)) / (mixin * 100.0));
      if (left)
      {
        temp = (long) ((clipVolume * clip->getFrame24 (x,0)) +
          (buffVolume * getFrame24 (i,0)));
        setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = (long) ((clipVolume * clip->getFrame24 (x,1)) +
          (buffVolume * getFrame24 (i,1)));
        setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = (long) ((clipVolume * clip->getFrame24 (x,2)) +
          (buffVolume * getFrame24 (i,2)));
        setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = (long) ((clipVolume * clip->getFrame24 (x,3)) +
          (buffVolume * getFrame24 (i,3)));
        setFrame24 (i,3,temp);
      }
    }
    else if (i > rangeEnd - mixout)
    {
      y = i - rangeStart;
      x = rangeEnd - i;
      clipVolume = (x * clipVol) / (mixout * 100.0);
      buffVolume = 1.0 - ((x * (100.0 - buffVol)) / (mixout * 100.0));
      if (left)
      {
        temp = (long) ((clipVolume * clip->getFrame24 (y,0)) +
          (buffVolume * getFrame24 (i,0)));
        setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = (long) ((clipVolume * clip->getFrame24 (y,1)) +
          (buffVolume * getFrame24 (i,1)));
        setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = (long) ((clipVolume * clip->getFrame24 (y,2)) +
          (buffVolume * getFrame24 (i,2)));
        setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = (long) ((clipVolume * clip->getFrame24 (y,3)) +
          (buffVolume * getFrame24 (i,3)));
        setFrame24 (i,3,temp);
      }
    }
    else
    {
      y = i - rangeStart;
      if (left)
      {
        temp = (long) (((clipVol / 100.0) * clip->getFrame24 (y,0)) +
          ((buffVol / 100.0) * getFrame24 (i,0)));
        setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = (long) (((clipVol / 100.0) * clip->getFrame24 (y,1)) +
          ((buffVol / 100.0) * getFrame24 (i,1)));
        setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = (long) (((clipVol / 100.0) * clip->getFrame24 (y,2)) +
          ((buffVol / 100.0) * getFrame24 (i,2)));
        setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = (long) (((clipVol / 100.0) * clip->getFrame24 (y,3)) +
          ((buffVol / 100.0) * getFrame24 (i,3)));
        setFrame24 (i,3,temp);
      }
    }
  }
  changed = TRUE;
}

void DPSample::overwrite (DPSample *clip,int mode)
{
  mix (clip,0,100,0,0,mode);
}

void DPSample::mix (
  DPSample *clip,
  double buffVol,
  double clipVol,
  long   mixin,
  long   mixout,
  int    mode)
{
  if (!valid || !rangeValid || !clip->valid) return;
  long rangeLen = clip->frames;
  if (rangeLen == 0) return;
  long temp = (rangeStart + rangeLen) - frames;
  if (temp < 0) temp = 0;
  long newBufferLen = frames + temp;
  DPSample newSamp;
  
  int stereoMono = (channels == 2 && mode != 2);
  int quadroMono = (channels == 4 && mode != 2);
  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));

  if (!newSamp.fresh (rate,width,channels,newBufferLen)) return;
  
  rangeEnd = rangeStart + rangeLen;
  
  long i;
  long x;
  long y;
  
  double clipVolume;
  double buffVolume;
  
  for (i=0; i<newBufferLen; i++)
  {
    if (i < rangeStart || i >= rangeEnd)
    {
      if (left)
      {
        temp = getFrame24 (i,0);
        newSamp.setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = getFrame24 (i,1);
        newSamp.setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = getFrame24 (i,2);
        newSamp.setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = getFrame24 (i,3);
        newSamp.setFrame24 (i,3,temp);
      }
    }
    else if (i < rangeStart + mixin)
    {
      x = i - rangeStart;
      clipVolume = (x * clipVol) / (mixin * 100.0);
      buffVolume = 1.0 - ((x * (100.0 - buffVol)) / (mixin * 100.0));
      if (left)
      {
        temp = (long) ((clipVolume * clip->getFrame24 (x,0)) +
          (buffVolume * getFrame24 (i,0)));
        newSamp.setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = (long) ((clipVolume * clip->getFrame24 (x,1)) +
          (buffVolume * getFrame24 (i,1)));
        newSamp.setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = (long) ((clipVolume * clip->getFrame24 (x,2)) +
          (buffVolume * getFrame24 (i,2)));
        newSamp.setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = (long) ((clipVolume * clip->getFrame24 (x,3)) +
          (buffVolume * getFrame24 (i,3)));
        newSamp.setFrame24 (i,3,temp);
      }
    }
    else if (i > rangeEnd - mixout)
    {
      y = i - rangeStart;
      x = rangeEnd - i;
      clipVolume = (x * clipVol) / (mixout * 100.0);
      buffVolume = 1.0 - ((x * (100.0 - buffVol)) / (mixout * 100.0));
      if (left)
      {
        temp = (long) ((clipVolume * clip->getFrame24 (y,0)) +
          (buffVolume * getFrame24 (i,0)));
        newSamp.setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = (long) ((clipVolume * clip->getFrame24 (y,1)) +
          (buffVolume * getFrame24 (i,1)));
        newSamp.setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = (long) ((clipVolume * clip->getFrame24 (y,2)) +
          (buffVolume * getFrame24 (i,2)));
        newSamp.setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = (long) ((clipVolume * clip->getFrame24 (y,3)) +
          (buffVolume * getFrame24 (i,3)));
        newSamp.setFrame24 (i,3,temp);
      }
    }
    else
    {
      y = i - rangeStart;
      if (left)
      {
        temp = (long) (((clipVol / 100.0) * clip->getFrame24 (y,0)) +
          ((buffVol / 100.0) * getFrame24 (i,0)));
        newSamp.setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = (long) (((clipVol / 100.0) * clip->getFrame24 (y,1)) +
          ((buffVol / 100.0) * getFrame24 (i,1)));
        newSamp.setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = (long) (((clipVol / 100.0) * clip->getFrame24 (y,2)) +
          ((buffVol / 100.0) * getFrame24 (i,2)));
        newSamp.setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = (long) (((clipVol / 100.0) * clip->getFrame24 (y,3)) +
          ((buffVol / 100.0) * getFrame24 (i,3)));
        newSamp.setFrame24 (i,3,temp);
      }
    }
  }
  
  COPYREST
  
  void *tempPointer = data;
  data = newSamp.data;
  newSamp.data = tempPointer;

  frames  = newBufferLen;
  samples = frames * channels;
  bytes   = samples * width;
  changed = TRUE;
}

/*---------------------------------------------------------------------------
| Range functions
---------------------------------------------------------------------------*/

void DPSample::reverse (int mode)
{
  if (!valid || !rangeValid) return;
  if (rangeStart == rangeEnd) return;
  long i = rangeStart;
  long j = rangeEnd - 1;
  long temp1;
  long temp2;

  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));
  
  while (i < j)
  {
    if (left)
    {
      temp1 = getFrame24 (i,0);
      temp2 = getFrame24 (j,0);
      setFrame24 (i,0,temp2);
      setFrame24 (j,0,temp1);
    }
    if (right)
    {
      temp1 = getFrame24 (i,1);
      temp2 = getFrame24 (j,1);
      setFrame24 (i,1,temp2);
      setFrame24 (j,1,temp1);
    }
    if (three)
    {
      temp1 = getFrame24 (i,2);
      temp2 = getFrame24 (j,2);
      setFrame24 (i,2,temp2);
      setFrame24 (j,2,temp1);
    }
    if (four)
    {
      temp1 = getFrame24 (i,3);
      temp2 = getFrame24 (j,3);
      setFrame24 (i,3,temp2);
      setFrame24 (j,3,temp1);
    }
    i++;
    j--;
  }
  changed = TRUE;
}

void DPSample::invert (int mode)
{
  if (!valid || !rangeValid) return;
  if (rangeStart == rangeEnd) return;
  long i;
  long temp;

  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));
  
  for (i=rangeStart; i<rangeEnd; i++)
  {
    if (left)
    {
      temp = getFrame24 (i,0);
      setFrame24 (i,0,-temp);
    }
    if (right)
    {
      temp = getFrame24 (i,1);
      setFrame24 (i,1,-temp);
    }
    if (three)
    {
      temp = getFrame24 (i,2);
      setFrame24 (i,2,-temp);
    }
    if (four)
    {
      temp = getFrame24 (i,3);
      setFrame24 (i,3,-temp);
    }
  }
  changed = TRUE;
}

void DPSample::zero (int mode)
{
  if (!valid || !rangeValid) return;
  if (rangeStart == rangeEnd) return;
  long i;

  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));
  
  for (i=rangeStart; i<rangeEnd; i++)
  {
    if (left)
      setFrame24 (i,0,0);
    if (right)
      setFrame24 (i,1,0);
    if (three)
      setFrame24 (i,2,0);
    if (four)
      setFrame24 (i,3,0);
  }
  changed = TRUE;
}

void DPSample::channelChange (int channelMode)
{
  // For Stereo
  // channelMode == 0 for right copied to left
  // channelMode == 1 for right exchanged with left
  // channelMode == 2 for left copied to right
  
  // For Quadro
  // channelMode == Binary number between 0 and 63
  // for channel 0 to 3 (from) 0 to 3 (to) mode (<-,<->,->) (0,1,2)
  
  if (channels == 1) return;

  if (!valid || !rangeValid) return;
  if (rangeStart == rangeEnd) return;

  if (channels == 2)
  {
    long i;
    long temp1;
    long temp2;
    int zero = (channelMode == 0);
    int one  = (channelMode == 1);

    for (i=rangeStart; i<rangeEnd; i++)
    {
      temp1 = getFrame24 (i,0);
      temp2 = getFrame24 (i,1);
      if (zero)
      {
        setFrame24 (i,0,temp2);
      }
      else if (one)
      {
        setFrame24 (i,0,temp2);
        setFrame24 (i,1,temp1);
      }
      else
      {
        setFrame24 (i,1,temp1);
      }
    }
  }
  else
  {
    long i;
    long temp [4];
    long tempSource;
    long tempDest;
    
    int source = channelMode & 3;
    int dest   = (channelMode >> 2) & 3;
    int mode   = (channelMode >> 4) & 3;
    
    for (i=rangeStart; i<rangeEnd; i++)
    {
      temp [0] = getFrame24 (i,0);
      temp [1] = getFrame24 (i,1);
      temp [2] = getFrame24 (i,2);
      temp [3] = getFrame24 (i,3);
      tempSource = temp [source];
      tempDest   = temp [dest];
      
      if (mode == 0)
      {
        setFrame24 (i,source,tempDest);
      }
      else if (mode == 1)
      {
        setFrame24 (i,source,tempDest);
        setFrame24 (i,dest,tempSource);
      }
      else
      {
        setFrame24 (i,dest,tempSource);
      }
    }
  }
  changed = TRUE;
}

void DPSample::duplicate (long times,int mode)
{
  if (!valid || !rangeValid) return;
  if (rangeStart == rangeEnd) return;
  if (times <= 1) return;
  long rangeLenBefore = rangeEnd - rangeStart;
  long rangeLenAfter  = times * rangeLenBefore;
  long newBufferLen = frames - rangeLenBefore + rangeLenAfter;
  DPSample newSamp;
  
  int stereoMono = (channels == 2 && mode != 2);
  int quadroMono = (channels == 4 && mode != 2);
  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));

  if (!newSamp.fresh (rate,width,channels,newBufferLen)) return;

  rangeEnd = rangeStart + rangeLenAfter;
  
  long i;
  long x;
  long temp;
  
  for (i=0; i<newBufferLen; i++)
  {
    if (i<rangeStart)
    {
      if (left)
      {
        temp = getFrame24 (i,0);
        newSamp.setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = getFrame24 (i,1);
        newSamp.setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = getFrame24 (i,2);
        newSamp.setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = getFrame24 (i,3);
        newSamp.setFrame24 (i,3,temp);
      }
    }
    else if (i >= rangeEnd)
    {
      x = i - rangeLenAfter + rangeLenBefore;
      if (left)
      {
        temp = getFrame24 (x,0);
        newSamp.setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = getFrame24 (x,1);
        newSamp.setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = getFrame24 (x,2);
        newSamp.setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = getFrame24 (x,3);
        newSamp.setFrame24 (i,3,temp);
      }
    }
    else
    {
      x = rangeStart + ((i - rangeStart) % rangeLenBefore);
      if (left)
      {
        temp = getFrame24 (x,0);
        newSamp.setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = getFrame24 (x,1);
        newSamp.setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = getFrame24 (x,2);
        newSamp.setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = getFrame24 (x,3);
        newSamp.setFrame24 (i,3,temp);
      }
    }
  }
  
  COPYREST
  
  void *tempPointer = data;
  data = newSamp.data;
  newSamp.data = tempPointer;

  frames  = newBufferLen;
  samples = frames * channels;
  bytes   = samples * width;
  
  // Sort out markers
  if (!stereoMono && !quadroMono)
  {
    if (susLoopStart >= rangeStart)
      susLoopStart += rangeLenAfter - rangeLenBefore;
    if (susLoopEnd   >= rangeStart)
      susLoopEnd   += rangeLenAfter - rangeLenBefore;
    if (relLoopStart >= rangeStart)
      relLoopStart += rangeLenAfter - rangeLenBefore;
    if (relLoopEnd   >= rangeStart)
      relLoopEnd   += rangeLenAfter - rangeLenBefore;
    if (displayEnd   >= rangeStart)
      displayEnd   += rangeLenAfter - rangeLenBefore;
  }
  changed = TRUE;
}

void DPSample::panorama (long amount,long ltor,long rtol,long panChannels)
{
  // Quadro
  // pan Channels == Binary number 0 to 15
  // 0 to 3 (left) 0 to 3 (right)
  
  if (channels == 1) return;
  if (!valid || !rangeValid) return;
  if (rangeStart == rangeEnd) return;
  
  long i;
  long tempLa;
  long tempRa;
  long tempLaDel;
  long tempRaDel;
  long tempLb;
  long tempRb;
  
  double mix1 = amount / 200.0;
  double mix2 = mix1;
  if (mix1 < 0.0) mix1 = -mix1;
  mix1 = 1.0 - mix1; 

  if (channels == 2)
  {
    for (i=rangeStart; i<rangeEnd; i++)
    {
      tempLa    = getFrame24 (i,0);
      tempRa    = getFrame24 (i,1);
      tempLaDel = getFrame24 (i-ltor,0);
      tempRaDel = getFrame24 (i-rtol,1);
      tempLb = (long) (mix1 * tempLa + mix2 * tempRaDel);
      tempRb = (long) (mix1 * tempRa + mix2 * tempLaDel);
      setFrame24 (i,0,tempLb);
      setFrame24 (i,1,tempRb);
    }
  }
  else
  {
    long left  = panChannels & 3;
    long right = (panChannels >> 2) & 3;
    
    for (i=rangeStart; i<rangeEnd; i++)
    {
      tempLa    = getFrame24 (i,left);
      tempRa    = getFrame24 (i,right);
      tempLaDel = getFrame24 (i-ltor,left);
      tempRaDel = getFrame24 (i-rtol,right);
      tempLb = (long) (mix1 * tempLa + mix2 * tempRaDel);
      tempRb = (long) (mix1 * tempRa + mix2 * tempLaDel);
      setFrame24 (i,left,tempLb);
      setFrame24 (i,right,tempRb);
    }
  }
  changed = TRUE;
}

void DPSample::adjustDC (long amount,int mode)
{
  // Note - can handle extra space (required for record)
  
  if (!valid || !rangeValid) return;
  if (rangeStart == rangeEnd) return;
  
  long i;
  long temp;
  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));
  
  for (i=rangeStart; i<rangeEnd; i++)
  {
    if (left)
    {
      temp = getFrame24Extra (i,0);
      temp += (amount * MAX23 / 100);
      limitValue (&temp,0);
      setFrame24Extra (i,0,temp);
    }
    if (right)
    {
      temp = getFrame24Extra (i,1);
      temp += (amount * MAX23 / 100);
      limitValue (&temp,0);
      setFrame24Extra (i,1,temp);
    }
    if (three)
    {
      temp = getFrame24Extra (i,2);
      temp += (amount * MAX23 / 100);
      limitValue (&temp,0);
      setFrame24Extra (i,2,temp);
    }
    if (four)
    {
      temp = getFrame24Extra (i,3);
      temp += (amount * MAX23 / 100);
      limitValue (&temp,0);
      setFrame24Extra (i,3,temp);
    }
  }
  changed = TRUE;
}

void DPSample::addWorkspace (long newFrames,int mode)
{
  if (!valid) return;
  if (newFrames <= 0) return;
  long newBufferLen = frames + newFrames;
  DPSample newSamp;
  
  int stereoMono = (channels == 2 && mode != 2);
  int quadroMono = (channels == 4 && mode != 2);
  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));
  
  if (!newSamp.fresh (rate,width,channels,newBufferLen)) return;

  if (!rangeValid)
  {
    rangeStart = frames;
    rangeValid = TRUE;
  }
  
  rangeEnd = rangeStart + newFrames;
  
  long i;
  long x;
  long temp;
  
  for (i=0; i<newBufferLen; i++)
  {
    if (i<rangeStart)
    {
      if (left)
      {
        temp = getFrame24 (i,0);
        newSamp.setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = getFrame24 (i,1);
        newSamp.setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = getFrame24 (i,2);
        newSamp.setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = getFrame24 (i,3);
        newSamp.setFrame24 (i,3,temp);
      }
    }
    else if (i >= rangeEnd)
    {
      x = i - newFrames;
      if (left)
      {
        temp = getFrame24 (x,0);
        newSamp.setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = getFrame24 (x,1);
        newSamp.setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = getFrame24 (x,2);
        newSamp.setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = getFrame24 (x,3);
        newSamp.setFrame24 (i,3,temp);
      }
    }
    else
    {
      if (left)
      {
        newSamp.setFrame24 (i,0,0);
      }
      if (right)
      {
        newSamp.setFrame24 (i,1,0);
      }
      if (three)
      {
        newSamp.setFrame24 (i,2,0);
      }
      if (four)
      {
        newSamp.setFrame24 (i,3,0);
      }
    }
  }

  void *tempPointer = data;
  data = newSamp.data;
  newSamp.data = tempPointer;

  frames  = newBufferLen;
  samples = frames * channels;
  bytes   = samples * width;
  
  COPYREST
  
  // Sort out markers
  if (!stereoMono && !quadroMono)
  {
    if (susLoopStart >= rangeStart) susLoopStart += newFrames;
    if (susLoopEnd   >= rangeStart) susLoopEnd   += newFrames;
    if (relLoopStart >= rangeStart) relLoopStart += newFrames;
    if (relLoopEnd   >= rangeStart) relLoopEnd   += newFrames;
  }
  changed = TRUE;
}

/*---------------------------------------------------------------------------
| Amplitude functions
---------------------------------------------------------------------------*/

void DPSample::percentage (
  double lamount,
  double ramount,
  double amount3,
  double amount4,
  int wrap,
  int mode)
{
  if (!valid || !rangeValid) return;
  if (rangeStart == rangeEnd) return;
  
  long i;
  long temp;
  
  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));
  
  for (i=rangeStart; i<rangeEnd; i++)
  {
    if (left)
    {
      temp = getFrame24 (i,0);
      temp = (long) rint ((double) temp * lamount / 100.0);
      limitValue (&temp,wrap);
      setFrame24 (i,0,temp);
    }
    if (right)
    {
      temp = getFrame24 (i,1);
      temp = (long) rint ((double) temp * ramount / 100.0);
      limitValue (&temp,wrap);
      setFrame24 (i,1,temp);
    }
    if (three)
    {
      temp = getFrame24 (i,2);
      temp = (long) rint ((double) temp * amount3 / 100.0);
      limitValue (&temp,wrap);
      setFrame24 (i,2,temp);
    }
    if (four)
    {
      temp = getFrame24 (i,3);
      temp = (long) rint ((double) temp * amount4 / 100.0);
      limitValue (&temp,wrap);
      setFrame24 (i,3,temp);
    }
  }
  changed = TRUE;
}

void DPSample::normalise (int mode)
{
  if (!valid || !rangeValid) return;
  if (rangeStart == rangeEnd) return;

  long i;
  long temp;
  long max = 0;
  long min = 0;
  double percent;
  
  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));
  
  for (i=rangeStart; i<rangeEnd; i++)
  {
    if (left)
    {
      temp = getFrame24 (i,0);
      if (temp < min) min = temp;
      else if (temp > max) max = temp;
    }
    if (right)
    {
      temp = getFrame24 (i,1);
      if (temp < min) min = temp;
      else if (temp > max) max = temp;
    }
    if (three)
    {
      temp = getFrame24 (i,2);
      if (temp < min) min = temp;
      else if (temp > max) max = temp;
    }
    if (four)
    {
      temp = getFrame24 (i,3);
      if (temp < min) min = temp;
      else if (temp > max) max = temp;
    }
  }
  
  if (max == 0 && min == 0) return;
  
  if (-min > max)
    percent = 100.0 * (double) MAX23 / -min;
  else
    percent = 100.0 * (double) MAX23_1 / max;
  
  percentage (percent,percent,percent,percent,FALSE,mode);
}

void DPSample::fadeIn (int mode)
{
  ramp (0.0,100.0,FALSE,mode);
}

void DPSample::fadeOut (int mode)
{
  ramp (100.0,0.0,FALSE,mode);
}

void DPSample::bounce (int bounceMode)
{
  // For Stereo
  // bounceMode == 0 for left to right
  // bounceMode == 1 for right to left
  
  // For Quadro
  // bounceMode == Binary number 0 to 31
  // for 0 to 3 (source) 0 to 3 (dest) mode (->,<-) (0,1)
  
  if (channels == 1) return;
  
  if (channels == 2)
  {
    if (bounceMode == 0)
    {
      ramp (100.0,000.0,FALSE,0);
      ramp (000.0,100.0,FALSE,1);
    }
    else
    {
      ramp (000.0,100.0,FALSE,0);
      ramp (100.0,000.0,FALSE,1);
    }
  }
  else
  {
    long source = bounceMode & 3;
    long dest   = (bounceMode >> 2) & 3;
    long mode   = (bounceMode >> 4) & 1;
    
    if (source == 2) source = 3;
    else if (source == 3) source = 4;
    if (dest == 2) dest = 3;
    else if (dest == 3) dest = 4;
    
    if (mode == 0)
    {
      ramp (100.0,000.0,FALSE,source);
      ramp (000.0,100.0,FALSE,dest);
    }
    else
    {
      ramp (100.0,000.0,FALSE,dest);
      ramp (000.0,100.0,FALSE,source);
    }
  }
}

void DPSample::ramp (
  double startamount,
  double endamount,
  int wrap,
  int mode)
{
  // mode == 0 for left only
  // mode == 1 for right only
  // mode == 2 for both left and right
  // mode == 3 for three only
  // mode == 4 for four only

  if (!valid || !rangeValid) return;
  long rangeLen = rangeEnd - rangeStart;
  if (rangeLen == 0) return;

  long i;
  long x;
  long temp;
  double amount;

  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));
  
  if (left)
  {
    temp = getFrame24 (rangeStart,0);
    temp = (long) rint ((double) temp * startamount / 100.0);
    limitValue (&temp,wrap);
    setFrame24 (rangeStart,0,temp);
  }
  if (right)
  {
    temp = getFrame24 (rangeStart,1);
    temp = (long) rint ((double) temp * startamount / 100.0);
    limitValue (&temp,wrap);
    setFrame24 (rangeStart,1,temp);
  }
  if (three)
  {
    temp = getFrame24 (rangeStart,2);
    temp = (long) rint ((double) temp * startamount / 100.0);
    limitValue (&temp,wrap);
    setFrame24 (rangeStart,2,temp);
  }
  if (four)
  {
    temp = getFrame24 (rangeStart,3);
    temp = (long) rint ((double) temp * startamount / 100.0);
    limitValue (&temp,wrap);
    setFrame24 (rangeStart,3,temp);
  }
  
  rangeLen--;
  if (rangeLen == 0) return;
  
  for (i=rangeStart+1; i<rangeEnd; i++)
  {
    x = i - rangeStart;
    amount = ((x * endamount / rangeLen)
      + ((rangeLen - x) * startamount / rangeLen)) / 100.0;
    if (left)
    {
      temp = getFrame24 (i,0);
      temp = (long) (temp * amount);
      limitValue (&temp,wrap);
      setFrame24 (i,0,temp);
    }
    if (right)
    {
      temp = getFrame24 (i,1);
      temp = (long) (temp * amount);
      limitValue (&temp,wrap);
      setFrame24 (i,1,temp);
    }
    if (three)
    {
      temp = getFrame24 (i,2);
      temp = (long) (temp * amount);
      limitValue (&temp,wrap);
      setFrame24 (i,2,temp);
    }
    if (four)
    {
      temp = getFrame24 (i,3);
      temp = (long) (temp * amount);
      limitValue (&temp,wrap);
      setFrame24 (i,3,temp);
    }
  }
  changed = TRUE;
}

void DPSample::draw (double *amounts,long *frames,long n,int wrap,int mode)
{
  if (!valid || !rangeValid) return;
  long rangeLen = rangeEnd - rangeStart;
  if (rangeLen == 0) return;
  if (n < 2) return;
  
  long i;
  long rangeValidBak = rangeStart;
  long rangeStartBak = rangeStart;
  long rangeEndBak   = rangeEnd;
  
  for (i=0; i<n-1; i++)
  {
    rangeStart = rangeStartBak + frames [i];
    rangeEnd   = rangeStartBak + frames [i+1];
    ramp (amounts [i],amounts [i+1],wrap,mode);
  }

  setRange (rangeStartBak,rangeEndBak);
  rangeValid = rangeValidBak;
}

void DPSample::limitValue (long *value,int wrap)
{
  while (*value < -MAX23 || *value > MAX23_1)
  {
    if (*value > MAX23_1)
      if (wrap) *value = MAX23_1 - (*value - MAX23_1);
      else *value = MAX23_1;
    if (*value < -MAX23)
      if (wrap) *value = -MAX23 + (-*value - MAX23);
      else *value = -MAX23;
  }
}

/*---------------------------------------------------------------------------
| FUNCTION DPSampleAudiofileError
---------------------------------------------------------------------------*/

void DPSampleAudiofileError (long,const char *)
{
}

/*---------------------------------------------------------------------------
| FUNCTION DPSampleStopCatch
---------------------------------------------------------------------------*/

void DPSampleStopCatch (...)
{
  DPSampleStop = TRUE;
}

/*---------------------------------------------------------------------------
| FUNCTION DPSampleReleaseCatch
---------------------------------------------------------------------------*/

void DPSampleReleaseCatch (...)
{
  if (!DPSampleRelease1)
  {
    DPSampleRelease1 = TRUE;
    DPSampleInc = 1;
  }
  else
  {
    DPSampleRelease2 = TRUE;
    DPSampleInc = 1;
  }
}

/*---------------------------------------------------------------------------
| Resampling functions
---------------------------------------------------------------------------*/

void DPSample::nonTimeStretch (long newFrames,double newRate,int mode)
{
  if (!valid || !rangeValid) return;
  long rangeLen = rangeEnd - rangeStart;
  if (rangeLen == 0) return;
  long newBufferLen = frames - rangeLen + newFrames;
  DPSample newSamp;
  
  int stereoMono = (channels == 2 && mode != 2);
  int quadroMono = (channels == 4 && mode != 2);
  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));
  
  if (stereoMono || quadroMono)
  {
    if (newBufferLen < frames) newBufferLen = frames;
    if (!newSamp.fresh (rate,width,channels,newBufferLen)) return;
  }
  else
  {
    if (!newSamp.fresh (rate,width,channels,newBufferLen)) return;
  }
  
  rangeEnd = rangeStart + newFrames;
  
  long i;
  long x;
  long temp;
  double y;
  
  for (i=0; i<newBufferLen; i++)
  {
    if (i < rangeStart)
    {
      if (left)
      {
        temp = getFrame24 (i,0);
        newSamp.setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = getFrame24 (i,1);
        newSamp.setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = getFrame24 (i,2);
        newSamp.setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = getFrame24 (i,3);
        newSamp.setFrame24 (i,3,temp);
      }
    }
    else if (i >= rangeEnd)
    {
      x = i - newFrames + rangeLen;
      if (left)
      {
        temp = getFrame24 (x,0);
        newSamp.setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = getFrame24 (x,1);
        newSamp.setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = getFrame24 (x,2);
        newSamp.setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = getFrame24 (x,3);
        newSamp.setFrame24 (i,3,temp);
      }
    }
    else
    {
      x = i - rangeStart;
      y = rangeStart + (double) x * rangeLen / newFrames;
      if (left)
      {
        temp = getFrame24Interp (y,0);
        newSamp.setFrame24 (i,0,temp);
      }
      if (right)
      {
        temp = getFrame24Interp (y,1);
        newSamp.setFrame24 (i,1,temp);
      }
      if (three)
      {
        temp = getFrame24Interp (y,2);
        newSamp.setFrame24 (i,2,temp);
      }
      if (four)
      {
        temp = getFrame24Interp (y,3);
        newSamp.setFrame24 (i,3,temp);
      }
    }
  }

  COPYREST
  
  void *tempPointer = data;
  data = newSamp.data;
  newSamp.data = tempPointer;

  frames  = newBufferLen;
  samples = frames * channels;
  bytes   = samples * width;
  
  // Sort out markers
  if (!stereoMono && !quadroMono)
  {
    rangeEnd = rangeStart + rangeLen;
    
    if (susLoopStart >= rangeEnd)
      susLoopStart += newFrames - rangeLen;
    else if (susLoopStart >= rangeStart)
      susLoopStart = 
        rangeStart + ((susLoopStart - rangeStart) * newFrames) / rangeLen;
    if (susLoopEnd >= rangeEnd)
      susLoopEnd += newFrames - rangeLen;
    else if (susLoopEnd >= rangeStart)
      susLoopEnd = 
        rangeStart + ((susLoopEnd - rangeStart) * newFrames) / rangeLen;
    if (relLoopStart >= rangeEnd)
      relLoopStart += newFrames - rangeLen;
    else if (relLoopStart >= rangeStart)
      relLoopStart = 
        rangeStart + ((relLoopStart - rangeStart) * newFrames) / rangeLen;
    if (relLoopEnd >= rangeEnd)
      relLoopEnd += newFrames - rangeLen;
    else if (relLoopEnd >= rangeStart)
      relLoopEnd = 
        rangeStart + ((relLoopEnd - rangeStart) * newFrames) / rangeLen;
    if (displayStart >= rangeEnd)
      displayStart += newFrames - rangeLen;
    else if (displayStart >= rangeStart)
      displayStart = 
        rangeStart + ((displayStart - rangeStart) * newFrames) / rangeLen;
    if (displayEnd >= rangeEnd)
      displayEnd += newFrames - rangeLen;
    else if (displayEnd >= rangeStart)
      displayEnd = 
        rangeStart + ((displayEnd - rangeStart) * newFrames) / rangeLen;
    
    rangeEnd = rangeStart + newFrames;
  }
  
  // Update sample rate
  setRate (newRate);
  
  changed = TRUE;
}

int DPSample::timeStretch (
  double pitchChange,
  double framesChange,
  double newRate,
  long   lowestFreq,
  long   quality,
  int    mode)
{
  if (!valid || !rangeValid) return 0;
  long rangeLen = rangeEnd - rangeStart;
  if (rangeLen == 0) return 0;
  
  // Must round framesChange so calculate vocoder parameters here !!
  long     R;
  long     N;
  long     Nw;
  long     D;
  long     I;
  double   P;
  double   synt;
  
  N            = 1;
  R            = (long) rate;
  while ((R / N) > lowestFreq) N = 2 * N;
  Nw           = quality * N;
  D            = Nw / 8;
  I            = (long)(D * framesChange / 100.0);
  P            = pitchChange / 100.0;
  synt         = 0.0;
  double newFramesChange = (double) I / D;
  
  long newFrames = (long) (rangeLen * newFramesChange);
  long newBufferLen = frames - rangeLen + newFrames;
  DPSample newSamp;
  
  int stereoMono = (channels == 2 && mode != 2);
  int quadroMono = (channels == 4 && mode != 2);
  int left       = (channels == 1 || (mode == 2 || mode == 0));
  int right      = (channels != 1 && (mode == 2 || mode == 1));
  int three      = (channels == 4 && (mode == 2 || mode == 3));
  int four       = (channels == 4 && (mode == 2 || mode == 4));
  
  if (stereoMono || quadroMono)
  {
    if (newBufferLen < frames) newBufferLen = frames;
    if (!newSamp.fresh (rate,width,channels,newBufferLen)) return 0;
  }
  else
  {
    if (!newSamp.fresh (rate,width,channels,newBufferLen)) return 0;
  }

  rangeEnd = rangeStart + newFrames;
  
  long i;
  long x;
  long temp;
  
  for (i=0; i<rangeStart; i++)
  {
    if (left)
    {
      temp = getFrame24 (i,0);
      newSamp.setFrame24 (i,0,temp);
    }
    if (right)
    {
      temp = getFrame24 (i,1);
      newSamp.setFrame24 (i,1,temp);
    }
    if (three)
    {
      temp = getFrame24 (i,2);
      newSamp.setFrame24 (i,2,temp);
    }
    if (four)
    {
      temp = getFrame24 (i,3);
      newSamp.setFrame24 (i,3,temp);
    }
  }

  for (i=rangeEnd; i<newBufferLen; i++)
  {
    x = i - newFrames + rangeLen;
    if (left)
    {
      temp = getFrame24 (x,0);
      newSamp.setFrame24 (i,0,temp);
    }
    if (right)
    {
      temp = getFrame24 (x,1);
      newSamp.setFrame24 (i,1,temp);
    }
    if (three)
    {
      temp = getFrame24 (x,2);
      newSamp.setFrame24 (i,2,temp);
    }
    if (four)
    {
      temp = getFrame24 (x,3);
      newSamp.setFrame24 (i,3,temp);
    }
  }
  
  rangeStart         = rangeStart;
  rangeEnd           = rangeStart + rangeLen;
  rangeValid         = TRUE;
  newSamp.rangeStart = rangeStart;
  newSamp.rangeEnd   = rangeStart + newFrames;
  newSamp.rangeValid = TRUE;
  
  // Channel 1 = Left
  if (left && newFrames > 0)
  {
    #ifndef NOFORMS
    if (channels == 4)
      fl_set_object_label
        (stretchForm->label,"Percent Complete - Channel 1");
    else if (channels == 2)
      fl_set_object_label
        (stretchForm->label,"Percent Complete - Left Channel");
    else
      fl_set_object_label
        (stretchForm->label,"Percent Complete");
    #endif
    if (!resample (this,&newSamp,0,R,N,Nw,D,I,P,synt)) return 0;
  }
  #ifndef NOFORMS
  if (stretchCancel) return 0;
  fl_set_slider_value (stretchForm->percentCompSlider,0.0);
  fl_set_object_label (stretchForm->percentComp,"0");
  #endif

  // Channel 2 = Right
  if (right && newFrames > 0)
  {
    #ifndef NOFORMS
    if (channels == 4)
      fl_set_object_label
        (stretchForm->label,"Percent Complete - Channel 2");
    else
      fl_set_object_label
        (stretchForm->label,"Percent Complete - Right Channel");
    #endif
    if (!resample (this,&newSamp,1,R,N,Nw,D,I,P,synt)) return 0;
  }
  #ifndef NOFORMS
  if (stretchCancel) return 0;
  fl_set_slider_value (stretchForm->percentCompSlider,0.0);
  fl_set_object_label (stretchForm->percentComp,"0");
  #endif
  
  // Channel 3
  if (three && newFrames > 0)
  {
    #ifndef NOFORMS
    fl_set_object_label
      (stretchForm->label,"Percent Complete - Channel 3");
    #endif
    if (!resample (this,&newSamp,2,R,N,Nw,D,I,P,synt)) return 0;
  }
  #ifndef NOFORMS
  if (stretchCancel) return 0;
  fl_set_slider_value (stretchForm->percentCompSlider,0.0);
  fl_set_object_label (stretchForm->percentComp,"0");
  #endif
  
  // Channel 4
  if (four && newFrames > 0)
  {
    #ifndef NOFORMS
    fl_set_object_label
      (stretchForm->label,"Percent Complete - Channel 4");
    #endif
    if (!resample (this,&newSamp,3,R,N,Nw,D,I,P,synt)) return 0;
  }
  #ifndef NOFORMS
  if (stretchCancel) return 0;
  fl_set_slider_value (stretchForm->percentCompSlider,0.0);
  fl_set_object_label (stretchForm->percentComp,"0");
  #endif
  
  rangeEnd = rangeStart + newFrames;

  COPYREST
  
  void *tempPointer = data;
  data = newSamp.data;
  newSamp.data = tempPointer;

  frames  = newBufferLen;
  samples = frames * channels;
  bytes   = samples * width;
  
  // Sort out markers
  if (!stereoMono && !quadroMono)
  {
    rangeEnd = rangeStart + rangeLen;
    
    if (susLoopStart >= rangeEnd)
      susLoopStart += newFrames - rangeLen;
    else if (susLoopStart >= rangeStart)
      susLoopStart = 
        rangeStart + ((susLoopStart - rangeStart) * newFrames) / rangeLen;
    if (susLoopEnd >= rangeEnd)
      susLoopEnd += newFrames - rangeLen;
    else if (susLoopEnd >= rangeStart)
      susLoopEnd = 
        rangeStart + ((susLoopEnd - rangeStart) * newFrames) / rangeLen;
    if (relLoopStart >= rangeEnd)
      relLoopStart += newFrames - rangeLen;
    else if (relLoopStart >= rangeStart)
      relLoopStart = 
        rangeStart + ((relLoopStart - rangeStart) * newFrames) / rangeLen;
    if (relLoopEnd >= rangeEnd)
      relLoopEnd += newFrames - rangeLen;
    else if (relLoopEnd >= rangeStart)
      relLoopEnd = 
        rangeStart + ((relLoopEnd - rangeStart) * newFrames) / rangeLen;
    if (displayStart >= rangeEnd)
      displayStart += newFrames - rangeLen;
    else if (displayStart >= rangeStart)
      displayStart = 
        rangeStart + ((displayStart - rangeStart) * newFrames) / rangeLen;
    if (displayEnd >= rangeEnd)
      displayEnd += newFrames - rangeLen;
    else if (displayEnd >= rangeStart)
      displayEnd = 
        rangeStart + ((displayEnd - rangeStart) * newFrames) / rangeLen;

    rangeEnd = rangeStart + newFrames;
  }
  
  // Update sample rate
  setRate (newRate);
  
  changed = TRUE;
  
  return 1;
}


/*---------------------------------------------------------------------------
| Dynamic memory functions
---------------------------------------------------------------------------*/

extraListEntry *DPSample::expand (double newTime)
{
  long newFrames;
  long newBytes;
  
  newFrames = (long) (newTime * rate);
  newBytes  = newFrames * channels * width;

  extraListEntry *e;
  extraListEntry *newE;
  long i;
  
  if (!extra)
  {
    e = new extraListEntry;
    e->start = frames;
    e->end   = e->start + newFrames - 1;
    e->data  = new char [newBytes];
    e->next  = 0;
    if (!(e->data))
    {
      delete e;
      return 0;
    }
    for (i=0; i<newBytes; i++) ((char *) (e->data)) [i] = 0;
    extra = e;
    return e;
  }
  
  e = extra;
  while (e->next) e = e->next;
  newE = new extraListEntry;
  newE->start = e->end + 1;
  newE->end   = newE->start + newFrames - 1;
  newE->data  = new char [newBytes];
  newE->next  = 0;
  if (!(newE->data))
  {
    delete newE;
    return 0;
  }
  for (i=0; i<newBytes; i++) ((char *) (newE->data)) [i] = 0;
  e->next = newE;
  return newE;
}

long DPSample::consolidate ()
{
  int stereo = (channels == 2);
  int quadro = (channels == 4);
  
  if (extra)
  {
    extraListEntry *e;
    e = extra;
    while (e->next) e = e->next;
    long newBufferLen = e->end + 1;
    DPSample newSamp;
    if (!(newSamp.fresh (rate,width,channels,newBufferLen))) return 0;
    
    long i;
    long temp;
    
    for (i=0; i<newBufferLen; i++)
    {
      temp = getFrame24Extra (i,0);
      newSamp.setFrame24 (i,0,temp);
      if (stereo || quadro)
      {
        temp = getFrame24Extra (i,1);
        newSamp.setFrame24 (i,1,temp);
      }
      if (quadro)
      {
        temp = getFrame24Extra (i,2);
        newSamp.setFrame24 (i,2,temp);
        temp = getFrame24Extra (i,3);
        newSamp.setFrame24 (i,3,temp);
      }
    }
    
    void *tempPointer = data;
    data = newSamp.data;
    newSamp.data = tempPointer;

    frames  = newBufferLen;
    samples = frames * channels;
    bytes   = samples * width;
    
    removeExtra ();
  }
  
  return 1;
}

void DPSample::removeExtra ()
{
  if (extra)
  {
    extraListEntry *e;
    extraListEntry *etemp;
    e = extra;
    
    while (e)
    {
      delete [] e->data;
      etemp = e->next;
      delete e;
      e = etemp;
    }
    extra = 0;
  }
}
        
/*---------------------------------------------------------------------------
| General routines
---------------------------------------------------------------------------*/

long getInputRate ()
{
  long pvbuffer [6];
  pvbuffer [0] = AL_INPUT_RATE;
  pvbuffer [2] = AL_INPUT_SOURCE;
  pvbuffer [4] = AL_DIGITAL_INPUT_RATE;
  ALgetparams (AL_DEFAULT_DEVICE,pvbuffer,6);
  if (pvbuffer [1] == AL_RATE_AES_1 || pvbuffer [3] == AL_INPUT_DIGITAL)
  {
    if (ALgetdefault (AL_DEFAULT_DEVICE,AL_DIGITAL_INPUT_RATE) >= 0)
      return pvbuffer [5];
  }
  else if (pvbuffer [1] > 0)
  {
    return pvbuffer [1];
  }
  return AL_RATE_UNDEFINED;
}

long getOutputRate ()
{
  long pvbuffer [4];
  pvbuffer [0] = AL_OUTPUT_RATE;
  pvbuffer [2] = AL_DIGITAL_INPUT_RATE;
  ALgetparams (AL_DEFAULT_DEVICE,pvbuffer,4);
  if (pvbuffer [1] > 0)
  {
    return pvbuffer [1];
  }
  else
  {
    if (pvbuffer [1] == AL_RATE_AES_1)
    {
      if (ALgetdefault (AL_DEFAULT_DEVICE,AL_DIGITAL_INPUT_RATE) >= 0)
        return pvbuffer [3];
    }
    else if (pvbuffer [1] == AL_RATE_INPUTRATE)
    {
      return getInputRate ();
    }
    return AL_RATE_UNDEFINED;
  }
}

void limitQueueSize (long channels,long *queueSize)
{
  switch (channels)
  {
    case 1 :
      if (*queueSize < 512) *queueSize = 512;
      else if (*queueSize > 131068) *queueSize = 131068;
      break;
    case 2 :
      if (*queueSize < 1024) *queueSize = 1024;
      else if (*queueSize > 262136) *queueSize = 262136;
      break;
    case 4 :
      if (*queueSize < 2048) *queueSize = 2048;
      else if (*queueSize > 262136) *queueSize = 262136;
      break;
    default :
      break;
  }
}

/***************************************************************************/
