/* chardev.c 
 * 
 * Create an input/output character device
 */


/* Copyright (C) 1998 by Ori Pomerantz */



/* The necessary header files */

/* Standard in kernel modules */
#include <linux/kernel.h>   /* We're doing kernel work */
#include <linux/module.h>   /* Specifically, a module */

/* Deal with CONFIG_MODVERSIONS */
#if CONFIG_MODVERSIONS==1
#define MODVERSIONS
#include <linux/modversions.h>
#endif        

/* For character devices */
#include <linux/fs.h>       /* The character device definitions are here */
#include <linux/wrapper.h>  /* A wrapper which does next to nothing at
                             * at present, but may help for compatibility
                             * with future versions of Linux */
			     
/* Our own ioctl numbers */
#include "chardev.h"


#define SUCCESS 0


/* Device Declarations *********************************************** */


/* The name for our device, as it will appear in /proc/devices */
#define DEVICE_NAME "char_dev"


/* The maximum length of the message for the device */
#define BUF_LEN 80

/* Is the device open right now? Used to prevent concurent access into
 * the same device */
static int Device_Open = 0;

/* The message the device will give when asked */
static char Message[BUF_LEN];

/* How far did the process reading the message get? Useful if the
 * message is larger than the size of the buffer we get to fill in
 * device_read. */
static char *Message_Ptr;


/* This function is called whenever a process attempts to open the device
 * file */
static int device_open(struct inode *inode, struct file *file)
{
#ifdef DEBUG
  printk ("device_open(%p,%p)\n", inode, file);
#endif

  /* We don't want to talk to two processes at the same time */
  if (Device_Open)
    return -EBUSY;

  /* If this was a process, we would have had to be more careful here,
   * because one process might have checked Device_Open right before the
   * other one tried to increment it. However, we're in the kernel, so
   * we're protected against context switches.
   */ 

  Device_Open++;

  /* Initialize the message */
  Message_Ptr = Message;

  MOD_INC_USE_COUNT;

  return SUCCESS;
}


/* This function is called when a process closes the device file. It
 * doesn't have a return value because it cannot fail. Regardless of what
 * else happens, you should always be able to close a device. */
static void device_release(struct inode *inode, struct file *file)
{
#ifdef DEBUG
  printk ("device_release(%p,%p)\n", inode, file);
#endif
 
  /* We're now ready for our next caller */
  Device_Open --;

  MOD_DEC_USE_COUNT;
}


/* This function is called whenever a process which already opened the
 * device file attempts to read from it. */
static int device_read(struct inode *inode,
                       struct file *file,
                       char *buffer,   /* The buffer to fill with the data */ 
                       int length)     /* The length of the buffer 
                                        * (mustn't write beyond that!) */

{
  /* Number of bytes actually written to the buffer */
  int bytes_read = 0;

#ifdef DEBUG
  printk("device_read(%p,%p,%p,%d)\n",
    inode, file, buffer, length);
#endif

  /* If we're at the end of the message, return 0 (which signifies end
   * of file) */
  if (*Message_Ptr == 0)
    return 0;

  /* Actually put the data into the buffer */
  while (length && *Message_Ptr)  {

    /* Because the buffer is in the user data segment, not the kernel 
     * data segment, assignment wouldn't work. Instead, we have to use
     * put_user which copies data from the kernel data segment to the user
     * data segment. */
    put_user(*(Message_Ptr++), buffer++);
    length --;
    bytes_read ++;
  }

#ifdef DEBUG
   printk ("Read %d bytes, %d left\n",
     bytes_read, length);
#endif

   /* Read functions are supposed to return the number of bytes actually
    * inserted into the buffer */
  return bytes_read;
}


/* This function is called when somebody tries to write into our device
 * file. */ 
static int device_write(struct inode *inode,
                        struct file *file,
                        const char *buffer,
                        int length)
{
  int i;

  return 1;

#ifdef DEBUG
  printk ("device_write(%p,%p,%s,%d)",
    inode, file, buffer, length);
#endif

  for(i=0; i<length && i<BUF_LEN; i++) 
    Message[i] = get_user(buffer+i);
  
  Message_Ptr = Message;

  /* Again, return the number of input characters used */
  return i;
}


/* This function is called whenever a process tries to do an ioctl on
 * our device file. We get two extra parameters (additional to the
 * inode and file structures, which all device functions get): the number
 * of the ioctl called and the parameter given to the ioctl function.
 *
 * If the ioctl is write or read/write (meaning output is returned to 
 * the calling process), the ioctl call returns the output of this 
 * function.
 */
int device_ioctl(struct inode *inode,
                 struct file *file,
                 unsigned int ioctl_num,    /* The number of the ioctl */
                 unsigned long ioctl_param) /* The parameter to it */
{
  int i;
  char *temp;

  /* Switch according to the ioctl called */
  switch (ioctl_num) {
    case IOCTL_SET_MSG:
      /* Receive a pointer to a message (in user space) and set that to
       * be the device's message. */ 

      /* Get the parameter given to ioctl by the process */
      temp = (char *) ioctl_param;
   
      /* Find the length of the message */
      for (i=0; get_user(temp) && i<BUF_LEN; i++, temp++)
        ;

      /* Don't reinvent the wheel - call device_write */
      device_write(inode, file, (char *) ioctl_param, i);
      break;

    case IOCTL_GET_MSG:
      /* Give the current message to the calling process - the parameter
       * we got is a pointer, fill it. */
      i = device_read(inode, file, (char *) ioctl_param, 99); 
      /* Warning - we assume here the buffer length is 100. If it's less
       * than that we might overflow the buffer, causing the process to
       * core dump. 
       *
       * The reason we only allow up to 99 characters is that the NULL
       * which terminates the string also needs room. */

      /* Put a zero at the end of the buffer, so it will be properly 
       * terminated */
      put_user('\0', (char *) ioctl_param+i);
      break;

    case IOCTL_GET_NTH_BYTE:
      /* This ioctl is both input (ioctl_param) and output (the return
       * value of this function) */
      return Message[ioctl_param];
      break;
  }

  return SUCCESS;
}


/* Module Declarations ********************************************** */


/* This structure will hold the functions to be called when 
 * a process does something to the device we created. Since a pointer to
 * this structure is kept in the devices table, it can't be local to
 * init_module. NULL is for unimplemented functions. */

struct file_operations Fops = {
  NULL,   /* seek */
  device_read, 
  device_write,
  NULL,   /* readdir */
  NULL,   /* select */
  device_ioctl,   /* ioctl */
  NULL,   /* mmap */
  device_open,
  device_release  /* a.k.a. close */
};


/* Initialize the module - Register the character device */
int init_module()
{
  int ret_val;

  /* Register the character device (atleast try) */
  ret_val = module_register_chrdev(MAJOR_NUM, 
                                 DEVICE_NAME,
                                 &Fops);

  /* Negative values signify an error */
  if (ret_val < 0) {
    printk ("Sorry, registering the character device failed with %d\n",
      ret_val);
    return ret_val;
  }

  printk ("Registeration is a success. The major device number is %d.\n",
    MAJOR_NUM);
  printk ("If you want to talk to the device driver, you'll have to\n");
  printk ("create a device file. We suggest you use:\n");
  printk ("mknod %s c %d 0\n", DEVICE_FILE_NAME, MAJOR_NUM);
  printk ("The device file name is important, because the ioctl program\n");
  printk ("assumes that's the file you'll use.\n");

  return 0;
}


/* Cleanup - unregister the appropriate file from /proc */
void cleanup_module()
{
  int ret;

  /* Unregister the device */
  ret = module_unregister_chrdev(MAJOR_NUM, DEVICE_NAME);
 
  /* If there's an error, report it */ 
  if (ret < 0)
    printk("Error in module_unregister_chrdev: %d\n", ret);
}  



