/*
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Brian W. Barrett, Arun F. Rodrigues, Jeffrey M. Squyres,
 * 	 and Andrew Lumsdaine
 *
 * This file is part of XMPI
 *
 * You should have received a copy of the License Agreement for XMPI 
 * along with the software; see the file LICENSE.  If not, contact 
 * Office of Research, University of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 *
 * Additional copyrights may follow.

 *
 *	$Id: xmpi_run.c,v 1.7 1999/11/08 06:20:26 bbarrett Exp $
 * 
 *	Function:	- application specification panel in
 *			  the builder dialog
 */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#define _NO_PROTO

#include <Xm/ArrowBG.h>
#include <Xm/DialogS.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/List.h>
#include <Xm/PanedW.h>
#include <Xm/PushBG.h>
#include <Xm/RowColumn.h>
#include <Xm/SelectioB.h>
#include <Xm/TextF.h>
#include <Xm/ToggleB.h>

#include "all_list.h"
#include "app_mgmt.h"
#include "app_schema.h"
#include "lam.h"
#include "xmpi.h"

/*
 * public functions
 */
void			xmpi_run_dialog();
int			xmpi_run();
void			xmpi_run_set_nodes();
void			xmpi_run_set_prog();
void			xmpi_run_delete();
void			xmpi_run_rerun();
void			xmpi_run_cleanup();
void			xmpi_run_busy();
void			xmpi_run_unbusy();

/*
 * external functions
 */
extern char *		getworkdir();

/*
 * private functions
 */
static void		run_run_cb();
static int		run_save();
static void		run_commit_cb();
static void		run_cancel_cb();
static void		run_save_cb();
static void		run_save_ok_cb();

/*
 * public variables
 */
int			xmpi_do_rerun = 0;	/* re-running application? */
int			xmpi_app_nprocs = 0;	/* number of procs in appl. */
struct _gps		*xmpi_app_procs = 0;	/* appl. GPS array */
Widget			xmpi_run_w = 0;		/* run dialog shell */

/*
 * external variables
 */
extern int		xmpi_trace;		/* tracing enabled? */

/*
 * private variables
 */
static char		*custom = 0;		/* custom aschema filename */
static char		prog_dir[XMPI_PATHMAX] = ""; /* dir. selected prog. */
static char		tempfile[32];
static Widget		dialog_mgr_w;		/* dialog manager */
static Widget		args_w;
static Widget		fullpath_w;
static Widget		prog_w;
static Widget		copies_w;
static Widget		node_w;
static Widget		list_w;
static Widget		transfer_w;

static char		delete_trans[] = "#override\n\
	<Key>osfDelete: xmpi_run_delete()\n\
	<Key>osfBackSpace: xmpi_run_delete()";

/*
 *	xmpi_run_dialog
 *
 *	Function:	- creates widgets for mpirun dialog window
 *			- pops up dialog
 *	Accepts:	- parent widget
 *			- initial boot schema
 */
void
xmpi_run_dialog(parent_w)

Widget			parent_w;

{
	Widget		action_mgr_w;	/* action area manager */
	Widget		prog_mgr_w;	/* program area manager */
	Widget		mgr_w;		/* prog/appl panel manager */
	Widget		w;
	Dimension	width;		/* alignment of labels */
	XmString	xstr;
	Arg		args[1];

	if (xmpi_run_w) {
/*
 * Dialog window already created, just pop it on the screen.
 */
		XtMapWidget(xmpi_wmshell(xmpi_run_w));
		XtPopup(xmpi_run_w, XtGrabNone);
		XtManageChild(dialog_mgr_w);
		return;
	}
/*
 * Create the dialog window.
 */
	xmpi_run_w = XtVaCreatePopupShell("buildrun_pop",
		xmDialogShellWidgetClass, xmpi_wmshell(parent_w),
		XmNdeleteResponse, XmUNMAP,
		XmNtitle, "XMPI Application Builder",
		XmNminHeight, 300,
		NULL);

	dialog_mgr_w = XtVaCreateWidget("buildrun_mgr",
		xmFormWidgetClass, xmpi_run_w,
		XmNmarginHeight, 1,
		XmNmarginWidth, 1,
		XmNfractionBase, 3,
		NULL);

	mgr_w = XtVaCreateWidget("app_mgr",
                xmPanedWindowWidgetClass, dialog_mgr_w,
                XmNsashWidth, 1,
                XmNsashHeight, 1,
                XmNseparatorOn, False,
                XmNleftAttachment, XmATTACH_FORM,
                XmNrightAttachment, XmATTACH_POSITION,
                XmNrightPosition, 1,
                XmNrightOffset, 5,
                XmNleftOffset, 1,
                XmNtopAttachment, XmATTACH_FORM,
                XmNbottomAttachment, XmATTACH_FORM,
                NULL);

	w = XtVaCreateManagedWidget("prog_frame",
		xmFrameWidgetClass, mgr_w,
		XmNmarginHeight, 3,
		XmNmarginWidth, 3,
		XmNshadowThickness, 2,
		NULL);

	prog_mgr_w = XtVaCreateWidget("prog_mgr",
		xmFormWidgetClass, w,
		XmNfractionBase, 7,
		NULL);
/*
 * Build the target nodes entry area.
 */
	w = XtVaCreateManagedWidget("Node:",
		xmLabelGadgetClass, prog_mgr_w,
		XmNalignment, XmALIGNMENT_BEGINNING,
		XmNtopAttachment, XmATTACH_POSITION,
		XmNtopPosition, 3,
		XmNbottomAttachment, XmATTACH_POSITION,
		XmNbottomPosition, 4,
		XmNleftAttachment, XmATTACH_FORM,
		NULL);
	XtVaGetValues(w, XmNwidth, &width, NULL);
	node_w = XtVaCreateManagedWidget("node_text",
		xmTextFieldWidgetClass, prog_mgr_w,
		XmNtopAttachment, XmATTACH_POSITION,
		XmNtopPosition, 3,
		XmNbottomAttachment, XmATTACH_POSITION,
		XmNbottomPosition, 4,
		XmNleftAttachment, XmATTACH_WIDGET,
		XmNleftWidget, w,
		XmNrightAttachment, XmATTACH_FORM,
		NULL);
/*
 * Build the program name entry area.
 */
	w = XtVaCreateManagedWidget("Prog:",
		xmLabelGadgetClass, prog_mgr_w,
		XmNalignment, XmALIGNMENT_BEGINNING,
		XmNtopAttachment, XmATTACH_POSITION,
		XmNtopPosition, 0,
		XmNbottomAttachment, XmATTACH_POSITION,
		XmNbottomPosition, 1,
		XmNleftAttachment, XmATTACH_FORM,
		XmNwidth, width,
		NULL);
	prog_w = XtVaCreateManagedWidget("prog_text",
		xmTextFieldWidgetClass, prog_mgr_w,
		XmNtopAttachment, XmATTACH_POSITION,
		XmNtopPosition, 0,
		XmNbottomAttachment, XmATTACH_POSITION,
		XmNbottomPosition, 1,
		XmNleftAttachment, XmATTACH_WIDGET,
		XmNleftWidget, w,
		XmNrightAttachment, XmATTACH_FORM,
		NULL);
/*
 * Build the number of copies entry area.
 */
	w = XtVaCreateManagedWidget("Copy:",
		xmLabelGadgetClass, prog_mgr_w,
		XmNalignment, XmALIGNMENT_BEGINNING,
		XmNtopAttachment, XmATTACH_POSITION,
		XmNtopPosition, 1,
		XmNbottomAttachment, XmATTACH_POSITION,
		XmNbottomPosition, 2,
		XmNleftAttachment, XmATTACH_FORM,
		XmNwidth, width,
		NULL);
	copies_w = xmpi_copies_create(prog_mgr_w);
	XtVaSetValues(copies_w,
		XmNtopAttachment, XmATTACH_POSITION,
		XmNtopPosition, 1,
		XmNbottomAttachment, XmATTACH_POSITION,
		XmNbottomPosition, 2,
		XmNleftAttachment, XmATTACH_WIDGET,
		XmNleftWidget, w,
		XmNrightAttachment, XmATTACH_FORM,
		NULL);
	XtManageChild(copies_w);
/*
 * Build the program arguments entry area.
 */
	w = XtVaCreateManagedWidget("Args:",
		xmLabelGadgetClass, prog_mgr_w,
		XmNalignment, XmALIGNMENT_BEGINNING,
		XmNtopAttachment, XmATTACH_POSITION,
		XmNtopPosition, 2,
		XmNbottomAttachment, XmATTACH_POSITION,
		XmNbottomPosition, 3,
		XmNleftAttachment, XmATTACH_FORM,
		XmNwidth, width,
		NULL);
	args_w = XtVaCreateManagedWidget("args_text",
		xmTextFieldWidgetClass, prog_mgr_w,
		XmNtopAttachment, XmATTACH_POSITION,
		XmNtopPosition, 2,
		XmNbottomAttachment, XmATTACH_POSITION,
		XmNbottomPosition, 3,
		XmNleftAttachment, XmATTACH_WIDGET,
		XmNleftWidget, w,
		XmNrightAttachment, XmATTACH_FORM,
		NULL);
/*
 * Build the "Transfer Program" toggle button.
 */
	xstr = XmStringCreateSimple("Transfer Program");
	transfer_w = XtVaCreateManagedWidget("transfer_toggle",
		xmToggleButtonWidgetClass, prog_mgr_w,
		XmNalignment, XmALIGNMENT_BEGINNING,
		XmNnavigationType, XmTAB_GROUP,
		XmNtopAttachment, XmATTACH_POSITION,
		XmNtopPosition, 4,
		XmNbottomAttachment, XmATTACH_POSITION,
		XmNbottomPosition, 5,
		XmNleftAttachment, XmATTACH_FORM,
		XmNlabelString, xstr,
		NULL);
	XmStringFree(xstr);
/*
 * Build the "Use Full Pathname" toggle button.
 */
	xstr = XmStringCreateSimple("Use Full Pathname");
	fullpath_w = XtVaCreateManagedWidget("fullpath_toggle",
		xmToggleButtonWidgetClass, prog_mgr_w,
		XmNalignment, XmALIGNMENT_BEGINNING,
		XmNnavigationType, XmTAB_GROUP,
		XmNtopAttachment, XmATTACH_POSITION,
		XmNtopPosition, 5,
		XmNbottomAttachment, XmATTACH_POSITION,
		XmNbottomPosition, 6,
		XmNleftAttachment, XmATTACH_FORM,
		XmNlabelString, xstr,
		NULL);
	XmStringFree(xstr);
/*
 * Build the commit button.
 */
	w = XtVaCreateManagedWidget("commit_btn",
		xmArrowButtonGadgetClass, prog_mgr_w,
		XmNarrowDirection, XmARROW_DOWN,
		XmNnavigationType, XmTAB_GROUP,
		XmNtopAttachment, XmATTACH_POSITION,
		XmNtopPosition, 6,
		XmNbottomAttachment, XmATTACH_POSITION,
		XmNbottomPosition, 7,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		NULL);
	XtAddCallback(w, XmNactivateCallback, run_commit_cb, xmpi_run_w);

	XtManageChild(prog_mgr_w);

	xstr = XmStringCreateSimple("Current Application");
	w = XtVaCreateManagedWidget("banner",
		xmLabelWidgetClass, mgr_w,
		XmNlabelString, xstr,
		NULL);
	XmStringFree(xstr);
/*
 * Build the application scrolled list.
 */
	XtSetArg(args[0], XmNlistSizePolicy, XmCONSTANT);
	list_w = XmCreateScrolledList(mgr_w, "app_list", args, 1);
	XtVaSetValues(list_w,
		XmNscrollBarDisplayPolicy, XmSTATIC,
		XmNselectionPolicy, XmEXTENDED_SELECT,
		NULL);
	XtOverrideTranslations(list_w, XtParseTranslationTable(delete_trans));

	XtManageChild(list_w);
/*
 * Create the dialog action area.
 */
	action_mgr_w = XtVaCreateWidget("action_mgr",
		xmRowColumnWidgetClass, mgr_w,
		XmNorientation, XmHORIZONTAL,
		XmNnumColumns, 1,
		XmNpacking, XmPACK_COLUMN,
		XmNisAligned, True,
		XmNentryAlignment, XmALIGNMENT_CENTER,
		XmNmarginWidth, 0,
		XmNspacing, 0,
		XmNskipAdjust, True,
		NULL);
/*
 * Create the "Run" button.
 */
	w = XtVaCreateManagedWidget("Run",
		xmPushButtonGadgetClass, action_mgr_w,
		XmNshowAsDefault, True,
		XmNdefaultButtonShadowThickness, 1,
		NULL);
	XtAddCallback(w, XmNactivateCallback, run_run_cb, xmpi_run_w);
/*
 * Create the "Save" button.
 */
	w = XtVaCreateManagedWidget("Save",
		xmPushButtonGadgetClass, action_mgr_w,
		XmNshowAsDefault, False,
		XmNdefaultButtonShadowThickness, 1,
		NULL);
	XtAddCallback(w, XmNactivateCallback, run_save_cb, xmpi_run_w);
/*
 * Create the "Cancel" button.
 */
	w = XtVaCreateManagedWidget("Cancel",
		xmPushButtonGadgetClass, action_mgr_w,
		XmNshowAsDefault, False,
		XmNdefaultButtonShadowThickness, 1,
		NULL);
	XtAddCallback(w, XmNactivateCallback, run_cancel_cb, xmpi_run_w);

	XtManageChild(action_mgr_w);
	xmpi_nosash(mgr_w);
	XtManageChild(mgr_w);
/*
 * Build program browser.
 */
	xmpi_browse_build(dialog_mgr_w);
/*
 * Build node browser.
 */
	xmpi_nodes_build(dialog_mgr_w);

	XtManageChild(dialog_mgr_w);
/*
 * Pop up the window.
 */
	XtPopup(xmpi_run_w, XtGrabNone);
}

/*
 *	xmpi_run
 *
 *	Function:	- runs the supplied application schema
 *	Accepts:	- application schema
 *			- popup widget invoking the run if any
 *	Returns:	- 0 or LAMERROR
 */
int
xmpi_run(aschema, w)

char			*aschema;
Widget			w;

{
/*
 * Clean away existing application and trace windows.
 */
	xmpi_ctl_setsensitive(XMPI_BKIVIAT, False);
	if (!xmpi_do_rerun) {
		xmpi_mat_destroy();
	}
	xmpi_kv_popdown();
	xmpi_tr_destroy();
	xmpi_vw_dbsetmode(0);
	xmpi_ctl_clean();
	xmpi_busy();

	if (w) {
		XtPopdown(w);
	}
/*
 * Run the application.
 */
	xmpi_ctl_setinfo("Starting application...");
	xmpi_flush();

	if (xmpi_sys_run(aschema, &xmpi_app_procs, &xmpi_app_nprocs)) {
		xmpi_do_rerun = 0;
		xmpi_unbusy();
		xmpi_ctl_resetinfo();
		return(LAMERROR);
	}
/*
 * Initialize and update the view window.
 */
	if (xmpi_do_rerun && xmpi_vw_getprocs()) {
		xmpi_vw_update(1);
	} else {
		xmpi_vw_init();
	}

	xmpi_do_rerun = 0;

	if (xmpi_trace) {
		xmpi_ctl_setsensitive(XMPI_BTRACE, True);
	}

	xmpi_ctl_setsensitive(XMPI_BSNAPSHOT, True);
	xmpi_ctl_setsensitive(XMPI_BRERUN, True);
	xmpi_ctl_setsensitive(XMPI_BMATRIX, True);
	xmpi_ctl_resetinfo();
	xmpi_unbusy();

	return(0);
}

/*
 *	xmpi_run_set_nodes
 *
 *	Function:	- sets the nodes text field
 *	Accepts:	- nodes string
 */
void
xmpi_run_set_nodes(p)

char			*p;

{
	XmTextFieldSetString(node_w, p);
}

/*
 *	xmpi_run_set_prog
 *
 *	Function:	- sets the prog text field
 *			- has side effect of changing base directory
 *	Accepts:	- program filename
 *			- program directory
 */
void
xmpi_run_set_prog(prog, dir)

char			*prog;
char			*dir;

{
	strcpy(prog_dir, dir);
	if (prog && *prog) {
		XmTextFieldSetString(prog_w, prog);
	}
}

/*
 *	xmpi_run_delete
 *
 *	Function:	- deletes all selected programs
 */
void
xmpi_run_delete()

{
	int		*pposn;		/* item position array */
	int		nposn;		/* # positions */

	while (XmListGetSelectedPos(list_w, &pposn, &nposn)) {
		XmListDeletePos(list_w, pposn[0]);
		XtFree((char*)pposn);
	}
}

/*
 *	xmpi_run_rerun
 *
 *	Function:	- reruns previous application schema
 */
void
xmpi_run_rerun()

{
	if (custom) {
		xmpi_do_rerun = 1;
		xmpi_run(custom, (Widget) 0);
	}
}

/*
 *	xmpi_run_cleanup
 *
 *	Function:	- removes custom app schema file, if any
 */
void
xmpi_run_cleanup()

{
	if (custom) {
		unlink(custom);
		custom = 0;
	}
}

/*
 *	run_run_cb
 *
 *	Function:	- creates application schema from appl list
 *			- runs the application
 *	Accepts:	- parent widget
 */
static void
run_run_cb(w)

Widget			w;

{
	xmpi_run_cleanup();

	strcpy(tempfile, TMP);
	strcat(tempfile, "/xmpiXXXXXX");
	mktemp(tempfile);
	custom = tempfile;

	if (!run_save(custom)) {
		XtUnmapWidget(xmpi_wmshell(w));

		if (!xmpi_run(custom, xmpi_run_w)) {
			xmpi_ctl_setapp("<custom built>");
		}
	}
}

/*
 *	run_save
 *
 *	Function:	- saves appl list contents to a file
 *	Accepts:	- save filename
 *	Returns:	- 0 or LAMERROR
 */
static int
run_save(filename)

char			*filename;

{
	int		fd;
	int		i;
	int		nitems;		/* # appl list items */
	XmString	*items;		/* appl list items */
	char		*progline;	/* one line in appl schema */

	XtVaGetValues(list_w,
		XmNitemCount, &nitems,
		XmNitems, &items,
		NULL);

	if (nitems <= 0) {
		errno = 0;
		xmpi_error(xmpi_run_w, "No programs specified.");
		
		return(LAMERROR);
	}

	fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, 0644);

	if (fd < 0) {
		xmpi_error(xmpi_run_w, filename);
		return(LAMERROR);
	}

	for (i = 0; i < nitems; ++i) {
		XmStringGetLtoR(items[i], XmSTRING_DEFAULT_CHARSET,
			&progline);
		write(fd, progline, strlen(progline));
		write(fd, "\n", 1);
		XtFree(progline);
	}

	close(fd);
	return(0);
}

/*
 *	run_commit_cb
 *
 *	Function:	- adds a line to the application schema
 */
static void
run_commit_cb()

{
	XmString	xstr;
	char		*args;
	char		*linebuf;	/* one line in appl schema */
	char		*nodes;
	char		*copies;
	char		*prog;
	int		ncopy = 0;
	unsigned int	size;		/* line buffer size */
/*
 * Retrieve program name.
 */
	prog = XmTextFieldGetString(prog_w);

	if ((prog == 0) || (*prog == '\0')) {
		errno = 0;
		xmpi_error(xmpi_run_w, "No program specified.");
		XtFree(prog);
		return;
	}
/*
 * Retrieve target nodes.
 */
	nodes = XmTextFieldGetString(node_w);

	if ((nodes == 0) || (*nodes == '\0')) {
		errno = 0;
		xmpi_error(xmpi_run_w, "No nodes specified.");
		XtFree(prog);
		return;
	}

	size = strlen(prog_dir) + strlen(prog) + strlen(nodes) + 2;
/*
 * Retrieve target copies. Add extra room for -c.
 */
	copies = xmpi_copies_get();

	if ((copies != 0) && (*copies != '\0')) {
		ncopy = stoi((unsigned char *)copies);
		if (ncopy >= 1) {
			size += strlen(copies) + 5;
		}
	}

	size = strlen(prog_dir) + strlen(prog) + strlen(nodes) + 2;
/*
 * Retrieve arguments.
 */
	args = XmTextFieldGetString(args_w);

	if ((args != 0) && (*args != '\0')) {
		size += strlen(args) + 4;
	}
/*
 * Give extra room for a possible transfer option.
 */
	size += 10;

	if ((linebuf = malloc(size)) == 0) xmpi_fail("xmpi (malloc)");

	strcpy(linebuf, nodes);
	if (ncopy >= 1) {
		strcat(linebuf, " -c ");
		strcat(linebuf, copies);
	}
	strcat(linebuf, " ");
/*
 * Retrieve transfer toggle.
 */
	if (XmToggleButtonGetState(transfer_w)) {
		strcat(linebuf, "-s h ");
	}
/*
 * Retrieve fullpath toggle.
 */
	if (XmToggleButtonGetState(fullpath_w)) {
		if (prog[0] != STRDIR && prog_dir[0]) {
			strcat(linebuf, prog_dir);
			if (prog_dir[strlen(prog_dir) - 1] != STRDIR) {
				strcat(linebuf, STRSDIR);
			}
		}
	}

	strcat(linebuf, prog);

	if ((args != 0) && (*args != '\0')) {
		strcat(linebuf, " -- ");
		strcat(linebuf, args);
	}
/*
 * Add full program line to application list.
 */
	xstr = XmStringCreateSimple(linebuf);
	XmListAddItem(list_w, xstr, 0);
	XmStringFree(xstr);
/*
 * Make the last item visible.
 */
	XmListSetBottomPos(list_w, 0);
	free(linebuf);

	/*
	  Free things
	 */
	XtFree(prog);
	XtFree(args);
	XtFree(nodes);
}

/*
 *	run_cancel_cb
 *
 *	Function:	- cancels the dialog with no processing
 */
static void
run_cancel_cb()

{
	XtPopdown(xmpi_run_w);
}

/*
 *	run_save_cb
 *
 *	Function:	- pops up prompt dialog for appl schema filename
 *	Accepts:	- called back from which widget
 */
static void
run_save_cb(from_w)

Widget			from_w;

{
	Widget		w;
	XmString	str1, str2;
	Arg		args[2];
	char		*cwd;
	char		cwdir[XMPI_PATHMAX];

	cwd = getworkdir();
	if (cwd) {
		strcpy(cwdir, cwd);
		strcat(cwdir, "/");
		free(cwd);
	} else {
		cwdir[0] = 0;
	}
	
	str1 = XmStringCreateSimple("Application Schema filename:");
	str2 = XmStringCreateSimple(cwdir);
	
	XtSetArg(args[0], XmNselectionLabelString, str1);
	XtSetArg(args[1], XmNtextString, str2);

	w = XmCreatePromptDialog(xmpi_wmshell(from_w), "save_dialog", args, 2);

	XmStringFree(str1);
	XmStringFree(str2);

	XtAddCallback(w, XmNcancelCallback,
			(XtCallbackProc) XtDestroyWidget, NULL);
	XtAddCallback(w, XmNokCallback, run_save_ok_cb, NULL);
	XtUnmanageChild(XmSelectionBoxGetChild(w, XmDIALOG_HELP_BUTTON));

	XtManageChild(w);
	XtPopup(XtParent(w), XtGrabNone);
}

/*
 *	run_save_ok_cb
 *
 *	Function:	- extracts selected filename
 *			- saves current appl schema into filename
 *	Accepts:	- selection dialog callback parameters
 */
static void
run_save_ok_cb(parent_w, cdata, cbs)

Widget			parent_w;
XtPointer		cdata;
XmSelectionBoxCallbackStruct
			*cbs;

{
	struct stat 	finfo;			/* file information */
	char		*filename;		/* appl schema filename */
	int		exists = 1;
	
	XmStringGetLtoR(cbs->value, XmSTRING_DEFAULT_CHARSET, &filename);

	if (!filename || (*filename == '\0')) return;
/*
 * Check if file already exists and if so ask user if they want to overwrite
 * it or not.
 */
	if (stat(filename, &finfo)) {
		if (errno != ENOENT) {
			xmpi_error(parent_w, filename);
			XtFree(filename);
			return;
		}
		exists = 0;
	}

	if (exists && !xmpi_yesno(parent_w, "File exists. Overwrite?")) {
		XtFree(filename);
		return;
	}
	
	(void) run_save(filename);
	XtFree(filename);
	XtDestroyWidget(parent_w);
}

/*
 *	xmpi_run_busy
 *
 *	Function:	- set the busy cursor for the Build&Run window
 */
void
xmpi_run_busy()

{
	if (xmpi_run_w) xmpi_busy_widget(xmpi_run_w);
}

/*
 *	xmpi_run_unbusy
 *
 *	Function:	- reset the busy cursor for the Build&Run window
 */
void
xmpi_run_unbusy()

{
	if (xmpi_run_w) xmpi_unbusy_widget(xmpi_run_w);
}
