(**
  Implements a handler for keyboards focus chains.
**)

MODULE VOKeyHandler;

(*
    Keyboard focus chains.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)


IMPORT F := VOFrame,
       E := VOEvent,
       G := VOGUIObject;

CONST
  (* special shortcut modes *)

  none    * = 0;
  return  * = 1;
  escape  * = 2;
  default * = 3;

TYPE
  ShortcutEntry     * = POINTER TO ShortcutEntryDesc;

  (**
    Store one shortcut entry.
  **)

  ShortcutEntryDesc * = RECORD
                          next      : ShortcutEntry;
                          gadget    : G.Gadget;
                          qualifier : SET;
                          char      : CHAR;
                          id,mode   : LONGINT;
                        END;

  KeyChainEntry     = POINTER TO KeyChainEntryDesc;

  (**
    Entry in the KeyChain list.
  **)

  KeyChainEntryDesc = RECORD
                        next,
                        last   : KeyChainEntry;
                        gadget : G.Gadget;
                      END;

  KeyChain        = POINTER TO KeyChainDesc;

  (**
    A KeyHandler has a list of KeyChains. Pressing the tabkey
    switches the focus to another object in the same keychain.
    If an objects gets activated the KeyHandler checks, if the object
    is in a KeyChain and makes this KeyChain the new KeyChain. Also
    the current KeyChain gets changed, when the current KeyChain holds
    no valid object. This happens when all objects in a keychain
    are hidden aka not currently visible.
  **)

  KeyChainDesc    = RECORD
                      first,
                      last   : KeyChainEntry;
                    END;

  KeyHandler*     = POINTER TO KeyHandlerDesc;

  (**
    Every window has a KeyHandler. The KeyHandler does handle
    keyevents. It is responsible for switching the keyboardfocus
    and the handling of hotkeys. The use of a Keyhandler it optional
    for a window. The KeyHandler accepts only Gadgets as objects.
  **)

  KeyHandlerDesc* = RECORD
                      first,
                      last    : KeyChain;

                      active  : BOOLEAN;

                      cChain  : KeyChain;
                      cEntry  : KeyChainEntry;

                      sFirst,
                      sLast,
                      current : ShortcutEntry;
                    END;

  (**
    Checks if the given shortcut entry matches the given key.
  **)

  PROCEDURE (sc : ShortcutEntry) Match(qualifier : SET; keysym : LONGINT; char  : CHAR):BOOLEAN;

  VAR
    found : BOOLEAN;

  BEGIN
    qualifier:=qualifier-E.shiftMask;

    found:=FALSE;
    IF (sc.qualifier=qualifier) & (sc.char=CAP(char)) THEN
      found:=TRUE;
    ELSIF ((sc.mode=return) & (keysym=E.return))
       OR ((sc.mode=escape) & (keysym=E.escape))
       OR ((sc.mode=default) & ((keysym=E.return) OR (keysym=E.escape))) THEN
      found:=TRUE;
    END;

    RETURN found & ((sc.gadget.visible) OR (G.scAlways IN sc.gadget.flags));
  END Match;

  (**
    Initializes the KeyChain. Must be called first after allocation.
  **)

  PROCEDURE (k : KeyChain) Init;

  BEGIN
    k.first:=NIL;
    k.last:=NIL;
  END Init;

  (**
    Adds a new gadget to the current KeyChain.
  **)

  PROCEDURE (k : KeyChain) Add(gadget : G.Gadget);

  VAR
    entry : KeyChainEntry;

  BEGIN
    ASSERT(gadget#NIL);

    NEW(entry);
    entry.gadget:=gadget;
    entry.next:=NIL;
    IF k.first=NIL THEN
      k.first:=entry;
      entry.last:=NIL;
    ELSE
      k.last.next:=entry;
      entry.last:=k.last;
    END;
    k.last:=entry;
  END Add;

  (**
    The user has pressed tab and wants the KeyHandler to deactivate the
    current and to activate the next valid object.
  **)

  PROCEDURE (k : KeyChain) NextObject(entry : KeyChainEntry):KeyChainEntry;

  VAR
    help : KeyChainEntry;

  BEGIN
    IF entry=NIL THEN
      entry:=k.first;
    END;

    help:=entry;

    IF help=NIL THEN
      RETURN NIL;
    END;

    REPEAT
      IF (help=NIL) OR (help.next=NIL) THEN
        help:=k.first;
      ELSE
        help:=help.next;
      END;
      IF help.gadget.visible & ~help.gadget.disabled THEN
        RETURN help;
      END;
    UNTIL help=entry;
    RETURN NIL;
  END NextObject;

  (**
    The user has pressed shift tab and wants the KeyHandler to deactivate the
    current and to activate the last valid object.
  **)

  PROCEDURE (k : KeyChain) LastObject(entry : KeyChainEntry):KeyChainEntry;

  VAR
    help : KeyChainEntry;

  BEGIN
    IF entry=NIL THEN
      help:=k.last;
    ELSE
      help:=entry;
    END;

    IF help=NIL THEN
      RETURN NIL;
    END;

    REPEAT
      help:=help.last;
      IF help=NIL THEN
        help:=k.last;
      END;
      IF help.gadget.visible & ~help.gadget.disabled THEN
        RETURN help;
      END;
    UNTIL help=entry;
    RETURN NIL;
  END LastObject;

  (**
    Searches for the given gadget in the KeyChain and returns the
    corresponding KeyChainEntry, if it exists.
  **)

  PROCEDURE (k : KeyChain) GetChainEntry(gadget : G.Gadget):KeyChainEntry;

  VAR
    entry : KeyChainEntry;

  BEGIN
    entry:=k.first;
    WHILE (entry#NIL) & (entry.gadget#gadget) DO
      entry:=entry.next;
    END;
    RETURN entry;
  END GetChainEntry;

  (**
    Initializes the KeyHander. Must be called first after allocation.
  **)

  PROCEDURE (k : KeyHandler) Init*;

  BEGIN
    NEW(k.first);
    k.first.Init;
    k.last:=k.first;

    k.cChain:=k.first;
    k.cEntry:=NIL;

    k.active:=FALSE;

    k.sFirst:=NIL;
    k.sLast:=NIL;

    k.current:=NIL;
  END Init;

  (**
    Adds a shortcut with the given qualifier and character for an object.
    Additional a specialmode can be given for the shortcut whith states
    additional situations the shortcut get evaluated. An object must
    be visible to get notified, so shortcuts can be shared as long as
    only one object at the time is visible.
  **)

  PROCEDURE (k : KeyHandler) AddShortcutObject*(gadget : G.Gadget; qualifier : SET;
                                                char : CHAR; id,mode : LONGINT);

  VAR
    entry : ShortcutEntry;

  BEGIN
    ASSERT(G.handleSC IN gadget.flags);

    NEW(entry);
    entry.next:=NIL;
    entry.qualifier:=qualifier;
    entry.char:=CAP(char);
    entry.gadget:=gadget;
    entry.id:=id;
    entry.mode:=mode;

    IF k.sLast#NIL THEN
      k.sLast.next:=entry;
    ELSE
      k.sFirst:=entry;
    END;
    k.sLast:=entry;
  END AddShortcutObject;

  (**
    Adds a new gadget to the current KeyChain.
  **)

  PROCEDURE (k : KeyHandler) AddFocusObject*(gadget : G.Gadget);

  VAR
    focus : F.Frame;

  BEGIN
    ASSERT(gadget.CanFocus()); (* The object must support foccusing *)
    INCL(gadget.flags,G.mayFocus);

    k.cChain.Add(gadget);

    IF G.stdFocus IN gadget.flags THEN
      NEW(focus);
      focus.Init;     (* We must of cause call Frame.Init of the frame *)
      (* The frame must be resizeable in all directions *)
      focus.SetFlags({G.horizontalFlex,G.verticalFlex});
      (* We need a frame-image for a button *)
      focus.SetFrame(F.focusFrame);
      gadget.focus:=focus;
    END;
  END AddFocusObject;

  (**
    hands the focus to the given entry.
  **)

  PROCEDURE (k : KeyHandler) SetObject(chain : KeyChain; entry : KeyChainEntry);

  BEGIN
    IF (entry#NIL) & ((k.cChain#chain) OR (k.cEntry#entry)) THEN
      IF k.cEntry#NIL THEN
        k.cEntry.gadget.LostFocus;
        k.cEntry:=NIL;
      END;

      k.cEntry:=entry;
      k.cChain:=chain;
      k.cEntry.gadget.CatchedFocus;
    END;
  END SetObject;


  (**
    The user has pressed tab and wants the KeyHandler to deactivate the
    current and to activate the next valid object.
  **)

  PROCEDURE (k : KeyHandler) NextObject*();

  VAR
    entry : KeyChainEntry;

  BEGIN
    entry:=k.cChain.NextObject(k.cEntry);
    IF entry#k.cEntry THEN
      k.SetObject(k.cChain,entry);
    END;
  END NextObject;

  (**
    The user has pressed shift tab and wants the KeyHandler to deactivate the
    current and to activate the last valid object.
  **)

  PROCEDURE (k : KeyHandler) LastObject*();

  VAR
    entry : KeyChainEntry;

  BEGIN
    entry:=k.cChain.LastObject(k.cEntry);
    IF entry#k.cEntry THEN
      k.SetObject(k.cChain,entry);
    END;
  END LastObject;

  (**
    The user has pressed alt tab and wants the KeyHandler to deactivate the
    current and to activate the next (valid) KeyChain.

    NOTE
    Not suported yet.
  **)

  PROCEDURE (k : KeyHandler) NextChain*();

  BEGIN

  END NextChain;

  (**
    The user has pressed shift alt tab and wants the KeyHandler to deactivate the
    current and to activate the last (valid) KeyChain.

    NOTE
    Not suported yet.
  **)

  PROCEDURE (k : KeyHandler) LastChain*();

  BEGIN

  END LastChain;

  (**
    Searches for the given gadget in the KeyChains and returns the
    corresponding KeyChainEntry, if it exists.

    NOTE
    If we support more than one KeyChain, we must modify the
    parameterlist to return the found KeyChain, too.
  **)

  PROCEDURE (k : KeyHandler) GetChainEntry(gadget : G.Gadget):KeyChainEntry;

  BEGIN
    RETURN k.cChain.GetChainEntry(gadget);
  END GetChainEntry;

  (**
    Returns the shortcut matching or NIL.
  **)

  PROCEDURE (k : KeyHandler) GetSCEntry(qualifier : SET; keysym : LONGINT; char  : CHAR):ShortcutEntry;

  VAR
    sc    : ShortcutEntry;
    found : BOOLEAN;

  BEGIN
    found:=FALSE;
    sc:=k.sFirst;

    qualifier:=qualifier-E.shiftMask;
    WHILE sc#NIL DO
      IF (sc.qualifier=qualifier) & (sc.char=CAP(char)) THEN
        found:=TRUE;
      ELSIF ((sc.mode=return) & (keysym=E.return))
         OR ((sc.mode=escape) & (keysym=E.escape))
         OR ((sc.mode=default) & ((keysym=E.return) OR (keysym=E.escape))) THEN
        found:=TRUE;
      END;

      IF found & ((sc.gadget.visible) OR (G.scAlways IN sc.gadget.flags)) THEN
        RETURN sc;
      ELSE
        found:=FALSE;
        sc:=sc.next;
      END;
    END;
    RETURN NIL;
  END GetSCEntry;

  (**
    Called by the window, when a new object gets the focus. The Keyhandler
    should look for this object in any of the keychains and make it the
    object witht he current keyboardfocus, if it is in.
  **)

  PROCEDURE (k : KeyHandler) SetCurrentObject*(gadget : G.Gadget);

  BEGIN
    IF (gadget#NIL) & (k.cEntry=NIL) OR (k.cEntry.gadget#gadget) THEN
      k.SetObject(k.cChain,k.cChain.GetChainEntry(gadget));
    END;
  END SetCurrentObject;

  (**
    The window got called by the object with the current keyboard focus
    that it hides itself. KeyHandler has now to find out a new object
    with keyboard focus.
  **)

  PROCEDURE (k : KeyHandler) FindNewObject*();

  BEGIN
    k.NextObject();
  END FindNewObject;

  (**
    If there is no current focus object, get a new one, else
    do nothing.
  **)

  PROCEDURE (k : KeyHandler) Refocus*;

  BEGIN
    IF ~(k.active & (k.cEntry#NIL)) THEN
      k.NextObject;
    END;
  END Refocus;

  (**
    Gets called when the window get activated.
  **)

  PROCEDURE (k : KeyHandler) Activate*();

  VAR
    help : KeyChainEntry;

  BEGIN
    k.active:=TRUE;
    IF k.cEntry=NIL THEN
      help:=k.cChain.first;
      WHILE (help#NIL) & (~help.gadget.visible OR help.gadget.disabled) DO
        help:=help.next;
      END;
      IF help#NIL THEN
        k.SetObject(k.cChain,help);
      END;
    ELSE
      k.cEntry.gadget.CatchedFocus;
    END;
  END Activate;

  (**

  **)

  PROCEDURE (k : KeyHandler) CancelCurrent;

  BEGIN
    IF k.current#NIL THEN
      k.current.gadget.HandleShortcutEvent(k.current.id,G.canceled);
    END;
    k.current:=NIL;
  END CancelCurrent;

  (**
    Gets called when the window has been deactivated.
  **)

  PROCEDURE (k : KeyHandler) Deactivate*();

  BEGIN
    k.active:=FALSE;
    IF k.cEntry#NIL THEN
      k.cEntry.gadget.LostFocus;
    END;

    k.CancelCurrent;
  END Deactivate;

  (**
    This method gets called by the window, if the window
    has no current gadget which has the focus. The window
    does only call this method for keyboard events.
  **)

  PROCEDURE (k : KeyHandler) HandleEvent*(event : E.KeyEvent; preObject : BOOLEAN):BOOLEAN;

  VAR
    keysym : LONGINT;
    sc     : ShortcutEntry;
    buffer : ARRAY 256 OF CHAR;

  BEGIN
    IF preObject THEN
      IF event.type=E.keyDown THEN
        keysym:=event.GetKey();
        CASE keysym OF
          E.tab:
            IF event.qualifier={} THEN
              k.NextObject;
              RETURN TRUE;
            END;
        | E.leftTab:
            IF event.qualifier=E.shiftMask THEN
              k.LastObject;
              RETURN TRUE;
            END;
        ELSE
        END;
      END;

      keysym:=event.GetKey();
      IF event.GetText(buffer)>0 THEN
        IF event.type=E.keyDown THEN

          sc:=k.GetSCEntry(event.qualifier,keysym,buffer[0]);

          IF sc#NIL THEN
            IF sc#k.current THEN
              k.CancelCurrent;
              k.current:=sc;
              k.current.gadget.HandleShortcutEvent(sc.id,G.pressed);
              RETURN TRUE;
            END;
          ELSE
            k.CancelCurrent;
          END;
        ELSIF k.current#NIL THEN
          IF ~k.current.Match(event.qualifier,keysym,buffer[0]) THEN
            k.CancelCurrent;
          ELSE
            k.current.gadget.HandleShortcutEvent(k.current.id,G.released);
            k.current:=NIL;
          END;
        END;
      ELSE
        k.CancelCurrent;
      END;
      RETURN FALSE;
    ELSE
      IF k.cEntry#NIL THEN
        RETURN k.cEntry.gadget.HandleFocusEvent(event);
      ELSE
        RETURN FALSE;
      END;
    END;
  END HandleEvent;

END VOKeyHandler.