(**
   A image class that implements a fuel gauge, needed for displaying the
   amount a process has yet proceded.
**)

MODULE VOFuelGauge;

(*
    A fuelgauge.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT  D := VODisplay,
        F := VOFrame,
        G := VOGUIObject,
        O := VOObject,
        P := VOPrefs,
        V := VOValue;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (P.PrefsDesc)
                 frame* : LONGINT; (* the frame to use for the button *)
               END;

  FuelGauge*     = POINTER TO FuelGaugeDesc;
  FuelGaugeDesc* = RECORD (G.ImageDesc)
                     prefs    : Prefs;
                     frame    : F.Frame;

                     current  : V.ValueModel;
                     min,max  : LONGINT;
                     vertical : BOOLEAN;
                   END;

VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.double3DIn;
  END Init;

  PROCEDURE (f : FuelGauge) Init*;

  BEGIN
    f.Init^;

    f.prefs:=prefs;

    f.current:=NIL;

    f.max:=100;
    f.vertical:=FALSE;

    NEW(f.frame);
    f.frame.Init;
    f.frame.SetFlags({G.horizontalFlex,G.verticalFlex});
  END Init;

  PROCEDURE (f : FuelGauge) SetDirection*(vertical : BOOLEAN);

  BEGIN
    f.vertical:=vertical;
  END SetDirection;

  (**
    Set the minimal and maximal value the fuel gauge should display.

    NOTE
    The image currently does not cut down the value to the given range,
    so be carefull.
  **)

  PROCEDURE (f : FuelGauge) SetRange*(min,max : LONGINT);

  BEGIN
    IF (f.min#min) OR (f.max#max) THEN
      f.min:=min;
      f.max:=max;
      IF f.visible THEN
        f.Redraw;
      END;
    END;
  END SetRange;

  (**
    Set the integer model which represents the current value.
  **)

  PROCEDURE (f : FuelGauge) SetModel*(model : V.ValueModel);

  BEGIN
    IF f.current#NIL THEN
      f.UnattachModel(f.current);
    END;
    f.current:=model;
    f.AttachModel(f.current);
    IF f.visible THEN
      f.Redraw;
    END;
  END SetModel;

  PROCEDURE (f : FuelGauge) CalcSize*(display : D.Display);

  BEGIN
    f.frame.SetInternalFrame(f.prefs.frame);
    f.frame.CalcSize(display);

    IF f.vertical THEN
      f.width:=2*display.spaceWidth;
      f.height:=4*display.spaceHeight;
    ELSE
      f.width:=4*display.spaceWidth;
      f.height:=2*display.spaceHeight;
    END;

    INC(f.width,f.frame.leftBorder+f.frame.rightBorder);
    INC(f.height,f.frame.topBorder+f.frame.bottomBorder);

    f.minWidth:=f.width;
    f.minHeight:=f.height;

    f.CalcSize^(display);
  END CalcSize;

  PROCEDURE (f : FuelGauge) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    size : LONGINT;

  BEGIN
    f.Draw^(x,y,draw);

    f.frame.Resize(f.width,f.height);
    f.frame.Draw(f.x,f.y,draw);

    IF f.vertical THEN
      size:=((f.height-f.frame.topBorder-f.frame.bottomBorder)*f.current.GetLongint()-f.min) DIV (f.max-f.min);
      draw.PushForeground(D.backgroundColor);
      draw.FillRectangle(f.x+f.frame.leftBorder,
                         f.y+f.frame.topBorder,
                         f.width-f.frame.leftBorder-f.frame.rightBorder,
                         f.height-f.frame.topBorder-f.frame.bottomBorder-size);
      draw.PopForeground;
      draw.PushForeground(D.fillColor);
      draw.FillRectangle(f.x+f.frame.leftBorder,
                        f.y+f.frame.topBorder+f.height-f.frame.topBorder-f.frame.bottomBorder-size,
                         f.width-f.frame.leftBorder-f.frame.rightBorder,
                         size);
      draw.PopForeground;
    ELSE
      size:=((f.width-f.frame.leftBorder-f.frame.rightBorder)*f.current.GetLongint()-f.min) DIV (f.max-f.min);
      draw.PushForeground(D.fillColor);
      draw.FillRectangle(f.x+f.frame.leftBorder,f.y+f.frame.topBorder,
                         size,f.height-f.frame.topBorder-f.frame.bottomBorder);
      draw.PopForeground;
      draw.PushForeground(D.backgroundColor);
      draw.FillRectangle(f.x+f.frame.leftBorder+size,f.y+f.frame.topBorder,
                         f.width-f.frame.leftBorder-f.frame.rightBorder-size,
                         f.height-f.frame.topBorder-f.frame.bottomBorder);
      draw.PopForeground;
    END;
  END Draw;

  PROCEDURE (f : FuelGauge) Hide*;

  BEGIN
    IF f.visible THEN
      f.frame.Hide;

      f.draw.PushForeground(D.backgroundColor);
      f.draw.FillRectangle(f.x+f.frame.leftBorder,f.y+f.frame.topBorder,
                           f.width-f.frame.leftBorder-f.frame.rightBorder,
                           f.height-f.frame.topBorder-f.frame.bottomBorder);
      f.draw.PopForeground;
      f.Hide^;
    END;
  END Hide;

  PROCEDURE (f : FuelGauge) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF f.visible THEN
      f.Redraw;
    END;
  END Resync;

BEGIN
  NEW(prefs);
  prefs.Init;
END VOFuelGauge.