(**
  This class defines a number of events - based on userinteraction -
  a object (gadget, window) can get. These messages are OS-independant,
  but offer a way to interpret the underlying OS-specific events, however
  the interface to that part of the events is not portable.

  NOTE
  * Not all GUIs can name the qualifiers extactly. F.e. X11 does not make a
    destinction between shift_left and shift_right in the qualifier field
    of an event. use the qualifier mask for that or handle key up and down for
    qualifiers explicitely..
**)

MODULE VOEvent;

(*
    Classhierachie defining a number of OS-independend messgaes.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT       C,

             X11,
       xk := Xkeysymdef,
       xu := Xutil;

CONST
  (* type *)
  keyUp       * = 0;
  keyDown     * = 1;
  mouseUp     * = 2;
  mouseDown   * = 3;

  (* qualifiers *)

  none          * =  0;

  button1       * =  1;
  button2       * =  2;
  button3       * =  3;
  button4       * =  4;
  button5       * =  5;

  dragDropButton * = button2;

  qShiftLeft    * =  6;
  qShiftRight   * =  7;
  qCapsLock     * =  8;
  qControlLeft  * =  9;
  qControlRight * = 10;
  qAltLeft      * = 11;
  qAltRight     * = 12;
  qMetaLeft     * = 13;
  qMetaRight    * = 14;
  qSuperLeft    * = 15;
  qSuperRight   * = 16;
  qHyperLeft    * = 17;
  qHyperRight   * = 18;

  (* qualifier masks *)

  shiftMask    * = {qShiftLeft,qShiftRight};
  altMask      * = {qAltLeft,qAltRight};
  controlMask  * = {qControlLeft,qControlRight};
  metaMask     * = {qMetaLeft,qMetaRight};          (* not supported *)
  superMask    * = {qSuperLeft,qSuperRight};        (* not supported *)
  hyperMask    * = {qHyperLeft,qHyperRight};        (* not supported *)

  buttonMask   * = {button1,button2,button3,button4,button5};
  keyMask      * = shiftMask+altMask+controlMask+metaMask+superMask+hyperMask;

  (* key constants *)

  shiftLeft    * = xk.XK_Shift_L;
  shiftRight   * = xk.XK_Shift_R;
  shiftLock    * = xk.XK_Shift_Lock;
  capsLock     * = xk.XK_Caps_Lock;
  scrollLock   * = xk.XK_Scroll_Lock;
  numLock      * = xk.XK_Num_Lock;
  controlLeft  * = xk.XK_Control_L;
  controlRight * = xk.XK_Control_R;
  altLeft      * = xk.XK_Alt_L;
  altRight     * = xk.XK_Alt_R;
  metaLeft     * = xk.XK_Meta_L;
  metaRight    * = xk.XK_Meta_R;
  superLeft    * = xk.XK_Super_L;
  superRight   * = xk.XK_Super_R;
  hyperLeft    * = xk.XK_Hyper_L;
  hyperRight   * = xk.XK_Hyper_R;

  backspace    * = xk.XK_BackSpace;
  delete       * = xk.XK_Delete;
  tab          * = xk.XK_Tab;
  leftTab      * = xk.XK_ISO_Left_Tab;
  return       * = xk.XK_Return;
  escape       * = xk.XK_Escape;
  space        * = xk.XK_space;

  home         * = xk.XK_Home;
  begin        * = xk.XK_Begin;
  end          * = xk.XK_End;
  left         * = xk.XK_Left;
  right        * = xk.XK_Right;
  up           * = xk.XK_Up;
  down         * = xk.XK_Down;
  pageUp       * = xk.XK_Page_Up;
  pageDown     * = xk.XK_Page_Down;
  prior        * = xk.XK_Prior;
  next         * = xk.XK_Next;

  print        * = xk.XK_Print;
  insert       * = xk.XK_Insert;

  f1           * = xk.XK_F1;
  f2           * = xk.XK_F2;
  f3           * = xk.XK_F3;
  f4           * = xk.XK_F4;
  f5           * = xk.XK_F5;
  f6           * = xk.XK_F6;
  f7           * = xk.XK_F7;
  f8           * = xk.XK_F8;
  f9           * = xk.XK_F9;
  f10          * = xk.XK_F10;
  f11          * = xk.XK_F11;
  f12          * = xk.XK_F12;

TYPE
  Event*     = POINTER TO EventDesc;

  (**
    Baseclass for events. Currently all objects get a instance
    of this baseclass and then have to analyse the message by
    evaluating the containing X11-event itself. This may change in the
    future. Display will send generate and send instances of inherited
    Classes thatd define abstract events.
  **)
  EventDesc*      = RECORD
                      event* : X11.XEvent;
                      reUse* : BOOLEAN;
                    END;

  KeyEvent*       = POINTER TO KeyEventDesc;

  (**
    Keyboard event. The application receives this event when a key
    has been pressed or raised.
  **)

  KeyEventDesc*   = RECORD (EventDesc)
                      type*      : INTEGER; (* type of key event *)
                      qualifier* : SET;
                    END;

  MouseEvent*     = POINTER TO MouseEventDesc;

  (**
    Mouse event. The application recieves this event when one or
    more mousebuttons have been pressed or released. You'll also
    get events when the mouse moves.
  **)

  MouseEventDesc* = RECORD (EventDesc)
                      type*      : INTEGER;  (* type of mouse event *)
                      button*    : INTEGER;  (* button that caused action *)
                      qualifier* : SET;      (* the qualifier *)
                      x*,y*      : LONGINT;  (* window relative position of mouse *)
                    END;

  MotionEvent*    = POINTER TO MotionEventDesc;

  (**
    Mouse event. The application recieves this event when one or
    more mousebuttons have been pressed or released. You'll also
    get events when the mouse moves.
  **)

  MotionEventDesc* = RECORD (EventDesc)
                       qualifier* : SET;      (* the qualifier *)
                       x*,y*      : LONGINT;  (* window relative position of mouse *)
                     END;

VAR
  simpleEvent : Event;
  mouseEvent  : MouseEvent;
  motionEvent : MotionEvent;
  keyEvent    : KeyEvent;

  (**
    Convert the given X11 qualifier description to a VisualOberon
    qualifier description.
  **)

  PROCEDURE (e : Event) EvaluateQualifier(state : X11.uintmask):SET;

  VAR
    qualifier : SET;

  BEGIN
    qualifier:={};
    IF state*X11.ShiftMask#{} THEN
      qualifier:=qualifier+shiftMask;
    END;
    IF state*X11.LockMask#{} THEN
      INCL(qualifier,qCapsLock);
    END;
    IF state*X11.ControlMask#{} THEN
      qualifier:=qualifier+controlMask;
    END;

    IF state*X11.Mod1Mask#{} THEN
      qualifier:=qualifier+altMask;
    END;

    IF state*X11.Button1Mask#{} THEN
      INCL(qualifier,button1);
    END;
    IF state*X11.Button2Mask#{} THEN
      INCL(qualifier,button2);
    END;
    IF state*X11.Button3Mask#{} THEN
      INCL(qualifier,button3);
    END;
    IF state*X11.Button4Mask#{} THEN
      INCL(qualifier,button4);
    END;
    IF state*X11.Button5Mask#{} THEN
      INCL(qualifier,button5);
    END;

    RETURN qualifier;
  END EvaluateQualifier;

  (**
    NOTE
      THis as a VODisplay-only access-method for initializing the MouseEvent.
      Do not call this method in your code, it is VODisplay only and does only
      exists on X11 implementations of VisualOberon.
  **)

  PROCEDURE (e : KeyEvent) SetX11Event*(event : X11.XKeyEvent);

  BEGIN
    e.event.xkey:=event;

    IF event.type=X11.KeyPress THEN
      e.type:=keyDown;
    ELSE
      e.type:=keyUp;
    END;

    e.qualifier:=e.EvaluateQualifier(event.state);
  END SetX11Event;

  (**
    Returns the ASCII-string corresponding to the keyboard event.

    RESULT
    The number of characters in the string.
  **)

  PROCEDURE (e : KeyEvent) GetText*(VAR string : ARRAY OF CHAR):LONGINT;

  VAR
    keysym  : X11.KeySym;
    compose : xu.XComposeStatus;
    count   : LONGINT;

  BEGIN
     count:=xu.XLookupString(e.event.xkey,string,LEN(string)-1,keysym,compose);
     string[count]:=0X;

     RETURN count;
  END GetText;

  (**
    Returns the ASCII-string corresponding to the keyboard event.

    RESULT
    The number of characters in the string.
  **)

  PROCEDURE (e : KeyEvent) GetKey*():LONGINT;

  VAR
    keysym  : X11.KeySym;
    compose : xu.XComposeStatus;
    buffer  : ARRAY 256 OF CHAR;
    length  : LONGINT;

  BEGIN
     buffer[0]:=0X;
     length:=xu.XLookupString(e.event.xkey,buffer,LEN(buffer),keysym,compose);

     RETURN keysym;
  END GetKey;

  (**
    Returns a string representing the name of key(s) pressed.
  **)

  PROCEDURE (e : KeyEvent) GetName*(keysym : LONGINT; VAR buffer : ARRAY OF CHAR);

  VAR
    count  : LONGINT;
    string : C.charPtr1d;

  BEGIN
    buffer[0]:=0X;
    string:=X11.XKeysymToString(keysym);
    IF string#NIL THEN
      count:=0;
      WHILE (string[count]#0X) & (count<LEN(buffer)-1) DO
        buffer[count]:=string[count];
        INC(count);
      END;
      buffer[count]:=0X;
    END;
  END GetName;

  (**
    NOTE
      THis as a VODisplay-only access-method for initializing the MouseEvent.
      Do not call this method in your code, it is VODisplay only and does only
      exists on X11 implementations of VisualOberon.
  **)

  PROCEDURE (e : MouseEvent) SetX11Event*(event : X11.XButtonEvent);

  BEGIN
    e.event.xbutton:=event;

    IF event.type=X11.ButtonPress THEN
      e.type:=mouseDown;
    ELSE
      e.type:=mouseUp;
    END;

    e.x:=event.x;
    e.y:=event.y;

    e.qualifier:=e.EvaluateQualifier(event.state);

    CASE event.button OF
      X11.Button1 : e.button:=button1;
    | X11.Button2 : e.button:=button2;
    | X11.Button3 : e.button:=button3;
    | X11.Button4 : e.button:=button4;
    | X11.Button5 : e.button:=button5;
    ELSE
      e.button:=none;
    END;
  END SetX11Event;

  (**
    NOTE
      THis as a VODisplay-only access-method for initializing the MouseEvent.
      Do not call this method in your code, it is VODisplay only and does only
      exists on X11 implementations of VisualOberon.
  **)

  PROCEDURE (e : MotionEvent) SetX11Event*(event : X11.XMotionEvent);

  BEGIN
    e.event.xmotion:=event;

    e.x:=event.x;
    e.y:=event.y;

    e.qualifier:=e.EvaluateQualifier(event.state);
  END SetX11Event;

  (**
    Convert the given X11 event to a VisualOberon event.
  **)

  PROCEDURE GetEvent*(event : X11.XEvent):Event;

  BEGIN
    CASE event.type OF
      X11.ButtonPress,
      X11.ButtonRelease:
        mouseEvent.SetX11Event(event.xbutton);
        RETURN mouseEvent;
    | X11.MotionNotify:
        motionEvent.SetX11Event(event.xmotion);
        RETURN motionEvent;
    | X11.KeyPress,
      X11.KeyRelease:
        keyEvent.SetX11Event(event.xkey);
        RETURN keyEvent;
    ELSE
      simpleEvent.event:=event;
      RETURN simpleEvent;
    END;
  END GetEvent;

BEGIN
  NEW(simpleEvent);
  NEW(keyEvent);
  NEW(mouseEvent);
  NEW(motionEvent);
END VOEvent.