;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  persistent object model
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-class <line-graphic> (<leaf-object>)
  (line-start type: <point>)
  (line-end type: <point>))

(define-method status-line-when-sel ((self <line-graphic>))
  (format #f "Line ~d" (id self)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;   <line-graphic>
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-method pick-list* ((self <line-graphic>) pt ctm)
  (pick-on-path self pt ctm (list (line-start self) (line-end self))))

(define-method paint-artwork* ((self <line-graphic>) dev)
  (moveto dev (line-start self))
  (lineto dev (line-end self))
  (stroke dev))

(define-method accum-handles ((self <line-graphic>) accum)
  (accum self (line-start self) 0)
  (accum self (line-end self) 1))

(define-method paint-object* ((self <line-graphic>) dev)
  (let ((sw (get-property self 'stroke-width #f)))
    ;
    (define (render)
      (moveto dev (line-start self))
      (lineto dev (line-end self))
      (if sw (setlinewidth dev sw))
      (stroke dev))
       ;;
       ;;
       (if sw
	   (with-gstate-saved dev render)
	   (render))))

(define-method start-active-drag ((self <line-graphic>) 
				  (in-view <open-view>)
				  (initial-pt <point>)) ;; device-sheet coords
  (let ((p0 (line-start self))
	(p1 (line-end self))
	(ctm (invert-transform (compute-view-ctm-for self in-view))))
    (lambda ((pt <point>) flags)
      (set! pt (window->sheet in-view pt))
      (let ((d (transform (point- pt initial-pt) ctm)))
	(set-line-start! self (point+ p0 d))
	(set-line-end! self (point+ p1 d))
	(mark-as-dirty (in-document in-view))
	; expensive, but effective
	(clear-all-areas (in-document in-view))))))

;;;  relocate the `near' point so that it is on a valid
;;;  shift-constrained line from `from'

(define $PI (atan 0 -1))

(define (shift-constraint-filter (near <point>) (from <point>))
  (let* ((d (point- near from))
	 (a (atan (dy d) (dx d)))
	 (near-a (modulo (inexact->exact (round (/ a (/ $PI 4)))) 8)))
    ;(dm "Near angle index: ~d" near-a)
    (case near-a
      ((0 4) (make-point (x near) (y from)))
      ((2 6) (make-point (x from) (y near)))
      ((1 5) (let ((t (/ (+ (dx d) (dy d)) 2)))
	       (point+ from (make-size t t))))
      ((3 7) (let ((t (/ (- (dx d) (dy d)) 2)))
	       (point+ from (make-size t (- t))))))))

;;

(define-method start-active-drag-handle ((self <line-graphic>) 
					 (in-view <open-view>)
					 handle-id
					 (initial-pt <point>)) ; sheetdevice
  (let* ((up0 (case handle-id
		((0) (line-start self))
		((1) (line-end self))))
	 (u->d-ctm (compute-view-ctm-for self in-view))
	 (d->u-ctm (invert-transform u->d-ctm))
	 (other-dp (transform (case handle-id
				((0) (line-end self))
				((1) (line-start self)))
			      u->d-ctm))
	 (win (content-window in-view))
	 (dp0 (transform up0 u->d-ctm))
	 (gc (transient-gc win))
	 (vo (view-origin (underlying-object in-view)))
	 (last #f))
    ;
    (define (shift-adj pt)
      ; do the `shift' constraint in user space
      (transform (shift-constraint-filter
		  (transform pt d->u-ctm)
		  (case handle-id
		    ((0) (line-end self))
		    ((1) (line-start self))))
		 u->d-ctm))
    (vector
     ; mouse-drag
     (lambda ((pt <point>) flags)
       (set! pt (window->sheet in-view pt))
       (if (shift-state? flags)
	   (set! pt (shift-adj pt)))
	; No need to convert to user coordinates to draw just a line,
	; even if the view is rotated.  True because lines are invariant 
	; under affine xforms.
       (if last
	   (draw-lines win gc last))
       (set! last (map inexact->exact (list (- (x other-dp) (x vo))
					    (- (y other-dp) (y vo))
					    (- (x pt) (x vo))
					    (- (y pt) (y vo)))))
       (draw-lines win gc last)
       (flush-client))
     ; mouse-up
     (lambda ((pt <point>) flags)
       (set! pt (window->sheet in-view pt))
       (if (shift-state? flags)
	   (set! pt (shift-adj pt)))
       ; this is a little confusing, the fact that we can't
       ; just inversely transform the given point to get the
       ; final point. (apparently, that doesn't work because 
       ; `d->u-ctm' includes the translation to get to the base of
       ; this object, and hence transforms the device point which
       ; is at the origin of this object to <0,0>)
       (let ((adj (transform (point- pt initial-pt) d->u-ctm)))
	 (case handle-id
	   ((0)
	    (set-line-start! self (point+ up0 adj)))
	   ((1)
	    (set-line-end! self (point+ up0 adj))))
	 (mark-as-dirty (in-document in-view))
	 (clear-all-areas (in-document in-view))
	 ; is this needed...?
	 (update-handles in-view))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;   line-drawing tool
;;;
;;;   (nb, this is really a graphic command functionality)
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (drawline-button-press (in-view <open-view>)
			       (dp0 <point>)
			       modifier-state)
  (let* ((fwd-ctm (view-ctm (underlying-object in-view)))
	 (ctm (translate (invert-transform fwd-ctm)
			 (view-origin (underlying-object in-view))))
	 (fwd-ctm (invert-transform ctm))
	 (win (content-window in-view))
	 (dp0 (tweak-point in-view dp0)) ;; snap to geometry
	 (up0 (transform dp0 ctm))
	 (gc (transient-gc win))
	 (last #f))
    (define (shift-adj pt)
      ; do the `shift' constraint in user space
      (transform (shift-constraint-filter (transform pt ctm) up0)
		 fwd-ctm))
    (set-active-drag-proc!
     in-view
     (vector
      (lambda ((dp2 <point>) flags)
	(if (shift-state? flags)
	    (set! dp2 (shift-adj dp2)))
	(let* ((up2 (transform dp2 ctm))
	       (dp1 (transform (make-point (x up2) (y up0)) fwd-ctm))
	       (dp3 (transform (make-point (x up0) (y up2)) fwd-ctm)))
	  (if last
	      (draw-lines win gc last))
	  (set! last (map inexact->exact
			  (list (x dp0) (y dp0)
				(x dp2) (y dp2))))
	  (draw-lines win gc last)
	  (flush-client)))
      (lambda ((dp2 <point>) flags)
	(if (shift-state? flags)
	    (set! dp2 (shift-adj dp2)))
	(let* ((up2 (transform dp2 ctm))
	       (par (page-contents
		     (view-page
		      (underlying-object in-view))))
	       (line (make <line-graphic>
			   in-document: (in-document par)
			   parent-object: par
			   line-start: (make-point (x up0) (y up0))
			   line-end: (make-point (x up2) (y up2))
			   graphic-bounding-box: (make-rect 0 0 0 0))))
	  (if last
	      (draw-lines win gc last))
	  (clear-all-areas (in-document in-view))
	  (do-select in-view line 0)
	  (update-handles in-view)))))))

(add-major-mode!
 (make <major-mode>
       name: 'draw-line
       button-press-proc: drawline-button-press))

(define-interactive (draw-line-mode view)
  (interactive (owner))
  (set-major-mode! view (get-major-mode 'draw-line)))

(graphic-set-key #\M-3 draw-line-mode)

;;;

(define-method externalize ((self <line-graphic>))
  `(line start-x: ,(x (line-start self))
	 start-y: ,(y (line-start self))
	 end-x: ,(x (line-end self))
	 end-y: ,(y (line-end self))))

(define (paste-line-from-extern extern group offset)
  (apply (lambda (#key start-x start-y end-x end-y
		       (stroke-width default: #f)) 
	   (let ((sw (or stroke-width 1)) ;; stroke weight
		 (ls (point+ (make-point start-x start-y) offset))
		 (le (point+ (make-point end-x end-y) offset)))
	     (make <line-graphic>
		   in-document: (in-document group)
		   parent-object: group
		   properties: (if stroke-width
				   (vector 'stroke-width stroke-width)
				   '#())
		   graphic-bounding-box: (inset-rect
					  (bbox-rect (x ls) (y ls)
						     (x le) (y le))
					  (- sw)
					  (- sw))
		   line-start: ls
		   line-end: le)))
	 (cdr extern)))


