/*
 * PCMTST.C - multiple children tests for PPC
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "ppc.h"

static int
 SC_DECLARE(process_end, (char *name, int *pr, int quiet)),
 SC_DECLARE(interrupt_mode, (char *name, int quiet)),
 SC_DECLARE(poll_mode, (char *name, int quiet)),
 SC_DECLARE(process_count, (void));

static void
 SC_DECLARE(child_has_txt, (int fd)),
 SC_DECLARE(tty_has_txt, (int fd)),
 SC_DECLARE(clean_up, (int sig));

int
 n_processes;

PROCESS
 **processes;

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* MAIN - test PPC with multiple children */

main(argc, argv, envp)
   int argc;
   char **argv, **envp;
   {int i, interrupts, ret, to;
    char mode[5], *prog, **argl;
    PROCESS *p;

/* process the command line arguments */
    interrupts  = TRUE;
    n_processes = 2;
    to          = 1000000;
    strcpy(mode, "wp");
    for (i = 1; i < argc; i++)
        {if (argv[i][0] == '-')
            {switch (argv[i][1])
                {case 'c' : to = SC_stoi(argv[++i]);
                            break;
	         case 'h' :
		      printf("\nUsage: pcmtst [-pst] [-c n] [-h] [-i] [-n <#>] [<prog> [<arg1> ...]]\n");
		      printf("   Options:\n");
		      printf("      c - timeout after n seconds\n");
		      printf("      h - print this help message\n");
		      printf("      i - poll explicitly instead of using system call\n");
		      printf("      n - # of copies to spawn\n");
		      printf("\n");
		      printf("      p - use pipes for communications\n");
		      printf("      s - use sockets for communications\n");
		      printf("      t - use pseudo terminals for communications\n");
		      printf("\n");
		      exit(1);

                 case 'i' :
                      interrupts = FALSE;
		      break;
                 case 'n' :
                      n_processes = SC_stoi(argv[++i]);
		      break;
                 case 'p' :
                      strcpy(mode, "wp");
		      break;
                 case 's' :
                      strcpy(mode, "ws");
		      break;
                 case 't' :
                      strcpy(mode, "wt");
		      break;};}
         else
            break;};

    SC_setbuf(stdout, NULL);

    argc -= i;
    if (argc < 1)
       {prog = SC_strsavef("pcctst", "char*:PCMTST:pcctst");
        argl = FMAKE_N(char *, 2, "PCMTST.C:argl");
        argl[0] = SC_strsavef("pcctst", "char*:PCMTST:argl0");
        argl[1] = NULL;}
    else
       {prog = argv[i];
	argl = argv + i;};

/* print this before we potentially go into RAW mode (PTY's do this) */
    PRINT(stdout, "\nRunning process: %s\n", prog);

    PC_unblock_file(stdin);

    processes = FMAKE_N(PROCESS *, n_processes,
                "PCMTST.C:processes");

    for (i = 0; i < n_processes; i++)
        {p = PC_open(argl, envp, mode);
	 if (p == NULL)
	    {PRINT(stdout, "\nError: failed to open copy #%d of %s\n\n",
		   i + 1, prog);
	     exit(1);};
	 processes[i] = p;
	 PRINT(stdout, "\n  Child #%d (%s) is %d", i + 1, prog, p->id);
	 PC_unblock(p);};

    PRINT(stdout, "\n");

/* set the alarm */
    PC_alarm(to, clean_up);

    if (interrupts)
       ret = interrupt_mode(prog, FALSE);
    else
       ret = poll_mode(prog, FALSE);

/* reset the alarm */
    PC_alarm(0, NULL);

    PRINT(stdout, "\nProcess test %s ended\n\n", prog);

    PC_block_file(stdin);

    exit(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* INTERRUPT_MODE - handle the child and tty I/O via interrupts */

static int interrupt_mode(name, quiet)
   char *name;
   int quiet;
   {int i, n, pi, ret;

    PC_io_interrupts_on = TRUE;

    pi = PC_io_callback_file(stdin, tty_has_txt);
    for (i = 0; i < n_processes; i++)
        if (processes[i] != NULL)
	   pi &= PC_io_callback_fd(processes[i]->in, child_has_txt);

    PC_io_interrupts_on = pi;

    if (pi)
       PC_catch_io_interrupts(PC_io_interrupts_on);

    while (process_count() > 0)
       {n = process_end(name, &ret, quiet);
        if (n != 0)
	   continue;

/* since all children are passive only activity on stdin will evoke action */
	PC_poll_descriptors(-1);

/* delay to minimize collisions between SIGIO and SIGCHLD handlers */
	sleep(1);};

    return(0);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* CHILD_HAS_TXT - handle text from the child process */

static void child_has_txt(fd)
   int fd;
   {int i;
    char s[MAXLINE];
    PROCESS *p;

    p = NULL;
    for (i = 0; i < n_processes; i++)
        if (processes[i] != NULL)
	   if (processes[i]->in == fd)
	      {p = processes[i];
	       break;};

    if (p != NULL)
       while (PC_gets(s, MAXLINE, p) != NULL)
          PRINT(stdout, "\n  From #%d | %s", i + 1, s);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TTY_HAS_TXT - handle text from the tty */

static void tty_has_txt(fd)
   int fd;
   {int i;
    char s[MAXLINE], *t, *pt;
    PROCESS *p;

    t = fgets(s, LRG_TXT_BUFFER, stdin);
    if (t != NULL)
       {t = SC_strtok(s, " ", pt);
	i = SC_stoi(t);
	p = processes[i - 1];
	if (p == NULL)
	   {PRINT(stdout, "\nError: child #%d previously terminated\n\n", i);
	    exit(1);};
	t = SC_strtok(NULL, "", pt);
	PC_printf(p, "%s", t);
        PRINT(stdout, "\n  To   #%d | %s", i, t);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* POLL_MODE - poll the process and the tty */

static int poll_mode(name, quiet)
   char *name;
   int quiet;
   {int i, ret;
    char s[LRG_TXT_BUFFER], *t, *pt;
    PROCESS *p;

    SIGNAL(SIGINT, clean_up);

    while (process_count() > 0)
       {for (i = 0; i < n_processes; i++)
            {p = processes[i];
	     if (p != NULL)
	        while (PC_gets(s, LRG_TXT_BUFFER, p) != NULL)
		   PRINT(stdout, "\n  From #%d | %s", i + 1, s);};

        if (process_end(name, &ret, quiet))
           continue;

        PC_unblock_file(stdin);
        PC_err[0] = '\0';

        t = fgets(s, LRG_TXT_BUFFER, stdin);

        PC_block_file(stdin);
        PC_err[0] = '\0';

        if (t != NULL)
	   {t = SC_strtok(s, " ", pt);
            i = SC_stoi(t);
	    p = processes[i - 1];
            if (p == NULL)
	       {PRINT(stdout, "\nError: child #%d previously terminated\n\n", i);
		exit(1);};
            t = SC_strtok(NULL, "", pt);
	    PC_printf(p, "%s", t);
            PRINT(stdout, "\n  To   #%d | %s", i, t);
	    sleep(1);};};

    return(0);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PROCESS_END - return the number of process which have ended
 *             - do the PC_close now since the TTY may be in
 *             - RAW mode and we want the output to
 *             - look nice (PTY's do this)
 */

static int process_end(name, pr, quiet)
   char *name;
   int *pr, quiet;
   {int i, status, reason, ret;
    PROCESS *p;

    *pr = 0;
    ret = 0;
    for (i = 0; i < n_processes; i++)
        {p = processes[i];
	 if (p != NULL)
	    if (PC_status(p) != RUNNING)
	       {status = p->status;
		reason = p->reason;

		*pr = reason;

/* get anything remaining from the child */
		child_has_txt(p->in);

		if (!quiet)
		   PRINT(stdout, "\n    Child #%d (%d) terminated (%d %d)\n",
			 i + 1, p->id, status, reason);

		PC_close(p);
		processes[i] = NULL;

		ret++;};};

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PROCESS_COUNT - return the number of active child processes */

static int process_count()
   {int i, ret;

    ret = 0;
    for (i = 0; i < n_processes; i++)
        if (processes[i] != NULL)
	   ret++;
    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* CLEAN_UP - clean up this session */

static void clean_up(sig)
   int sig;
   {PC_block_file(stdin);

    if (sig == SIGALRM)
       {PRINT(stdout, "Multiple child test timed out\n");
	exit(123);}
    else
       exit(1);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
