/*
 * MLRAND.C - random number generation routines
 *          - most of the routines were written by Jim Rathkopf, LLNL
 *          - with modifications for portability and efficiency by SAB
 *          - THREADSAFE
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pml.h"

#ifndef RAND_MAX
#define RAND_MAX 2147483647.0
#endif

#define BASE24        16777216.0                                    /* 2^24 */
#define TWO8               256.0                                     /* 2^8 */
#define TWO16            65536.0                                    /* 2^16 */
#define TWO48  281474976710656.0                                    /* 2^48 */

#define UPPERPART(r)            floor(r/BASE24)
#define LOWERPART(r, rupper)    (r - rupper*BASE24)

#define DSEED(_id)    _PM_rng[_id].dseed
#define DMULT(_id)    _PM_rng[_id].dmult
#define DADDER(_id)   _PM_rng[_id].dadder
#define MULTPLR(_id)  _PM_rng[_id].multplr

typedef union u_rand_seed rand_seed;
typedef struct s_PM_rng_info PM_rng_info;

union u_rand_seed
   {double sd;
    unsigned short sa[4];};

struct s_PM_rng_info
   {double dseed[2];
    double dmult[2];
    double dadder;
    unsigned short multplr[3];};

static PM_rng_info
 *_PM_rng;

SC_THREAD_LOCK(PM_rng_lock);

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PM_INIT_RAND_PAR - initialize the SMP parallel global data for
 *                   - the random number generator
 */

void _PM_init_rand_par()
   {int i, nt;
    double *pd, *pm;
    unsigned short *pl;
    static int first = TRUE;

    if (first)
       {first = FALSE;

	SC_LOCKON(PM_rng_lock);

	nt = SC_n_threads;
	nt = max(nt, 1);

	_PM_rng = NMAKE_N(PM_rng_info, nt, "_PM_INIT_RAND_PAR:_PM_rng");

	for (i = 0; i < nt; i++)
	    {pd = DSEED(i);
	     pm = DMULT(i);
	     pl = MULTPLR(i);

	     pd[0] = 16555217.0;
	     pd[1] = 9732691.0;

	     pm[0] = 15184245.0;
	     pm[1] = 2651554.0;

	     pl[0] = 0xb175;
	     pl[1] = 0xa2e7;
	     pl[2] = 0x2875;

	     DADDER(i) = 0.0;};

	SC_LOCKOFF(PM_rng_lock);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PM_RANDOM_48 - generate pseudo-random numbers through a 48-bit linear
 *               - congruential algorithm
 *               - this emulates the drand48 library
 *               - of random number generators available on many, but not all, 
 *               - UNIX machines
 *               - return a double x such that 0.0 <= x < 1.0
 *
 * algorithm:
 *    x(n+1) = (a*x(n) + c)mod m
 *
 *    where
 *
 *    defaults, for standard unix rand48, Cray defaults used in coding
 *
 *         double name               hexdecimal         decimal
 *   x - seed-dseed[1], [2]         1234abcd330e      20017429951246 
 *   a - multiplier-dmult[1], [2]    5deece66d         25214903917
 *   c - adder-dadder                  b                  11
 *   m -                        2**481000000000000  281474976710656
 *
 *    24-bit defaults (decimal) (lower bits listed first)
 *   x - dseed[1], [2]- 13447950.0, 1193131.0
 *   a - dmult[1], [2]- 15525485.0, 1502.0
 *   c - dadder- 11.0
 *
 */

static double _PM_random_48()
   {int tid;
    double t1_48, t2_48, t1_24[2], t2_24[2];
    double d;
    double *pd, *pm;
    static int first = TRUE;

    if (first)
       {first = FALSE;
	_PM_init_rand_par();};

    SC_TID(tid);

    pd = DSEED(tid);
    pm = DMULT(tid);

/* perform 48-bit arithmetic using two part data */
    t1_48 = pm[0]*pd[0] + DADDER(tid);
    t2_48 = pm[1]*pd[0] + pm[0]*pd[1];

    t1_24[1] = UPPERPART(t1_48);
    t1_24[0] = LOWERPART(t1_48, t1_24[1]);

    t2_24[1] = UPPERPART(t2_48);
    t2_24[0] = LOWERPART(t2_48, t2_24[1]);

    t2_48 = t2_24[0] + t1_24[1];

    t2_24[1] = UPPERPART(t2_48);
    t2_24[0] = LOWERPART(t2_48, t2_24[1]);

    d = (pd[0] + pd[1]*BASE24)/TWO48;

    pd[0] = t1_24[0];
    pd[1] = t2_24[0];

    return(d);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PM_RAND48_16TO24 - take a number stored in 3 16-bit shorts and move to
 *                   - 2 doubles each containing 24 bits of data
 *                   -   X16, the 3 16-bit shorts
 *                   -   X24, the 2 doubles of 24 bits returned
 *
 */

static void _PM_rand48_16to24(x16, x24)
   unsigned short x16[3];
   double x24[2];
   {double t0, t1, t2, t1u, t1l;

    t0 = (double) x16[0];
    t1 = (double) x16[1];
    t2 = (double) x16[2];

    t1u = floor(t1/TWO8);
    t1l = t1 - t1u*TWO8;

    x24[0] = t0 + t1l*TWO16;
    x24[1] = t1u + t2*TWO8;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PM_RAND48_24TO16 - take a number stored in 2 doubles each containing
 *                   - 24 bits of data move to 3 16-bit shorts
 *                   -   X24, the 2 doubles of 24 bits
 *                   -   X16, the 3 16-bit shorts returned
 */

static void _PM_rand48_24to16(x24, x16)
   double x24[2];
   unsigned short x16[3];
   {double t0u, t0l, t1u, t1l;

    t0u = floor(x24[0]/TWO16);
    t0l = x24[0] - t0u*TWO16;

    t1u = floor(x24[1]/TWO8);
    t1l = x24[1] - t1u*TWO8;

    x16[0] = t0l;
    x16[1] = t0u + t1l*TWO8;
    x16[2] = t1u;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PM_SEED_48 - convert a 3 by 16 representation of a seed
 *             - return a pointer to old seed
 *             -   RS, new seed
 *
 */

static void _PM_seed_48(rs, aso)
   unsigned short rs[3], aso[3];
   {int tid;
    double sdo[2], *pd;

    SC_TID(tid);

    pd = DSEED(tid);

    sdo[0] = pd[0];
    sdo[1] = pd[1];

    _PM_rand48_24to16(sdo, aso);
    _PM_rand48_16to24(rs, pd);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PM_RNSET - set random number generator seed, multiplier
 *           -   S, the random number seed
 *           -   MLT, the random number multiplier
 */

static void _PM_rnset(s, mlt)
   unsigned short s[3];
   unsigned short mlt[3];
   {int tid;
    double *pd, *pm;

    SC_TID(tid);

    pd = DSEED(tid);
    pm = DMULT(tid);

/* construct 24-bit versions of 'seed' and 'mlt'
 * and store in 64-bit doubles
 */
    _PM_rand48_16to24(s, pd);
    _PM_rand48_16to24(mlt, pm);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_SRAND_48 - change the value of the random number seed
 *             - ranset must be passed a 64 bit double
 */

void PM_srand_48(x)
   double x;
   {int j, tid;
    unsigned short wrk_seed[3];
    rand_seed seed;

    SC_TID(tid);

    if (x == 0.0)
       {wrk_seed[0] = 0x9cd1;
        wrk_seed[1] = 0x53fc;
        wrk_seed[2] = 0x9482;}

/* store x in wrk_seed */
    else
       {seed.sd = x;
        for (j = 0; j < 3; j++)
            wrk_seed[j] = seed.sa[j];};

   _PM_rnset(wrk_seed, MULTPLR(tid));
  
   return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_GRAND_48 - return the value of the current seed */

double PM_grand_48()
   {int j, tid;
    unsigned short wrk_seed[3];
    rand_seed seed;
    static unsigned short SeedDum[3] = {0, 0, 0};

    SC_TID(tid);

/* get current seed */
    _PM_seed_48(SeedDum, wrk_seed);

/* set seed and multiplier */
    _PM_rnset(wrk_seed, MULTPLR(tid));

/* store wrk_seed in seed */
    for (j = 0; j < 3; j++)
        seed.sa[j] = wrk_seed[j];

    seed.sa[3] = 0x0;
  
    return(seed.sd);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_RANDOM_48 - generate a pseudo-random number
 *              - return a pseudo-random number between -1 and 1
 *              - use the linear congruential sequence function
 */

double PM_random_48(x)
   double x;
   {double v;

    v = 2.0*_PM_random_48() - 1.0;
        
    return(v);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PM_RANDOM_S - return a random number between -1 and 1
 *             - use the system rand function
 */

double PM_random_s(x)
   double x;
   {double v;
    static double rn = 0.0;

    if (rn == 0.0)
       rn = 2.0/RAND_MAX;

    v = rn*rand() - 1.0;
        
    return(v);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PMRNDF - F77 interface to PM_random_48 */

REAL F77_ID(pmrndf_, pmrndf, PMRNDF)(px)
   REAL *px;
   {

    return((REAL) PM_random_48((double) *px));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PMRNDF - F77 interface to PM_random_s */

REAL F77_ID(pmrnds_, pmrnds, PMRNDS)(px)
   REAL *px;
   {

    return((REAL) PM_random_s((double) *px));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

