/*
 * GSFIA.C - FORTRAN interface routines for PGS
 *         - NOTE: let's keep these in alphabetical order
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pgs.h"

static int
 _pg_view_attr_indx = 0,
 _pg_view_attr_max_indx = 0;

static PG_view_attributes
 **_pg_view_attr_list = NULL;

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_VIEW_ATTRIBUTES_POINTER - return the PG_view_attributes pointer
 *                            - associated with the index if the index
 *                            - is valid and NULL otherwise
 */

PG_view_attributes *PG_view_attributes_pointer(vwatid)
   int vwatid;
   {if ((vwatid < 0) || (vwatid >= _pg_view_attr_indx))
       return(NULL);

    return(_pg_view_attr_list[vwatid]);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGAXIS - draw an axis set */

FIXNUM F77_ID(pgaxis_, pgaxis, PGAXIS)(devid, paxt)
   FIXNUM *devid, *paxt;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_axis(dev, (int) *paxt);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSAXA - set the parameters which control the look
 *        - of the axes being drawn
 *        - see PG_set_axis_attributes description
 *        - for character valued attributes the number of characters
 *        - in the value is in the ATTR (PAT) value position and
 *        - the characters are stored consecutively in PATC
 */

FIXNUM F77_ID(pgsaxa_, pgsaxa, PGSAXA)(devid, pn, pat, patc)
   FIXNUM *devid, *pn;
   REAL *pat;
   F77_string patc;
   {int i, nc, nn, type, linecolor, txtcolor, prec;
    REAL *attr;
    REAL charspace, chupx, chupy, chpthx, chpthy;
    char *pc, bf[MAXLINE];
    PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);

/* load default values for external variables */
    linecolor     = dev->WHITE;
    txtcolor      = dev->WHITE;
    prec          = TEXT_CHARACTER_PRECISION;

    if (_PG_axis_label_x_format == NULL)
       _PG_axis_label_x_format = SC_strsavef("%10.2g",
                                 "char*:PGSAXA:xformat");

    if (_PG_axis_label_y_format == NULL)
       _PG_axis_label_y_format = SC_strsavef("%10.2g",
                                 "char*:PGSAXA:yformat");

    if (_PG_axis_type_face == NULL)
       _PG_axis_type_face = SC_strsavef("helvetica",
                            "char*:PGSAXA:type_face");

    charspace    = 0.0;
    chpthx       = 1.0;
    chpthy       = 0.0;
    chupx        = 0.0;
    chupy        = 1.0;

    _PG_axis_grid_on = FALSE;

/* the type of the second part of the pair is dependent on the
 * value of type 
 */
    nn   = *pn;
    pc   = SC_F77_C_STRING(patc);
    attr = pat;
    for (i = 0; i < nn; i++)
        {type = *attr++;
	 switch (type)
            {case AXIS_LINESTYLE :
	          _PG_axis_grid_style = *attr++;
		  break;

	     case AXIS_LINETHICK :
	          _PG_axis_line_width = *attr++;
		  break;

             case AXIS_LINECOLOR :
	          linecolor = *attr++;
		  break;

             case AXIS_LABELCOLOR :
	          txtcolor = *attr++;
		  break;

	     case AXIS_LABELSIZE :
	          _PG_axis_char_size = *attr++;
		  break;

             case AXIS_CHAR_ANGLE :
	          _PG_axis_char_angle = *attr++;
		  break;

             case AXIS_LABELPREC :
	          prec = *attr++;
		  break;

             case AXIS_TICKSIZE :
	          _PG_axis_major_tick_size = *attr++;
		  break;

             case AXIS_GRID_ON :
	          _PG_axis_grid_on = *attr++;
		  break;

             case AXIS_LABELFONT :
	          SFREE(_PG_axis_type_face);
		  nc = *attr++;
		  strncpy(bf, pc, nc);
		  bf[nc] = '\0';
                  pc += nc;
		  _PG_axis_type_face = SC_strsavef(bf,
                                       "char*:PGSAXA:bf");
		  break;

             case AXIS_X_FORMAT :
		  nc = *attr++;
		  strncpy(bf, pc, nc);
		  bf[nc] = '\0';
                  pc += nc;
	          strcpy(_PG_axis_label_x_format, bf);
		  break;

             case AXIS_Y_FORMAT :
		  nc = *attr++;
		  strncpy(bf, pc, nc);
		  bf[nc] = '\0';
                  pc += nc;
	          strcpy(_PG_axis_label_y_format, bf);
		  break;

             default :
	          return(FALSE);};};

    chpthx = sin(_PG_axis_char_angle);
    chpthy = cos(_PG_axis_char_angle);

/* set attribute values */
    PG_set_clipping(dev, FALSE);
    PG_set_color_text(dev, txtcolor, TRUE);
    PG_set_font(dev, _PG_axis_type_face, dev->type_style, dev->type_size);
    PG_set_char_precision(dev, prec);
    PG_set_char_path(dev, chpthx, chpthy);
    PG_set_char_up(dev, chupx, chupy);
    PG_set_char_space(dev, charspace);
    PG_set_color_line(dev, linecolor, TRUE);
    PG_set_line_style(dev, _PG_axis_line_style);
    PG_set_line_width(dev, _PG_axis_line_width);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGAXD - get axis decades */

FIXNUM F77_ID(pggaxd_, pggaxd, PGGAXD)(pd)
   REAL *pd;
   {

    PG_get_axis_decades(*pd);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGAXL - get log axis flags */

FIXNUM F77_ID(pggaxl_, pggaxl, PGGAXL)(devid, pxls, pyls)
   FIXNUM *devid, *pxls, *pyls;
   {int xls, yls;
    PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);

    PG_get_axis_log_scale(dev, &xls, &yls);

    *pxls = xls;
    *pyls = yls;

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGCLOS - close the PG_device associated with the integer index */

FIXNUM F77_ID(pgclos_, pgclos, PGCLOS)(devid)
   FIXNUM *devid;
   {PG_device *dev;

    dev = SC_DEL_POINTER(PG_device, *devid);
    PG_close_device(dev);

    *devid = -1;

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGCLPG - clear the page */

FIXNUM F77_ID(pgclpg_, pgclpg, PGCLPG)(devid, pi)
   FIXNUM *devid, *pi;
   {PG_device *dev;
    int i;

    dev = SC_GET_POINTER(PG_device, *devid);
    i   = *pi;

    PG_clear_page(dev, i);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGCLRG - clear the specified region */

FIXNUM F77_ID(pgclrg_, pgclrg, PGCLRG)(devid, pxn, pxx, pyn, pyx, pad)
   FIXNUM *devid;
   REAL *pxn, *pxx, *pyn, *pyx;
   FIXNUM *pad;
   {PG_device *dev;
    double xmn, xmx, ymn, ymx;
    int pd;

    dev = SC_GET_POINTER(PG_device, *devid);
    pd  = *pad;

    xmn = *pxn;
    xmx = *pxx;
    ymn = *pyn;
    ymx = *pyx;

    PG_clear_region_NDC(dev, xmn, xmx, ymn, ymx, pd);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGCLSC - clear the screen (necessary for CGM devices) */

FIXNUM F77_ID(pgclsc_, pgclsc, PGCLSC)(devid)
   FIXNUM *devid;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_clear_window(dev);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGCLVP - clear the viewport */

FIXNUM F77_ID(pgclvp_, pgclvp, PGCLVP)(devid)
   FIXNUM *devid;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_clear_viewport(dev);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGDDP2 - draw 2d disjoint polyline */

FIXNUM F77_ID(pgddp2_, pgddp2, PGDDP2)(devid, px, py, pn, pf, pc)
   FIXNUM *devid;
   REAL *px, *py;
   FIXNUM *pn, *pf, *pc;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_draw_disjoint_polyline_2(dev, px, py, (long) *pn,
                                (int) *pf, (int) *pc);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGDPL2 - draw 2d polyline */

FIXNUM F77_ID(pgdpl2_, pgdpl2, PGDPL2)(devid, px, py, pn, pc)
   FIXNUM *devid;
   REAL *px, *py;
   FIXNUM *pn, *pc;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_draw_polyline(dev, px, py, (long) *pn, (int) *pc);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGDDP3 - draw 3d disjoint polyline */

FIXNUM F77_ID(pgddp3_, pgddp3, PGDDP3)(devid, px, py, pz, pth, pph, pch,
				       pn, pf, pnrm)
   FIXNUM *devid;
   REAL *px, *py, *pz;
   REAL *pth, *pph, *pch;
   FIXNUM *pn, *pf, *pnrm;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_draw_disjoint_polyline_3(dev, px, py, pz,
                                (double) *pth, (double) *pph, (double) *pch,
                                (long) *pn, (int) *pf, (int) *pnrm);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGDRAX - draw a single axis */

FIXNUM F77_ID(pgdrax_, pgdrax, PGDRAX)(devid, pxl, pyl, pxr, pyr,
                                       pt1, pt2, pv1, pv2, psc, pnc, format,
                                       ptd, ptt, plt)
   FIXNUM *devid;
   REAL *pxl, *pyl, *pxr, *pyr, *pt1, *pt2, *pv1, *pv2, *psc;
   FIXNUM *pnc;
   F77_string format;
   FIXNUM *ptd, *ptt, *plt;
   {PG_device *dev;
    char lformat[MAXLINE];

    SC_FORTRAN_STR_C(lformat, format, *pnc);

    dev = SC_GET_POINTER(PG_device, *devid);

    switch (*ptd)
       {case MAJOR             :
             PG_draw_axis(dev, *pxl, *pyl, *pxr, *pyr,
                          *pt1, *pt2, *pv1, *pv2, *psc, lformat,
                          (int) *ptt, (int) *plt, FALSE,
                          MAJOR, 0);
             break;

        case MINOR             :
             PG_draw_axis(dev, *pxl, *pyl, *pxr, *pyr,
                          *pt1, *pt2, *pv1, *pv2, *psc, lformat,
                          (int) *ptt, (int) *plt, FALSE,
                          MINOR, 0);
             break;

        case LABEL             :
             PG_draw_axis(dev, *pxl, *pyl, *pxr, *pyr,
                          *pt1, *pt2, *pv1, *pv2, *psc, lformat,
                          (int) *ptt, (int) *plt, FALSE,
                          LABEL, 0);
             break;

        case MAJOR_MINOR       :
             PG_draw_axis(dev, *pxl, *pyl, *pxr, *pyr,
                          *pt1, *pt2, *pv1, *pv2, *psc, lformat,
                          (int) *ptt, (int) *plt, FALSE,
                          MAJOR, MINOR, 0);
             break;

        case MAJOR_LABEL       :
             PG_draw_axis(dev, *pxl, *pyl, *pxr, *pyr,
                          *pt1, *pt2, *pv1, *pv2, *psc, lformat,
                          (int) *ptt, (int) *plt, FALSE,
                          MAJOR, LABEL, 0);
             break;

        case MINOR_LABEL       :
             PG_draw_axis(dev, *pxl, *pyl, *pxr, *pyr,
                          *pt1, *pt2, *pv1, *pv2, *psc, lformat,
                          (int) *ptt, (int) *plt, FALSE,
                          MINOR, LABEL, 0);
             break;

        case MAJOR_MINOR_LABEL :
             PG_draw_axis(dev, *pxl, *pyl, *pxr, *pyr,
                          *pt1, *pt2, *pv1, *pv2, *psc, lformat,
                          (int) *ptt, (int) *plt, FALSE,
                          MAJOR, MINOR, LABEL, 0);
             break;};

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGDAX3 - draw a 3D axis set */

FIXNUM F77_ID(pgdax3_, pgdax3, PGDAX3)(devid, pxl, pyl, pzl, pnp,
                                       pth, pph, pch, pnrm)
   FIXNUM *devid;
   REAL *pxl, *pyl, *pzl;
   FIXNUM *pnp;
   REAL *pth, *pph, *pch;
   FIXNUM *pnrm;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);

    PG_axis_3d(dev, pxl, pyl, pzl, (int) *pnp,
	       (double) *pth, (double) *pph, (double) *pch,
               0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
               (int) *pnrm);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGDRMK - draw marker characters */

FIXNUM F77_ID(pgdrmk_, pgdrmk, PGDRMK)(devid, pn, px, py, pmrk)
   FIXNUM *devid, *pn;
   REAL *px, *py;
   FIXNUM *pmrk;
   {PG_device *dev;
    int n, mrk;

    dev = SC_GET_POINTER(PG_device, *devid);
    n   = *pn;
    mrk = *pmrk;

    PG_draw_markers(dev, n, px, py, mrk);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGDMRK - define a marker character */

FIXNUM F77_ID(pgdmrk_, pgdmrk, PGDMRK)(pn, px1, py1, px2, py2)
   FIXNUM *pn;
   REAL *px1, *py1, *px2, *py2;
   {int n;

    n = *pn;

    return((FIXNUM) PG_def_marker(n, px1, py1, px2, py2));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGDPLT - draw a mesh - plot a domain */

FIXNUM F77_ID(pgdplt_, pgdplt, PGDPLT)(devid, dom)
   FIXNUM *devid, *dom;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_domain_plot(dev, *(PM_set **) dom, NULL);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGDRBX - draw a box */

FIXNUM F77_ID(pgdrbx_, pgdrbx, PGDRBX)(devid, px1, px2, py1, py2)
   FIXNUM *devid;
   REAL *px1, *px2, *py1, *py2;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_draw_box(dev, *px1, *px2, *py1, *py2);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGDRIF - draw the set of interface objects for the device */

FIXNUM F77_ID(pgdrif_, pgdrif, PGDRIF)(devid)
   FIXNUM *devid;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_draw_interface_objects(dev);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGDRLN - draw a line between the specified points */

FIXNUM F77_ID(pgdrln_, pgdrln, PGDRLN)(devid, px1, py1, px2, py2)
   FIXNUM *devid;
   REAL *px1, *py1, *px2, *py2;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_draw_line(dev, *px1, *py1, *px2, *py2);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGDRPA - draw the current palette */

FIXNUM F77_ID(pgdrpa_, pgdrpa, PGDRPA)(devid, px1, py1, px2, py2, pz1, pz2, pw)
   FIXNUM *devid;
   REAL *px1, *py1, *px2, *py2, *pz1, *pz2, *pw;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_draw_palette(dev, *px1, *py1, *px2, *py2, *pz1, *pz2, *pw);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGFNPL - finish the plot (necessary for CGM devices) */

FIXNUM F77_ID(pgfnpl_, pgfnpl, PGFNPL)(devid)
   FIXNUM *devid;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_finish_plot(dev);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGFPLY - draw and fill the specified polygon with the specified color */

FIXNUM F77_ID(pgfply_, pgfply, PGFPLY)(devid, px, py, pn, pc)
   FIXNUM *devid;
   REAL *px, *py;
   FIXNUM *pn, *pc;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_set_color_fill(dev, (int) *pc, TRUE);
    PG_shade_poly(dev, px, py, *pn);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGCLM - get the clear mode */

FIXNUM F77_ID(pggclm_, pggclm, PGGCLM)(devid, pc)
   FIXNUM *devid, *pc;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_get_clear_mode(*pc);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGCLP - get the clipping */

FIXNUM F77_ID(pggclp_, pggclp, PGGCLP)(devid, pc)
   FIXNUM *devid, *pc;
   {PG_device *dev;
    int flg;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_get_clipping(dev, &flg);

    *pc = flg;

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGCPW - get the char path direction in WC */

FIXNUM F77_ID(pggcpw_, pggcpw, PGGCPW)(devid, px, py)
   FIXNUM *devid;
   REAL *px, *py;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_get_char_path(dev, px, py);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGCSS - get the char size in NDC */

FIXNUM F77_ID(pggcss_, pggcss, PGGCSS)(devid, pw, ph)
   FIXNUM *devid;
   REAL *pw, *ph;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_get_char_size_NDC(dev, pw, ph);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGCSW - get the char size in WC */

FIXNUM F77_ID(pggcsw_, pggcsw, PGGCSW)(devid, pw, ph)
   FIXNUM *devid;
   REAL *pw, *ph;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_get_char_size(dev, pw, ph);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGCUW - get the char up direction in WC */

FIXNUM F77_ID(pggcuw_, pggcuw, PGGCUW)(devid, px, py)
   FIXNUM *devid;
   REAL *px, *py;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_get_char_up(dev, px, py);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGDPI - get the PostScript dots per inch for 8.5 x 11 page */

FIXNUM F77_ID(pggdpi_, pggdpi, PGGDPI)(dpi)
   REAL *dpi;
   {

    PG_get_ps_dots_inch(*dpi);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGFIN - get the finished status */

FIXNUM F77_ID(pggfin_, pggfin, PGGFIN)(devid, pc)
   FIXNUM *devid, *pc;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    *pc = dev->finished;

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGLNC - get the line color */

FIXNUM F77_ID(pgglnc_, pgglnc, PGGLNC)(devid, pc)
   FIXNUM *devid, *pc;
   {PG_device *dev;
    int lc;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_get_line_color(dev, &lc);

    *pc = lc;

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGLOP - get the logical operation */

FIXNUM F77_ID(pgglop_, pgglop, PGGLOP)(devid, plop)
   FIXNUM *devid, *plop;
   {PG_device *dev;
    int lop;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_get_logical_op(dev, &lop);

    *plop = lop;

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGLNS - get the line style */

FIXNUM F77_ID(pgglns_, pgglns, PGGLNS)(devid, ps)
   FIXNUM *devid, *ps;
   {PG_device *dev;
    int ls;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_get_line_style(dev, &ls);

    *ps = ls;

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGLNW - get the line width */

FIXNUM F77_ID(pgglnw_, pgglnw, PGGLNW)(devid, pw)
   FIXNUM *devid;
   REAL *pw;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_get_line_width(dev, pw);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGMKO - get the marker orientation */

FIXNUM F77_ID(pggmko_, pggmko, PGGMKO)(devid, pw)
   FIXNUM *devid;
   REAL *pw;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_get_marker_orientation(dev, *pw);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGMKS - get the marker scale */

FIXNUM F77_ID(pggmks_, pggmks, PGGMKS)(devid, ps)
   FIXNUM *devid;
   REAL *ps;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_get_marker_scale(dev, *ps);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGTLN - get a line of text from the input descriptor */

FIXNUM F77_ID(pggtln_, pggtln, PGGTLN)(pnc, s, pfd)
   FIXNUM *pnc;
   F77_string s;
   FIXNUM *pfd;
   {char *t;
    long fd;
    SC_address f;

    fd = (long) *pfd;

    if (fd == 0L)
       t = GETLN(SC_F77_C_STRING(s), (int) *pnc, stdin);

    else
       {f.diskaddr = fd;
        t = GETLN(SC_F77_C_STRING(s), (int) *pnc, (FILE *) f.memaddr);};

    return((FIXNUM) (t != NULL));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGTXC - get the text color */

FIXNUM F77_ID(pggtxc_, pggtxc, PGGTXC)(devid, pc)
   FIXNUM *devid, *pc;
   {PG_device *dev;
    int lc;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_get_text_color(dev, &lc);

    *pc = lc;

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGTXF - get the font */

FIXNUM F77_ID(pggtxf_, pggtxf, PGGTXF)(devid, pncf, pf, pncs, pst, psz)
   FIXNUM *devid, *pncf;
   F77_string pf;
   FIXNUM *pncs;
   F77_string pst;
   FIXNUM *psz;
   {PG_device *dev;
    char *f, *st;
    int ls, nf, ns, lnf, lns;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_get_font(dev, &f, &st, &ls);

    nf = strlen(f);
    ns = strlen(st);

/* get the buffer sizes */
    lnf = *pncf;
    lns = *pncs;

/* return the actual string lengths */
    *pncf = nf;
    *pncs = ns;

    if ((nf > lnf) || (ns > lns))
       return((FIXNUM) FALSE);

    else
       {strncpy(SC_F77_C_STRING(pf), f, lnf);
        strncpy(SC_F77_C_STRING(pst), st, lns);
        *psz = ls;

        SFREE(f);
        SFREE(st);

        return((FIXNUM) TRUE);};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGTES - get the text extent in NDC */

FIXNUM F77_ID(pggtes_, pggtes, PGGTES)(devid, pnc, s, pdx, pdy)
   FIXNUM *devid, *pnc;
   F77_string s;
   REAL *pdx, *pdy;
   {PG_device *dev;
    char ls[MAXLINE];

    SC_FORTRAN_STR_C(ls, s, *pnc);

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_get_text_ext_NDC(dev, ls, pdx, pdy);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGTEW - get the text extent in WC */

FIXNUM F77_ID(pggtew_, pggtew, PGGTEW)(devid, pnc, s, pdx, pdy)
   FIXNUM *devid, *pnc;
   F77_string s;
   REAL *pdx, *pdy;
   {PG_device *dev;
    char ls[MAXLINE];

    SC_FORTRAN_STR_C(ls, s, *pnc);

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_get_text_ext(dev, ls, pdx, pdy);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGVWP - get the viewport */

FIXNUM F77_ID(pggvwp_, pggvwp, PGGVWP)(devid, px1, px2, py1, py2)
   FIXNUM *devid;
   REAL *px1, *px2, *py1, *py2;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_get_viewport(dev, px1, px2, py1, py2);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGGWCS - get the world coordinate system */

FIXNUM F77_ID(pggwcs_, pggwcs, PGGWCS)(devid, px1, px2, py1, py2)
   FIXNUM *devid;
   REAL *px1, *px2, *py1, *py2;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_get_viewport_WC(dev, px1, px2, py1, py2);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGMDVC - make the device current */

FIXNUM F77_ID(pgmdvc_, pgmdvc, PGMDVC)(devid)
   FIXNUM *devid;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_make_device_current(dev);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGMKDV - make a PG_device
 *        - save the PG_device pointer in an internal array
 *        - and return an integer index to the pointer if successful
 *        - return -1 otherwise
 */

FIXNUM F77_ID(pgmkdv_, pgmkdv, PGMKDV)(pncn, name, pnct, type, pncl, title)
   FIXNUM *pncn;
   F77_string name;
   FIXNUM *pnct;
   F77_string type;
   FIXNUM *pncl;
   F77_string title;
   {char lname[MAXLINE], ltype[MAXLINE], ltitle[MAXLINE];
    PG_device *dev;

    SC_FORTRAN_STR_C(lname, name, *pncn);
    SC_FORTRAN_STR_C(ltype, type, *pnct);
    SC_FORTRAN_STR_C(ltitle, title, *pncl);

    dev = PG_make_device(lname, ltype, ltitle);
    if (dev == NULL)
       return(-1);

    else
       return((FIXNUM) SC_ADD_POINTER(dev));}
    
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGOPEN - open the specified PG_device */

FIXNUM F77_ID(pgopen_, pgopen, PGOPEN)(devid, pxf, pyf, pdxf, pdyf)
   FIXNUM *devid;
   REAL *pxf, *pyf, *pdxf, *pdyf;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    if (PG_open_device(dev, *pxf, *pyf, *pdxf, *pdyf) == NULL)
       return((FIXNUM) FALSE);
    else
       return((FIXNUM) TRUE);}
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGPTIN - given a pointer, return its index
 *        - return -1 otherwise
 */

FIXNUM F77_ID(pgptin_, pgptin, PGPTIN)(pntr, indx)
   FIXNUM *pntr;
   FIXNUM *indx;
   {return(*indx = SC_GET_INDEX((byte *)pntr));}
        
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGQDEV - get some physical parameters from the specified PG_device */

FIXNUM F77_ID(pgqdev_, pgqdev, PGQDEV)(devid, pdx, pdy, pnc)
   FIXNUM *devid, *pdx, *pdy, *pnc;
   {PG_device *dev;
    int dx, dy, nc;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_query_device(dev, &dx, &dy, &nc);

    *pdx = dx;
    *pdy = dy;
    *pnc = nc;

    return((FIXNUM) TRUE);}
    
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGQWIN - get the window shape of the specified device */

FIXNUM F77_ID(pgqwin_, pgqwin, PGQWIN)(devid, pdx, pdy)
   FIXNUM *devid, *pdx, *pdy;
   {PG_device *dev;
    int dx, dy;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_query_window(dev, &dx, &dy);

    *pdx = dx;
    *pdy = dy;

    return((FIXNUM) TRUE);}
    
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGPLLN - low level line plot routine */

FIXNUM F77_ID(pgplln_, pgplln, PGPLLN)(devid, px, py, pn, pmod, paxt,
				       pcol, pwid,
				       psty, psca, pmrk, psta, pl)
   FIXNUM *devid;
   REAL *px, *py;
   FIXNUM *pn, *pmod, *paxt, *pcol;
   REAL *pwid;
   FIXNUM *psty, *psca, *pmrk, *psta, *pl;
   {PG_device *dev;
    pcons *info;

    dev = SC_GET_POINTER(PG_device, *devid);

    info = PG_set_line_info(NULL, (int) *pmod, (int) *paxt,
			    (int) *psty, (int) *psca, (int) *pmrk,
			    (int) *pcol, (int) *psta, *pwid);
    PG_plot_curve(dev, px, py, (int) *pn, info, (int) *pl);

    SC_free_alist(info, 2);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGPTOS - convert from PC to NDC */

FIXNUM F77_ID(pgptos_, pgptos, PGPTOS)(devid, pix, piy, px, py)
   FIXNUM *devid, *pix, *piy;
   REAL *px, *py;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);

    PtoS(dev, *pix, *piy, *px, *py);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGRDIF - read an interface description file */

FIXNUM F77_ID(pgrdif_, pgrdif, PGRDIF)(devid, pnc, name)
   FIXNUM *devid, *pnc;
   F77_string name;
   {PG_device *dev;
    char lname[MAXLINE];

    dev = SC_GET_POINTER(PG_device, *devid);

    SC_FORTRAN_STR_C(lname, name, *pnc);

    return((FIXNUM) PG_read_interface(dev, lname));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGRDVC - release device as the device current */

FIXNUM F77_ID(pgrdvc_, pgrdvc, PGRDVC)(devid)
   FIXNUM *devid;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_release_current_device(dev);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGRGDV - register a device */

FIXNUM F77_ID(pgrgdv_, pgrgdv, PGRGDV)(pnc, name)
   FIXNUM *pnc;
   F77_string name;
   {char lname[MAXLINE];

    extern void
      SC_DECLARE(PG_setup_image_device, (PG_device *d));

#ifdef HAVE_JPEGLIB
    extern void
      SC_DECLARE(PG_setup_jpeg_device, (PG_device *d));
#endif
    
    SC_FORTRAN_STR_C(lname, name, *pnc);

    if (!strcmp(lname, "IMAGE"))
       {PG_register_device("IMAGE", PG_setup_image_device);
        return((FIXNUM) TRUE);}

#ifdef HAVE_JPEGLIB
    else if (!strcmp(lname, "JPEG"))
       {PG_register_device("JPEG", PG_setup_jpeg_device);
        return((FIXNUM) TRUE);}
#endif 

    return((FIXNUM) FALSE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGRGFN - register a call back function with the device */

FIXNUM F77_ID(pgrgfn_, pgrgfn, PGRGFN)(pnc, name, fnc)
   FIXNUM *pnc;
   F77_string name;
   PFVoid fnc;
   {char lname[MAXLINE];

    SC_FORTRAN_STR_C(lname, name, *pnc);
    PG_register_callback(lname, fnc);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGRGVR - register a variable with the device */

FIXNUM F77_ID(pgrgvr_, pgrgvr, PGRGVR)(pnc, name, pnt, type, vr, vn, vx)
   FIXNUM *pnc;
   F77_string name;
   FIXNUM *pnt;
   F77_string type;
   byte *vr, *vn, *vx;
   {char lname[MAXLINE], ltype[MAXLINE];

    SC_FORTRAN_STR_C(lname, name, *pnc);
    SC_FORTRAN_STR_C(ltype, type, *pnt);
    PG_register_variable(lname, SC_strsavef(ltype,
                            "char*:PGRGVR:ltype"), vr, vn, vx);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGRVPA - restore the PG_view_attributes for the specified device
 *        - return TRUE iff successful
 */

FIXNUM F77_ID(pgrvpa_, pgrvpa, PGRVPA)(devid, vwatid)
   FIXNUM *devid, *vwatid;
   {PG_device *dev;
    PG_view_attributes *d;

    dev = SC_GET_POINTER(PG_device, *devid);
    d   = PG_view_attributes_pointer((int) *vwatid);

    PG_restore_view_attributes(dev, d);

    return((FIXNUM) TRUE);}
    
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSADR - set the auto domain and range */

FIXNUM F77_ID(pgsadr_, pgsadr, PGSADR)(devid, pd, pr)
   FIXNUM *devid, *pd, *pr;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);

    dev->autodomain = *pd;
    dev->autorange  = *pr;

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSADM - set the auto domain */

FIXNUM F77_ID(pgsadm_, pgsadm, PGSADM)(devid, pd)
   FIXNUM *devid, *pd;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);

    dev->autodomain = *pd;

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSARN - set the auto range */

FIXNUM F77_ID(pgsarn_, pgsarn, PGSARN)(devid, pr)
   FIXNUM *devid, *pr;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);

    dev->autorange  = *pr;

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSAXD - set axis decades */

FIXNUM F77_ID(pgsaxd_, pgsaxd, PGSAXD)(pd)
   REAL *pd;
   {REAL d;

    d = *pd;

    PG_set_axis_decades(d);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSAXL - set log axis flags */

FIXNUM F77_ID(pgsaxl_, pgsaxl, PGSAXL)(devid, pxls, pyls)
   FIXNUM *devid, *pxls, *pyls;
   {int xls, yls;
    PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    xls = *pxls;
    yls = *pyls;

    PG_set_axis_log_scale(dev, xls, yls);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSBWD - set the border width */

FIXNUM F77_ID(pgsbwd_, pgsbwd, PGSBWD)(devid, pw)
   FIXNUM *devid, *pw;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);

    PG_set_border_width(dev, *pw);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSCLM - set the clear mode */

FIXNUM F77_ID(pgsclm_, pgsclm, PGSCLM)(devid, pc)
   FIXNUM *devid, *pc;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_set_clear_mode(*pc);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSCLP - set the clipping */

FIXNUM F77_ID(pgsclp_, pgsclp, PGSCLP)(devid, pc)
   FIXNUM *devid, *pc;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_set_clipping(dev, (int) *pc);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSCPW - set the char path direction in WC */

FIXNUM F77_ID(pgscpw_, pgscpw, PGSCPW)(devid, px, py)
   FIXNUM *devid;
   REAL *px, *py;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_set_char_path(dev, (double) *px, (double) *py);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSCUW - set the char up direction in WC */

FIXNUM F77_ID(pgscuw_, pgscuw, PGSCUW)(devid, px, py)
   FIXNUM *devid;
   REAL *px, *py;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_set_char_up(dev, (double) *px, (double) *py);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSDPI - set the PostScript dots per inch for 8.5 x 11 page */

FIXNUM F77_ID(pgsdpi_, pgsdpi, PGSDPI)(dpi)
   REAL *dpi;
   {

    PG_set_ps_dots_inch(*dpi);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSELC - select a color from the n-dimensional palette */

FIXNUM F77_ID(pgselc_, pgselc, PGSELC)(devid, n, av, an, ax)
   FIXNUM *devid, *n;
   REAL *av, *an, *ax;
   {REAL extr[2];
    PG_device *dev;

    extr[0] = *an;
    extr[1] = *ax;
    
    dev = SC_GET_POINTER(PG_device, *devid);

    return((FIXNUM) PG_select_color(dev, *n, av, extr));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSFCL - set the fill color */

FIXNUM F77_ID(pgsfcl_, pgsfcl, PGSFCL)(devid, pc)
   FIXNUM *devid, *pc;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_set_fill_color(dev, *pc);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSFIN - set the finished status */

FIXNUM F77_ID(pgsfin_, pgsfin, PGSFIN)(devid, pc)
   FIXNUM *devid, *pc;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    dev->finished = *pc;

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSLNC - set the line color */

FIXNUM F77_ID(pgslnc_, pgslnc, PGSLNC)(devid, pc)
   FIXNUM *devid, *pc;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_set_color_line(dev, (int) *pc, TRUE);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSLOP - set the logical operation */

FIXNUM F77_ID(pgslop_, pgslop, PGSLOP)(devid, plop)
   FIXNUM *devid, *plop;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_set_logical_op(dev, (int) *plop);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSLNS - set the line style */

FIXNUM F77_ID(pgslns_, pgslns, PGSLNS)(devid, ps)
   FIXNUM *devid, *ps;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_set_line_style(dev, (int) *ps);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSLNW - set the line width */

FIXNUM F77_ID(pgslnw_, pgslnw, PGSLNW)(devid, pw)
   FIXNUM *devid;
   REAL *pw;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_set_line_width(dev, (double) *pw);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSMKO - set the marker orientation */

FIXNUM F77_ID(pgsmko_, pgsmko, PGSMKO)(devid, pw)
   FIXNUM *devid;
   REAL *pw;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_set_marker_orientation(dev, (double) *pw);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSMKS - set the marker scale */

FIXNUM F77_ID(pgsmks_, pgsmks, PGSMKS)(devid, ps)
   FIXNUM *devid;
   REAL *ps;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_set_marker_scale(dev, (double) *ps);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSPAL - set the current palette */

FIXNUM F77_ID(pgspal_, pgspal, PGSPAL)(devid, pnc, pname)
   FIXNUM *devid, *pnc;
   F77_string pname;
   {PG_device *dev;
    char lname[MAXLINE];

    dev = SC_GET_POINTER(PG_device, *devid);
    SC_FORTRAN_STR_C(lname, pname, *pnc);

    return((PG_set_palette(dev, lname) == NULL) ?
           (FIXNUM) FALSE : (FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSTOP - convert from NDC to PC */

FIXNUM F77_ID(pgstop_, pgstop, PGSTOP)(devid, px, py, pix, piy)
   FIXNUM *devid;
   REAL *px, *py;
   FIXNUM *pix, *piy;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);

    StoP(dev, *px, *py, *pix, *piy);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSTOW - convert from NDC to WC */

FIXNUM F77_ID(pgstow_, pgstow, PGSTOW)(devid, px, py)
   FIXNUM *devid;
   REAL *px, *py;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);

    StoW(dev, *px, *py);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSTXC - set the text color */

FIXNUM F77_ID(pgstxc_, pgstxc, PGSTXC)(devid, pc)
   FIXNUM *devid, *pc;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_set_color_text(dev, (int) *pc, TRUE);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSTXF - set the font */

FIXNUM F77_ID(pgstxf_, pgstxf, PGSTXF)(devid, pncf, face, pncs, style, psize)
   FIXNUM *devid, *pncf;
   F77_string face;
   FIXNUM *pncs;
   F77_string style;
   FIXNUM *psize;
   {PG_device *dev;
    char lface[MAXLINE], lstyle[MAXLINE];

    dev = SC_GET_POINTER(PG_device, *devid);

    SC_FORTRAN_STR_C(lface, face, *pncf);
    SC_FORTRAN_STR_C(lstyle, style, *pncs);

    PG_set_font(dev, lface, lstyle, (int) *psize);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSVPA - save the PG_view_attributes for the specified device
 *        - save the PG_view_attributes pointer in an internal array
 *        - return an integer index to the pointer if successful (via PN)
 *        - return TRUE if successful
 *        - if N < 0 allocate a new PG_view_attributes
 *        - otherwise N is taken to be the index of an existing structure
 *        - to be reused
 */

FIXNUM F77_ID(pgsvpa_, pgsvpa, PGSVPA)(devid, pn)
   FIXNUM *devid, *pn;
   {PG_device *dev;
    PG_view_attributes *d;
    int n;

    dev = SC_GET_POINTER(PG_device, *devid);
    n   = *pn;
    if ((n < 0) || (n >= _pg_view_attr_indx))

/* the first time out allocate the lists of pointers and disk addresses */
       {if (_pg_view_attr_list == NULL)
           {_pg_view_attr_indx = 0;
            _pg_view_attr_max_indx = 10;
            _pg_view_attr_list = FMAKE_N(PG_view_attributes *,
                                 _pg_view_attr_max_indx,
                                 "PGSVPA:_pg_view_attr_list");}

        if (_pg_view_attr_max_indx <= _pg_view_attr_indx)
           {_pg_view_attr_max_indx += 10;
            REMAKE_N(_pg_view_attr_list,
                     PG_view_attributes *,
                     _pg_view_attr_max_indx);};

        d = PG_make_view_attributes(dev);
        if (d == NULL)
           return((FIXNUM) -1);
        else
           {_pg_view_attr_list[_pg_view_attr_indx] = d;
            n = _pg_view_attr_indx++;};}

    else
       {d = _pg_view_attr_list[n];
        PG_save_view_attributes(d, dev);};

    *pn = n;

    return((FIXNUM) TRUE);}
    
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSVWP - set the viewport */

FIXNUM F77_ID(pgsvwp_, pgsvwp, PGSVWP)(devid, px1, px2, py1, py2)
   FIXNUM *devid;
   REAL *px1, *px2, *py1, *py2;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_set_viewport(dev, *px1, *px2, *py1, *py2);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSVPS - set viewport position */

FIXNUM F77_ID(pgsvps_, pgsvps, PGSVPS)(devid, px, py)
   FIXNUM *devid;
   REAL *px, *py;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);

    PG_set_viewport_pos(dev, *px, *py);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSVSH - set viewport shape */

FIXNUM F77_ID(pgsvsh_, pgsvsh, PGSVSH)(devid, pw, ph, pa)
   FIXNUM *devid;
   REAL *pw, *ph, *pa;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);

    PG_set_viewport_shape(dev, *pw, *ph, *pa);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGSWCS - set the world coordinate system */

FIXNUM F77_ID(pgswcs_, pgswcs, PGSWCS)(devid, px1, px2, py1, py2)
   FIXNUM *devid;
   REAL *px1, *px2, *py1, *py2;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_set_window(dev, *px1, *px2, *py1, *py2);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGUPVS - update the view surface of the PG_device */

FIXNUM F77_ID(pgupvs_, pgupvs, PGUPVS)(devid)
   FIXNUM *devid;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);
    PG_update_vs(dev);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGWRCL - write label centered wrt x */

FIXNUM F77_ID(pgwrcl_, pgwrcl, PGWRCL)(devid, psy, pnc, label)
   FIXNUM *devid;
   REAL *psy;
   FIXNUM *pnc;
   F77_string label;
   {PG_device *dev;
    char llabel[MAXLINE];

    dev = SC_GET_POINTER(PG_device, *devid);

    SC_FORTRAN_STR_C(llabel, label, *pnc);

    PG_center_label(dev, *psy, llabel);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGWRIF - write an interface description file */

FIXNUM F77_ID(pgwrif_, pgwrif, PGWRIF)(devid, pnc, name)
   FIXNUM *devid, *pnc;
   F77_string name;
   {PG_device *dev;
    char lname[MAXLINE];

    dev = SC_GET_POINTER(PG_device, *devid);

    SC_FORTRAN_STR_C(lname, name, *pnc);

    return((FIXNUM) PG_write_interface(dev, lname));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGWRTA - print the given message
 *        - at the specified screen coordinate in WC
 */

FIXNUM F77_ID(pgwrta_, pgwrta, PGWRTA)(devid, px, py, pnc, msg)
   FIXNUM *devid;
   REAL *px, *py;
   FIXNUM *pnc;
   F77_string msg;
   {char lmsg[MAXLINE];
    PG_device *dev;
    double x, y;

    x = *px;
    y = *py;
    SC_FORTRAN_STR_C(lmsg, msg, *pnc);

    dev = SC_GET_POINTER(PG_device, *devid);
    if (dev->gprint_flag)
       {PG_move_tx_abs(dev, x, y);
        PG_write_text(dev, stdscr, lmsg);};

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PGWTOS - convert from WC to NDC */

FIXNUM F77_ID(pgwtos_, pgwtos, PGWTOS)(devid, px, py)
   FIXNUM *devid;
   REAL *px, *py;
   {PG_device *dev;

    dev = SC_GET_POINTER(PG_device, *devid);

    WtoS(dev, *px, *py);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
