(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                program.ml                                *)
(****************************************************************************)

open Std;;
open Pp;;

open Names;;
open Vectops;;
open Generic;;
open Term;;
open Reduction;;
open Termenv;;
open More_util;;
open Machops;;
open Proof_trees;;
open Refiner;;
open Tacmach;;
open Progmach;;
open Constrtypes;;
open Clenv;;
open Initial;;
open Indtypes;;
open Printer;;
open Pfedit;;
open Himsg;;

open Wcclausenv;;
open Tacticals;;
open Tactics;;
open Elim;;


(* Checks if a term is a variable or not. *)
let rec is_var = function
  VAR _ -> true
| DOP2(Cast,c,_) -> is_var c
| DOPN(AppL,args) -> is_var (hd_vect args) 
| _ -> false ;;

(* Checks if a term is a constant or not. *)
let rec is_const = function
  DOPN(Const _,_) -> true
| DOP2(Cast,c,_) -> is_const c
| _ -> false ;;

(* A quoi sert cette metamap? elle ne contient que les metas introduites
 * par la fonction substn_meta, ce qui ne semble pas presenter grand
 * interet...
 *)
let mETAMAP = ref ([]:((int*constr) list));;
let clear_METAMAP () = mETAMAP:=[];;

(* ====================== *)
(* List of new exceptions *)
(* ====================== *)

(* raised when specif_of_annot fails *)
exception NotProgram ;;

(* raised when conclusion needs an expansion *)
exception Expand ;;

(* raised when program needs an expansion *)
exception Generalize of int ;;

(* raised when there are not two imbricated eliminations *)
exception Elim;;

(* raised when a program must not be modified by the modif_app_ind function *)
exception NoModif ;;
exception Rewrite;;

(* ================================== *)
(* List of different useful functions *)
(* ================================== *)

let rec nb_lam lf = 
  match strip_outer_cast lf with
    DOP2(Lambda,a,DLAM(x,b)) -> (nb_lam b)+1
  | x -> 0 ;;

(* A more appropriated function to check the dependency between two terms *) 
let dependant =
 let rec deprec m t =
     (eq_term m t)
     or (match t with
     VAR _       -> false
   | DOP1(_,c)   -> deprec m c
   | DOP2(_,c,t) -> deprec m c or deprec m t
   | DOPN(Const _,_) -> false
   | DOPN(MutInd _,_) -> false
   | DOPN(MutConstruct _,_) -> false
   | DOPN(_,cl)  -> exists_vect (deprec m) cl
   | DOPL(_,cl)  -> List.exists (deprec m) cl
   | DLAM(_,c)   -> deprec (lift 1 m) c
   | DLAMV(_,v)  -> exists_vect (deprec (lift 1 m)) v
   | _ -> false)
 in deprec;;

(* Checks if a term t = (MutInd() a1 ... an) *)
let rec is_mind = function
  DOPN(MutInd _,_) -> true
| DOP2(Cast,t,_) -> is_mind t
| DOPN(AppL,args) -> is_mind args.(0)
| _ -> false ;;

(* Checks if a term is a constructor or a mutconstructor or not *)
let rec is_construct = function
    DOPN(MutConstruct _,_) -> true 
  | DOP2(Cast,t,_) -> is_construct t
  | _ -> false ;;

let rec is_construct_head = function
    DOPN(MutConstruct _,_) -> true 
  | DOP2(Cast,t,_) -> is_construct_head t
  | DOPN(AppL,args) -> is_construct_head args.(0)
  | _ -> false ;;

(* Checks if a sort is informative or not *)
let rec informative_sort = function
  DOP0(Sort(Prop Pos)) -> true
| DOP0(Sort(Type(_))) -> true
| DOP0(Sort(Prop Null)) -> false
| DOP2(Prod,_,DLAM(_,b)) -> informative_sort b
| DOP2(Cast,t,_) -> informative_sort t
| _ -> anomaly"informative_sort" ;;

(* Gives the list of the types of the constructors of an inductive type
   Entry : ity = inductive type
   Output : list of the types of the constructors *)
let mconstructor_types ity =
  let lc = mis_lc (mind_specif_of_mind ity)
  in sAPPV lc ity
;;


(* Suppression of the outermost annotation of a term if it exists *)
let rec without_annot = function
  (DOP2(XTRA("ANNOT",[]),c,t))  -> c
| DOP2(Cast,t,_) -> without_annot t
| t -> t ;;

(* Extraction of the annotation of a term *)
let rec keep_only_annot = function
  (DOP2(XTRA("ANNOT",[]),c,t))  -> t
| DOP2(Cast,t,_) -> keep_only_annot t
| t -> t ;;

(* Suppression of the outermost comment of a term if it exists *)
let rec without_comment = function
  (DOP1(XTRA("COMMENT",[]),c))  -> c
| DOP2(Cast,t,_) -> without_comment t
| t -> t ;;


(* Constructs the ith constructor of ind applied to its parameters.
   Entries : i : number of the constructor
             ity : inductive type applied to at least its params
   Output : constructor. *)
let val_mconstruct sigma i ity = 
  match decomp_app (whd_betadeltaiota sigma ity) with
    (DOPN(MutInd(sp,j),cl) as ind, largs) ->
      let nparams = mind_nparams ind in
      let (globargs,_) = chop_list nparams largs in
      applist(DOPN(MutConstruct((sp,j),i),cl),globargs)
  | _ -> error "val_mconstruct" ;;

(* Tries to execute the term t as a pure term of the CCI.
   If it fails executes t as a program 
   Entries : sigma = ex.var. 
             sign = signature
             t = term
   Output : judgement *)
let try_execute sigma sign t = 
  try execute_pure sigma sign t
  with UserError _ -> execute_meta_prog sigma sign !mETAMAP t ;;

let try_reduce gls t =
  try whd_beta (pf_one_step_reduce gls t)
  with UserError _ -> t
;;

(* List of functions to create a clausal environment with a program *)
let mk_clenv_hnf_constr_type_of_prog gls t =
  let (wc,kont) = startWalk gls in
  let ty = pf_hnf_constr gls (type_of_prog (project gls) (pf_hyps gls) t) in
  (mk_clenv_from wc (t,ty), kont)
;;

(* Entries : gls = goals
             arg = program *)
let start_clenv arg gls =
  let sigma = project gls in
  let sign = pf_hyps gls in
  let ccival = (try_execute sigma sign (keep_only_annot arg))._VAL in
  try
    let (wc,kont) = startWalk gls in
    (mk_clenv_hnf_constr_type_of wc ccival, kont)
  with UserError _ -> mk_clenv_hnf_constr_type_of_prog gls ccival
     
;;

let start_connect clenv gls =
  let (wc,kont) = startWalk gls in
  (connect_clenv wc clenv, kont)
;;

(* Add a new metavariable in clenv. *)
let new_meta_pg clenv (na,ty) =
  let mv = newMETA() in
  let clenv' = clenv_pose (na,mv,ty) clenv in
  (clenv',mv)
;;


(* Weak-head beta-delta-iota reduction from a clenv *)
let clenv_whd_betadeltaiota_stack clenv c stack =
   whd_betadeltaiota_stack (w_Underlying clenv.hook) c stack ;;

let clenv_whd_betadeltaiota clenv c =
   whd_betadeltaiota (w_Underlying clenv.hook) c ;;

(* signature of a clenv *)
let clenv_sign clenv = w_hyps clenv.hook ;;
(* ex. var. of a clenv *)
let clenv_evc clenv = (w_Underlying clenv.hook) ;;


let pf_informative_type gls t = 
    is_info_cast_type (project gls) (mkCast t (pf_type_of gls t));;

let clenv_is_info_type clenv c t = 
    is_info_cast_type (w_Underlying clenv.hook) (mkCast c t);;


let metamap_of_clenv ce =
  let sign = clenv_sign ce in
  let evdecls = clenv_evc ce in
  let envpair = gENV evdecls sign in
  let metamap =
    List.map
      (function
	  (n,CLVAL(_,typ)) -> (n,(typ.rebus,Logic))
        | (n,CLTYP typ) -> (n,(typ.rebus,Logic)))
      (Intavm.toList ce.env) in
  let metamap2 =
    List.map 
      (function
	  (n,CLVAL(_,typ)) -> 
            if occur_meta typ.rebus then (n,(typ.rebus,Logic))
            else
	      (n,(typ.rebus,snd(infexemeta evdecls metamap envpair typ.rebus)))
        | (n,CLTYP typ) -> 
	    if occur_meta typ.rebus 
	    then (n,(typ.rebus,Logic))
	    else (n,(typ.rebus,
		     snd(infexemeta evdecls metamap envpair typ.rebus))))
      (Intavm.toList ce.env) in
  metamap2
;;


(* Same as execute_nocheck but in the context of a clenv *)
let clenv_execute_nc ce t =
  let sigma = clenv_evc ce in
  let sign = clenv_sign ce in
  (* simpler than metamap_of_clenv ? *)
  let metamap =
    List.map (function (n,CLVAL(_,typ)) -> (n,typ.rebus)
           | (n,CLTYP typ) -> (n,typ.rebus)) (Intavm.toList ce.env) in
  (execute_nocheck sigma metamap sign t)._VAL
;;


(* Same as pf_extraction but in the context of a clenv.
   Entries : ce : clenv
             t : term to be extracted
   Output : extracted term *)
let rec clenv_extraction ce t =
  let sign = clenv_sign ce in
  let sigma = clenv_evc ce in
  let metamap = metamap_of_clenv ce in
  match snd(infexemeta sigma metamap (gENV sigma sign) t) with
    Logic -> error "clenv_extraction : not informative"
  | Inf j -> j._VAL
;;


(* Extracts the kind of a term 
   Entries : sigma = ex. var.
             sign = signature
             t = term 
   Output : kind *)
let kind_of sigma sign t = (execute_constr sigma sign t)._KIND ;;

(* The same with different entries
   Entries : gls = goal
             t = term *)
let pf_kind_of gls t = kind_of (project gls) (pf_hyps gls) t ;;

(* Extraction of a term
   Entries : gls = goal
             t = term *)
let pf_extraction gls t =
  match snd (pf_infexecute gls t) with
  Logic -> error "pf_extraction : not informative"
| Inf j -> j._VAL
;;

(* Conversion in a goal *)
let pf_conv1_x gls m n = conv1_x (project gls) m n;;

(* Checks if a term is logical or not
   Entries : l = list of bounded var.
             P = term to be checked
             gls = goal *)
let is_logical l p gls = 
 let rec isrec lv p =
let sigma = project gls and sign = pf_hyps gls in
 match p with
  DOP2(Lambda,a,DLAM(_,b)) -> isrec ((isrec lv a)::lv) b
| DOP2(Cast,_,DOP0(Sort(Prop c))) -> c=Null
| DOP2(Cast,c,_) -> isrec lv c
| DOP2(Prod,a,DLAM(_,b)) -> isrec ((isrec lv a)::lv) b
| DOPN(AppL,args) -> isrec lv (args.(0))
| (DOPN((Const _|MutInd _),_) | VAR _) ->
    let j =
      try execute_constr sigma sign p
      with UserError _ -> execute_meta_prog sigma sign !mETAMAP p in
    (not (is_info_judge sigma j))
| Rel(n) -> List.nth lv (n-1)
| DOP0(Sort(Prop c)) -> (c=Null)
| DOP0(Sort(Type(_))) -> false
| DOPN(MutCase _,_) -> let (_,_,c,_) = destCase p in isrec lv c
| DOPN(Fix _,args) -> isrec lv (args.(0))
| DOPN(CoFix _,args) -> isrec lv (args.(0))
| DOP2(XTRA("ANNOT",[]),c,_) -> isrec lv c
| DOP1(XTRA("COMMENT",[]),c) -> true
| _ -> error"is_logical"
 in isrec l p  ;;

(* Split a term with a head and its arguments
   Output : (hd,args) = head, list of arguments *)
let list_arg gls = function
    DOPN(AppL,args) -> 
      let rec lrec l = 
        (match l with
          x::rest -> 
            let sigma = project gls and sign = pf_hyps gls in
            let j = execute_meta_prog sigma sign !mETAMAP x in
            if not(is_info_judge sigma j) then lrec rest else x::(lrec rest)
        | [] -> []) in
      lrec (Array.to_list args), List.tl(Array.to_list args)               
  | x -> [x],[] ;;

(* Gives the number of informative arguments corresponding to a type *)
let nb_info_arg gls = 
  let rec nrec ni l = function
    DOP2(Prod,a,DLAM(na,b)) ->
      if (is_logical l a gls) then nrec ni (true::l) b
      else nrec (ni+1) (false::l) b
  | DOP2(Cast,c,_) -> nrec ni l c                                
  | _ -> ni
  in nrec 0 [] ;;

(* Reduces completely a type
   Output : (b,t) = boolean, term
   the boolean is true if the head of the type is bounded
   the term is the initial type completely reduced *)
let rec red_all_lie gls = function
  DOP2(Prod,c1,DLAM(na,c2)) -> let (b,c2') = red_all_lie gls c2 in 
                                  b, DOP2(Prod,c1,DLAM(na,c2'))
| DOP2(Cast,c,_) -> red_all_lie gls c
| t -> let (x,l) = pf_whd_betadeltaiota_stack gls t [] in  
        match strip_outer_cast x with
          DOP2(Prod,_,_) -> red_all_lie gls x
        | Rel(_) -> true, applist(x,l)
        | _ -> false, applist(x,l) ;;

(* Extracts only the reduced type from the previous function *)
let red_all gls = (comp snd (red_all_lie gls)) ;;

let red_intros =
  tclTHEN (fun gls -> change_in_concl (red_all gls (pf_concl gls)) gls)
    intros
;;

(* [cut_THEN t tac1 tac2] introduces an hypothesis of type t, applies tac1
 * to it, and applies tac2 to solve the subgoal t (needn't to finish the
 * proof).
 *)
let cut_THEN t tac1 tac2 =
  tclTHENS (cut t)
    [ tclTHEN intro
      	(onLastHyp
	   (fun cl ->
	     let id = outSOME cl in
	     tclTHEN (tac1 (mkVar id)) (clear_one id)));
      tac2 ]
;;


(* Suppresses as many products in a constructor type as parameters
   Entries : nparams = number of parameters
             t = type where the products have to be suppressed
   Output : new type *)            
let without_nparams nparams t = 
 let rec crec n = function
  DOP2(Cast,c,_) -> crec n c
| (DOP2(Prod,a,DLAM(_,b)) as c) -> if n=0 then c else crec (n-1) b
| c -> c
 in crec nparams t  ;;

(* Checks if a mutual inductive type ity is a singleton 
   type with respect to a given type typ
   Entries : sigma = ex. var.
             env = signature
             typ = underlying type of the singleton type
             ityp = inductive type with its arguments
             gls = goal
   Output : boolean *)
let is_singleton sigma sign typ (ity,largs) =
  try
    match mis_singl (mind_specif_of_mind ity) with 
      None -> false 
    | Some _ ->
        let (_,inf) = infexecute sigma sign (applist(ity,largs))
        in 
          (match inf with 
              Inf {_VAL=inf_Cty} -> conv1_x sigma inf_Cty typ
            | _ -> anomaly "Program.is_singleton: not informative")
  with Invalid_argument _ | UserError _ -> false;;

let is_singleton_meta clenv typ ityp =
  let sign = clenv_sign clenv in
  let sigma = clenv_evc clenv in
  let (ity,largs) = decomp_app (whd_betadeltaiota sigma ityp) in
  try
    match mis_singl (mind_specif_of_mind ity) with 
      None -> false 
    | Some _ ->
        let inf_Cty = clenv_extraction clenv ityp in
        conv1_x sigma inf_Cty typ
  with UserError _ | Invalid_argument _ -> false;;

(* Checks if a  mutual inductive type is a tuple *)
let is_tuple ity =
  let mis = mind_specif_of_mind ity in
    mis.mib.mINDNTYPES =1 & Array.length (mis_consnames mis) = 1
;;

let clenv_dunbind dflt t b clenv =
  let (sign',b') = dunbind dflt (clenv_sign clenv) t b in
  let clenv' = clenv_add_sign (hd_sign sign') clenv in
  (b',clenv')
;;

(* Lists of function to do a resolution as before but for the second order.
   The final used function is assoc_THEN2 with the same principle as 
   assoc_THEN *)
let collect_metas_with_safe_item mvtac clenv =
  let mvs = collect_metas (clenv_instance_template clenv) in
  (fun i -> mvtac (try List.nth mvs i with Failure _ -> invalid_arg "item"))
;;

(* Applies the refine tactic and connects the corresponding clenv 
   by applying the continuation KONT. Applies the tactic mvtac to 
   the list of generated subgoals.
   Entries : KONT : continuation
             clenv : clenv to be connected
             mvtac : tactic to applies after the resolution
             gls : current goal *)
let res_THEN kONT clenv mvtac gls =
  let clenv' =
    clenv_unify (clenv_instance_template_type clenv) (pf_concl gls) clenv in
  let mvtac' = collect_metas_with_safe_item mvtac clenv' in
  (tclTHEN_i (clenv_refine kONT clenv') mvtac' 0) gls
;;

(* Applies the precedent res_THEN function with a tactic thingtac 
   taking arguments in the assocmap. In practice, thingtac is Realizer 
   and assocmap the association list between metavariables and programs.
   Entries : KONT : continuation
             clenv : clenv to be connected
             assocmap : association list between meta variables and terms
             thingtac : tactic to be applied *)
let assoc_THEN kONT clenv assocmap thingtac =
  res_THEN kONT clenv
    (fun mv ->
      let arg = try Some(List.assoc mv assocmap) with Not_found -> None
      in thingtac arg)
;;

let flexHeadSubstitution gl p oplist clause =
  let (clause',cllist) =
    constrain_clenv_using_subterm_list false clause oplist (pf_concl gl) in
  let typp = clenv_instance_type clause' p in
  clenv_unify (mkMeta p)
    (abstract_list_all gl typp (pf_concl gl) cllist) clause'
;;

let abstract_then_resolve_THEN kONT clause mvtac gl =
  match decomp_app (clenv_instance_template_type clause) with
   (DOP0(Meta(p)),oplist) ->
    let clause' = flexHeadSubstitution gl p oplist clause
    in res_pf_THEN_i kONT clause' mvtac 0 gl
  | _ -> error "A resolution failed"

let refinew_scheme_THEN kONT clause mvtac =
  let mvtac' = collect_metas_with_safe_item mvtac in
  tclORELSE
    (res_pf_THEN_i kONT clause mvtac' 0)
    (abstract_then_resolve_THEN kONT clause mvtac')
;;

let assoc_THEN2 kONT clenv assocmap thingtac gls =
  (refinew_scheme_THEN kONT clenv
  (fun mv -> let arg = try Some(List.assoc mv assocmap)
                       with Not_found -> None
             in thingtac arg))
  gls
;;

(* Counts the number of informative products in a term depending on a 
   list of arguments. 
   Entries : args : list of arguments
             gls : current goal
             term
   Output : number of counted products *)             
let nb_intros_liees_list args gls t =
  let rec nrec ni = function
      DOP2(Prod,a,DLAM(_,b)) ->
      	if (pf_informative_type gls a)
            or (List.exists (fun t -> dependent (strip_outer_cast t) a) args)
      	then ni
      	else nrec (ni+1) b
    | DOP2(Cast,c,_) -> nrec ni c
    | _ -> ni in
  if List.exists (fun t -> not(closed0 t)) args then 0 else nrec 0 t
;;

(* The same for dependence w.r.t. only one term *)
let nb_intros_liees c = nb_intros_liees_list [c] ;;

(* Repeat intro on every logical product of the conclusion.
   Entries : gls : current goal
   Output : subgoals *)  
let rec do_nb_intros gls  =
  match strip_outer_cast (pf_concl gls) with
    DOP2(Prod,a,_) when not (pf_informative_type gls a) ->
      (tclTHEN Tactics.intro do_nb_intros) gls
  | _ -> tclIDTAC gls
;;                     



(* =============================== *)
(* Associating a program to a goal *)
(* =============================== *)

let realizer pg gls =
  let metamap = !mETAMAP in
  let ty = type_meta_of_prog (project gls) (pf_hyps gls) metamap pg in
  let infconcl =
    (match snd(pf_infexecute gls (pf_concl gls)) with
         Logic -> error "Cannot attach a realizer to a logical goal"
       | Inf conclj -> conclj._VAL) in
  if pf_conv1_x gls infconcl ty then
    let octxt = pf_ctxt gls in
    let nctxt = set_pgm (Some pg) octxt in
    context nctxt gls
  else error
      "Type of program and informative extraction of goal do not coincide"
;;

let realizer_top pg gls =
  let pg' = if is_const pg then try_reduce gls pg else pg in
  let ty = type_of_prog (project gls) (pf_hyps gls) pg' in
  let infconcl =
    (match snd(pf_infexecute gls (pf_concl gls)) with
         Logic -> error "Cannot attach a realizer to a logical goal"
       | Inf conclj -> conclj._VAL)
  in
    if pf_conv1_x gls infconcl ty then
      let octxt = pf_ctxt gls in
      let nctxt = set_pgm (Some pg') octxt in
        clear_METAMAP (); context nctxt gls
    else error
      "Type of program and informative extraction of goal do not coincide"
;;

let dyn_realizer = 
  fun [(COMMAND com)] gl ->
    realizer_top (raw_prog_of_progcom (project gl) (pf_hyps gl) com) gl
;;

let realizer_tac = hide_tactic "Realizer" dyn_realizer;;

(* ============================= *)
(* Finding a program from a goal *)
(* ============================= *)

let extract_pg gls = 
  let ctxt = pf_ctxt gls in
  match ctxt.pgm with
    Some pg -> pg
  | None -> error"no program associated to this subgoal" ;;

(* ================ *)
(* Abstraction case *)
(* ================ *)

(* makes a lambda-abstraction [xbar:Tbar]body from
  the arguments xbar, Tbar, body, assuming that the
  variables are bound by name (VAR) NOT by Rel.
 *)
let lambda_list ids types body =
  List.fold_right2
    (fun id j body -> mkNamedLambda id j._TYPE body) ids types body
;;

let fw_args gls names args =
  List.fold_right2
    (fun x arg (xl,fargs) ->
       match pf_infexecute gls arg with
           (_,Inf infj) -> ((x::xl), (infj :: fargs))
         | _ -> (xl,fargs)) names args ([],[])
;;

(* Precondition:
   Goal: Gamma [x:A] |- Phi
   Program: (y:progA)P[y]

   Want the assumption x to be pure, otherwise want to
   eliminate it and find the new program to associate to the subgoal.
 *)

let finishIntroOnProgram id pg gl =
  let a = pf_type_of gl (VAR id) in
  let hd,largs = decomp_app a in
  match pg with
    DOP2(Lambda,progA,b) ->
      (match strip_outer_cast hd with
           (DOPN(MutInd _,_) as hdm) ->
        let sign = pf_hyps gl and sigma = project gl in
        let a' = try (execute_pure sigma sign progA)._VAL with 
                    UserError _ -> propify (gLOB sign) progA in
        if (not(pf_conv1_x gl a' a))
          & is_singleton sigma sign progA (hdm,largs)
        then (tclTHEN (simplest_case (VAR id)) (realizer pg)) gl
        else if is_tuple hd then
        (* Why this ? Program works fine without breaking all the records *)
          try (simple_elimination_then
                 (introCaseAssumsThen
                    (fun ba gls ->
              let cargs = List.map mkVar ba.cargs in
              let (sp,i,cl) = destMutInd ba.ba.ity in
	      let fwsp = fwsp_of sp in
              let chd = mkMutConstruct fwsp i ba.ba.branchnum cl in
              let (names,argsj) = fw_args gls ba.cargs cargs in
              let y' = applist(chd, List.map j_val_only argsj) in
              let subst_pg = nf_betaiota (sAPP b y') in
              let lam_pg = lambda_list names argsj subst_pg in
              let clsl = List.map inSOME ba.assums in
              (tclTHENLIST [ bring_hyps clsl;
			     clear_clauses clsl;
			     realizer lam_pg ]) gls))

                 (VAR id) gl)
          with UserError _ -> realizer (sAPP b (VAR id)) gl
        else realizer (sAPP b (VAR id)) gl
         | _ -> realizer (sAPP b (VAR id)) gl)
    | _ -> error "FinishIntroOnProgram"
;;

(* Precondition : the program is a lambda-abstraction.
   If it is a logical one, introduces as many hypotheses as needed 
   until arriving to the first compatible one.
   If it is an informative, introduces as many hypotheses as needed
   until arriving to the first informative compatible one *)
let rec deroule_intro pg gls = 
  let concl = pf_concl gls in
  match (pg,strip_outer_cast concl) with
    (DOP2(Lambda,DOP1(XTRA("COMMENT",[]),a),(DLAM(Name(ide),_) as b)),
     DOP2(Prod,a',_)) ->
      if (pf_informative_type gls a') then
        error "invalid form of goal"
      else
        if pf_conv_x gls a a' then
          (tclTHEN (intro_using ide)
             (onLastHyp (fun (Some id) -> realizer (sAPP b (VAR id))))) gls
        else
          (try (tclTHEN Tactics.intro (deroule_intro pg)) gls
           with UserError _ ->
             (tclTHEN (tclTHENL (cut a) Auto.full_trivial) (realizer b)) gls)
  | (DOP2(Lambda,a,(DLAM(Name(ide),_) as b)),DOP2(Prod,a',_)) ->
      if pf_informative_type gls a' then
        (tclTHEN (intro_using ide)
            (onLastHyp (fun (Some id) -> finishIntroOnProgram id pg))) gls
      else
        (tclTHEN Tactics.intro (deroule_intro pg)) gls
  | _ -> (tclTHEN red_in_concl (deroule_intro pg)) gls ;;
   
(* ========================================= *)
(* Retrieving the specification of a program *)
(* ========================================= *)

(* Finds the inductive type at the end of a conclusion 
   Entries : clenv : clenv in which type checking can be executed
             term of the conclusion
   Ouput : Inductive type if success *)
(* Incorrect vis-a-vis des de Bruijn... *)
let rec clenv_find_ind clenv = function
    (DOP2(Prod,c1,DLAM(_,c2))) ->  clenv_find_ind clenv c2
  | (DOP2(Cast,c,_))           ->  clenv_find_ind clenv c
  | (DOPN(Const _,_) as val_0)   ->  
      if evaluable_const (clenv_evc clenv) val_0 then
	clenv_find_ind clenv (const_value (clenv_evc clenv) val_0)
      else error"find_ind"
  | (DOPN(AppL,cl) as val_0)     ->  
      (match clenv_whd_betadeltaiota_stack clenv val_0 [] with
        (DOPN(MutInd _,_) as ind),args ->
	  (ind,fst(chop_list(mind_nparams ind) args))    
      | (DOP2(Prod,c1,DLAM(_,c2))),_ -> clenv_find_ind clenv c2              
      | _ -> error"find_ind")
  | (DOPN(MutInd _,_) as ind) -> find_minductype (clenv_evc clenv) ind
  | _ -> error"find_ind" ;;

(* Tries to match the head product of hdty with argty.
   Creates a proof term in three parts arg, list and clenv' 
   by adding metas until the good logical argument arg is found.
   Entries : clenv : clenv in which proof type checking can be executed 
             head : term in the CoC
             hdty : type of this term
             arg : corresponding program term
             argty : type of this program term
   Outputs : arg : proof term correspongind to arg
            list : list of metavariables introduced as logical arguments,
                   the head is head (with eventually new metavariables)
            clenv' : new clenv containing introduced metavariables *) 
let clausal_form_logic clenv head hdty arg argty = 
  let rec clrec argstack clenv = function
      DOP2(Cast,c,_) -> clrec argstack clenv c
    | DOP2(Prod,c1,DLAM(na,c2)) -> 
    	(try
	  let clenv' = clenv_unify c1 argty clenv in
          arg,List.map (clenv_instance_term clenv') (List.rev argstack),clenv'
     	with UserError _ -> 
          if not(clenv_is_info_type clenv c1 (clenv_type_of clenv c1)) then 
            let (clenv',mv) = new_meta_pg clenv (na,c1) in
            let (head,args,clenv'') =
	      clrec ((mkMeta mv)::argstack) clenv' (subst1 (mkMeta mv) c2) in
            (head, (mkMeta mv)::args,clenv'')
          else error"bad logical argument")
    | _ -> error"bad logical argument"
  in clrec [head] clenv hdty
;;
              
(* Tests if a type is the one of a program or not *)
let rec is_type_of_program = function
  Inf{_VAL=DOP0(Sort(Prop _))} -> true
| _ -> false ;;

(* Creates a list of metas corresponding to the logical products in a type.
   Returns the head of the type, the list of metavariables and the new clenv.
   Entries : clenv : clenv in which proof type checking can be executed 
             typ : type in the CoC
   Outputs : concl : conclusion of the type including new metavariables
            list : list of metavariables introduced as logical arguments
            clenv' : new clenv containing introduced metavariables *) 
(* Rq: also removes products corresponding to the first sets only! *)
let rec clausal_form_info clenv head =
  match head with
    DOP2(Cast,c,_) -> clausal_form_info clenv c
  | DOP2(Prod,c1,DLAM(na,c2)) -> 
      let t = clenv_type_of clenv c1 in
      let (_,inf) = infexecute (clenv_evc clenv) (clenv_sign clenv) t in
      if is_type_of_program inf then (head,[],clenv)
      else
      	let (clenv',mv) = new_meta_pg clenv (na,c1) in
	let nhead = subst1 (mkMeta mv) c2 in
	if not (clenv_is_info_type clenv c1 t) then
          let (head',args,clenv') = clausal_form_info clenv' nhead in
          (head',(mkMeta mv)::args,clenv')
	else (nhead,[mkMeta mv],clenv')
  | _ -> (head,[],clenv)
;;

(* Tries to unify arg1 and arg2 in clenv.
   Reduces the terms if necessary
   Entries : arg1, arg2, clenv
   Ouput : new clenv with metavariables instanciated *)
let rec unify_red arg1 arg2 clenv = 
  let sigma = clenv_evc clenv in
  if conv_x sigma arg1 arg2 then clenv
  else
    try clenv_typed_unify arg1 arg2 clenv
    with UserError _ -> 
      (try
 	let arg1' = whd_beta (one_step_reduce sigma arg1) in
      	unify_red arg2 arg1' clenv
      with UserError _ -> 
    	let arg2' = whd_beta (one_step_reduce sigma arg2) in
      	unify_red arg1 arg2' clenv )
;;

(* Creates a list of metas corresponding to the logical products in the type
   argty. Tries then to match the current conclusion with the head of argty
   containing metavariables.
   Then instanciates the current proof term head and the current argument.
   Returns the new head, the new list of arguments and a new clenv.
   Entries : clenv : clenv in which proof type checking can be executed 
             head : list containing the current head and eventually its 
                    logical arguments
             concl : conclusion
             arg : proof term in CoC
             argty : type of this proof term
   Outputs : head' : new head of the constructed proof term
            arg' : list of proof terms arguments to be applied to head'
            clenv' : new clenv containing introduced metavariables *) 
let try_clausal_form_match clenv head concl arg argty =
 
  let rec clrec clenv arg argty =
    try let clenv' = unify_red concl argty clenv in
      (List.map (clenv_instance_term clenv') head,
       clenv_instance_term clenv' arg,
       clenv')
    with UserError _ -> 
    (match strip_outer_cast argty with
      DOP2(Prod,c1,DLAM(na,c2)) -> 
        if not(clenv_is_info_type clenv c1 (clenv_type_of clenv c1)) then
          let (clenv',mv) = new_meta_pg clenv (na,c1) in
          clrec clenv' (DOPN(AppL,[|arg; mkMeta mv|])) (subst1 (mkMeta mv) c2)
        else error"try_clausal_form_match"
    | _ -> error"try_clausal_form_match")
  in clrec clenv arg argty  ;;


(* Tests if a type is an arity or not *)
let rec is_arity clenv p = match clenv_whd_betadeltaiota clenv p with
  DOP0(Sort(Prop _)) -> true
| DOP2(Prod,_,DLAM(_,b)) -> is_arity clenv b
| DOP2(Cast,c,_) -> is_arity clenv c
| _ -> false ;;

(* Forgets arities in head in a type and replaces them by metavariables.
   Entries : clenv : clenv in which type checking is possible 
             headfw : head of program term corresponding to the current type
             argstack : list of arguments for this program term
             typ : current type
   Outputs : concl : new type with eventually metavariables to replace arities
            argstack' : new list with added metavariables at the end
            clenv' : new clenv *)
let rec without_arity_in_head clenv headfw argstack = function
    (DOP2(Prod,c1,DLAM(na,c2)) as prod) -> 
      if is_arity clenv c1 then
	begin match strip_outer_cast headfw with
	     DOPN(AppL,[| _; arg |]) ->
	       let (clenv',mv) = new_meta_pg clenv (Anonymous,c1) in
	       (subst1 (mkMeta mv) c2, argstack@[mkMeta mv], clenv')
	   | _ -> (prod,argstack,clenv) 
	end
      else (prod,argstack,clenv)
  | DOP2(Cast,c,_) -> without_arity_in_head clenv headfw argstack c
  | cl -> (cl,argstack,clenv) ;;

(* Find the index of the first false place in a list of boolean.
   Entry : list of booleans
   Output : number *)
let find_info_prod = 
 let rec frec n = function
  true::l -> frec (n+1) l
| false::_ -> n
| [] -> error"find_info_prod"
 in frec 1 ;;

(* Checks if a type is a dependent type or not *)
let rec is_dependent_type = function
  DOP2(Cast,c,_) -> is_dependent_type c 
| DOP2(Prod,_,DLAM(_,b)) -> if noccurn 1 b then 
                              is_dependent_type b
                            else (true,b)
| x -> false,x ;;

(* For a singleton type, construction of the corresponding lambda-expression
   to appear as an argument of a match in an elimination
   on this singleton type.
   Entries : lc : type of the constructor of the singleton type
             ind : singleton type
             nparams : number of parameters
             head : term to be placed at the end of the lambda-expression.
                    It is a list; if the list is empty, this is to say
                    that there is no term to place at the end.
   Output : lambda-expression *)
let construct_arg_sig lc ind nparams head gls = 

 let rec crec n l a = 
match a with
  DOP2(Cast,c,_) -> crec n l c
| DOP2(Prod,a_0,DLAM(na,b))  -> 
    if noccurn n a_0 then
      Environ.lambda_name (na, a_0, crec (n+1) ((is_logical l a_0 gls)::l) b)
    else 
      Environ.lambda_name (na, subst1 ind a_0,
      Environ.lambda_create (subst1 ind a,
                             crec (n+1)
                               ((is_logical l (subst1 ind a) gls)::
                                ((is_logical l (subst1 ind a_0) gls)::l))
                               b))
| _ -> if head = [] then
        Rel(find_info_prod l)
       else applist(List.hd head,[Rel(find_info_prod l)])
 in crec (1+nparams) [] lc  ;;

(* From a list of integer i, creates a list of Rel(n-i).
   Entries : n : integer
             l : list of integers
   Ouput : list of Rel(n-i) *)
let rec make_rel_list n = function
  a::l -> Rel(n-a)::(make_rel_list n l)
| [] -> [] ;;

(* Same as construct_arg_sig but for a tuple and not a singleton.
   Entries : lc : type of the constructor of the singleton type
             ind : singleton type
             nparams : number of parameters
             head : term to be placed at the end of the lambda-expression.
                    It is a list; if the list is empty, this is to say
                    that there is no term to place at the end.
             concl : current conclusion
   Output : lambda-expression *)

let construct_arg_ind lc ind nparams head concl gls = 
  let (headconcl,_) = decomp_app concl in
    try 
      let (ind',largs) = find_minductype (project gls) headconcl in
      let (sp,i,cl) = destMutInd ind' in
      let mis = mind_specif_of_mind ind' in
      let (_,lc_of_cl) = decomp_all_DLAMV_name (mis_lc mis) in
	
      let rec drec n l lc lc_of_cl =
	let nparams = mis_nparams mis in 
	let (globargs,_) = chop_list nparams largs in
	  begin match (lc,lc_of_cl) with
	      (DOP2(Cast,c,_),_) -> drec n l c lc_of_cl
	    | (_,DOP2(Cast,c,_)) -> drec n l lc c
	    | (DOP2(Prod,b,DLAM(na,c)),(DOP2(Prod,b',DLAM(_,c')))) -> 
		let (l',lc') =
		  if pf_conv_x gls b b' then (n::l, c') else (l,lc_of_cl) in
		if noccurn n b then
		  DOP2(Lambda,b,DLAM(na,drec (n+1) l' c lc'))
		else 
		  Environ.lambda_name
		    (na, subst1 ind b,
		     Environ.lambda_create
		       (subst1 ind lc, drec (n+1) l' c lc'))
	    | _ ->
		let constr = DOPN(MutConstruct((sp,i),1),cl) in
		if head = [] then
                  applist(constr,globargs@(List.rev (make_rel_list n l)))
              	else 
		  applist(constr,
			  globargs@[applist(List.hd head,
					    (List.rev (make_rel_list n l)))])
	  end
      in drec (1+nparams) [] lc (hd_vect lc_of_cl) 

    with Induc -> error "construct_arg_ind" ;;

(* Combination of the previous functions to eliminate an argument
   those type is a singleton or a tuple when needed.
   Entries : concl : current conclusion
             ccihead : proof term to be eliminated
             cciheadty : its type
             cciarg : program term to be eliminated
             cciargty : its type
             gls : current goal
   Output : new proof term transformed in an elimination *)
let elim_sig_arg concl ccihead cciheadty cciarg cciargty typarg gls = 
  let sigma = project gls and sign = pf_hyps gls in
  let (head,args) = decomp_app (try_reduce gls cciargty) in
  try 
    let (t,largs,lc,nparams) =
           let (t,largs) = find_minductype sigma cciargty in
           let mis = mind_specif_of_mind t in
           let (_,lc) = decomp_all_DLAMV_name(mis_lc mis)
           in (t,largs,lc,mis_nparams mis) in
    if is_singleton sigma sign typarg (t,largs) then
      let lf =
 	whd_beta
	  (applist(construct_arg_sig (hd_vect lc) t nparams [ccihead] gls,
		   largs)) in
      let (b,p) = is_dependent_type cciheadty in
      let typ =
      	if b then
	  let lf' =
	    whd_beta (applist(construct_arg_sig (hd_vect lc) t nparams [] gls,
			      largs)) in
          let proj = mkMutCaseA (ci_of_mind t) concl (Rel 1) [|lf'|]
          in Environ.lambda_create (cciargty, subst1 proj p)
      	else match cciheadty with
                DOP2(Prod,_,DLAM(_,b_0)) -> b_0
              | _ -> error"elim_sig_arg"
      in mkMutCaseA (ci_of_mind t) typ cciarg [|lf|]
  
    else if is_tuple head then
      let hdapp =
 	construct_arg_ind (hd_vect lc) t nparams [ccihead] concl gls in
      let lf = whd_beta (applist(hdapp, largs)) in
      let (b,p) = is_dependent_type cciheadty in
      let typ =
      	if b then 
	  let hdapp = construct_arg_ind (hd_vect lc) t nparams [] concl gls in
      	  let lf' = whd_beta (applist(hdapp, largs)) in
          let proj = mkMutCaseA (ci_of_mind t) concl (Rel 1) [|lf'|] in
          Environ.lambda_create (cciargty, subst1 proj p)
      	else (match cciheadty with
          DOP2(Prod,_,DLAM(_,b_0)) -> b_0
        | _ -> error"elim_sig_arg") in
      mkMutCaseA (ci_of_mind t) typ cciarg [|lf|]
    else error"elim_sig_arg" 
  with Induc | Not_found | Invalid_argument _ -> error"elim_sig_arg";;

(* Same as construct_arg_sig.
   But this function eliminates the head of an application and no more 
   an argument.
   Entries : lc : type of the constructor of the singleton type
             ind : singleton type
             nparams : number of parameters
             arg : term to be placed at the end of the lambda-expression,
                    It is a list; if the list is empty, this is to say
                    that there is no term to place at the end.
   Output : lambda-expression *)
let construct_head_sig lc ind nparams arg gls = 

 let rec drec n l a =
match a with
  DOP2(Cast,c,_) -> drec n l c
| DOP2(Prod,a_0,DLAM(na,b)) -> 
      if noccurn n a_0 then
        Environ.lambda_name (na, a_0, drec (n+1) ((is_logical l a_0 gls)::l) b)
      else 
        Environ.lambda_name (na, subst1 ind a_0, 
        Environ.lambda_create
          (subst1 ind a,
           drec (n+1) ((is_logical l (subst1 ind a) gls)
                       ::((is_logical l (subst1 ind a_0) gls)::l))
             b))
| _ -> if arg = [] then
        Rel(find_info_prod l)
       else applist(Rel(find_info_prod l),arg)
 in drec (1+nparams) [] lc  ;;

let force_cast sigma sign = function 
    (DOP2(Cast,_,_) as a) -> a
  | a -> let j = execute_constr sigma sign (strip_all_casts a) in
         DOP2(Cast,j._VAL,j._TYPE);; 

let force_cast_type sigma sign c =
  match c with
      DOP2(Cast,c,DOP0(Sort s)) -> {body=c;typ=s}
    | a -> let j = execute_constr sigma sign (strip_all_casts a) in
	match j._TYPE with
	    DOP0(Sort s) -> {body=j._VAL;typ=s}
	  | _ -> anomaly "force_cast_type";;

(* Gives the type of the tail of a lambda-expression.
   Entries : sigma : metavariables environment
             sign : current signature
             term : lambda-expression
   Output : typ : type of the tail *)
let rec type_tail_lf sigma sign = function
  DOP2(Cast,c,_) -> type_tail_lf sigma sign c 
| DOP2(Lambda,a,(DLAM(Name(x),b) as c)) -> let a' = force_cast_type sigma sign a in
     type_tail_lf sigma (add_sign (x,a') sign) (sAPP c (VAR x))
| DOP2(Lambda,a,DLAM(_,b)) -> type_tail_lf sigma sign b
| x -> (execute_constr sigma sign x)._TYPE ;;

(* Combination of the two previous functions to eliminate an head of an
   application if needed.
   Entries : ccival : proof term to be transformed (head of an application)
             ccityp : its type
             arg : argument applied to the head
             gls : current goal
   Output : new proof term with an elimination *)
let elim_sig_head ccival ccityp arg gls = 
  let sigma = project gls and sign = pf_hyps gls in
  let (head,largs) = decomp_app (try_reduce gls ccityp) in
  try 
  let tyarg = pf_extraction gls ccityp in
  let (t,largs) = find_minductype sigma ccityp in
  if is_singleton sigma sign tyarg (t,largs) then
    let (t,largs,lc,nparams) = 
        let mis = mind_specif_of_mind t in
        let (_,lc) = decomp_all_DLAMV_name(mis_lc mis) in 
          (t,largs,lc,mis_nparams mis) in
    let lf = propify (gLOB sign) 
               (whd_beta
                  (applist(construct_head_sig (hd_vect lc) t nparams [arg] gls,
                           largs))) in
    let typP = type_tail_lf sigma sign lf in
    mkMutCaseA (ci_of_mind t) typP ccival [|lf|]
  else error"elim_sig_head"
 with Induc | Not_found | Invalid_argument _ -> error"elim_sig_head" ;;

let elim_sig ccival ccityp gls = 
  let sigma = project gls and sign = pf_hyps gls in
  let (head,largs) = decomp_app (try_reduce gls ccityp) in
  try 
  let tyarg = pf_extraction gls ccityp in
  let (t,largs) = find_minductype sigma ccityp in
    if is_singleton sigma sign tyarg (t,largs) then
        let mis = mind_specif_of_mind t in 
        let (_,lc) = decomp_all_DLAMV_name(mis_lc mis) in 
        let nparams = mis_nparams mis in
	let hdapp = construct_head_sig (hd_vect lc) t nparams [] gls in
    	let lf = propify (gLOB sign) (whd_beta (applist(hdapp, largs))) in
    	let typP = type_tail_lf sigma sign lf in
    	mkMutCaseA (ci_of_mind t) typP ccival [|lf|]
    else error"elim_sig"
  with Induc | Not_found | Invalid_argument _ -> error"elim_sig" ;;

let unfold_head_fconst sigma = 
 let rec unfrec = function
    DOPN(Const _,_) as k -> const_value sigma k 
  | DOP2(Lambda,t,DLAM(na,b)) -> DOP2(Lambda,t,DLAM(na,unfrec b))
  | DOPN(AppL,v) -> DOPN(AppL,cons_vect (unfrec (hd_vect v)) (tl_vect v))
  | x -> x
 in unfrec
    
;;

(* Checks if a term contains a MutCase. *)
let rec is_rec = function
  DOP2(Cast,c,_) -> is_rec c
| DOPN(AppL,args) -> let l = Array.to_list args in
                     if (List.length l = 1) then (is_rec (List.hd l))
                     else
                      (is_rec (List.hd l)) or (is_rec (List.hd(List.tl l)))
| DOP2(Lambda,c,DLAM(_,c')) -> (is_rec c) or (is_rec c')
| DOP2(Prod,c,DLAM(_,c')) -> (is_rec c) or (is_rec c')
| DOPN(MutInd _,c) -> (List.exists is_rec (Array.to_list c))
| DOPN(MutConstruct _,c) -> (List.exists is_rec (Array.to_list c))
| DOPN(MutCase _,_) -> true
| DOPN((Fix _|CoFix _),b) -> is_rec (last_vect b)
| DOP2(XTRA("ANNOT",_),c,t) -> (is_rec c) or (is_rec t)
| _ -> false ;;

(* Checks if a term contains a MutCase. *)
let rec is_rec_or_fix = function
  DOP2(Cast,c,_) -> is_rec_or_fix c
| DOPN(AppL,args) ->
    let l = Array.to_list args in
    if (List.length l = 1) then (is_rec_or_fix (List.hd l))
    else (is_rec_or_fix (List.hd l)) or (is_rec_or_fix (List.hd(List.tl l)))
| DOP2(Lambda,c,DLAM(_,c')) -> (is_rec_or_fix c) or (is_rec_or_fix c')
| DOP2(Prod,c,DLAM(_,c')) -> (is_rec_or_fix c) or (is_rec_or_fix c')
| DOPN(MutInd _,c) -> (List.exists is_rec_or_fix (Array.to_list c))
| DOPN(MutConstruct _,c) -> (List.exists is_rec_or_fix (Array.to_list c))
| DOPN(MutCase _,_) -> true
| DOPN(Fix _,args) -> true
| DOPN(CoFix _,args) -> true
| DOP2(XTRA("ANNOT",_),c,t) -> (is_rec_or_fix c) or (is_rec_or_fix t)
| _ -> false ;;


let is_sing_nocheck ty =
  try mis_singl (mind_specif_of_mind ty) <> None
  with Invalid_argument _ -> false
;;

(* Basic function to find the specification of a program.
   Entries : gls : current goal
             clenv : current clausal environment
             pmap : association list between metavariables and programs
             cl : current conclusion
             pg : current program
   Output : proof term with metavariables correspondint to pg *)
let specif_of_annot gls (clenv,pmap) cl pg = 
  (* k is the number of terms that must be applied to pg in order to have a
   * realizer of type cl (i.e. the number of generalizations to do) *)
  let rec specrec k (clenv,pmap) cl pg  = 
    let val_0 = telescope_appl (strip_outer_cast pg) in
    match val_0 with
      VAR id ->
 	(val_0,(snd(lookup_sign id (clenv_sign clenv))).body,(clenv,pmap))
    | DOPN(Const sp,_) -> 
        let ccival = propify (gLOB(clenv_sign clenv)) val_0 in
        let cont = pf_extraction gls ccival in
        if opaque_const (project gls) ccival then
                (ccival,clenv_type_of clenv ccival,(clenv,pmap))
        else (try let j = execute_pure (clenv_evc clenv) (clenv_sign clenv)
	                    (unfold_head_fconst (project gls) cont) in
                  if (is_rec_or_fix j._VAL) then
                    (ccival,clenv_type_of clenv ccival,(clenv,pmap))
                  else (j._VAL,j._TYPE,(clenv,pmap))
             with UserError _ ->
	       (ccival,clenv_type_of clenv ccival,(clenv,pmap)))
    | DOP0(Meta p) -> (val_0, cl, (clenv,pmap))
    | DOPN(MutConstruct((sp,j),i),largs) -> 
      (match cl with
        DOP0(Sort(Prop Pos)) ->
          let j = execute_pure (clenv_evc clenv) (clenv_sign clenv) val_0 in
          (j._VAL,j._TYPE,(clenv,pmap))
      | _ ->
        if occur_meta cl then
          let ccival = propify (gLOB(clenv_sign clenv)) val_0 in
            (ccival,clenv_type_of clenv ccival,(clenv,pmap))
        else
          let (DOPN(MutInd(sp',num),cl') as ind,globargs) =
	    clenv_find_ind clenv cl in
        if is_sing_nocheck ind then
          let nparams = mind_nparams(DOPN(MutInd(sp,j),largs)) in
          if (nparams = 0) or (k>nparams) then
            let ccival = propify (gLOB(clenv_sign clenv)) val_0 in
            (ccival,clenv_type_of clenv ccival,(clenv,pmap))
          else
          let ccival=applist(DOPN(MutConstruct((sp',num),1),cl'),globargs) in 
          let ccityp = clenv_type_of clenv ccival in 
          let (clenv',pmap',tail_typ,args) = 
              one_arg (clenv,pmap,ccityp) k val_0 in
          let (_,args',clenv'') = clausal_form_info clenv' tail_typ in
          let ccival' = applist(ccival, args@args') in
          (ccival',clenv_type_of clenv'' ccival',(clenv'',pmap'))
        else
          let val_0 = applist(DOPN(MutConstruct((sp',num),i),cl'),globargs) in 
          let t = clenv_type_of clenv val_0 in
          let (_,args,clenv') = clausal_form_info clenv t in
          let ccival = applist(val_0,args) in
          (ccival,clenv_type_of clenv' ccival,(clenv',pmap)))
    | DOP2(XTRA("ANNOT",[]),c,t) -> 
        let (clenv',mv) = new_meta_pg clenv (Anonymous,t) in
        (mkMeta mv,t,(clenv',(mv,c)::pmap))
    | DOP2(Lambda,c1,(DLAM _ as b)) -> 
      let c1assum=
	(try execute_pure_type (clenv_evc clenv) (clenv_sign clenv) c1 
         with UserError _ -> raise NotProgram) in
      let (b',clenv') = clenv_dunbind (id_of_string "H") c1assum b clenv in
      let (ccival,ccityp,(clenv'',pmap')) = specrec k (clenv',pmap) cl b' in
      let ccival' = clenv_instance_term clenv'' ccival and
          ccityp' = clenv_instance_term clenv'' ccityp in
      if occur_meta ccival' then raise NotProgram
      else let (ty,b') = dbind (clenv_sign clenv'') ccival' in
           let (_,tB') = dbind (clenv_sign clenv'') ccityp' in
           (DOP2(Lambda,incast_type ty,b'),
	    DOP2(Prod,incast_type ty,tB'),
	    (clenv'',pmap'))
    | DOPN(AppL, [| t1; t2 |]) ->
	 let (ccivalt1,ccitypt1,(clenv',pmap')) = 
	   specrec (k+1) (clenv,pmap) cl t1 in 
	 let sigma = clenv_evc clenv' in 
	 let j = execute_meta_prog sigma (clenv_sign clenv') !mETAMAP t2 in
	   if not(is_info_judge sigma j) then
             let (arg,t1'::mvlist,clenv'') = 
               clausal_form_logic clenv' ccivalt1 ccitypt1 j._VAL j._TYPE in
             let ccival = applist (t1',mvlist@[without_comment arg]) in
             (ccival,clenv_type_of clenv ccival,(clenv'',pmap'))
	   else
             let (concl,args,clenv'') =
	       clausal_form_info clenv' (nf_beta ccitypt1) in
             if is_arity clenv j._TYPE then
	       let ccival = applist(ccivalt1,args) in
	       begin match args with
		 [DOP0(Meta(mv))] -> 
		   (ccival,clenv_type_of clenv'' ccival,
		    (clenv'',(mv,j._VAL)::pmap')) 
	       | _ -> (ccival,clenv_type_of clenv'' ccival,(clenv'',pmap'))
	       end
	     else
	       (try match_with_cl (clenv'',pmap') ccivalt1 t1 concl args t2 k
	       with Expand -> 
		 (try match_with_cl (clenv'',pmap') ccivalt1 t1
		     (whd_beta (one_step_reduce (clenv_evc clenv'') concl))
		     args t2 k
		 with UserError _ -> raise (Generalize k)))
    | _ -> raise NotProgram

  and one_arg (clenv,pmap,t) k c = 
    let (concl,args,clenv') = clausal_form_info clenv t in
    let sigma = clenv_evc clenv' in
    if is_arity clenv (type_meta_of_prog sigma (clenv_sign clenv) !mETAMAP c)
    then (clenv',pmap,concl,args) 
    else
      (match concl with
        DOP2(Prod,c1,DLAM(na,c2)) ->
          (try 
            let (ccival',ccityp',(clenv'',pmap')) =
	      specrec k (clenv',pmap) c1 c in
	    let (c2'::l',c',clenv''') =
	      try_clausal_form_match clenv'' (c2::args) c1 ccival' ccityp' in
	    (clenv''',pmap',(subst1 c' c2'),l'@[c'])
          with NotProgram | UserError _ ->
            let (clenv'',mv) = new_meta_pg clenv' (na,c1) in
            let pmap' = (mv,c)::pmap in
            (clenv'',pmap',concl,args@[DOP0(Meta(mv))]))
      | _ -> anomaly"specif")

  and match_with_cl (clenv,pmap) head headfw concl argstack arg k =
    let (concl',argstack',clenv1) =
      without_arity_in_head clenv headfw argstack concl in
    match concl' with
      DOP2(Prod,c1,DLAM(na,c2)) ->
  	(try
	  let (ccival,ccityp,(clenv2,pmap')) =
	    specrec 0 (clenv1,pmap) c1 arg in
	  let (_,argstack'',clenv3) = clausal_form_info clenv2 ccityp in
	  let ccival'= applist(ccival,argstack'') in
	  let ccityp' = clenv_type_of clenv3 ccival' in
          let val_0 = applist(head,argstack') in
	  (try
	    let (head',arg',clenv4) =
	      try_clausal_form_match clenv3 [val_0] c1 ccival' ccityp' in
	    let ccival = DOPN(AppL,Array.of_list(head'@[arg'])) in
	    (ccival,clenv_type_of clenv4 ccival,(clenv4,pmap'))
          with UserError _ -> 
	    let ty =
	      type_meta_of_prog (project gls) (clenv_sign clenv) !mETAMAP
		arg in
            let newpg =
	      elim_sig_arg c1 val_0 (clenv_type_of clenv1 val_0)
                ccival' ccityp' ty gls in 
            (newpg,clenv_type_of clenv3 newpg,(clenv3,pmap')))
   	with (NotProgram | UserError _ | Failure _) -> 
	  let (clenv2,mv) = new_meta_pg clenv1 (na,c1) in
	  let pmap' = (mv,arg)::pmap in
	  let ccival = applist(head,argstack'@[DOP0(Meta(mv))]) in
          (ccival,clenv_type_of clenv2 ccival,(clenv2,pmap')))
    | _ -> (try
 	let newpg = elim_sig_head head (clenv_type_of clenv head) arg gls in 
        (newpg,clenv_type_of clenv newpg,(clenv,pmap))
    with UserError _ -> raise Expand)  in

  let (ccival,ccityp,(clenv,pmap)) = specrec 0 (clenv,pmap) cl pg in
  (nf_beta ccival, ccityp,(clenv,pmap)) ;;


let pf_specif_of_annot gls cl pg =
  let (clenv,kONT) = start_clenv pg gls in
  (specif_of_annot gls (clenv,[]) cl pg, kONT)
;;


let specif_of_annot_pure gls (clenv,pmap) cl pg =
  let (ccival,ccityp,(clenv',pmap')) =
    let sign = clenv_sign clenv in
    try
      let j = execute_constr (clenv_evc clenv) sign (propify (gLOB sign) pg) in
      (j._VAL,j._TYPE,(clenv,pmap))
    with UserError _ -> specif_of_annot gls (clenv,pmap) cl pg in
  (nf_beta ccival, ccityp, (clenv',pmap'))
;;

(* ================ *)
(* Application Case *)
(* ================ *)

(* Checks if the head of a type is bound or not and gives this head.
   Entries : gls : current goal
             term : proof type
   Output : (b,t) : (boolean,head of the type) *)             
let rec find_bound gls = function
  DOP2(Prod,_,DLAM(_,b)) -> find_bound gls b
| DOP2(Cast,c,_) -> find_bound gls c
| concl -> let (x,args) = pf_whd_betadeltaiota_stack gls concl [] in
            match strip_outer_cast x with
              Rel(_) -> (true,concl)
            | DOP0(Meta _) -> (true,concl)
            | _ -> (false, concl) ;;

(* Checks if the head of an application is bound or not. *)
let rec is_head_rel = function
  DOPN(AppL,args) -> is_head_rel (hd_vect args)
| DOP2(Cast,c,_) -> is_head_rel c
| Rel(_) -> true
| DOP0(Meta _) -> true
| _ -> false ;;

(* Generalizes a goal gls n times with respect to the n arguments of a program
   if the program is an application with at least n arguments.
   Associated to this new goal the program without the n generalized arguments.
   Applied in the case of a Rec or a MutCase at the head of the program.
   Entries : n : number of generalizations
             pg : program
             gls : current goal
   Output : generalized goal *)
let rec generalize_term n pg gls = 
  if n=0 then tclIDTAC gls else 
  
 let genrec pg gls =
  match telescope_appl(strip_outer_cast pg) with
    DOPN(AppL, [| t1; t2 |]) -> 
      let ((ccival,ccityp,(clenv,pmap)),kONT) =
        try pf_specif_of_annot gls mkSet t2
        with NotProgram -> error"insufficient specification of the program" in
      let newconcl =
        Environ.prod_create (ccityp,subst_term ccival (pf_concl gls))
      in (tclTHEN (kONT clenv.hook)
         (tclTHENS (cut newconcl)
            [tclTHEN Tactics.intro
               (fun gls ->
              let (id,t) = hd_sign (pf_untyped_hyps gls) in
              let (DOP2(Prod,_,b)) = pf_whd_betadeltaiota gls t in
              let (clenv',kONT) = start_connect clenv gls in
              let clenv'' = clenv_change_head (applist(VAR id,[ccival]),
                                                  (sAPP b ccival)) clenv' in


              assoc_THEN kONT clenv'' pmap
                         (fun opg gl -> match opg with
                              Some pg -> realizer pg gl
                            | None -> tclIDTAC gl) gls);
             realizer t1])) gls
  | _ -> anomaly"generalize_term"
 in (tclTHEN (genrec pg)
       (fun gl -> let pg = extract_pg gl in
          generalize_term (n-1) pg gl))
      gls
   ;;        

(* Checks if an application has no more than one argument. *)
let rec with_max_one_arg = function
  DOP2(Cast,t,_) -> with_max_one_arg t
| DOPN(AppL,args) -> ((Array.length args)<= 2)
| _ -> true ;;

(* Creates a list with the names of logical hypotheses in the current context
   that depend on c (c has to be a pure program term).
   Entries : c : program term
             gls : current goal 
   Output : list of hypotheses names *)
let one_generalize c gls = 
  try let sign = pf_hyps gls and sigma = project gls in
      let j = execute_pure sigma sign c in
      let ccival = j._VAL in
      let rec fullrec sign n = 
        if n=0 then [] else
        let (signid,signval) = hd_sign sign in
	let signval = signval.body in
        if (dependent (strip_outer_cast ccival) signval) & (with_max_one_arg (collapse_appl signval)) & (is_logical [] signval gls) then
          (global (gLOB sign) signid)::(fullrec (tl_sign sign) (n-1))
        else fullrec (tl_sign sign) (n-1)
      in fullrec sign (sign_length sign)
  with UserError _ -> [] ;;

(* Treats a list of hypotheses names and suppresses all the ones whose
   type already appears in the current conclusion.
   Entries : concl : current conclusion
             l : list of hypotheses names
             gls : current goal
   Output : list of hypotheses names (passed through a filter) *)
let rec keep_only_not_in_concl concl l gls = match l with
    x::l' -> let t = pf_type_of gls x in
             if dependent t concl then
              keep_only_not_in_concl concl l' gls
             else
              x::(keep_only_not_in_concl concl l' gls)
  | [] -> [] ;;

(* Generalizes a conclusion first with respect to a list of hypotheses
   and then with respect to a value val of type typ.
   Entries : val : proof value
             typ : its type
             l : list of hypotheses names
             concl : current conclusion
             gls : current goal 
   Output : new generalized conclusion *)
let rec create_new_concl val_0 typ l concl gls = 
let rec crec l = (match l with
  x::l' -> let t = pf_type_of gls x in Environ.prod_create (t,crec l')
| [] -> concl)
in Environ.prod_create (typ, subst_term val_0 (crec l)) ;;

(* Gives the type of the nth product of a type t.
   Entries : n : number
             t : type
   Output : type *)
let rec good_type n t = 
  match t with
    DOP2(Prod,_,DLAM(_,a)) -> if n=0 then a else good_type (n-1) a
  | _ -> anomaly "shouldn't appear" ;;

(* Generalizes a goal gls n times with respect to the n arguments of a program
   if the program is an application with at least n arguments.
   Associated to this new goal the program without the n generalized arguments.
   This is only applied in the case of a fixpoint at the head of the program.
   Entries : n : number of generalizations
             pg : program
             gls : current goal
   Output : generalized goal *)
let rec generalize_fix_term n pg gls = 
  if n=0 then tclIDTAC gls else 
  
 let genrec pg gls =
  match telescope_appl(strip_outer_cast pg) with
    DOPN(AppL, [| t1; t2 |] ) ->
      (try
 	let ((ccival,ccityp,(clenv,pmap)),kONT) =
	  pf_specif_of_annot gls mkSet t2 in
 	let l = one_generalize t2 gls in
 	let l' = keep_only_not_in_concl (pf_concl gls) l gls in
 	let newconcl = create_new_concl ccival ccityp l' (pf_concl gls) gls in
 	(tclTHENS (tclTHEN (kONT clenv.hook) (cut newconcl))
           [tclTHEN Tactics.intro
               (fun gls ->
		 let (id,t) = hd_sign (pf_hyps gls) in
		 let t = t.body in
		 let len = (List.length l') in
		 let typ = good_type len (pf_whd_betadeltaiota gls t) in
		 let (clenv',kONT) = start_connect clenv gls in
		 let clenv'' =
		   clenv_change_head
		     (applist(VAR id,[ccival]@l'),
		      (subst1 ccival (lift (-len) typ))) clenv' in
		 assoc_THEN kONT clenv'' pmap
                   (fun opg gl -> match opg with
		     Some pg -> realizer pg gl
                   | None -> tclIDTAC gl) gls);
             realizer t1]) gls
      with NotProgram -> error"insufficient specification of the program")
  | _ -> anomaly "generalize_fix_term"
 in (tclTHEN (genrec pg)
       (fun gl -> let pg = extract_pg gl in generalize_fix_term (n-1) pg gl))
   gls
;;        

(* Does as many preliminary logical introductions as necessary in the case of
   a program corresponding to an application.
   Entries : concltyp : head of the conclusion
             concl : conclusion
             gls : current goal
   Output : new goal with introductions done *)
let refinew_intros concltyp concl gls =
  let ni = if is_head_rel concltyp then
            let (_,args) = pf_whd_betadeltaiota_stack gls concltyp [] in
              Some(nb_intros_liees_list args gls concl)
           else None
  in match ni with
      None -> intros gls
    | Some(ni) -> tclDO ni intros gls ;;

(* Refines the current goal with the proof term ccival of type ccityp.
   clenv is the current clausal environment.
   pmap the association list between metas and programs.
   The head of the clenv is transformed to e the good one.
   Then, a first order refinement is tried and then a second order if it fails.
   Entries : ccival : proof term to refine
             ccityp : its type
             clenv : current clausal environment
             pmap : current association list between metas and programs
             gls : current goal
   Output : new refined goal(s) with program(s) associated to *)
let apply_tac kONT (ccival,ccityp) (clenv,pmap) gls = 
  let clenv' = clenv_change_head (ccival,ccityp) clenv in   
  let clenv'' = clenv_apply_n_times (nb_prod ccityp) clenv' in
  let templtyp = clenv_instance_template_type clenv'' in
  let clenv''' = clenv_apply_n_times (nb_prod templtyp) clenv'' in
    try assoc_THEN kONT clenv''' pmap
 	  (fun opg gl -> match opg with
	    Some pg -> realizer pg gl
          | None -> tclIDTAC gl) gls
    with UserError _ ->
        assoc_THEN2 kONT clenv''' pmap
 	  (fun opg gl -> match opg with
	    Some pg -> realizer pg gl
          | None -> tclIDTAC gl) gls
;;

(* Applies the previous tactics but first decides the number of logical
   introductions to be done.
   The current program is head applied to args. The corresponding proof term
   is calculated.
   Entries : head : head of the current program
             args : args of the current program
             gls : current goal
   Output : new refined goal(s) with program(s) associated to *)
let refine_pg head args gls = 
  let concl = pf_concl gls in
  let (clenv0,kONT) = start_clenv head gls in
  let (ccival,ccityp,(clenv,pmap)) = 
    specif_of_annot gls (clenv0,[]) concl (applist(head,args)) in
  let (b,concltyp) = find_bound gls ccityp in
  let intro_tac =
    if b then refinew_intros concltyp concl else do_nb_intros in
  (tclTHENLIST
     [ kONT clenv.hook;
       intro_tac; 
       (fun gl ->
	 let (clenv',kONT) = start_connect clenv gl in
	 apply_tac kONT (ccival,ccityp) (clenv',pmap) gl)]) gls
;;

(* General tactics to refine a goal with a program which is an application.
   Applies the previous tactics by deciding if generalizations are needed.
   The current program is head applied to args. The corresponding proof term
   is calculated.
   Entries : head : head of the current program
             args : args of the current program
             gls : current goal
   Output : new refined goal(s) with program(s) associated to *)
let program_app head args gls = 
  let concl = pf_concl gls in
  match (strip_outer_cast head) with
    DOPN(MutCase _,_) ->
      generalize_term (List.length args) (applist(head,args)) gls
  | DOPN((Fix _|CoFix _),_) ->
      generalize_fix_term (List.length args) (applist(head,args)) gls
  | _ -> 
      let ((_,ccityp,_),_) = pf_specif_of_annot gls concl head in
      let (b,concltyp) = find_bound gls ccityp in
      try if b then
        tclTHEN (refinew_intros concltyp concl) (refine_pg head args) gls
      else 
        let concl' = (red_all gls concl) in
        tclTHENLIST [ change_in_concl concl';
		      do_nb_intros;
		      refine_pg head args ] gls
      with Generalize k -> generalize_term k (applist(head,args)) gls
 ;;

(* ============== *)
(* Recursion Case *)
(* ============== *)

(* Calculates a list of integers.
   This list represents for each constructor type of the number of 
   corresponding lambda-abstraction. Its depends on the fact that
   the type is recursive or not.
   Entries : recursive : if the type is recursive or not
             l : list of constructors types
   Output : list of number of lamba-expression for each constructor *)
let rec search_nb_lam recursive l = 
  match l with
    sTR::l' ->
      let rec mrec n m sTR = 
        match strip_outer_cast sTR with
          DOP2(Lambda,_,DLAM(_,b)) -> mrec (n+1) m b
        | DOP2(Prod,a,DLAM(_,b))   ->
	    if noccurn (n+1) a then mrec (n+1) m b
            else (match b with
              DOP2(Prod,_,DLAM(_,b')) -> mrec (n+2) (m+1) b'
            | _ -> n-m)
        | _ -> if recursive then n-2 else n-m-1
      in (match sTR with
        DLAMV(_,l) -> (List.map (mrec 1 0) (Array.to_list l))
      | DLAM(_) ->
	  let (x,l) = decomp_all_DLAMV_name sTR in
          List.map (mrec (List.length x) ((List.length x)-1)) (Array.to_list l)
      | _ -> (mrec 0 0 sTR)::search_nb_lam recursive l')
  | [] -> [] ;;

(* Adds a lambda-expression [name:val]... after the nth abstraction
   of the lambda-expression f.
   Entries : n : number of abstractions to pass
             f : lambda-expression
             name : name to put in the new abstraction
             val : type of name
   Output : new lambda-expression *)
let rec change_lam_at n f name val_0 gls =
  let sigma = project gls and sign = pf_hyps gls in
  if n=0 then DOP2(Lambda,val_0,DLAM(Name(name),subst_var name (lift 1 f)))
  else match strip_outer_cast f with
        DOP2(Lambda,a,DLAM(name',b)) -> 
          DOP2(Lambda,a,DLAM(name',change_lam_at (n-1) b name val_0 gls ))
      | x -> (match red_all gls (type_meta_of_prog sigma sign !mETAMAP f)
              with 
                  DOP2(Prod,a,DLAM(na,b)) ->
                    change_lam_at n
                      (Environ.lambda_name (na,a,DOPN(AppL,[|f; Rel 1|])))
                      name val_0 gls
                | _ -> anomaly "shouldn't appear") ;;

(* Applies the precedent function on a list of number and a list of
   lambda-expressions.
   Entries : name : name to be add in each lambda-expression
             ln : list of numbers
             lf : list of lambda-expressions
             val : type of name
   Output : list of new lambda-expressions *)
let rec modif_lf name ln lf val_0 gls =
  match (lf,ln) with
    (f::lf',n::ln') ->
      (change_lam_at n f name val_0 gls)::modif_lf name ln' lf' val_0 gls
  | ([],[]) -> [] ;;

(* Suppress the integer n on each number element of the input list.
   Entries : n : number
             l : list of integers
   Output : list of integers *)
let rec suppress_params n = function
  m::l -> (m-n)::(suppress_params n l)
| [] -> [] ;;

(* Looks in the current signature if they are hypotheses that depend
   on the term c. If yes, adds these hypotheses as lambda-abstractions
   in each element of the list lf at the right place by using the precedent
   functions.
   Entries : recursive : if the type of c is recursive or not
             c : term on which hypotheses have to depend
             lf : list of lambda-expressions
             gls : current goal
   Output : (l',lf') : (list of dependent hypotheses names, modified lf) *)
let full_generalize recursive c lf gls =
  try let sign = pf_hyps gls and sigma = project gls in
      let j = execute_pure sigma sign c in
      let ccival = j._VAL in
      let rec fullrec sign n l = 
        if n=0 then [],l else
        let (signid,signval) = hd_sign sign in
	let signval = signval.body in
        if (dependent (strip_outer_cast ccival) signval) then
          let inf = snd(pf_infexecute gls signval) in
          match inf with
            Logic ->
	      (global (gLOB sign) signid)
	      ::(fst (fullrec (tl_sign sign) (n-1) l)),l
          | Inf j ->
	      let t = (type_meta_of_prog sigma sign !mETAMAP c) in
	      let m =
		(match t with
                  DOPN(MutInd _,_) ->
                    search_nb_lam recursive [mis_lc(mind_specif_of_mind t)]
                | _ -> 
                    let (t',largs) = find_minductype sigma t in
		    let m' = search_nb_lam recursive
			[mis_lc(mind_specif_of_mind t')] in
		    suppress_params (List.length largs) m')  in
	      let l' = modif_lf signid m l j._VAL gls in
              let result = fullrec (tl_sign sign) (n-1) l' in
              (global (gLOB sign) signid)::(fst(result)),snd(result)
        else fullrec (tl_sign sign) (n-1) l in
      fullrec sign (sign_length sign) lf
  with UserError _ -> [],lf ;;

(* Pass through all the lambda-abstractions of a term and gives the end. *)
(* Cette fonction a toute les chances de faire n'importe quoi avec les dB *)
let rec split_lf  = function
    DOP2(Lambda,_,DLAM(_,b)) -> split_lf b
  | DOP2(Cast,c,_) -> split_lf c
  | x -> x ;;

(* Replaces the end of succession of lambda-abstractions by a term. *)
let rec replace_in_lf term = function
  DOP2(Lambda,a,DLAM(na,b)) -> DOP2(Lambda,a,DLAM(na,replace_in_lf term b))
| DOP2(Cast,c,_) -> replace_in_lf term c
| x -> term ;;

let rec replace_in_lf_lift term n = function
    DOP2(Lambda,a,DLAM(na,b)) ->
      DOP2(Lambda,a,DLAM(na,replace_in_lf_lift term (n+1) b))
  | DOP2(Cast,c,_) -> replace_in_lf_lift term n c
  | x -> lift (n-1) term ;;

(* Same but with a list of terms to be replaced in a list of terms. *)
let rec replace_in_lf_one term l gls = function
    DOP2(Lambda,a,DLAM(na,b)) ->
      DOP2(Lambda,a,DLAM(na,replace_in_lf_one term l gls b))
  | DOPN(MutCase _,_) as mc -> 
      let (ci,p,c,lf) = destCase mc in
      let lf = Array.to_list lf in
      let sigma = project gls and sign = pf_hyps gls in
      mkMutCase ci (type_meta_of_prog sigma sign !mETAMAP term) c
      	(replace_in_lf_list l lf gls)
  | DOPN(MutConstruct(_,i),_) -> List.nth l (i-1)
  | DOP2(Cast,c,_) -> replace_in_lf_one term l gls c
  | x ->
      (try
 	let x' = nf_beta (pf_one_step_reduce gls x) in
      	replace_in_lf_one term l gls x'
      with UserError _ -> term) (* term or x ? *)

and replace_in_lf_list l l' gls = 
  let rec lrec = function
      a::m,b::n -> (replace_in_lf_one a l gls b)::lrec(m,n)
    | [],[] -> []
  in lrec (l,l')  ;;

(* Checks if a term is pure or not. If yes, returns the corresponding proof
   term. If no, returns the same program term.
   Entries : term : term to be checked
             gls : current goal
   Output : (b,val) = (boolean,pure or not pure term) *)
let is_pure term gls = 
  let sign = pf_hyps gls in
  let sigma = project gls in
  try let j = execute_pure sigma sign term in
      let (_,inf) = pf_infexecute gls j._VAL in
      match inf with
        Logic -> false, term
      | Inf{_TYPE=typ} -> let j' = execute_pure sigma sign typ in
                          if pf_conv_x gls j._TYPE j'._VAL then
                              true, j._VAL
                          else false, term
  with UserError _ -> false, term ;; 

(* Same but in the context of a clenv and not a goal. *)
let clenv_is_pure term clenv = 
  let sign = clenv_sign clenv in
  let sigma = clenv_evc clenv in
  let envpair = gENV sigma sign in
  let metamap = metamap_of_clenv clenv in
  try (* Pourquoi jeter l'info pour la recalculer apres ??? *)
      let (j,_) =
 	infexemeta sigma metamap envpair (propify (gLOB sign) term) in
      let (_,inf) = infexemeta sigma metamap envpair j._VAL in
      match inf with
        Logic -> false, term
      | Inf{_TYPE=typ} ->
          let (j',_) =
            infexemeta sigma metamap envpair (propify (gLOB sign) typ) in
            if conv_x sigma j._TYPE j'._VAL then true, j._VAL
            else false, term
  with UserError _ -> false, term ;; 

(* Register to store the indexes of informtive subgoals. *)
let rEG_INF = ref ([]:(int list));;
(* Clears this register in a given goal. *)
let clear_REG_INF gls = (rEG_INF:=[]; tclIDTAC gls);;

(* Associates a list of programs to a list of subgoals *)
let assoc_arg lf =
  (fun i gls -> 
          (match snd(pf_infexecute gls (pf_concl gls)) with
            Logic -> tclIDTAC gls
          | _ -> rEG_INF := !rEG_INF@[i];
                 let j = index i !rEG_INF in
                 try let f = List.nth lf (j-1) in
                      realizer f gls
                 with Failure "nth" -> tclIDTAC gls)) ;;

(* Applies Elim or Case with ccival and associates the lf to the subgoals.
   Entries : recursive : boolean
             ccival : value to be eliminated
             lf : list of programs to be associated
   Output : tactic *)
let elim_pg recursive ccival lf = 
  tclTHEN clear_REG_INF
    (tclTHEN_i
       (if recursive then simplest_elim ccival else simplest_case ccival)
       (assoc_arg lf) 0)
;;
let case_pg = elim_pg false;;

(* Applies ElimType with ccityp and associates the lf to the subgoals.
   Entries : recursive : boolean
             ccityp : type to be eliminated
             lf : list of programs to be associated
   Output : tactic *)
let elim_type_pg ccityp lf = 
  tclTHEN clear_REG_INF (tclTHEN_i (elim_type ccityp) (assoc_arg lf) 0)
;;

(* Suppress all the products at the head of a term *)
let rec cut_prod = function
  DOP2(Prod,_,DLAM(_,b)) -> cut_prod b
| DOP2(Cast,t,_) -> cut_prod t
| x -> x ;;

(* Applies CaseType for mutual inductive types
   with typ and associates the lf to the subgoals.
   Entries : typ : type to be eliminated
             lf : list of programs to be associated
   Output : tactic *)
let case_type_pg typ lf gls =
  (tclTHEN clear_REG_INF (tclTHEN_i (case_type typ) (assoc_arg lf) 0)) gls
;;

(* Same as before, but possibly do one more case if typ is a singleton.
   Entries : typ : type to be eliminated
             lf : list of programs to be associated
   Output : tactic *)
(* TODO: compare this with elim_sig... *)
let case_type_singl_pg typ lf gls =
  let sigma = project gls in
  let (ity,_) = decomp_app (whd_betadeltaiota sigma typ) in
  match mis_singl (mind_specif_of_mind ity) with
    None ->
      (tclTHEN clear_REG_INF (tclTHEN_i (case_type typ) (assoc_arg lf) 0)) gls
  | Some _ ->
      (tclTHENS (case_type typ)
	 [ realizer (List.hd lf);
	   tclTHENLIST
	     [ do_nb_intros;
	       intro;
	       onLastHyp (fun oh -> case_pg (mkVar(outSOME oh)) (List.tl lf)) ]
	 ])
	gls
;;

(* Transforms a goal Gamma |- G into Gamma |- (x:ccityp)(ccival=x)G *)
let elim_pattern ccival ccityp gls =
  let prop =
    applist(pf_global gls (id_of_string "eq"),[ccityp;ccival;ccival]) in 
  (tclTHEN (tclTHENL (cut prop) Auto.full_trivial)
     (pattern_option [[2],ccival,ccityp] None)) gls ;;


(* Replaces the metas of a term with the corresponding program in pmap.
   Entries : pmap : association list between metas and programs
             t : term where metas have to be replaced
   Output : new term *)
let replace_meta_with_prog pmap t = 
 let rec prec = function
  DOP0(Meta n) -> List.assoc n pmap 
| DOP2(Lambda,a,DLAM(na,b)) -> DOP2(Lambda,a,DLAM(na,prec b))
| DOP2(Cast,c,_) -> prec c
| DOP2(Prod,a,DLAM(na,b)) ->  DOP2(Prod,a,DLAM(na,prec b))
| DOPN(AppL,args) -> DOPN(AppL,Array.map prec args)
| DOPN(MutCase ci,args) -> DOPN(MutCase ci,Array.map prec args)
| DOPN(Fix(vn,i),args) -> DOPN(Fix(vn,i),Array.map prec args)
| DOPN(CoFix(i),args) -> DOPN(CoFix(i),Array.map prec args)
| c -> c
 in if pmap = [] then t else prec t  ;;

(* Replaces the metas of a term with the corresponding program in pmap
   preceded by the right singleton type constructor.
   Fails if not possible with UserError.
   Entries : pmap : association list between metas and programs
             t : term where metas have to be replaced
   Output : new term *)
let replace_meta_with_construct (clenv,pmap) val_0 typ = 
  let sigma = clenv_evc clenv in
  let sign = clenv_sign clenv in
  let rec prec typ val_0 =
    match (telescope_appl val_0) with 
      DOP0(Meta n) ->
 	let newval = List.assoc n pmap in
        let newtyp = clenv_extraction clenv (clenv_type_of clenv newval) in
        (try
          if (is_singleton_meta clenv newtyp typ)
          then applist(val_mconstruct sigma 1 typ,[newval])
          else error "replace_meta_with_construct"
        with UserError _ -> error "replace_meta_with_construct")
    | DOP2(Lambda,a,DLAM(na,b)) -> DOP2(Lambda,a,DLAM(na,prec typ b))
    | DOP2(Cast,c,_) -> prec typ c
    | DOP2(Prod,a,DLAM(na,b)) ->  DOP2(Prod,a,DLAM(na,prec typ b))
    | DOPN(AppL,[|a;b|]) ->
 	let (DOP2(Prod,typ',_)) = clenv_type_of clenv a in
        DOPN(AppL,[|(prec typ a); (prec typ' b)|])
    | DOPN(MutCase ci,args) -> DOPN(MutCase ci,Array.map (prec typ) args)
    | DOPN(Fix(vn,i),args) -> DOPN(Fix(vn,i),Array.map (prec typ) args)
    | DOPN(CoFix(i),args) -> DOPN(CoFix(i),Array.map (prec typ) args)
    | c -> c
 in prec typ val_0  ;;

(* Suppress all the logical products of a term.
 * BOGUS: fait n'importe quoi avec les dB: le resultat de tail_type
 * peut avoir un nombre arbitraire de nouvelles variables libres.
 *)
let tail_type gls  t = 
 let rec trec t l =
  match t with
    DOP2(Cast,c,_) -> trec c l
  | DOP2(Prod,a,DLAM(_,b)) -> if (is_logical l a gls) then
                                  trec b (true::l)
                              else b
  | _ -> error"tail_type"
 in trec t []  ;;


(* The move_arg_in_lf* functions are BOGUS. What we need is applying
   arg under the abstractions corresponding to the pattern variables.
   1- We may need to do some eta-expansion.
   2- In [Case c of <pat> => [x]M end], arg will be applied to M!!!
 *)

(* Applies a term at the end of a succession of lambda-abstractions.
   [xi:Ti]t -> [xi:Ti](t arg).
   Entries : lf : term with lambda-abstractions
             arg : argument to be applied
   Ouput : new term *)
let rec move_arg_in_lf lf arg = 
  match strip_outer_cast lf with
    DOP2(Lambda,a,DLAM(x,b)) -> DOP2(Lambda,a,DLAM(x,move_arg_in_lf b arg))
  | x -> applist(x,[arg]) ;;

(* Applies a term at the end of a succession of lambda-abstractions.
   [xi:Ti]t -> [xi:Ti](arg t).
   Entries : lf : term with lambda-abstractions
             arg : argument to be applied
   Ouput : new term *)
let rec move_arg_in_lf2 lf arg = 
  match strip_outer_cast lf with
    DOP2(Lambda,a,DLAM(x,b)) -> DOP2(Lambda,a,DLAM(x,move_arg_in_lf2 b arg))
  | x -> applist(arg,[x]) ;;

(* Applies a term at the end of a succession of lambda-abstractions.
   [xi:Ti]t -> [xi-1:Ti-1]t[xi/arg].
   Entries : lf : term with lambda-abstractions
             arg : argument to be applied
   Ouput : new term *)
let rec move_arg_in_lf3 lf n m arg = 
  match strip_outer_cast lf with
    DOP2(Lambda,a,DLAM(x,b)) ->
      if n=m then (nf_beta (applist(lf,[arg])))
      else DOP2(Lambda,a,DLAM(x,move_arg_in_lf3 b n (m+1) arg))
  | x -> applist(x,[arg]) ;;

(* Moves a MutCase in an application back to the head in order
   to have a MutCase and no more an application.
   Entries : clenv : local clenv in which typechecking is possible
             gls : current goal
   Output : new term *)
let rec move_rec clenv gls = 
  let rec mrec x = match (telescope_appl x) with
      DOPN(AppL, [| DOPN(MutCase _,_) as t1; t2 |]) as val_0 ->
        let (ci,p,c,lf) = destCase t1 in
          if Array.length lf = 1
          then let f = lf.(0) in
               let p' = (match p with
                             DOP2(Prod,_,DLAM(_,b)) -> b
                           | DOP2(Lambda,_,DLAM(_,b)) -> b
                           | _ -> error"move_rec") in
               let lf' =
		 try clenv_execute_nc clenv (move_arg_in_lf f t2)
                 with UserError _ -> move_arg_in_lf3 f (nb_lam f) 1 t2 in
               mrec (mkMutCaseA ci p' c [|lf'|])
          else val_0
    |  DOPN(AppL, [| t1; DOPN(MutCase _,_) as t2 |]) as val_0 ->
         let (ci,p,c,lf) = destCase t2 in
           if Array.length lf = 1
           then let f = lf.(0) in
		let b = tail_type gls (clenv_type_of clenv t1) in
		let arg = subst1 (mkMutCaseA ci p (Rel 1) [|f|]) b in
		let t = clenv_type_of clenv c in
		let p' = Environ.lambda_create (t,arg) in
		let lf' = move_arg_in_lf2 f t1 in
		mrec (mkMutCaseA ci p' c [|lf'|])
           else val_0
    | DOPN(AppL, [| t1; t2 |]) -> 
	(match (is_rec t1, is_rec t2) with
          (true,true) -> mrec (telescope_appl (applist (mrec t1,[mrec t2])))
	| (true,false) -> mrec (telescope_appl (applist (mrec t1,[t2])))
	| (false,true) -> mrec (telescope_appl (applist (t1,[mrec t2])))
        | (false,false) -> applist (t1,[t2]))
    | x -> x
  in mrec  ;;


(* Same as before but for mutual inductive types. No recursive parameters,
   since an elimination is always non recursive in this case. *)
let rec case_app ci p c lf pm gls = 
  let concl = pf_concl gls and sign = pf_hyps gls and sigma = project gls in
  let (b,ccival) = is_pure c gls in
  if b then
    if dependant (strip_outer_cast ccival) concl 
       & (List.for_all (fun f -> not (dependant c f)) lf) 
                 (* Optimizations for interpreting terms generated by Cases 
                    in Case x of f1 .. fk end keep information on x when 
                    it occurs in on of the fi *)
    then 
      let ni = nb_intros_liees ccival gls concl in
      (tclTHEN (tclDO ni Tactics.intro) (case_pg ccival lf)) gls
    else
      (tclTHENLIST [ do_nb_intros;
		     elim_pattern ccival (pf_type_of gls ccival);
		     case_pg ccival lf ]) gls
  else 
    (* Pourquoi ne pas utiliser start_clenv ? *)
    let (wc,kONT) = startWalk gls in
    let clenv0 = mk_clenv_hnf_constr_type_of wc concl in
    let (ccival',ccityp,(clenv,pmap)) = 
      specif_of_annot_pure gls (clenv0,pm) concl c in
    let (ccityp',_,clenv') = clausal_form_info clenv ccityp in
    match ccival' with
      DOPN(MutCase _,_) ->
          let (ci,_,d,lf') = destCase ccival' in
          let newc = clenv_extraction clenv d in 
          let lf' = lf'.(0) in
          let newlf' = clenv_extraction clenv lf' in
          let fin = replace_meta_with_prog pmap (split_lf newlf') in
          let term = mkMutCase ci p fin (List.map (subst_term newc) lf) in
          let newlf = replace_in_lf term newlf' in
          case_app ci p newc [newlf] pmap gls
    | _ ->
 	if is_rec ccival' then
          let term = move_rec clenv' gls ccival' in
          (match term with
            DOPN(MutCase _,_) ->
              let (ci,_,d,lf') = destCase term in
              let c_of_c = clenv_extraction clenv' d in
              let lf_of_c = lf'.(0) in
              let lf_of_c' = clenv_extraction clenv' lf_of_c in
              let fin = replace_meta_with_prog pmap (split_lf lf_of_c') in
              let term = mkMutCase ci p fin lf in
              let newlf =
		lift (-1) (subst_term c_of_c (replace_in_lf term lf_of_c')) in
              case_app ci p c_of_c [newlf] pmap gls
          | _ -> error"case_app")
        else 
          (try 
            let j = execute_meta_prog sigma sign !mETAMAP c in
            if (pf_conv_x gls ccityp' (execute_pure sigma sign j._TYPE)._VAL)
	    then
              (try
		let newc = nf_beta (pf_one_step_reduce gls c) in
              	(match newc with
                  DOPN(MutCase _,_) ->
                    let (ci,newP,newc',newlf) = destCase newc in
                    let args' =
		      replace_in_lf_list lf (Array.to_list newlf) gls in
                    case_app ci p newc' args' pmap gls
                | _ -> error "not a rec")
              with UserError _ -> raise Elim)
            else raise Elim
          with Elim | UserError _ ->
            (try
              if occur_meta ccival' then
		let head = without_annot c in
		(tclTHEN do_nb_intros
		   (case_type_singl_pg ccityp' (head::lf))) gls
              else 
		(* Pb: ccityp peut etre un singleton... *)
		(tclTHEN do_nb_intros (case_pg ccival' lf)) gls 
            with (UserError _) as err1 -> 
              if is_singleton_meta clenv'
                    (clenv_extraction clenv' ccityp') ccityp' then
                let newpg = elim_sig ccival' ccityp' gls in
                (match newpg with
                  DOPN(MutCase _,_) ->
                    let (ci',newP,newc',newlf) = destCase newpg in
                    let args' =
                      replace_in_lf_lift
                        (mkMutCase ci p (mkRel 1) lf) 0
                        (clenv_extraction clenv newlf.(0)) in
		    case_app ci' p newc' [args'] pmap gls
                | _ -> raise err1)
              else 
                (try
                  let newccival =
                    replace_meta_with_construct (clenv,pmap) ccival' ccityp' in
                  (tclTHEN do_nb_intros (case_pg newccival lf)) gls 
                with Not_found | UserError _ -> raise err1)));;

(* Same as before but for mutual inductive types. *)
let program_case ci p lf c gls =
  let sigma = project gls and sign = pf_hyps gls in
  let j = execute_meta_prog sigma sign !mETAMAP c in
  if (is_var c) then
    let (l',lf') = full_generalize false c lf gls in
    (tclTHEN (generalize l') (case_app ci p c lf' [])) gls
  else case_app ci p c lf [] gls ;;

(* ========= *)
(* Injection *)
(* ========= *)

(* Checks if a term is specified or not. *)
let rec non_specified = function
    DOP2(Cast,c,_) -> non_specified c
  | DOPN(AppL,args) -> non_specified (hd_vect args)
  | DOP2(Lambda,_,DLAM(_,c)) -> non_specified c
  | DOP2(XTRA("ANNOT",[]),_,_) -> false
  | _ -> true ;;

(* Exception raised to trap particular cases of injection.
   Ie when [x:T]y needs to be transformed into (exists T' [x:T]y),
   for example. *)
exception Particular;;

(* Build a constructor applied to its arguments and to metavariables
 * corresponding to the first logical arguments
 *)
let constructor_info clenv typ k gls =
  let cstr = val_mconstruct (project gls) k typ in
  let (_,args,clenv') = clausal_form_info clenv (pf_type_of gls cstr) in
  (clenv', mkAppList cstr args)
;;


(* When a program must be injected, it cannot contain metavariables (because
 * the injected program appears applied to the predicate of sig (or other).
 * Instead, we replace Meta by VARs by simply doing Cut on all the missing
 * proofs.
 * This trick should be generalized: Program often fails because it produces
 * terms with metavariables corresponding to the logical subgoals.
 *
 * TODO: understand how to handle properly the walking constraints...
 *)
(*
let clenv_cut_products head (ccival,ccityp) (clenv,pmap) =
  let (_,dom,_) =
    destProd (whd_betadeltaiota (clenv_evc clenv)
		(clenv_type_of clenv head)) in
  let rec intro_logical tac ccival ccityp gls =
    if pf_conv1_x gls ccityp dom then tac ccival gls else
    match ccityp with
      DOP2(Cast,c,_) -> intro_logical tac ccival c gls
    | DOP2(Prod,c1,DLAM(na,c2)) ->
	(cut_THEN c1
	  (fun t -> intro_logical tac (mkAppList ccival [t]) (subst1 t c2))
	  tclIDTAC) gls
    | _ -> tac ccival gls in
  let app_tac val' gls =
    let (clenv',kONT) = start_connect clenv gls in
    let ccival = mkAppList head [val'] in
    apply_tac kONT (ccival, pf_type_of gls ccival) (clenv',pmap) gls in
  intro_logical app_tac ccival ccityp
;;
*)

(* Transforms a term ccival into (Construct(1,ind) args ccival).
   Corresponds to an injection in the good singleton type.
   Entries : n : integer to trap errors
             ccival : value to be injected
             ccityp : its type
             clenv : local clenv
             pmap : association list of metas and programs
             gls : current goal
   Output : new goal with good program associated to *)
let injection_red n (ccival,ccityp) (clenv,pmap) gls =
  let sigma = project gls and sign = pf_hyps gls and concl = pf_concl gls in
  let infty = clenv_extraction clenv ccityp in
  try
    let _ = if not (is_singleton_meta clenv infty concl) then error "inj" in
    let _ = if (n=0) & (occur_meta ccityp) then raise Particular in
    let (clenv',val') = constructor_info clenv concl 1 gls in
    (match whd_betadeltaiota sigma (clenv_type_of clenv' val') with
      (DOP2(Prod,c1,_)) ->
	if is_singleton_meta clenv' infty c1 then
          let (clenv'',val1') = constructor_info clenv' c1 1 gls in
	  let trm = nf_beta (applist(val',[applist(val1',[ccival])])) in
          let typ = clenv_type_of clenv'' trm in
          let (clenv''',kONT) = start_connect clenv'' gls in
          apply_tac kONT (trm,typ) (clenv''',pmap) gls
        else
          let (clenv'',kONT) = start_connect clenv' gls in
	  let trm = nf_beta (applist(val',[ccival])) in
	  let typ = clenv_type_of clenv' trm in
          apply_tac kONT (trm,typ) (clenv'',pmap) gls
(*        clenv_cut_products val'
	  (ccival, clenv_type_of clenv' ccival) (clenv',pmap) gls*)
    | _ -> error "inj")
  with UserError _ | Invalid_argument _ ->
    if n=0 then error"injection"
    else
      let pg = clenv_extraction clenv ccival in
      realizer (replace_meta_with_prog pmap pg) gls
;;

let injection n (ccival,ccityp) (clenv,pmap) =
  tclTHEN red_intros (injection_red n (ccival,ccityp) (clenv,pmap))
;;


(* Transforms the Case predicate P' when moving Cases out:
 * ((<P>Cases ... of pat => br end) arg)  -->
 * <P'>Cases ... of pat => (br arg) end
 *)
let app_case_predicate p arg =
  match kind_of_term p with
    IsLambda(x,a,b) ->
      let (_,_,rng) = destProd (strip_outer_cast b) in
      mkLambda x a (subst1 (lift 1 arg) rng)
  | _ -> let (_,_,rng) = destProd (strip_outer_cast p) in (subst1 arg rng)
;;

(* Takes a program on the form (head args). 
   Calculates the corresponding proof term ccival.
   Looks if transformations are first needed on ccival (to be injected),
   that is if MutCase need to be moved first. Moves them if necessary
   and applies the appropriated tactics. Then, branches on the previous
   injection function.
   Entries : head : head of the program
             args : list of arguments of the program
             gls : current goal
   Output : new subgoal *)
let program_inj head args gls = 
  let concl = pf_concl gls and sign = pf_hyps gls in
  let (clenv0,kONT) = start_clenv head gls in
  let (ccival,ccityp,(clenv,pmap)) = 
    specif_of_annot gls (clenv0,[]) concl (applist(head,args))
  in 
 let rec prec n ccival clenv gls =
  match telescope_appl(strip_outer_cast ccival) with
     DOPN(MutCase _,_) as mut -> 
      let (_,_,c,lf) = destCase mut in
      let cty = clenv_type_of clenv c in
        (match lf with
          [|f|] -> if occur_meta c then
                  let cfw = clenv_extraction clenv c in
                  (tclTHENS 
                   (case_type cty)
		      ([realizer (replace_meta_with_prog pmap cfw);
			 (fun gl ->
                          let (clenv',_) = start_connect clenv gl in
                          prec (n+1) f clenv' gl)])) gls
                 else (tclTHEN
			 (simplest_case c)
			 (fun gl ->
                           let (clenv',_) = start_connect clenv gls in
                           prec (n+1) f clenv' gl)) gls
        | _ -> injection n (ccival,ccityp) (clenv,pmap) gls)
  | DOP2(Lambda,_,b) -> 
      (tclTHEN Tactics.intro
	 (fun gl ->
           let (clenv',_) = start_connect clenv gl in
           onLastHyp (fun cl -> prec (n+1) (sAPP b (VAR(outSOME cl))) clenv')
	     gl)) gls
  | DOPN(AppL, [| DOPN(MutCase _,_) as t1; t2 |]) ->
      let (mind,p,c,lf) = destCase t1 in
      (match lf with
        [|f|] -> let p' = app_case_predicate p t2 in
                 let lf' = 
		   try clenv_execute_nc clenv (move_arg_in_lf f t2)
                   with UserError _ -> move_arg_in_lf3 f (nb_lam f) 1 t2 in
		 let term = mkMutCaseA mind p' c [|lf'|] in
		 prec (n+1) term clenv gls
      | _ -> injection n (ccival,ccityp) (clenv,pmap) gls)
  | DOPN(AppL, [| t1; DOPN(MutCase _,_) as t2 |]) ->
      let (mind,p,c,lf) = destCase t2 in
        (match lf with
           [|f|] -> (* ce code contient un nombre incroyable de bugs... *)
                    let b = tail_type gls (clenv_type_of clenv t1) in
                    let t = clenv_type_of clenv c in
                    let (mi,_) = find_minductype (Evd.mt_evd ()) t in
                    let arg = subst1 (mkMutCaseA (ci_of_mind mi) p (Rel 1) 
					[|f|]) b in
                    let p' = Environ.lambda_create (t,arg) in
                    let lf' = move_arg_in_lf2 f t1 in
                    let term = mkMutCaseA mind p' c [|lf'|] in
                    prec (n+1) term clenv gls
           | _ -> injection n (ccival,ccityp) (clenv,pmap) gls)
  | DOPN(AppL, [| t1; t2 |]) -> 
      if (is_rec t1) then
        let term = move_rec clenv gls t1 in
        prec (n+1) (applist(term,[t2])) clenv gls
      else if (is_rec t2) then
        let term = move_rec clenv gls t2 in
        prec (n+1) (applist(t1,[term])) clenv gls
      else injection n (ccival,ccityp) (clenv,pmap) gls
  | _ -> injection n (ccival,ccityp) (clenv,pmap) gls
 in prec 0 ccival clenv gls  ;;

(* Calls the previous function but first looks if previous logical
   introductions are needed. *)
let program_inj_intros head args gls =
  let concl = pf_concl gls in
  let (clenv0,kONT) = start_clenv head gls in
  let (ccival,ccityp,(clenv,pmap)) = 
    specif_of_annot gls (clenv0,[]) concl (applist(head,args)) in
  let intro_tac =
    match telescope_appl(strip_outer_cast ccival) with
      DOPN(MutCase _,_) as mut ->
        let (_,_,_,lf) = destCase mut in
        (match lf with
          [|f|] -> do_nb_intros
        | _   -> tclIDTAC)
    | DOPN(AppL,[|_; DOPN(MutCase _,_) as mut|]) -> 
        let (_,_,_,lf) = destCase mut in
        (match lf with
          [|f|] -> do_nb_intros
        | _   -> tclIDTAC)
    | _ -> tclIDTAC
  in (tclTHEN intro_tac (program_inj head args)) gls
;;


(* ================================================ *)
(* Simplification : b <-> if b then true else false *)
(* ================================================ *)

(* Constructs a list of (MutConstruct(P,i) params) for 1<=i<=n. *)

let mconstruct_lf n p largs =
  let (sp,i,cl) = destMutInd p in
  let nparams = mind_nparams p in
  let (globargs,_) = chop_list nparams largs in
  let rec make_list_ind k l = 
    if k=0 then l
    else make_list_ind (k-1) (mkAppList (mkMutConstruct sp i k cl) globargs::l)
  in make_list_ind n []  ;;

(* Calls the previous mutconstruct_lf with n = length of constructors
   and P = ind. *)
let construct_abs_mconstr ind largs = 
   let l = mis_consnames(mind_specif_of_mind ind) in
   let n = List.length (Array.to_list l) in
          mconstruct_lf n ind largs ;;

(* Checks if a MutCase is hidden behind a constant. *)
let is_hidden_rec gls t = 
  match whd_betadeltaiota (project gls) t with
    DOPN(MutCase _,_) -> true
  | _ -> false
;;
let is_const_red t args gls = 
  let b =
    try let t' =
      nf_betadeltaiota (project gls)
 	(pf_one_step_reduce gls (applist(t,args))) in 
    is_construct_head t'
    with UserError _ -> false in
  (is_const t) & b ;;

(* Checks if a term is a modifiable.
   A term is modifiable 
   (i.e. b can be transformed into if b then true else false)
   if it has no hidden MutCase or if it is a variable. 
   It must not be a reducible constant too.  <-- ??? pourquoi ?
   The term has to be (head args). *)
(* So a function returning a nat is modifiable ??? *)
let is_modifiable head args gls = 
  (is_var head or not(List.exists (is_hidden_rec gls) args))
    & not(is_const_red head args gls)
;;

(* Checks if a product is anonymous or not. *)
let rec is_anonymous_prod gls = function
  DOP2(Prod,_,DLAM(Anonymous,_))-> true
| DOP2(Prod,a,DLAM(_,b)) -> (not(is_logical [] a gls)) & noccurn 1 b
| DOP2(Cast,c,_) -> is_anonymous_prod gls c
| _ -> false ;;

(* Creates a list of n functions applying IDTAC to a goal.
   The first function of the list is first_tac. *)
let make_IDTAC_list first_tac n = 
let rec mrec n = 
  if n=0 then []
  else (fun gl -> tclIDTAC gl)::(mrec (n-1)) 
in first_tac::(mrec n);;

(* Compares two conclusion concl1 and concl2.
   If they just differs in some arguments, these ones are kept in a list
   of the form (argconcl1=argconcl2).
   Entries : concl1, concl2 : two conclusions
             gls : current goal
   Output : list of arguments.*)
let rec compare_two_concl concl1 concl2 gls = 
  match (collapse_appl concl1,collapse_appl concl2) with
      (DOPN(AppL,args1),DOPN(AppL,args2)) -> 
	if eq_constr args1.(0) args2.(0) then
          let rec mrec l = function
              x::l1,y::l2 -> 
		if eq_constr x y then mrec l (l1,l2)
		else 
		  let prop = applist(pf_global gls (id_of_string "eq"),
                                     (pf_type_of gls x)::(x::[y])) 
		  in mrec (prop::l) (l1,l2)
            | [],[] -> l
          in mrec [] (Array.to_list (tl_vect args1),
		      Array.to_list (tl_vect args2))
	else error "compare_two_concl"
	  
    |  (DOP2(Cast,c1,_),DOP2(Cast,c2,_)) -> compare_two_concl c1 c2 gls
    |  (DOP2(Cast,c1,_),_) -> compare_two_concl c1 concl2 gls
    |  (_,DOP2(Cast,c2,_)) -> compare_two_concl concl1 c2 gls
    |  _ -> error "compare_two_concl"
;;

(* Checks if a term is a constructor or a mutconstructor or an application
   beginning with a constructor or a mutconstructor. *)
let rec is_a_construct gls = function
    DOP2(Cast,c,_) -> is_a_construct gls c
  | DOPN(Const _,_) as c ->
      evaluable_const (project gls) c
	& is_a_construct gls (const_value (project gls) c)
  | DOP2(Lambda,_,DLAM(_,b)) -> is_a_construct gls b
  | DOPN(MutConstruct _,_) -> true
  | DOPN(AppL,args) -> is_a_construct gls args.(0)
  | _ -> false ;;

(* Creates a list of functions from a list whose elements have the form
   (arg1=arg2). Each function cuts an element (arg1=arg2) in the
   current goal, introduces this new hypothesis and eliminates it. *)
let rec make_cut_list list_cut =
  match list_cut with
    a::l -> cut_THEN a simplest_elim Auto.default_full_auto
  | [] -> tclIDTAC;;

(* Global function with modifies a program b into if b then true else false.
   The program has the form (head args).
   Ce qu'elle n'arrive pas du tout a faire.
 *)
let modif_app_ind head args gls = 
  let pg = applist(head,args) in
  let concl = pf_concl gls and sigma = project gls 
  and sign = pf_hyps gls in
  let (clenv0,kONT) = start_clenv head gls in
  let (ccival,ccityp,(clenv,pmap)) =
    specif_of_annot gls (clenv0,[]) concl pg in
  let typg = type_meta_of_prog sigma sign !mETAMAP pg in
  let (ccityp',_,clenv') = clausal_form_info clenv ccityp in
  try
    let lf =
      if is_singleton_meta clenv' typg ccityp' then
        [Environ.lambda_create (typg, Rel 1)]
      else if (is_modifiable head args gls) &
	      not(fst(clenv_is_pure ccival clenv'))
      then
        let (t,largs) = find_minductype sigma typg in
        construct_abs_mconstr t largs
      else raise Rewrite in
    if occur_meta ccival then
      let first_tac gl =
	let concl = pf_concl gl in
        if is_anonymous_prod gl concl then raise NoModif
        else tclIDTAC gl in
      let tclIDTAC_list = make_IDTAC_list first_tac (List.length lf) in 
      (tclTHENS 
         (tclTHEN (kONT clenv'.hook) (case_type_pg ccityp' (pg::lf)))
	 tclIDTAC_list) gls
    else (tclTHEN (kONT clenv'.hook) (case_pg ccival lf)) gls
  with NoModif -> error"modif_app_ind"
  | Induc | Invalid_argument _ | Rewrite | UserError _-> 
      let list_cut = compare_two_concl ccityp' concl gls in
      (tclTHENLIST
	 [ kONT clenv'.hook;
	   clear_REG_INF;
	   tclTHEN_i (make_cut_list list_cut) (assoc_arg [pg]) 0 ])
	gls
;;

(* Injection of a program which can either not be specified or 
   not be injected with the standard method. *)
let inject_bad_program pg gls = 
  (* Pourquoi ne pas utiliser start_clenv ? *)
  let (wc,_) = startWalk gls in
  let sigma = project gls 
  and sign = pf_hyps gls and concl = pf_concl gls in
  try
    let j = execute_pure sigma sign pg in
    let clenv0 = mk_clenv_hnf_constr_type_of wc j._VAL in
    let (ccival,ccityp,(clenv,pmap)) = 
      specif_of_annot gls (clenv0,[]) concl pg
    in injection 0 (ccival,ccityp) (clenv,pmap) gls 
  with UserError _ ->
    let j = execute_meta_prog sigma sign !mETAMAP pg in
    let clenv = mk_clenv_from wc (j._VAL,j._TYPE) in
    let t = clenv_extraction clenv concl in
    let (clenv',mv) = new_meta_pg clenv (Anonymous,t) in
    let ccival = mkMeta mv in
    let ccityp = clenv_type_of clenv' ccival in
    let pmap = [(mv,pg)] in
    injection 0 (ccival,ccityp) (clenv',pmap) gls ;;

(* Checks if the head of an application if a meta or not. *)
let rec is_head_ex = function
  DOP0(Meta _) -> true
| DOP2(Cast,c,_) -> is_head_ex c
| DOPN(AppL,args) -> is_head_ex args.(0)
| _ -> false;;

(*===============*)
(* Fixpoint case *)
(*===============*)

(* Looks for the good index of the decreasing argument in a fixpoint. 
   Entries : i : index from the program point of view.
             concl : current conclusion
             gls : current goal
   Output : new index *)
let real_decreasing_number i concl gls = 
 let rec nrec n m l concl =
  match concl with
    DOP2(Prod,a,DLAM(_,b)) ->   if (is_logical l a gls) then
                                  nrec n (m+1) (true::l) b
                                else if n=i+1 then m-1 
                                     else nrec (n+1) (m+1) (false::l) b
  | DOP2(Cast,c,_) -> nrec n m l c
  | _ -> m-1
 in nrec 1 1 [] concl 
 ;;

(* Finds the ith products in a term.
   Returns the term without the ith first products and a list associating
   de Bruijn indexes and booleans (the boolean says if the free variable
   corresponding to the de Bruijn index is logical or not.
   Entries : i : number
             concl : conclusion
   Ouput : (A,l) = (term, boolean list) *)
let find_ind_arg i concl = 
 let rec nrec n l concl =
  match concl with
    DOP2(Prod,a,DLAM(_,b)) -> if n=i then
                                a,l
                              else nrec (n+1) (false::l) b
  | DOP2(Cast,c,_) -> nrec n l c
  | _ -> concl,l
 in nrec 0 [] concl 
 ;;


(* Generalizes the current goal with respect to the type of arg and 
   associates pg to the new goal.
   Entries : arg : argument to generalize
             pg : program to be associated
             gls : current goal
   Output : new goal *)
let generalize_concl arg pg gls = 
  let typ = pf_type_of gls arg in
  let newconcl = Environ.prod_create (typ,subst_term arg (pf_concl gls)) in
  (tclTHENS (cut newconcl) [Auto.default_full_auto; realizer pg]) gls ;;

let substn_meta laml typ kind n =
    
 let rec substrec depth = function 
     (Rel k as x)     -> if k<=depth then x
                         else if k-depth <= 1 then
                           let p = newMETA() in
                           mETAMAP := !mETAMAP@[(p,mkCast typ kind)];
                           lift_substituend depth
			     (make_substituend
				(applist(laml,[mkCast (mkMeta p) typ])))
                         else Rel(k-1)
   | VAR id           -> VAR id
   | DOPN(oper,cl)    -> DOPN(oper,Array.map (substrec depth) cl)
   | DOPL(oper,cl)    -> DOPL(oper,List.map (substrec depth) cl)
   | DOP1(oper,c)     -> DOP1(oper,substrec depth c)
   | DOP2(oper,c1,c2) -> DOP2(oper,substrec depth c1,substrec depth c2)
   | DLAM(na,c)       -> DLAM(na,substrec (depth+1) c)
   | DLAMV(na,v)      -> DLAMV(na,Array.map (substrec (depth+1)) v)
   | x                -> x
 in substrec n
    ;;

let subst1_meta laml typ kind = substn_meta laml typ kind 0;;

let substn laml n = substn_many [|make_substituend laml|] (n-1);;

(* Generalizes the current goal as many times as they are logical arguments
   in the list largs which are not parameters and are closed.
   Entries : globargs : parameters
             largs : list of arguments
             lrel : list of boolean to typecheck
             pg : program to be associated
             gls : current goal
   Output : new goal *)
let simple_count_needed globargs largs lrel pg gls = 
  let sigma = project gls and sign = pf_hyps gls in

  let rec crec l pg gls = 
  match l with
    a::l' ->
      if (not(List.exists (eq_constr a) globargs)) & (closed0 a) then
        if is_logical lrel (pf_type_of gls a) gls then
          (tclTHEN (generalize_concl a pg) (crec l' pg)) gls
        else
          let j1 = execute_meta_prog sigma sign !mETAMAP a in
          let typ = j1._TYPE and kind = j1._KIND in
          let newpg = match pg with
            DOPN(Fix(vn,i),cl) -> 
              let j = execute_meta_prog sigma sign !mETAMAP
                  (Environ.prod_create
                     (typ,lift 1 (strip_outer_cast cl.(0)))) in
              let newcl0 = DOP2(Cast,j._VAL,j._TYPE) in
              let newvn = [| vn.(0)+1 |] in
              let newcl1 = match cl.(1) with
                DLAMV(name,clvect) ->
                  let newtyp = DOP2(Cast,j1._TYPE,j1._KIND) in
		  DLAMV(name,[| DOP2(Cast,
                    Environ.lambda_create
                      (newtyp, lift 1 (subst1_meta (Rel(1)) typ kind
                                         (strip_outer_cast (hd_vect clvect)))),
                    j._VAL) |])
              | _ -> error"count_needed"  in
              DOPN(Fix(newvn,i),[| newcl0;newcl1 |])
          | _ -> error "count_needed" in
          (tclTHEN (generalize_concl a newpg) (crec l' newpg)) gls
      else crec l' pg gls
  | [] -> tclIDTAC gls
  in crec largs pg gls  ;;

let count_needed globargs largs lrel pg gls = 
  let sigma = project gls and sign = pf_hyps gls in

 let rec crec l pg gls = 
match l with
  a::l' -> if (not((List.exists (eq_constr a) globargs))) & (closed0 a) then
            if (is_logical lrel (pf_type_of gls a) gls) then
              ((tclTHEN (generalize_concl a pg) (crec l' pg))) gls
            else
              let j1 = execute_meta_prog sigma sign !mETAMAP a in
              let typ = j1._TYPE and kind = j1._KIND in
              let newpg = match pg with
                DOPN(Fix(vn,i),cl) -> 
                  let j = execute_meta_prog sigma sign !mETAMAP
                            (Environ.prod_create
                               (typ,lift 1 (strip_outer_cast cl.(i)))) in
                  let newcl = Array.of_list(except (last_vect cl) (Array.to_list cl)) in
                  let newcl0 = map_i_vect 
                                 (fun k t -> if k=i then
                                              DOP2(Cast,j._VAL,j._TYPE) 
                                             else t) 0 newcl in
                  let newvn = map_i_vect 
                                 (fun k t -> if k=i then
                                              t+1 
                                             else t) 0 vn in
                  let newcl1 = 
                      let rec nrec cl = match cl with
                        DLAM(name,rest) -> DLAM(name,nrec rest)
                      | DLAMV(name,clvect) ->
                          let newtyp = DOP2(Cast,j1._TYPE,j1._KIND) in
  DLAMV(name,map_i_vect 
                (fun k t -> if k=i then
                 DOP2(Cast,
                      Environ.lambda_create
                        (newtyp,lift 1 (subst1_meta (Rel(1)) typ kind
                                          (strip_outer_cast t))),j._VAL)
                            else t) 0 clvect)
                      | _ -> error"count_needed"  in
                        nrec (last_vect cl) in
                   DOPN(Fix(newvn,i),Array.append newcl0 [|newcl1|])
              | _ -> error "count_needed" in
                ((tclTHEN (generalize_concl a newpg) (crec l' newpg))) gls
          else crec l' pg gls
| [] -> tclIDTAC gls
 in crec largs pg gls  ;;


(* Same but does not the generalizations but counts the numbers that have 
   to be done. *)
let count_added globargs largs lrel gls = 

 let rec crec n l = 
match l with
  a::l' -> if (not(List.exists (eq_constr a) globargs)) & (closed0 a) then
            crec (n+1) l'
          else crec n l' 
| [] -> n
 in crec 0 largs  ;;

(* Applies count_needed with the good parameters and then applies
   the fixpoint tactic with the good arguments.
   Entries : f : identifier of the fixpoint
             i : program number of the decreasing argument
             l : list of booleans to typecheck
             concl : current conclusion
             pg : program to be associated
             gls : current goal
   Output : new goal *)
let rec simple_nb_needed_generalize f i l concl pg gls = 
  match collapse_appl(strip_outer_cast concl) with
    DOPN(AppL,args) -> 
      let t1::largs = Array.to_list args in
        begin match t1 with
            DOPN(MutInd _,_) -> 
              let nparams = mind_nparams t1 in
              let (globargs,_) = chop_list nparams largs in
              let k = count_added globargs largs l gls in
              (tclTHEN (simple_count_needed globargs largs l pg) 
		 (fix f (i+k+1))) gls
          | DOPN(Const _,_) -> simple_nb_needed_generalize f i l 
		(whd_beta (pf_one_step_reduce gls concl)) pg gls
          | _ -> anomaly"simple_nb_needed_generalize"
	end
    | _ -> fix f (i+1) gls;;

let rec nb_needed_generalize lf ln ltyp lrel i concl pg gls = 
  match collapse_appl(strip_outer_cast concl) with
      DOPN(AppL,args) -> 
	let t1::largs = Array.to_list args in
          begin match t1 with
               DOPN(MutInd _,_) -> 
                 let nparams = mind_nparams t1 in
                 let (globargs,_) = chop_list nparams largs in
                 let k = count_added globargs largs lrel gls in
                 let newln = map_i (fun j t -> if j=i then t+k
                                    else t) 1 ln in
                 (tclTHEN (count_needed globargs largs lrel pg)
		    (mutual_fix lf newln ltyp)) gls
             | DOPN(Const _,_) -> nb_needed_generalize lf ln ltyp lrel i 
		   (whd_beta (pf_one_step_reduce gls concl)) pg gls
             | _ -> anomaly"nb_needed_generalize"
	  end
    | _ -> mutual_fix lf ln ltyp gls;;

(* Global function to computes a program fixpoint.
   Entries : i, vn, cl : arguments of the fixpoint
             pg : program to be associated
             gls : current goal
   Output : new goal *)
let apply_simple_fixpoint vn cl pg gls = 
  let k = real_decreasing_number vn.(0) (pf_concl gls) gls in
  let ldef = last_vect cl in
  let f = match ldef with
                 DLAMV(Name(f),_) -> f
              | _ -> anomaly "shouldn't appear" in
  let newf =  let avoid = ids_of_sign (pf_hyps gls) in 
              (next_ident_away_from f avoid) in
  let (a,lrel) = find_ind_arg k (pf_concl gls) in
  (tclTHEN (simple_nb_needed_generalize newf k lrel a pg) 
     (fun gl ->
       let pg = try extract_pg gl with UserError _ -> pg in
       let t = match strip_outer_cast pg with
         DOPN(Fix(_),cl) -> 
           (match last_vect cl with
             DLAMV(_,t) -> t
           | _ -> anomaly "shouldn't appear")
       | _ -> anomaly "shouldn't appear" in
       onLastHyp
	 (fun cl -> realizer (subst1 (VAR(outSOME cl)) (hd_vect t))) gl))
    gls
;;

let get_annot l = 
 let rec grec n = function
     (DOP2(XTRA("ANNOT",[]),c,t))::l -> t::(grec (n+1) l)
   | _::_ -> if n=0 then grec (n+1) l else error
  "Each non-first function of a mutual program (co)fixpoint must be annotated"
   | [] -> []
 in grec 0 l ;;

let get_back_fix_annot pg = 
 let rec grec l pg = match pg with
  DOP2(Lambda,a,DLAM(_,b)) -> grec l b
| DOP2(Cast,c,_) -> grec l c
| DOPN(AppL,args) -> grec l (args.(0))
| DOPN(Fix _,args) -> grec l (last_vect args) 
| DOPN(CoFix _,args) -> grec l (last_vect args) 
| DLAM(_,a) -> grec l a
| DLAMV(_,restv) -> get_annot (Array.to_list restv)
| DOP2(XTRA("ANNOT",[]),c,_) -> grec l c
| _ -> l
 in grec [] pg  ;;

let apply_fixpoint vn i cl pg gls = 
  if Array.length vn = 1 then apply_simple_fixpoint vn cl pg gls
  else
      let ltyp = get_back_fix_annot pg in
      let vnlist = Array.to_list vn in
      let ltyp_util =
 	let rec nrec n l = match l with
          [] -> if n=i+2 then [pf_concl gls] else []
        | a::l' ->
	    if n=i+1 then (pf_concl gls)::a::l'
            else a::(nrec (n+1) l') in
        nrec 1 ltyp in
      let ln =
 	List.map (fun (n,t) -> real_decreasing_number n t gls)
	  (List.combine vnlist ltyp_util) in
      let k = List.nth ln i in
      let realln = List.map (fun t -> t+1) ln in
      let (lna,_) = decomp_all_DLAMV_name (last_vect cl) in
      let avoid = ids_of_sign (pf_hyps gls) in
      let lf =
 	List.map
	  (function Name(f) -> next_ident_away_from f avoid
	    | _ -> anomaly "shouldn't appear")
	  (List.rev lna) in
      let (a,lrel) = find_ind_arg k (pf_concl gls) in
      (tclTHEN_i (nb_needed_generalize lf realln ltyp lrel k a pg)
         (fun i gl ->
	   let pg = try extract_pg gl with UserError _ -> pg in
           let t = 
             match strip_outer_cast pg with
               DOPN(Fix(_),cl) -> 
                 let (_,t) = decomp_all_DLAMV_name (last_vect cl)
                 in t
             | _ -> anomaly "shouldn't appear" in
           let n = List.length realln in
           let newpg = 
             let rec srec j l t = match l with
               a_0::l' -> srec (j+1) l' (substn (VAR a_0) (n-j+1) t)
             | [] -> t
             in srec 1 lf (without_annot t.(i)) in
           realizer newpg gl) 0) gls ;;

(* Global function to computes a program cofixpoint.
   Entries : cl : argument of the cofixpoint
             pg : program to be associated
             gls : current goal
   Output : new goal *)
let apply_simple_cofixpoint cl pg gls = 
  let ldef = last_vect cl in
  let f = match ldef with
                 DLAMV(Name(f),_) -> f
              | _ -> anomaly "shouldn't appear" in
  let newf =  let avoid = ids_of_sign (pf_hyps gls) in 
              (next_ident_away_from f avoid) in
    ((tclTHEN (cofix newf) 
        ((fun gl -> let t = match strip_outer_cast pg with
                            DOPN(CoFix(_),cl) -> 
                                (match last_vect cl with
                                    DLAMV(_,t) -> t
                                  | _ -> anomaly "shouldn't appear")
                          | _ -> anomaly "shouldn't appear" in
     onLastHyp (fun (Some id) -> realizer (subst1 (VAR id) (hd_vect t))) gl)))) gls ;;

let apply_cofixpoint cl pg gls = 
let (lna,_) = decomp_all_DLAMV_name (last_vect cl) in
match lna with
  [a] -> apply_simple_cofixpoint cl pg gls
|  _  ->
  let ltyp = get_back_fix_annot pg in
  let avoid = ids_of_sign (pf_hyps gls) in
  let lf = List.map (function (Name(f)) -> next_ident_away_from f avoid | _ -> anomaly "shouldn't appear") (List.rev lna) in
    (tclTHEN_i (mutual_cofix lf ltyp)
            (fun i gl -> let t = match strip_outer_cast pg with
                              DOPN(CoFix(_),cl) -> 
                               let (_,t) = decomp_all_DLAMV_name (last_vect cl)
                               in t
                            | _ -> anomaly "shouldn't appear" in
                         let n = List.length lf in
                         let newpg = 
                              let rec srec j l t = match l with
                                   a::l' -> srec (j+1) l' (substn (VAR a) (n-j+1) t)
                                 | [] -> t
                              in srec 1 lf (without_annot t.(i)) in
                           realizer newpg gl) 0) gls ;;

(* ============= *)
(* Global tactic *)
(* ============= *)

let rec program_app_tac head args gls =
  let concl = pf_concl gls in
  try 
    let (clenv0,kONT) = start_clenv head gls in
    let (ccival,ccityp,(clenv,pmap)) = 
      specif_of_annot gls (clenv0,[]) concl (applist(head,args)) in
    if is_head_ex ccival then
      tclTHENS 
        (cut (clenv_type_of clenv (DOP0(Meta (fst (List.hd pmap)))))) 
	[(try (tclTHEN (realizer (snd (List.hd (List.tl pmap)))) 
		 Auto.default_full_auto)
	  with Failure _ -> Auto.default_full_auto);
	 realizer (snd(List.hd pmap))]
	gls
    else 
      try
	let lf = List.map snd pmap in
        (tclTHENS (Tactics.exact ccival) (List.map realizer lf)) gls
      with UserError _ ->
        try  program_app head args gls
        with UserError _ -> 
          try
	    if non_specified ccival then 
	      try program_inj_intros head args gls
	      with Particular -> inject_bad_program (applist(head,args)) gls
            else error"program_app_tac" 
          with UserError _ -> 
	    (tclTHEN do_nb_intros (modif_app_ind head args)) gls
  with (NotProgram | Generalize _ | UserError _) -> program_app head args gls
;;

let program_app_ext pg gls =
  let sigma = (project gls) and sign = (pf_hyps gls) in
  let (linfargs,allargs)= list_arg gls (collapse_appl pg) in
  let head = List.hd linfargs and infargs = List.tl linfargs in
  let j = try_execute sigma sign head in
  let n = nb_info_arg gls j._TYPE in
  if (n>(List.length infargs)) then
    match red_all gls (type_meta_of_prog sigma sign !mETAMAP pg) with
      DOP2(Prod,a,DLAM(na,b)) ->
        let pg' = Environ.lambda_name(na,a,mkAppList pg [mkRel 1]) in
 	(try deroule_intro pg' gls
        with UserError _ -> program_app_tac head allargs gls)
    | _ -> program_app_tac head allargs gls
  else program_app_tac head allargs gls ;;


let rec program_tactic pg gls = 
  match pg with
    DOP2(Lambda,_,_) -> 
      (try deroule_intro pg gls
      with UserError _ -> inject_bad_program pg gls)
  | DOPN(MutCase _,_) -> 
      let (ci,p,c,lf) = destCase pg in
      (try program_case ci p (Array.to_list lf) c gls
      with NotProgram -> error"Insufficient specification of the program")
  | DOPN(Fix(vn,i),cl)  -> apply_fixpoint vn i cl pg gls
  | DOPN(CoFix(_),cl) -> apply_cofixpoint cl pg gls
  | DOP2(Cast,t,_) -> program_tactic t gls
  | _ ->
      (try
    	let j = execute_pure (project gls) (pf_hyps gls) pg in 
    	try Tactics.exact j._VAL gls
    	with UserError _ ->
	  let (sp,i,cl) = destMutInd (pf_concl gls) in
	  let c1 = mkMutConstruct sp i 1 cl in
	  let (_,a,_) = destProd (pf_type_of gls c1) in
	  if (pf_conv_x gls j._TYPE a) 
	  then Tactics.exact (mkAppList c1 [j._VAL]) gls
	  else error "program_tactic"
      with Invalid_argument _ | UserError _ | Induc ->
    	(try program_app_ext pg gls
    	with (UserError _) as err -> 
	  let pg' = (try nf_beta (pf_one_step_reduce gls pg) 
	  with UserError _ -> raise err) in
	  if eq_constr pg pg' then raise err
	  else program_tactic pg' gls))
;;

let program gls = 
 let deb = Logic.val_debug() in Logic.set_debug true; 
  let pg = extract_pg gls in
  let res = 
    (tclTHEN (context (set_pgm None (pf_ctxt gls))) (program_tactic pg)) gls
  in Logic.set_debug deb;res 
;;

let program_tac = hide_atomic_tactic "Program" program ;;

let program_expand gls = 
  let pg = extract_pg gls in
    (tclTHEN (context (set_pgm None (pf_ctxt gls)))
       (realizer (nf_beta (pf_one_step_reduce gls pg)))) gls ;;

let program_expand_tac = hide_atomic_tactic "Program_Expand" program_expand;;

let program_all = tclTHEN (tclREPEAT program_tac) Auto.default_full_auto;;

let program_all_tac = hide_atomic_tactic "Program_all" program_all ;;

let show_nth_program n =
    let pf = proof_of_pftreestate (get_pftreestate ()) in
    let gllist = fst(frontier pf) in
       try pr_ctxt (get_ctxt (List.nth gllist (n-1)))
       with Failure _ -> [< 'sTR "No such program" >] ;;

open Vernacinterp;;

vinterp_add("ShowProgram",
    function [] -> (fun () -> pPNL(show_nth_program 1))
      | [VARG_NUMBER n] -> (fun () -> pPNL(show_nth_program n)));;


(* $Id: program.ml,v 1.56 1999/11/30 19:25:45 mohring Exp $ *)
