(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                              reduction.mli                               *)
(****************************************************************************)

open Impuniv
open Names
open Generic
open Term
open Closure
open Evd

exception Redelimination
exception Induc
exception Elimconst


type reduction_function = constr -> constr
type stack_reduction_function = constr -> constr list -> constr * constr list

val whd_stack : stack_reduction_function

(* Reduction Function Operators *)
val under_casts      : reduction_function -> reduction_function
val under_outer_cast : reduction_function -> reduction_function
val strong           : reduction_function -> reduction_function
val strong_prodspine : reduction_function -> reduction_function
val stack_reduction_of_reduction : reduction_function -> stack_reduction_function

(*************************
 ** Reduction Functions **
 *************************)

(* Generic Optimized Reduction Functions using Closures *)

(* 1. lazy strategy *)
val clos_norm_flags : Closure.flags -> 'c evar_map -> reduction_function
(* Same as (strong whd_beta[delta][iota]), but much faster on big terms *) 
val nf_beta : reduction_function
val nf_betaiota : reduction_function
val nf_betadeltaiota : 'c evar_map -> reduction_function

(* 2. call by value strategy *)
val cbv_norm_flags : flags -> 'c evar_map -> reduction_function
val cbv_beta : reduction_function
val cbv_betaiota : reduction_function
val cbv_betadeltaiota : 'c evar_map -> reduction_function

(* 3. lazy strategy, weak head reduction *)
val whd_beta : reduction_function
val whd_betaiota : reduction_function
val whd_betadeltaiota : 'c evar_map -> reduction_function

val whd_beta_stack : stack_reduction_function
val whd_betaiota_stack : stack_reduction_function
val whd_betadeltaiota_stack : 'c evar_map -> stack_reduction_function


(* Head normal forms
 *)
val whd_const_stack : section_path list -> 'c evar_map -> stack_reduction_function
val whd_const : section_path list -> 'c evar_map -> reduction_function
val whd_delta_stack : 'c evar_map -> stack_reduction_function
val whd_delta : 'c evar_map -> reduction_function
val whd_betadelta_stack : 'c evar_map -> stack_reduction_function
val whd_betadelta : 'c evar_map -> reduction_function
val whd_betadeltat_stack : 'c evar_map -> stack_reduction_function
val whd_betadeltat : 'c evar_map -> reduction_function
val whd_betadeltatiota_stack : 'c evar_map -> stack_reduction_function
val whd_betadeltatiota : 'c evar_map -> reduction_function
val whd_betadeltaiotaeta_stack : 'c evar_map -> stack_reduction_function
val whd_betadeltaiotaeta : 'c evar_map -> reduction_function

val beta_applist : (constr * constr list) -> constr


(* Special-Purpose Reduction Functions
 *)
val whd_meta : (int * constr) list -> reduction_function
val plain_instance : (int * constr) list -> reduction_function
val instance : (int * constr) list -> reduction_function

val whd_ise : 'a evar_map -> reduction_function
val whd_ise1 : 'a evar_map -> reduction_function
val nf_ise1 : 'a evar_map -> reduction_function
val whd_ise1_metas : 'a evar_map -> reduction_function

val red_elim_const : 'c evar_map -> stack_reduction_function
val one_step_reduce : 'c evar_map -> constr -> constr

val hnf_prod_app : 'c evar_map -> string -> constr -> constr -> constr
val hnf_prod_appvect : 'c evar_map -> string -> constr -> constr array -> constr
val hnf_prod_applist : 'c evar_map -> string -> constr -> constr list -> constr
val hnf_lam_app : 'c evar_map -> string -> constr -> constr -> constr
val hnf_lam_appvect : 'c evar_map -> string -> constr -> constr array -> constr
val hnf_lam_applist : 'c evar_map -> string -> constr -> constr list -> constr
val splay_prod : 'c evar_map -> constr -> (name * constr) list * constr
val decomp_prod : 'c evar_map -> constr -> int * constr
val decomp_n_prod : 'c evar_map -> int -> constr -> ((name * constr) list) * constr

val is_info_arity : 'c evar_map -> constr -> bool
val is_info_sort : constr -> bool
val is_logic_arity : 'c evar_map -> constr -> bool
val is_type_arity : 'c evar_map -> constr -> bool
val is_info_judge : 'c evar_map -> judgement -> bool
val is_info_type : 'c evar_map -> type_judgement -> bool
val is_info_cast_type : 'c evar_map -> constr -> bool
val contents_of_cast_type : 'c evar_map -> constr -> contents
val contents_of_type : 'c evar_map -> type_judgement -> contents
val poly_args : constr -> int list
val poly_args_type : type_judgement -> int list
val reduce_to_mind : 'c evar_map -> constr -> constr * constr * constr
val reduce_to_ind  : 'c evar_map -> constr -> 
                                section_path*constr*constr

val whd_programs : reduction_function

(* Generic reduction: reduction functions associated to tactics *)
type red_expr =
  | Red
  | Hnf
  | Simpl
  | Cbv of flags
  | Lazy of flags
  | Unfold of (int list * section_path) list
  | Fold of constr list
(*  | Change of constr *)
  | Pattern of (int list * constr * constr) list
;;

val nf : 'c evar_map -> reduction_function
val unfoldn : (int list * section_path) list -> 'c evar_map -> reduction_function
val fold_one_com : 'c evar_map -> constr -> reduction_function
val fold_commands : constr list -> 'c evar_map -> reduction_function
val subst_term_occ : int list -> constr -> constr -> constr
val pattern_occs : (int list * constr * constr) list -> constr -> constr
val hnf_constr : 'c evar_map -> constr -> constr
val compute : 'c evar_map -> reduction_function
val reduction_of_redexp : red_expr -> 'c evar_map -> constr -> constr



(* Conversion Functions (uses closures, lazy strategy)
 *)
val fconv : conv_pb -> 'c evar_map -> constr -> constr -> bool
(* fconv has 4 instances:
 * conv   = fconv CONV   : conversion test, and adjust universes constraints
 * conv_x = fconv CONV_X :     id.        , without adjusting univ (used in tactics)
 * conv_leq   = fconv CONV_LEQ   : cumulativity test, adjust universes
 * conv_x_leq = fconv CONV_X_LEQ :       id.        , without adjusting (in tactics)
 *)
val conv : 'c evar_map -> constr -> constr -> bool
val conv_leq : 'c evar_map -> constr -> constr -> bool
val conv_x : 'c evar_map -> constr -> constr -> bool
val conv_x_leq : 'c evar_map -> constr -> constr -> bool

(* recursive part of conversion tests *)
val ccnv : conv_pb -> 'a clos_infos -> lift_spec -> lift_spec -> fconstr -> fconstr -> bool


(* Obsolete Reduction Functions *)

val hnf : 'c evar_map -> constr -> constr * constr list
val apprec : 'c evar_map -> stack_reduction_function
val red_product : 'c evar_map -> reduction_function
val find_mrectype : 'c evar_map -> constr -> constr * constr list
val find_minductype : 'c evar_map -> constr -> constr * constr list
val find_mcoinductype : 'c evar_map -> constr -> constr * constr list
val check_mrectype_spec : 'c evar_map -> constr -> constr
val minductype_spec : 'c evar_map -> constr -> constr
val mrectype_spec : 'c evar_map -> constr -> constr

(* Fonction spciale qui laisse les cast cls sous les Fix ou les MutCase *)
val strip_all_casts : constr -> constr

(* $Id: reduction.mli,v 1.12 1999/11/08 15:21:25 mohring Exp $ *)
