(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                macros.ml                                 *)
(****************************************************************************)


open Pp;;
open Std;;
open Names;;
open Term;;
open Proof_trees;;

open Libobject;;
open Library;;
open CoqAst;;
open Pcoq;;
open Ast;;


(* This file contains the table of macro tactics. The body of the
   defined tactics is stored as an ast with variables, which are
   substituted by the real argumnts at calling time.

   User defined tactics are stored in a separate table so that they
   are sensible to the Reset and Require commands.

   -- Eduardo (25/9/97).
*)

(* The data stored in the table *)

type macro_data =
    {mac_args : string list;
     mac_body : Ast.act}
;;

(* Summary and Object declaration *)

type t        = (string,macro_data) Fmavm.t;;
type frozen_t = (string,macro_data) Fmavm.frozen_t;;

let mactab      = (Fmavm.create (Pervasives.compare,17) : t);;
let lookup id   = Fmavm.map mactab id;;

let init ()     = Fmavm.empty mactab  in
let freeze ()   = Fmavm.freeze mactab in
let unfreeze fs = Fmavm.unfreeze fs mactab       
in   Summary.declare_summary "tactic-macro"
        {Summary.freeze_function   = freeze;
         Summary.unfreeze_function = unfreeze;
         Summary.init_function     = init}
;;

let (inMD,outMD) =
  let add (na,md) = Fmavm.add mactab (na,md) in
  let cache_md (_,(na,md)) = add (na,md)     in 
  let specification_md x   = x               
  in  declare_object ("TACTIC-MACRO-DATA",
                      {load_function = (fun _ -> ());
                       cache_function         = cache_md;
                       specification_function = specification_md});;


let add_macro_hint na (ids,body) =
  if Fmavm.in_dom mactab na 
  then errorlabstrm "add_macro_hint"
    [< 'sTR "There is already a Tactic Macro named "; 'sTR na >]
  else
    add_named_object (id_of_string na, OBJ)
      (inMD(na,{mac_args = ids; mac_body = body}))
;;


let macro_expand macro_loc macro argcoms = 
  let md =
    (try lookup macro
     with Not_found ->
       user_err_loc(macro_loc,"macro_expand",
		    [< 'sTR"Tactic macro ";'sTR macro; 'sPC;
		       'sTR"not defined" >])) in
  let transform = 
    List.map 
    (function (COMMAND c)   -> c
       | _ -> user_err_loc(macro_loc,"macro_expand", 
			   [<'sTR "The arguments of a tactic macro";
			     'sPC; 'sTR"must be terms">])) in
  let argcoms = transform argcoms in
    if List.length argcoms <> List.length md.mac_args 
    then user_err_loc
      (macro_loc,"macro_expand",
       [< 'sTR "Tactic macro "; 'sTR macro; 'sPC;
          'sTR "applied to the wrong number of arguments:"; 'sPC;
          'iNT (List.length argcoms) ; 'sTR" instead of ";
          'iNT (List.length md.mac_args) >])
    else let astb = List.map2 (fun id argcom -> (id, Vast argcom))
                      md.mac_args argcoms in
      (match Ast.eval_act macro_loc astb md.mac_body with
           (Vast ast) ->  ast
	 | _ -> anomaly "expand_macro : eval_act did not return a Vast")
;;

(* $Id: macros.ml,v 1.12 1999/10/29 23:19:23 barras Exp $ *)
