(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                vectops.ml                                *)
(****************************************************************************)
open Std;;

type 'a vectlist = 'a array list;;
type 'a vectstack = ('a array * int) list;;

let vect_it f v a = 
 let rec vect_it_f n =
  if n >= Array.length v then a else f v.(n) (vect_it_f (succ n))
 in vect_it_f 0
 
;;

let vect_it2 f v1 v2 a =

 let rec vect_it_f n =
  if n >= Array.length v1 then a else f v1.(n) v2.(n) (vect_it_f (succ n))
 in if (not((Array.length v1 = Array.length v2))) then invalid_arg "vect_it2"
else vect_it_f 0
 
;;

let it_vect f a v = 
 let rec it_vect_f a n =
  if n >= Array.length v then a
   else it_vect_f (f a v.(n)) (succ n)
 in it_vect_f a 0
 
;;

let it_vect2 f a v1 v2 =

 let rec it_vect_f a n =
  if n >= Array.length v1 then a
   else it_vect_f (f a v1.(n) v2.(n)) (succ n)
 in if (not((Array.length v1 = Array.length v2))) then invalid_arg "it_vect2"
else it_vect_f a 0
 
;;

let vlist_length l =
    List.fold_left (fun sofar v -> sofar+(Array.length v)) 0 l
;;

let join_vect l =
if l = [] then Array.of_list[] else
let len = vlist_length l in
if len = 0 then Array.of_list[] else
let result = Array.create len (List.hd l).(0) in
let rec aux n = function
    [] -> ()
  | h::t -> let vlen = Array.length h
            in (Array.blit h 0 result n (Array.length h);
                aux (n+vlen) t)
in aux 0 l; result
;;

let vlitem l n = if n<0 then failwith "item" else
 (
 let rec item_op = 
   function (hd::t,n) -> let hdlen = Array.length hd
                    in if n < hdlen then hd.(n)
                       else item_op(t,n-hdlen)
   |   ([],_) -> failwith "item"
 in item_op (l,n) );;

let vlnth l n = vlitem l (pred n);;

let chop_vect n v =
    let vlen = Array.length v
    in if n > vlen  then failwith "chop_vect"
    else (Array.sub v 0 n,Array.sub v n (vlen-n))
;;

let safe_chop_vect n v =
    let vlen = Array.length v
    in if n > vlen  then (v,Inr(n-vlen))
    else (Array.sub v 0 n,Inl(Array.sub v n (vlen-n)))
;;

let chop_vect_list = 
 let rec chop_aux =
  fun p_0 p_1 -> match p_0,p_1 with (0, (l1,l2)) -> List.rev l1,l2
    | (_, (_,[])) -> failwith "chop_list"
    | (n, (l1,h::t)) ->
      match safe_chop_vect n h with
      (v,Inl v') -> v::(List.rev l1),v'::t
    | (v,Inr more) -> chop_aux more (v::l1,t)
 in (fun n l -> chop_aux n ([],l))
 
;;

let exists_vect f v = 
 let rec exrec = function
    (-1) -> false
    | n -> (f v.(n)) or (exrec (n-1))
 in exrec ((Array.length v)-1) 
;;

let exists2_i_vect_LR f start v1 v2 =
if Array.length v1 <> Array.length v2 then
    invalid_arg "exists2_i_LR"
else (
 let rec exrec n =
      if n = Array.length v1 then false
      else f (start+n) v1.(n) v2.(n) or exrec (n+1)
 in exrec 0 )
;;

let exists2_vect_LR f v1 v2 =
if Array.length v1 <> Array.length v2 then
    invalid_arg "exists2_i_LR"
else (
 let rec exrec n =
      if n = Array.length v1 then false
      else f v1.(n) v2.(n) or exrec (n+1)
 in exrec 0 )
;;

let for_all_vect f v = 
 let rec allrec = function
    (-1) -> true
    | n -> (f v.(n)) & (allrec (n-1))
 in allrec ((Array.length v)-1) 
;;

let for_all_vect_i f base v = 
 let rec allrec = function
    (-1) -> true
    | n -> (f (base+n) v.(n)) & (allrec (n-1))
 in allrec ((Array.length v)-1) 
;;

let for_all2eq_vect f v1 v2 =
  let vlv1 = (Array.length v1)
  in
    
 let rec allrec = function
    (-1) -> true
    | n -> (f v1.(n) v2.(n)) & (allrec (n-1))
 in vlv1 = (Array.length v2)
    & allrec (pred vlv1) 
;;

let list_of_vectlist vl =
let rec aux = function
    [] -> []
  | v::vl -> (Array.to_list v)@(aux vl)
in aux vl
;;

let vstack1 v =
    if Array.length v = 0 then [] else [(v,0)]
;;

let push_vstack v vs =
    if Array.length v = 0 then vs
    else (v,0)::vs
;;

let vstack_empty = function
    [] -> true
    | _ -> false
;;

let pop_vstack = function
    (v,n)::vs ->
    (v.(n),if n = (Array.length v) then vs else (v,n+1)::vs)
  | _ -> invalid_arg "pop_vstack"
;;

let vstack_length l =
    List.fold_left (fun sofar (v,n) -> sofar+(Array.length v)-n) 0 l
;;

let safe_chop_vfrag n (v,start) =
    let vlen = (Array.length v)-start
    in if n > vlen  then ((v,start),Inr(n-vlen))
    else ((Array.sub v start n,0),Inl(v,n+start))
;;

let chop_vstack = 
 let rec chop_aux =
  fun p_0 p_1 -> match p_0,p_1 with (0, (l1,l2)) -> List.rev l1,l2
    | (_, (_,[])) -> failwith "chop_list"
    | (n, (l1,h::t)) ->
      match safe_chop_vfrag n h with
      (v,Inl v') -> v::(List.rev l1),v'::t
    | (v,Inr more) -> chop_aux more (v::l1,t)
 in (fun n l -> chop_aux n ([],l))
 
;;

let join_vstack l =
if l = [] then Array.of_list[] else
let len = vstack_length l in
let result = Array.create len (fst(List.hd l)).(0) in
let rec aux n = function
    [] -> ()
  | (h,st)::t -> let vlen = (Array.length h)-st
            in (Array.blit h st result n (Array.length h);
                aux (n+vlen) t)
in aux 0 l; result
;;

let tabulate_vect f n =
    if n = 0 then Array.of_list []
    else
        let zeroval = f 0 in
        let rslt = Array.create n zeroval in
        let rec aux = function
            0 -> ()
          | n -> (rslt.(n) <- (f n);
                  aux (n-1))
        in aux (n-1);rslt
;;

let rec map_vstack f = function
    [] -> []
    | (v,n)::l ->
      (tabulate_vect (fun i -> f v.(n+i)) ((Array.length v)-n),0)::(map_vstack f l)
;;

let rec map_i_vstack f startnum = 
 let rec map_rec idx = function
    [] -> []
    | (v,n)::l ->
      let vlen = ((Array.length v)-n)
      in (tabulate_vect (fun i -> f (idx+i) v.(n+i)) vlen,0)::(map_rec (idx+vlen) l)
 in map_rec startnum 
;;

let rec map_vstack_list f = function
    [] -> []
    | vs -> let e,rs = pop_vstack vs
            in (f e)::(map_vstack_list f rs)
;;

let for_all_vstack f = 
 let rec allrec = function
    [] -> true
    | vs -> let (e,rs) = pop_vstack vs
            in f e & (allrec rs)
 in allrec 
;;

let for_all2eq_vstack f = 
 let rec allrec = fun
    p_0 p_1 -> match p_0,p_1 with ([], []) -> true
  | ([], vs2) -> vs2=[]
  | (vs1, []) -> vs1=[]
  | (vs1, vs2) -> let e1,rs1 = pop_vstack vs1
               and e2,rs2 = pop_vstack vs2
               in f e1 e2 & allrec rs1 rs2
 in allrec 
;;

let concat_vstack vs1 vs2 = vs1@vs2;;

let app_vect v l =
    vect_it (fun e l -> e::l) v l
;;

let it_vect_from n f a v = 
 let rec it_vect_f a n =
  if n >= Array.length v then a
   else it_vect_f (f a v.(n)) (succ n)
 in it_vect_f a n
 
;;

let vect_it_from n f v a = 
 let rec vect_it_f n =
  if n >= Array.length v then a else f v.(n) (vect_it_f (succ n))
 in vect_it_f n
 
;;

let it_vect2_from n f a v1 v2 =

 let rec it_vect_f a n =
  if n >= Array.length v1 then a
   else it_vect_f (f a v1.(n) v2.(n)) (succ n)
 in if (not((Array.length v1 = Array.length v2))) then invalid_arg "it_vect2"
else it_vect_f a n
 
;;

let app_tl_vect v l =
if Array.length v = 0 then invalid_arg "app_tl_vect"
else vect_it_from 1 (fun e l -> e::l) v l
;;

let list_of_tl_vect v =
if Array.length v = 0 then invalid_arg "app_tl_vect"
else vect_it_from 1 (fun e l -> e::l) v []
;;

let app_vectpair (v1,v2) l =
    vect_it2 (fun e1 e2 l -> (e1,e2)::l) v1 v2 l
;;

let hd_vect v = v.(0);;
let tl_vect v = Array.sub v 1 ((Array.length v) - 1);;
let cons_vect e v2 = Array.append [|e|] v2;;
let car_vect = hd_vect;;
let cdr_vect = tl_vect;;
let cadr_vect v = v.(1);;
let cddr_vect v = Array.sub v 2 ((Array.length v) - 2);;
let last_vect v = v.((Array.length v) - 1);;

let map2_vect f v1 v2 =
  if Array.length v1 <> Array.length v2 then
      invalid_arg "map2_vect"
  else if Array.length v1 == 0 then [| |] else begin
    let res = Array.create (Array.length v1) (f v1.(0)v2.(0)) in
      for i = 1 to pred(Array.length v1) do
        res.(i) <- f v1.(i) v2.(i)
      done;
      res
    end
;;

let map3_vect f v1 v2 v3 =
  if Array.length v1 <> Array.length v2 or Array.length v1 <> Array.length v3 then
      invalid_arg "map3_vect"
  else if Array.length v1 == 0 then [| |] else begin
    let res = Array.create (Array.length v1) (f v1.(0)v2.(0) v3.(0)) in
      for i = 1 to pred(Array.length v1) do
        res.(i) <- f v1.(i) v2.(i) v3.(i)
      done;
      res
    end
;;

let map4_vect f v1 v2 v3 v4 =
   if Array.length v1 <> Array.length v2 or
      Array.length v1 <> Array.length v3 or
      Array.length v1 <> Array.length v4 
   then invalid_arg "map4_vect"
   else if Array.length v1 == 0 then [| |] else begin
     let res = Array.create (Array.length v1) (f v1.(0)v2.(0) v3.(0) v4.(0)) in
       for i = 1 to pred(Array.length v1) do
         res.(i) <- f v1.(i) v2.(i) v3.(i) v4.(i)
      done;
      res
    end
;;

let map_i_vect f start v =
    tabulate_vect (fun i -> f (start+i) v.(i)) (Array.length v)
;;

let split_vect v =
    let n = Array.length v in
        (tabulate_vect (fun i -> fst v.(i)) n,
         tabulate_vect (fun i -> snd v.(i)) n)
;;

let combine_vect_list (v1,v2) =
    if Array.length v1 <> Array.length v2 then
        invalid_arg "combine_vect_list"
    else (
 let rec comb n =
          if n = Array.length v1 then [] else (v1.(n),v2.(n))::(comb (n+1))
 in comb 0 )
;;

let map_vect_list f v =
  let rec map i =
    if i >= Array.length v then [] else f v.(i) :: map (succ i)
  in
    map 0
;;

let map_tl_vect_list f v =
  let rec map i =
    if i >= Array.length v then [] else f v.(i) :: map (succ i)
  in
    map 1
;;

let first_vect p v =
    
 let rec
    firstrec n =
    if n = Array.length v then raise Not_found
    else if p v.(n) then n
         else firstrec (n+1)
 in firstrec 0 
;;

let vect_index e v = 
 let rec indrec n =
    if n = Array.length v then failwith "vect_index"
    else if e = v.(n) then n+1 else indrec (succ n)
 in indrec 0
    
;;


let vect_index_pred p v = 
 let rec indrec n =
    if n = Array.length v then failwith "index_pred"
    else if p v.(n) then n
         else indrec (n+1)
 in indrec 0
    
;;

let first_result_vect_i f start v = let n = Array.length v in 
 let rec aux k = if k > n then failwith "first_result_vect_i"
                  else  try (f k v.(k-1))
                        with Failure _ | UserError _ -> aux (k+1)
 in aux start

;;

(* $Id: vectops.ml,v 1.6 1999/06/29 07:47:21 loiseleu Exp $ *)
