(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                intset.mli                                *)
(****************************************************************************)
(* Sets over ordered types *)

(* This module implements the set data structure, given a total ordering
   function over the set elements. All operations over sets
   are purely applicative (no side-effects).
   The implementation uses balanced binary trees, and is therefore
   reasonably efficient: insertion and membership take time
   logarithmic in the size of the set, for instance. *)

type t;;
        (* The type of sets containing elements of type ['a]. *)

val empty: t
        (* The empty set.
           The argument is a total ordering function over the set elements.
           This is a two-argument function [f] such that
           [f e1 e2] is zero if the elements [e1] and [e2] are equal,
           [f e1 e2] is strictly negative if [e1] is smaller than [e2],
           and [f e1 e2] is strictly positive if [e1] is greater than [e2].
           Examples: a suitable ordering function for type [int]
           is [prefix -]. For type [string], you could use
           [compare_strings]. *)
  val is_empty: t -> bool
        (* Test whether a set is empty or not. *)
  val mem: int -> t -> bool
        (* [mem x s] tests whether [x] belongs to the set [s]. *)
  val add: int -> t -> t
        (* [add x s] returns a set containing all elements of [s],
           plus [x]. If [x] was already in [s], [s] is returned unchanged. *)
  val remove: int -> t -> t
        (* [remove x s] returns a set containing all elements of [s],
           except [x]. If [x] was not in [s], [s] is returned unchanged. *)
  val union: t -> t -> t
  val inter: t -> t -> t
  val diff: t -> t -> t
        (* Union, intersection and set difference. *)
  val equal: t -> t -> bool
        (* [equal s1 s2] tests whether the sets [s1] and [s2] are
           equal, that is, contain the same elements. *)
  val compare: t -> t -> int
        (* Total ordering between sets. Can be used as the ordering function
           for doing sets of sets. *)
  val elements: t -> int list
        (* Return the list of all elements of the given set.
           The elements appear in the list in some non-specified order. *)
  val iter: (int -> 'b) -> t -> unit
        (* [iter f s] applies [f] in turn to all elements of [s], and
           discards the results. The elements of [s] are presented to [f]
           in a non-specified order. *)
  val fold: (int -> 'b -> 'b) -> t -> 'b -> 'b
        (* [fold f s a] computes [(f xN ... (f x2 (f x1 a))...)],
           where [x1 ... xN] are the elements of [s].
           The order in which elements of [s] are presented to [f] is
           not specified. *)
;;

(* $Id: intset.mli,v 1.4 1999/06/29 07:47:18 loiseleu Exp $ *)
